/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/command/axis.cpp,v 1.4 2004/09/09 10:45:13 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#include "libgpm.h"

#ifndef M_PI
# define M_PI 3.14159265358979323846
#endif

#define NON  0
#define REAL  1
#define INTEGER 2

#define ON_THE_AXIS  0
#define UNDER_THE_AXIS  1

#define LNR_LNR 0
#define LOG_LNR 1
#define LNR_LOG 2
#define LOG_LOG 3

#ifdef __cplusplus
extern "C" {
#endif

static int  LabelFormat(int side, const char *axis_label, int label_type);
static void AxisType(int side, float *x, float *y, int *label_pos,
                     float *theta);
static int  errorCheck(int tbuf, char typ, const char *fig1, const char *fig2,
                       int grid, int form);


DLLEXPORT int mod_gpm_axis(){
  int    err;
  int    label_pos, igrx, igry, itype;
  int    index, label_type, labFormat;
  int    title_no;
  float  theta;
  float  x, y, fmin, fmax, chr_size, scale_x, scale_y;
  char   *ttl_p, *axis_label, *axis_bar;
  
  /* LOAD SYSTEM PARAMETER */
  regpm();

  title_no         = (int)  GetScalar(0);
  GpmCont.axisDraw = (char) GetScalar(1);
  axis_bar         =        GetString(2);
  axis_label       =        GetString(3);
  chr_size         = (float)GetScalar(4);
  itype            = (int)  GetScalar(5);
  label_type       = (int)  GetScalar(6);
  scale_x          = (float)GetScalar(7);
  scale_y          = (float)GetScalar(8);
  index            = (int)  GetScalar(9);

  err = errorCheck(title_no, GpmCont.axisDraw, axis_bar, axis_label, 
                   itype, label_type);
  if(err != 0)
    return err;
  
  title_no -= 1;
  igrx = (itype == 1 || itype == 3) ? 1 : 0;
  igry = (itype == 2 || itype == 3) ? 1 : 0;
  igrx *= (int)(GpmCont.fLineType + 1);
  igry *= (int)(GpmCont.fLineType + 1);
  
  if(gopen(GpmCont.paper, GpmCont.orientation, GpmCont.device, 
           GpmCont.paperWidth, GpmCont.paperHeight) != 0)
    return 6; /* window does not exist */

  if(checktodvi() != 0)
    libgpm_dvi_write_param(&gpmdev_cont.dvi, GpmCont.factor, 
                           GpmCont.xOrigin, GpmCont.yOrigin);

  gfactor(GpmCont.factor);
  gnewpen(GpmCont.fColor);
  gpen(GpmCont.fLineWidth, GpmCont.fLineType, 0);
  gorigin(GpmCont.xOrigin, GpmCont.yOrigin);
  gfont(GpmCont.fontType);
  
  /* DRAW X-AXIS */
  if(!strcmp(axis_bar, "X") || !strcmp(axis_bar, "XY")){
    ttl_p = GpmCont.title_x[title_no];
    
    AxisType('X', &x, &y, &label_pos, &theta);
    labFormat = LabelFormat('X', axis_label, label_type);
    if(GpmCont.axisType == LNR_LNR || GpmCont.axisType == LNR_LOG){
      if(index == 1)
        labFormat *= -1;
      fmin = GpmCont.xMin;
      fmax = GpmCont.xMax;
      gaxis(x, y, GpmCont.xSize, theta, fmin, fmax, scale_x, labFormat, 
            label_pos, ttl_p, chr_size, igrx, GpmCont.ySize);
    }else{
      fmin = (GpmCont.xMin > 0.0 ) ?
        (float)log10(fabs((double)(GpmCont.xMin))) : (float)1.0e-37;
      fmax = (GpmCont.xMax > 0.0 ) ?
        (float)log10(fabs((double)(GpmCont.xMax))) : (float)1.0e-37;
      gaxisl(x, y, GpmCont.xSize, theta, fmin, fmax, labFormat,
             label_pos, ttl_p, chr_size, igrx,  GpmCont.ySize);
    }
  }
  /* DRAW Y-AXIS */
  if(!strcmp(axis_bar, "Y") || !strcmp(axis_bar, "XY")){
    ttl_p = GpmCont.title_y[title_no];
    
    AxisType('Y', &x, &y, &label_pos, &theta);
    labFormat = LabelFormat('Y', axis_label, label_type);
    if(GpmCont.axisType == LNR_LNR || GpmCont.axisType == LOG_LNR) {
      if(index == 1)
        labFormat *= -1;
      fmin = GpmCont.yMin;
      fmax = GpmCont.yMax;
      gaxis(x, y, GpmCont.ySize, theta, fmin, fmax,
      scale_y, labFormat, label_pos, ttl_p, chr_size,
      igry, GpmCont.xSize);
    }else{
      fmin = (GpmCont.yMin > 0.0 ) ?
        (float)log10(fabs((double)(GpmCont.yMin))) : (float)1.0e-37;
      fmax = (GpmCont.yMax > 0.0 ) ?
        (float)log10(fabs((double)(GpmCont.yMax))) : (float)1.0e-37;

      gaxisl(x, y, GpmCont.ySize, theta, fmin, fmax, labFormat,
             label_pos, ttl_p, chr_size, igry, GpmCont.xSize);
    }
    
  }
  gflush();
  gclose();
  
  /* STORE SYSTEM PARAMETER */
  wrgpm();
  return 0;
}


#define  OUT_OF_RANGE(x,MiN,MaX)  (x < MiN || x > MaX)

static int errorCheck(int tbuf, char typ, const char *fig1, const char *fig2,
                      int grid, int form){
  /* title noumber (1,2,3) */
  if(OUT_OF_RANGE(tbuf, 1, 3))
    return 3; /* illegal parameter */

  /* AXIS DRAW STYLE   (1 .. 4) */
  if(OUT_OF_RANGE((int)typ, 1, 4))
    return 11; /* illegal axis type */
  
  /* Axis bar (fig1) */
  if(fig1 == NULL)
    return 3; /* illegal parameter */

  /* Axis label (fig2) */
  if(fig2 == NULL)
    return 3; /* illegal parameter */

  /* GRID DRAW   (0 ... 3)  */
  if(OUT_OF_RANGE(grid, 0, 3))
    return 12; /* illegal grid number */
  
  /* SCALE FORMAT TYPE  (0 ... 3) */
  if(OUT_OF_RANGE(form, 0, 3))
    return 13; /* illegal scale type */

  return 0; /* success */
}


static void AxisType(int side, float *x, float *y, int *label_pos, 
                     float *theta){
  *x = *y = 0.0;
  
  if(side == 'X'){
    *label_pos = UNDER_THE_AXIS;
    if(GpmCont.axisDraw == GPM_AXIS_DRAW_LT || 
       GpmCont.axisDraw == GPM_AXIS_DRAW_RT){
      *y = GpmCont.ySize;
      *label_pos = ON_THE_AXIS;
    }
    *theta = 0.0;
    
  }else if (side == 'Y'){
    *label_pos = ON_THE_AXIS;
    if(GpmCont.axisDraw == GPM_AXIS_DRAW_RB ||
       GpmCont.axisDraw == GPM_AXIS_DRAW_RT){
      *x = GpmCont.xSize;
      *label_pos = UNDER_THE_AXIS;
    }
    *theta = (float)(M_PI/2.0);
  }
}

static int LabelFormat(int side, const char *axis_label, int label_type){
  int labFormat = REAL;
  
  switch(label_type){
  case GPM_AXIS_TYPE_RR:
    labFormat = REAL;
    break;
  case GPM_AXIS_TYPE_IR:
    labFormat = (side == 'X') ? INTEGER : REAL;
    break;
  case GPM_AXIS_TYPE_RI:
    labFormat = (side == 'X') ? REAL : INTEGER;
    break;
  case GPM_AXIS_TYPE_II:
    labFormat = INTEGER;
    break;
  }
  
  /* Mismatch */
  if(axis_label[0] != side && axis_label[1] != side)
    labFormat = NON;

  return labFormat;
}

#ifdef __cplusplus
}
#endif
