/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: text2buffer.cpp,v 1.2 2004/04/02 06:02:15 orrisroot Exp $ */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#include "dcm_fifo.h"

#define NEWLINE 0
#define END     1
#define OTHER   2
#undef  fscanf
#define isDelmit(x) (x=='\t'||x=='\r'||x=='\n'||x==' '||x==','||x==0xa0)

#ifdef __cplusplus
extern "C" {
#endif

typedef struct {
  int dim;
  int *index;
  double *data;
} Buffer_Struct;

int SkipNotNumber(FILE *fp){
  int c;
  for(c=fgetc(fp);!isDelmit(c);c=fgetc(fp))
    if(feof(fp)) return END;
  if(c=='\n'||c=='\r')return NEWLINE;
  ungetc(c,fp);
  return OTHER;
}

int GetToken(FILE *fp, DCM_FIFO *fifo){
  double d;
  int c,tmp;
  while(fscanf(fp,"%lf",&d)==1){
    DCM_FIFO_push(fifo,d);
    for(c=fgetc(fp);isDelmit(c);c=fgetc(fp)){
      if(c=='\n')return NEWLINE;
      if(c=='\r'){
        tmp=fgetc(fp);
        if(tmp!='\n') ungetc(tmp,fp); /* for mac */
        return NEWLINE; /* for dos */
      }
    }
    if(feof(fp))return END;
    ungetc(c,fp);
  }
  for(c=fgetc(fp);isDelmit(c);c=fgetc(fp));
  if(feof(fp))return END;
  return OTHER;
}

DCM_FIFO_LIST *ReadDataFromFile(char *file){
  int stat;
  FILE *fp;
  DCM_FIFO_LIST *list;
  DCM_FIFO      *fifo;
  list=DCM_FIFO_LIST_New();
  if(list == NULL) return NULL;
  fp=fopen(file,"r");
  if(fp==NULL) return NULL; /* file open error */
  fifo=DCM_FIFO_New();
  
  for(stat=GetToken(fp,fifo);stat!=END;stat=GetToken(fp,fifo)){
    switch(stat){
    case NEWLINE:
      if(DCM_FIFO_LIST_Add(list,fifo) != 0) return NULL;
      fifo=DCM_FIFO_New();
      if(fifo == NULL) return NULL;
      break;
    case OTHER:
      DCM_FIFO_push(fifo,0.0);
      printf("warning : datum is not number near (L:%ld,C:%ld), set zero.\n",
	     list->size,fifo->size-1);
      if(SkipNotNumber(fp)==NEWLINE){
	if(DCM_FIFO_LIST_Add(list,fifo) != 0) return NULL;
	fifo=DCM_FIFO_New();
        if(fifo == NULL) return NULL;
      }
      break;
    }
  }
  fclose(fp);
  if(fifo->size==0){
    DCM_FIFO_Delete(fifo);
  }else{
    DCM_FIFO_LIST_Add(list,fifo);
  }
  return list;
}

int DataFormat(DCM_FIFO_LIST *fifo_list){
  DCM_FIFO_LIST_CELL *it;
  char fflag;
  unsigned long maxcol;
  maxcol=0;
  fflag=1;
  for(it=fifo_list->top;it!=NULL;it=it->next){
    if(maxcol<it->fifo->size) maxcol=it->fifo->size;
  }
  for(it=fifo_list->top;it!=NULL;it=it->next){
    while(it->fifo->size!=maxcol){
      if(fflag){
	printf("warning : change data for matrix.\n");
	fflag=0;
      }
      DCM_FIFO_push(it->fifo,0.0);
    }
  }
  return maxcol;
}


#ifdef _DEBUG_DCM
void PrintSeries(double *data,int dim,int *index){
  int i,j;
  printf("Return Series\n");
  printf("dim : %d\n",dim);
  switch(dim){
  case 1:
    printf("length : %d\n",index[0]);
    for(i=0;i<index[0];i++){
      if(i%8==0) printf("[%d]%%    ",i);
      printf("%6.1f ",data[i]);
      if(i%8==7) printf("\n");
    }
    break;
  case 2:
    printf("length : %d\n",index[0]);
    for(i=0;i<index[0];i++){
     for(j=0;j<index[1];j++){
       if(j%9==0) printf("[%2d]:[%2d]%%",i,j);
       printf(" %6.1f",data[i*index[1]+j]);
       if(j%9==8 && index[1]!=j+1) printf("\n");
      }
     if(i+1!=index[0])printf("\n");
    }
    break;
  default:
    printf("no supported dimension : %d\n",dim);
    break;
  }
  puts("");
}
#endif

DLLEXPORT int mod_dcm_text2buffer(int argc,char *argv[]){
  Buffer_Struct buffer;
  DCM_FIFO_LIST *list;
  DCM_FIFO_LIST_CELL *it;
  char *fname;
  unsigned long n;
  int m_flag;
  m_flag=1;

  /* get input file name*/
#ifdef _DEBUG_DCM
  if(argc<2){
    printf("usage : %s filename\n",argv[0]);
    return (2);
  }
  fname=argv[1];
  if(argc>2){
    m_flag=(int)(argv[2][0]-'0');
  }
#else
  if(GetArgNum()==0) return (1);
  fname=GetString(0);
  if(fname==NULL)
    return (1);
  if(GetArgNum()>1){
    m_flag=(int)GetScalar(1);
  }
#endif
  
  /* initialize */
  buffer.index=(int*)malloc(sizeof(int)*2);
  if(buffer.index==NULL) return 8;

  /* data read */
  list=ReadDataFromFile(fname);
  if(list == NULL) return (2);

  /* data format and header setting */
  if(list->size==0){
    buffer.index[0]=0;
    buffer.index[1]=0;
    buffer.data=NULL;
  }else{
    if(m_flag==0){ /* liner data */
      buffer.index[0]=DCM_FIFO_LIST_TotalSize(list);
      buffer.index[1]=1;
    }else{ /* matrix data */
      buffer.index[0]=list->size;
      buffer.index[1]=DataFormat(list);
    }
    buffer.data=(double *)AllocBuffer(buffer.index[0]*buffer.index[1]);
    if(buffer.data==NULL){
      DCM_FIFO_LIST_Delete(list);
      return (8); /* out of memory */
    }
    n=0;
    for(it=list->top;it!=NULL;it=it->next){
      int stat;
      while(it->fifo->size!=0){
	buffer.data[n++]=DCM_FIFO_pop(it->fifo, &stat);
        if(stat != 0) return stat;
      } 
    }
  }
  DCM_FIFO_LIST_Delete(list);
  if(buffer.index[0]==1){
    buffer.index[0]=buffer.index[1];
    buffer.index[1]=1;
  }
  if(buffer.index[0]==1 && buffer.index[1]==1)
#ifdef _DEBUG_DCM
    printf("Return Scalar\n%f\n",buffer.data[0]);
#else
    ReturnScalar(buffer.data[0]);
#endif
  else {
    if(buffer.index[1]==1){
      buffer.index[1]=0;
      buffer.dim=1;
    }else{
      buffer.dim=2;
    }
#ifdef _DEBUG_DCM
    PrintSeries(buffer.data,buffer.dim,buffer.index);
#else
    ReturnSeries(buffer.data,buffer.dim,buffer.index);
#endif
  }
  /* free data */
  free(buffer.index);
  //FreeBuffer(buffer.data);
  return 0;
}

#ifdef __cplusplus
}
#endif
