/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: matfile.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $ */

#include <windows.h>
#include <stdio.h>
#include <malloc.h>
#include "matfile.h"

static char RTSimMemAllocError[] = "Memory allocation error";
static const char RTSimStructLogVarFieldNames[] =
                  "time\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                  "signals\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                  "blockName\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0";
static const char RTSimLocalLoggingSignalsStructFieldNames[] = 
                  "values\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                  "label\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                  "title\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                  "plotStyle\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0";
static const char RTSimGlobalLoggingSignalsStructFieldNames[] = 
                  "values\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                  "label\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
                  "blockName\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0";

#define TIME_FIELD_NAME      (RTSimStructLogVarFieldNames[0*mxMAXNAM])
#define SIGNALS_FIELD_NAME   (RTSimStructLogVarFieldNames[1*mxMAXNAM])
#define BLOCKNAME_FIELD_NAME (RTSimStructLogVarFieldNames[2*mxMAXNAM])

#define VALUES_FIELD_NAME    (RTSimLocalLoggingSignalsStructFieldNames[0*mxMAXNAM])
#define LABEL_FIELD_NAME     (RTSimLocalLoggingSignalsStructFieldNames[1*mxMAXNAM])
#define TITLE_FIELD_NAME     (RTSimLocalLoggingSignalsStructFieldNames[2*mxMAXNAM])
#define PLOTSTYLE_FIELD_NAME (RTSimLocalLoggingSignalsStructFieldNames[3*mxMAXNAM])

#ifdef __cplusplus
extern "C" {
#endif

LogVar	g_logVar;
static FILE*	g_fptr;
static BOOL	g_emptyFile    = TRUE; /* assume */
static BOOL	g_errFlag      = FALSE;

BOOL MatfileCreateHeader( char* matname )
{
	char	filename[_MAX_PATH];
	char	msgbuf[_MAX_PATH];
	strcpy(filename, matname);

    /*******************************
     * Create MAT file with header *
     *******************************/
    if ((g_fptr=fopen(filename,"wb")) == NULL) {
        sprintf(msgbuf,"*** Error opening %s",filename);
		printf("%s\n",msgbuf);
        return FALSE;
    }
    if (RTSim_WriteMat5FileHeader(g_fptr)) {
        printf("*** Error writing to %s\n",filename);
		MatfileCreateClose( matname );
        return FALSE;
    }
	return TRUE;
}

void MatfileCreateClose(char* matname)
{
	char	filename[_MAX_PATH];
	strcpy(filename, matname);

    /******************
     * Close the file *
     ******************/
    fclose(g_fptr);
    if (g_emptyFile || g_errFlag) {
        (void)remove(filename);
    } else {
        //sprintf(msgbuf, "** created %s **", filename);
		//printf("%s\n",msgbuf);
    }
}


void MatfileCreate( char* tag,
	double* re, double* im, int nRows, int nCols)
{
	strcpy(g_logVar.data.name, tag);
	if(im == NULL) {
		// Real_T̏ꍇ
		g_logVar.data.re = re;
		g_logVar.data.im = im;
		g_logVar.data.nRows = nRows;
		g_logVar.data.nCols = nCols;
		g_logVar.data.dTypeID = 0;
		g_logVar.data.mxID = mxDOUBLE_CLASS;
		g_logVar.data.elSize = sizeof(double);
		g_logVar.data.logical = 0x0;
		g_logVar.data.complex = 0x0;
		g_logVar.wrapped = 0;
	}
	else {
		// CReal_T̏ꍇ
		g_logVar.data.re = re;
		g_logVar.data.im = im;
		g_logVar.data.nRows = nRows;
		g_logVar.data.nCols = nCols;
		g_logVar.data.dTypeID = 0;
		g_logVar.data.mxID = mxDOUBLE_CLASS;
		g_logVar.data.elSize = sizeof(double);
		g_logVar.data.logical = 0x0;
		g_logVar.data.complex = 0x800;
		g_logVar.wrapped = 0;
	}
	RTSim_StopDataLogging(g_fptr);

	//ŁÄ
	//Matlab`ɓŕύXĂ邽߁Ä悪󂳂
	if(g_logVar.data.re)
		free(g_logVar.data.re);
	if(g_logVar.data.im)
		free(g_logVar.data.im);

}

/* Function: RTSim_WriteItemToMatFile =============================================
 * Abstract:
 *      Entry function for writing out a mat item to the mat file.
 *
 *      Return values is
 *          == 0 : upon success
 *          <> 0 : upon failure
 */
int RTSim_WriteItemToMatFile(FILE *fp, MatItem *pItem, ItemDataKind itemKind)
{
    /* Determine the item size */
    if (pItem->type == matMATRIX) {
        if (RTSim_ProcessMatItem(fp, pItem, itemKind, 0)) return(1);
    }

    /* Write the item tag and data */
    if (pItem->nbytes > 4) {
        int nAlignBytes;

        if (fwrite(pItem, 1, matTAG_SIZE, fp) != matTAG_SIZE) return(1);

        if (pItem->type == matMATRIX) {
            if (RTSim_ProcessMatItem(fp, pItem, itemKind, 1)) return(1);
        } else {
            if ( fwrite(pItem->data, 1, pItem->nbytes, fp) !=
                                                    ((size_t) pItem->nbytes) ) {
                return(1);
            }
        }

        /* Add offset for 8-byte alignment */
        nAlignBytes = matINT64_ALIGN(pItem->nbytes) - pItem->nbytes;
        if (nAlignBytes > 0) {
            int pad[2] = {0, 0};
            if ( fwrite(pad,1,nAlignBytes,fp) != ((size_t) nAlignBytes) ) {
                return(1);
            }
        }
    } else {
        MatItem item = {0, 0, NULL};
        item.type = ((unsigned int)(pItem->type))|(((unsigned int)(pItem->nbytes))<<16);
        (void)memcpy(&item.nbytes, pItem->data, pItem->nbytes);
        if (fwrite(&item, 1, matTAG_SIZE, fp) != matTAG_SIZE) return(1);
    }

    return(0);

} /* end: RTSim_WriteItemToMatFile */

/* Function: RTSim_ProcessMatItem =================================================
 * Abstract:
 *      This routine along with RTSim_WriteItemToMatFile() write out a specified
 *      mat-item the the .mat file. Note that if the input argument
 *          cmd == 0, then this function just calculates the size of the item.
 *          cmd <> 0, this function writes the mat-item to the file.
 *      Return values is
 *           -1 : coding/logic error
 *            0 : upon success
 *          > 0 : upon write failure (1)
 */
int RTSim_ProcessMatItem(	   FILE         *fp,
                           MatItem      *pItem,
                           ItemDataKind itemKind,
                           int          cmd)
{
    mxClassID    mxID          = mxUNKNOWN_CLASS;
    unsigned int     arrayFlags[2] = {0, 0};
    int      dims[2] = {0, 0};
    int      nBytesInItem = 0;
    char	 *itemName;
    MatItem  item;
	char	 msgbuf[_MAX_PATH];

    switch (itemKind) {
      case DATA_ITEM: {
          sprintf(msgbuf,"Unexpected itemKind = DATA_ITEM in RTSim_ProcessMatItem @A");
		  printf("%s\n",msgbuf);
          return(-1);
      }
      case MATRIX_ITEM: {
          const MatrixData *var = (const MatrixData *) pItem->data;

          mxID           = var->mxID;
          arrayFlags[0]  = mxID;
          arrayFlags[0] |= var->logical;
          arrayFlags[0] |= var->complex;
          dims[0]        = var->nRows;
          dims[1]        = var->nCols;
          itemName       = (char *)var->name;
          break;
      }
      case STRUCT_LOG_VAR_ITEM: {
          const StructLogVar *var = (const StructLogVar *) pItem->data;

          mxID          = mxSTRUCT_CLASS;
          arrayFlags[0] = mxID;
          dims[0]       = 1;
          dims[1]       = 1;
          itemName      = (char *)var->name;
          break;
      }
      case SIGNALS_STRUCT_ITEM: {
          const SignalsStruct *var = (const SignalsStruct *) pItem->data;

          mxID          = mxSTRUCT_CLASS;
          arrayFlags[0] = mxID;
          dims[0]       = 1;
          dims[1]       = var->numSignals;
          itemName      = (char *)&SIGNALS_FIELD_NAME;
          break;
      }
      default:
        sprintf(msgbuf,"Unexpected itemKind=%d in RTSim_ProcessMatItem @B",itemKind);
		printf("%s\n",msgbuf);
        return(-1);
    }

    /* array flags */
    item.nbytes = 2*sizeof(unsigned int);
    if (cmd) {
        item.type = matUINT32;
        item.data = arrayFlags;
        if (RTSim_WriteItemToMatFile(fp,&item, DATA_ITEM))
			return(1);

    } else {
        /*LINTED E_CAST_INT_TO_SMALL_INT*/
        nBytesInItem += matINT64_ALIGN(matTAG_SIZE + item.nbytes);
    }
    /* dimensions */
    item.nbytes = 2*sizeof(int);
    if (cmd) {
        item.type = matINT32;
        item.data = dims;
        if (RTSim_WriteItemToMatFile(fp,&item, DATA_ITEM)) return(1);
    } else {
        /*LINTED E_CAST_INT_TO_SMALL_INT*/
        nBytesInItem += matINT64_ALIGN(matTAG_SIZE + item.nbytes);
    }
    /* name */
    item.nbytes = strlen(itemName);
    if (cmd) {
        item.type = matINT8;
        item.data = (const char *) itemName;
        if (RTSim_WriteItemToMatFile(fp,&item, DATA_ITEM)) return(1);
    } else {
        nBytesInItem += (item.nbytes <= 4) ? /*LINTED E_CAST_INT_TO_SMALL_INT*/
                        matTAG_SIZE : matINT64_ALIGN(matTAG_SIZE + item.nbytes);
    }

    if (itemKind == MATRIX_ITEM) {
        const MatrixData *var   = (const MatrixData*) pItem->data;
        int              matID  = RTSim_GetMatIdFromMxId(mxID);
        size_t           elSize = var->elSize;

        /* data */
        item.nbytes = var->nRows * var->nCols * elSize;
        if (cmd) {
            item.type = matID;
            item.data = var->re;
            if (RTSim_WriteItemToMatFile(fp, &item, DATA_ITEM)) return(1);
        } else {
            nBytesInItem += (item.nbytes <= 4) ? /*LINTED*/
                        matTAG_SIZE : matINT64_ALIGN(matTAG_SIZE + item.nbytes);
        }
        /* imaginary part */
        if (var->complex) {
            item.nbytes = var->nRows * var->nCols * elSize;
            if (cmd) {
                item.type = matID;
                item.data = var->im;
                if (RTSim_WriteItemToMatFile(fp, &item, DATA_ITEM)) return(1);
            } else {
                nBytesInItem += (item.nbytes <= 4) ? /*LINTED*/
                        matTAG_SIZE : matINT64_ALIGN(matTAG_SIZE + item.nbytes);
            }
        }
    } else {  /* some type of structure item */
        const char *fieldNames;
        int        sizeofFieldNames;

        /* field names */
        switch (itemKind) {
          case STRUCT_LOG_VAR_ITEM: {
              const StructLogVar *var = (const StructLogVar *) pItem->data;
              fieldNames        = RTSimStructLogVarFieldNames;
              sizeofFieldNames  = var->numActiveFields * mxMAXNAM;
              break;
          }
          case SIGNALS_STRUCT_ITEM: {
              const SignalsStruct *var = (const SignalsStruct *) pItem->data;
              fieldNames        = var->fieldNames;
              sizeofFieldNames  = var->numActiveFields * mxMAXNAM;
              break;
          }
          default:
            sprintf(msgbuf, "Unexpected itemKind=%d in RTSim_ProcessMatItem @C", itemKind);
			printf("%s\n",msgbuf);
            return(-1);
        }

        /* write field names */
        if (cmd) {
            int tmpInt = mxMAXNAM;

            item.nbytes = sizeof(int);
            item.type   = matINT32;
            item.data   = &tmpInt;
            if (RTSim_WriteItemToMatFile(fp,&item, DATA_ITEM)) return(1);
            
            item.nbytes = sizeofFieldNames;
            item.type   = matINT8;
            item.data   = (const char *) fieldNames;
            if (RTSim_WriteItemToMatFile(fp,&item, DATA_ITEM)) return(1);
        } else {
            /*LINTED E_CAST_INT_TO_SMALL_INT*/
            nBytesInItem += matINT64_ALIGN( matTAG_SIZE + matTAG_SIZE +
                                            sizeofFieldNames );
        }

        /* process each field of the structure */
        switch (itemKind) {
          case STRUCT_LOG_VAR_ITEM: {
              const StructLogVar *var = (struct StructLogVar_Tag *)pItem->data;

              /* time */
              {
                  const void *data = var->time;

                  if (var->logTime) { /* time is a LogVar, get the MatrixData */
                      data = &(((const LogVar*) (var->time))->data);
                  }

                  item.type = matMATRIX;
                  item.data = data;
                  if (cmd) {
                      if (RTSim_WriteItemToMatFile(fp,&item,MATRIX_ITEM))return(1);
                  } else {
                      if (RTSim_ProcessMatItem(fp, &item, MATRIX_ITEM,0))return(1);
                      nBytesInItem += item.nbytes + matTAG_SIZE;
                  }
              }

              /* signals */
              item.type = matMATRIX;
              item.data = &(var->signals);
              if (cmd) {
                  if (RTSim_WriteItemToMatFile(fp,&item,SIGNALS_STRUCT_ITEM)) {
                      return(1);
                  }
              } else {
                  if (RTSim_ProcessMatItem(fp, &item, SIGNALS_STRUCT_ITEM,0)) {
                      return(1);
                  }
                  nBytesInItem += item.nbytes + matTAG_SIZE;
              }

              /* block name */
              if (var->blockName != NULL) {
                  item.type = matMATRIX;
                  item.data = var->blockName;
                  if (cmd) {
                      if (RTSim_WriteItemToMatFile(fp, &item, MATRIX_ITEM)) {
                          return(1);
                      }
                  } else {
                      if (RTSim_ProcessMatItem(fp, &item, MATRIX_ITEM, 0)) {
                          return(1);
                      }
                      nBytesInItem += item.nbytes + matTAG_SIZE;
                  }
              }
              break;
          }
          case SIGNALS_STRUCT_ITEM: {
              const SignalsStruct *var        = (SignalsStruct *)pItem->data;
              const LogVar        *values     = var->values;
              const MatrixData    *labels     = var->labels;
              const MatrixData    *plotStyles = var->plotStyles;
              const MatrixData    *titles     = var->titles;
              const MatrixData    *blockNames = var->blockNames;
              int                 i;

              for (i = 0; i < var->numSignals; i++) {
                  /* values */
                  item.type = matMATRIX;
                  item.data = &(values->data);
                  if (cmd) {
                      if (RTSim_WriteItemToMatFile(fp, &item,MATRIX_ITEM)) {
                          return(1);
                      }
                  } else {
                      if (RTSim_ProcessMatItem(fp, &item, MATRIX_ITEM, 0)) {
                          return(1);
                      }
                      nBytesInItem += item.nbytes + matTAG_SIZE;
                  }
                  values = values->next;

                  /* label */
                  item.type = matMATRIX;
                  item.data = &(labels[i]);
                  if (cmd) {
                      if (RTSim_WriteItemToMatFile(fp, &item,MATRIX_ITEM)) {
                          return(1);
                      }
                  } else {
                      if (RTSim_ProcessMatItem(fp, &item, MATRIX_ITEM, 0)) {
                          return(1);
                      }
                      nBytesInItem += item.nbytes + matTAG_SIZE;
                  }
                  /* title */
                  if (titles != NULL) {
                      item.type = matMATRIX;
                      item.data = &(titles[i]);
                      if (cmd) {
                          if (RTSim_WriteItemToMatFile(fp, &item, MATRIX_ITEM)) {
                              return(1);
                          }
                      } else {
                          if (RTSim_ProcessMatItem(fp, &item, MATRIX_ITEM, 0)) {
                              return(1);
                          }
                          nBytesInItem += item.nbytes + matTAG_SIZE;
                      }
                  }
                  /* plot style */
                  if (plotStyles != NULL) {
                      item.type = matMATRIX;
                      item.data = &(plotStyles[i]);
                      if (cmd) {
                          if (RTSim_WriteItemToMatFile(fp,&item, MATRIX_ITEM)) {
                              return(1);
                          }
                      } else {
                          if (RTSim_ProcessMatItem(fp, &item, MATRIX_ITEM, 0)) {
                              return(1);
                          }
                          nBytesInItem += item.nbytes + matTAG_SIZE;
                      }
                  }
                  /* block name */
                  if (blockNames != NULL) {
                      item.type = matMATRIX;
                      item.data = &(blockNames[i]);
                      if (cmd) {
                          if (RTSim_WriteItemToMatFile(fp, &item, MATRIX_ITEM)) {
                              return(1);
                          }
                      } else {
                          if (RTSim_ProcessMatItem(fp, &item, MATRIX_ITEM, 0)) {
                              return(1);
                          }
                          nBytesInItem += item.nbytes + matTAG_SIZE;
                      }
                  }
              } /* for i=1:numSignals */
              break;
          }
          default:
            sprintf(msgbuf, "Unexpected itemKind=%d in "
                          "RTSim_ProcessMatItem @D", itemKind);
			printf("%s\n",msgbuf);
            return(-1);
        }
    } /* end struct item */

    if (!cmd) {
        pItem->nbytes = nBytesInItem;
    }

    return(0);

} /* end: RTSim_ProcessMatItem */

/* Function: RTSim_GetMatIdFromMxId ===============================================
 * Abstract:
 *      Get the MatId given the mxClassID.
 */
int RTSim_GetMatIdFromMxId(mxClassID mxID)
{
    int matID;

    switch (mxID) {
      case mxCELL_CLASS:
      case mxSTRUCT_CLASS:
      case mxOBJECT_CLASS:
        matID = -1;
        break;
      case mxCHAR_CLASS:
        matID = matUINT16;
        break;
      case mxSPARSE_CLASS:
      case mxDOUBLE_CLASS:
        matID = matDOUBLE;
        break;
      case mxSINGLE_CLASS:
        matID = matFLOAT;
        break;
      case mxINT8_CLASS:
        matID = matINT8;
        break;
      case mxUINT8_CLASS:
        matID = matUINT8;
        break;
      case mxINT16_CLASS:
        matID = matINT16;
        break;
      case mxUINT16_CLASS:
        matID = matUINT16;
        break;
      case mxINT32_CLASS:
        matID = matINT32;
        break;
      case mxUINT32_CLASS:
        matID = matUINT32;
        break;
      default:
        matID = matUNKNOWN;
        break;
    }
    return(matID);

} /* end: RTSim_GetMatIdFromMxId */


/* Function: RTSim_WriteMat5FileHeader ============================================
 * Abstract:
 *      Function to write the mat file header.
 *      Return values is
 *          == 0 : upon success
 *          <> 0 : upon failure
 */
int RTSim_WriteMat5FileHeader(FILE *fp)
{
    int        nbytes;
    int        nspaces;
    int        i, n;
    unsigned short     ver[2];
    char       spaces[16];
    const char *matversion = "MATLAB 5.0 MAT-file";

    (void)memset(spaces, ' ', sizeof(spaces));

    n = strlen(matversion);
    nbytes = fwrite(matversion, 1, n, fp);
    /*LINTED E_ASSIGN_INT_TO_SMALL_INT*/
    nspaces = matVERSION_INFO_OFFSET - nbytes;
    /*LINTED E_ASSIGN_INT_TO_SMALL_INT*/
    n = nspaces % sizeof(spaces);
    nbytes += fwrite(spaces, 1, n, fp);
    /*LINTED E_ASSIGN_INT_TO_SMALL_INT*/
    n = nspaces / sizeof(spaces);
    for (i = 0; i < n; ++i) {
        nbytes += fwrite(spaces, 1, sizeof(spaces), fp);
    }
    if (nbytes == matVERSION_INFO_OFFSET) {
        ver[0] = matVERSION;
        ver[1] = matKEY;
        nbytes += fwrite(ver, 1, sizeof(ver), fp);
    }
    return(nbytes != matVERSION_INFO_OFFSET + sizeof(ver));

} /* end: RTSim_WriteMat5FileHeader */

/* Function: RTSim_FixupLogVar ====================================================
 * Abstract:
 *	Make the logged variable suitable for MATLAB.
 */
char* RTSim_FixupLogVar(LogVar *var)
{
    int  nCols   = var->data.nCols;
    int  maxRows = var->data.nRows;
    size_t elSize  = var->data.elSize;
    //int  nRows   = (var->wrapped ?  maxRows : var->rowIdx);
	int	nRows = maxRows;
    //var->nDataPoints = var->rowIdx + var->wrapped * maxRows;
	var->nDataPoints = maxRows;
    if (nRows == 0) return(NULL);

    if (nCols > 1) {  /* Transpose? */
        int	   nEl    = nRows*nCols;
        char   *src   = (char *)var->data.re;
        char   *pmT;
        int	   k;

        if ((pmT = (char *)malloc(nEl*elSize)) == NULL) {
            return(RTSimMemAllocError);
        }

        for (k=0; k<nEl; k++) {
            int kT   = nRows*(k%nCols) + (k/nCols);
            char  *dst = pmT + kT*elSize;
            (void)memcpy(dst, src, elSize);
            src += elSize;
        }
        if (var->data.complex) {
            char *pmiT = (char *)var->data.re;
            src  = (char *)var->data.im;
            for (k=0; k<nEl; k++) {
                int kT   = nRows*(k%nCols) + (k/nCols);
                char  *dst = pmiT + kT*elSize;
                (void)memcpy(dst, src, elSize);
                src += elSize;
            }
            var->data.re = var->data.im;
            var->data.im = pmiT;
        }
        free(var->data.re);
        var->data.re = pmT;

    } /* Transpose? */
    /*
     * We might have allocated more number of rows than the number of data
     * points that have been logged, in which case set nRows to nDataPoints
     * so that only these values get saved.
     */
    if (var->nDataPoints < var->data.nRows) {
        var->data.nRows = var->nDataPoints;
    }

     return(NULL);

} /* end RTSim_FixupLogVar */

/* Function: RTSim_StopDataLogging ================================================
 * Abstract:
 *	Write logged data to model.mat and free memory.
 */
void RTSim_StopDataLogging(FILE *fptr)
{
    LogVar        *var;         //= logInfo->logVarsList;

    //BOOL	matrixFormat;		// int_T = (ssGetLogFormat(S) == 0);
    char	msgbuf[_MAX_PATH];
	char	*msg;

    /**************************************************
     * First log all the variables in the LogVar list *
     **************************************************/
	var = &g_logVar;
    while (var != NULL) {
        if ( (msg = RTSim_FixupLogVar(var)) != NULL ) {
            sprintf(msgbuf,"*** Error writing RTSim_StopDataLogging() due to: %s\n", msg);
			printf("%s\n",msgbuf);
            g_errFlag = 1;
            break;
        }
        if (var->nDataPoints > 0) {
            MatItem item;

            item.type   = matMATRIX;
            item.nbytes = 0; /* not yet known */
            item.data   = &(var->data);
            if (RTSim_WriteItemToMatFile(fptr, &item, MATRIX_ITEM)) {
                sprintf(msgbuf,"*** Error writing log variable %s to "
                              "file RTSim_StopDataLogging()",var->data.name);
				printf("%s\n",msgbuf);
                g_errFlag = 1;
                break;
            }
            g_emptyFile = 0;
        }
        var = var->next;
    }

	return;
} /* end RTSim_StopDataLogging */

#ifdef __cplusplus
}
#endif

