/*
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/bps/command/trace.cpp,v 1.8 2005/03/28 09:45:43 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libbps.h"
#include "libgpm.h"

/************************************************
*                                               *
*       Back Propergation Simurator(BPS)        *
*             subroutine package                *
*               Version                         *
*         coded  in Nuv.10 1990                 *
*         coded by      K.Kuroda                *
*                                               *
*************************************************
*                                               *
*       filename trace.c                        *
*            trace of network                   *
*                                               *
*************************************************/

#ifdef __cplusplus
extern "C" {
#endif

static float  BoxWidth;
static float  XPlus;
static float  YPlus;
static float  XLeng;
static float  XLengHalf;
static float  MaxScale;
static float  MinScale;
  /* static int    DisplayMode = 0; */

static  char  *LayerName[] = {
  "INPUT LAYER", "OUTPUT LAYER", "TEACH DATA", "HIDDEN LAYER" 
};

static double norm_color(double data);
static double norm_size(double data);
static void size(double x1, double y1, double x2, double y2, double level);
static void draw_teach(float *teach_data, int dmode);

#define  K  1.5
#define  W  7.0  


/************************************************
  draw network structure
  ************************************************/
int DrawNetwork(int mode)
{
  int   lay, datapoint, i, unit, unit_to, unit_from, nx, ny;
  int   max_num_unit, bias;
  float hx, hy, gx_point, gy_point, gx_to, gx_from, w, x, gx_increase;
  float box_width_half, *sigmoid_data, next_gx_point, next_gy_point;
  char  no[5], layer_name[BPS_MAX_LAYER][20];
  int   DisplayMode;

  /* SET PARAMETERS */
  DisplayMode  = mode;
  max_num_unit = 0;
  bias = 0;
  for (lay=0; lay < bps_cont.NumOfLayer; lay++){
    if (bps_cont.NumOfCell[lay] > max_num_unit)
      max_num_unit = bps_cont.NumOfCell[lay];
    if (CheckBias(lay)) bias = 1;
  }
  nx = (max_num_unit + bias) * 2 - 1; 
  ny = bps_cont.NumOfLayer * 3 - 2;
  hx = GpmCont.xSize / (float)nx;
  hy = GpmCont.ySize / (float)ny;
  if (hx < hy){
    BoxWidth = hx;
    XPlus = hx;
    YPlus = (GpmCont.ySize - bps_cont.NumOfLayer * hx) / 
      (bps_cont.NumOfLayer - 1);
  }
  else {
    BoxWidth = hy;
    XPlus = (GpmCont.xSize - (max_num_unit + bias) * hy) / 
      (max_num_unit + bias - 1);
    YPlus = hy * 2;
  }
  box_width_half = BoxWidth / 2;
  
  if (bias) XLeng = GpmCont.xSize - BoxWidth - XPlus; 
  else XLeng = GpmCont.xSize;
  XLengHalf = XLeng / 2;
  
  /* SET LAYERNAME */
  strcpy(layer_name[0], LayerName[0]);
  strcpy(layer_name[bps_cont.NumOfLayer-1], LayerName[1]);
  if (bps_cont.NumOfLayer > 3){
    for (lay=1; lay<bps_cont.NumOfLayer-1; lay++){
      strcpy(layer_name[lay], LayerName[3]);
      sprintf(no, " %d", lay);
      strcat(layer_name[lay], no);
    }
  } else {
    strcpy(layer_name[1], LayerName[3]);
  }
  
  /* SET SIGMOID DATA */
  datapoint = (int)(BoxWidth * K);
  gx_increase = BoxWidth / (float)datapoint;
  sigmoid_data = (float*)emalloc((datapoint+1)*sizeof(float));
  if (sigmoid_data == NULL)
		return 46; /* Can't Allocate To \"sigmoid_data\" */
  
  w = (float)((W * 2) / (float)datapoint);
  for (i=0; i<=datapoint; i++){
    x = (float)(-W + i * w); 
    sigmoid_data[i] = (float)(box_width_half/2 + Sigmoid(x)*box_width_half);
  }  
  
  /* DRAW NETWORK */
  gnewpen(GpmCont.fColor);
  gpen(GpmCont.fLineWidth,GpmCont.fLineType, 0);
  gorigin(GpmCont.xOrigin, GpmCont.yOrigin);
  gfactor(GpmCont.factor);
  
  /* DRAW BOX */
  gx_point = 0.0;  
  gy_point = 0.0;
  for (lay=0; lay<bps_cont.NumOfLayer; lay++){
    gx_point = (GpmCont.xSize - XLeng) + XLengHalf
      - (bps_cont.NumOfCell[lay]*BoxWidth + (bps_cont.NumOfCell[lay]-1)*XPlus)/2;  
    for (unit=1; unit<=bps_cont.NumOfCell[lay]; unit++){
      gbox(gx_point, gy_point, gx_point+BoxWidth, gy_point+BoxWidth, 1, -1.0);
      if (DisplayMode != BPS_TRACEMODE_NETWORK_ONLY){
  /* DRAW FUNCTION MARK */
  switch(bps_cont.FuncBias[lay][0]){
    /* SIGMOID */
  case BPS_BIAS_SIGMOID :
    next_gx_point = gx_point;
    next_gy_point = gy_point + sigmoid_data[0];
    gplot(next_gx_point, next_gy_point, 0);  
    for (i=1; i<=datapoint; i++) {
      next_gx_point += gx_increase;
      next_gy_point = gy_point + sigmoid_data[i];
      gplot(next_gx_point, next_gy_point, 2);
    }
    break;
    
    /* LINEAR */
  case BPS_BIAS_LINEAR :
    gplot(gx_point, gy_point, 0);
    gplot(gx_point+BoxWidth, gy_point+BoxWidth, 2);
    break;
  default :
    break;
  }
      }
      gx_point += BoxWidth + XPlus;  
    }
    
    gnewpen(GpmCont.fColor);
    if (lay+1 != bps_cont.NumOfLayer){
      if (CheckBias(lay+1)){
  gbox(0.0, gy_point+BoxWidth+YPlus/4, BoxWidth,
       gy_point+BoxWidth*2+YPlus/4, 1, -1.0); 
  }
    }  
    gy_point += BoxWidth + YPlus;
  }
  
  /* DRAW LINE */
  gy_point = BoxWidth; 
  
  for (lay=1; lay<bps_cont.NumOfLayer; lay++){
    gx_to = (GpmCont.xSize-XLeng) + XLengHalf
      - (bps_cont.NumOfCell[lay]*BoxWidth + (bps_cont.NumOfCell[lay]-1)*XPlus)/2;  
    for (unit_to=1; unit_to<=bps_cont.NumOfCell[lay]; unit_to++){
      if (CheckBias(lay)){
  gplot(box_width_half, gy_point+BoxWidth+YPlus/4, 0);
  gplot(gx_to+box_width_half, gy_point+YPlus, 2);
      }
      
      gx_from = (GpmCont.xSize-XLeng) + XLengHalf
  - (bps_cont.NumOfCell[lay-1]*BoxWidth + (bps_cont.NumOfCell[lay-1]-1)*XPlus)/2;
      for (unit_from=1; unit_from<=bps_cont.NumOfCell[lay-1]; unit_from++){
  gplot(gx_from+box_width_half, gy_point, 0);
  gplot(gx_to+box_width_half, gy_point+YPlus, 2);
  gx_from += BoxWidth + XPlus;
      }
      
      gx_to += BoxWidth + XPlus;
    }
    gy_point += BoxWidth + YPlus;
  }
  
  efree(sigmoid_data);
  gflush();

	return 0;
}


/************************************************
  draw teach data
  ************************************************/
void DrawTeachData()
{
  int  unit;
  float  gx_point, gy_point;
  
  gnewpen(GpmCont.fColor);
  gpen(GpmCont.fLineWidth,GpmCont.fLineType, 0);
  gorigin(GpmCont.xOrigin, GpmCont.yOrigin);
  gfactor(GpmCont.factor);
  
  gy_point = GpmCont.ySize + BoxWidth;
  gx_point = (GpmCont.xSize-XLeng) + XLengHalf
    - (bps_cont.NumOfCell[bps_cont.NumOfLayer-1]*BoxWidth + (bps_cont.NumOfCell[bps_cont.NumOfLayer-1]-1)*XPlus)/2;
  
  for (unit=1; unit<=bps_cont.NumOfCell[bps_cont.NumOfLayer-1]; unit++){ 
    gbox(gx_point, gy_point, gx_point+BoxWidth, gy_point+BoxWidth, 1, -1.0);
    gx_point += BoxWidth + XPlus;  
  }
  
  gnewpen(GpmCont.gColor);
  gflush();
}


/************************************************
 normalize data
************************************************/
static double norm_color(double data)
{
  double  level;
  
  level = (data - MinScale) / (MaxScale - MinScale);
  return level;
  
}


/************************************************
 normalize data
************************************************/
static double norm_size(double data)
{
  double  level, max, min;
  
  if ((MinScale < 0) && (MaxScale > 0)){
    max = (double)((-MinScale < MaxScale) ? MaxScale : -MinScale);
    min = 0.0;
    level = (data - min) / (max - min);

    if (level >  1.0) return  1.0;
    if (level < -1.0) return -1.0;
    
    return level;
  }  
  level = (data - MinScale) / (MaxScale - MinScale);

  if (level > 1.0) return 1.0;
  if (level < 0.0) return 0.0;
  
  return level;
  
}


/************************************************
  count size
  ************************************************/
static void size(double x1, double y1, double x2, double y2, double level)
{
  double size, size_delta;
  
  if (level > 0){
    gnewpen(2);
    size = BoxWidth * level;
  }
  else{
    gnewpen(1);
    size = BoxWidth * -level;
  }
  size_delta = (BoxWidth - size) / 2;
  gbox(x1+size_delta, y1+size_delta,
       x1+size_delta+size, y1+size_delta+size, 2, -1.0);
}


/************************************************
  draw teach data
  ************************************************/
static void draw_teach(float *teach_data, int dmode)
{
  int  unit;
  float  gx_point, gy_point, level;
  
  gy_point = GpmCont.ySize + BoxWidth;
  gx_point = (GpmCont.xSize - XLeng) + XLengHalf
    - (bps_cont.NumOfCell[bps_cont.NumOfLayer-1]*BoxWidth 
       + (bps_cont.NumOfCell[bps_cont.NumOfLayer-1]-1)*XPlus)/2;
  
  for (unit=0; unit<bps_cont.NumOfCell[bps_cont.NumOfLayer-1]; unit++){ 
    if (dmode == BPS_TRACEMODE_COLOR){
      level = (float)norm_color(teach_data[unit]);
      grainbow(level);
      gbox(gx_point, gy_point,
     gx_point+BoxWidth, gy_point+BoxWidth, 2, level);
      gnewpen(GpmCont.fColor);
      gbox(gx_point, gy_point, gx_point+BoxWidth, gy_point+BoxWidth, 1,-1.0);
    } else {
      level = (float)norm_size(teach_data[unit]);
      gnewpen(0);
      gbox(gx_point, gy_point, gx_point+BoxWidth, gy_point+BoxWidth, 2,-1.0);
      size(gx_point, gy_point, gx_point+BoxWidth, gy_point+BoxWidth, level);
      gnewpen(GpmCont.fColor);
      gbox(gx_point, gy_point, gx_point+BoxWidth, gy_point+BoxWidth, 1,-1.0);
    }
    gx_point += BoxWidth + XPlus;  
  }
}


/************************************************
  draw box of network structure
  ************************************************/
void DrawBox(int ptrn_no, int teach_flag, int inlayer, int outlayer, int dmode)
{
  int        lay, unit;
  double     gx_point, gy_point, level;
  bps_cel_t *cell_pt = NULL;
  
  gnewpen(GpmCont.fColor);
  gpen(GpmCont.fLineWidth,GpmCont.fLineType, 0);
  gorigin(GpmCont.xOrigin, GpmCont.yOrigin);
  gfactor(GpmCont.factor);
  
  gx_point = 0.0;  
  gy_point = 0.0;
  
  for (lay=0; lay<bps_cont.NumOfLayer; lay++){
    gx_point = (GpmCont.xSize - XLeng) + XLengHalf
      - (bps_cont.NumOfCell[lay]*BoxWidth 
         + (bps_cont.NumOfCell[lay]-1)*XPlus) / 2;
    if ((inlayer <= lay) && (lay <= outlayer)){ 
      for (unit=1; unit<=bps_cont.NumOfCell[lay]; unit++){ 
        cell_pt = bps_cont.BPNet[lay][unit].CellNode;
        if (dmode == BPS_TRACEMODE_COLOR){
          level = norm_color(cell_pt->Active);
          grainbow(level);
          gbox(gx_point, gy_point,
               gx_point+BoxWidth, gy_point+BoxWidth, 2, level);
          gnewpen(GpmCont.fColor);
          gbox(gx_point, gy_point,
               gx_point+BoxWidth, gy_point+BoxWidth, 1, -1.0);
        } else {
          level = norm_size(cell_pt->Active);
          gnewpen(0);
          gbox(gx_point, gy_point,
               gx_point+BoxWidth, gy_point+BoxWidth, 2, -1.0);
          size(gx_point, gy_point,
               gx_point+BoxWidth, gy_point+BoxWidth, level);
          gnewpen(GpmCont.fColor);
          gbox(gx_point, gy_point,
               gx_point+BoxWidth, gy_point+BoxWidth, 1, -1.0);
        }
        gx_point += BoxWidth + XPlus;  
      }
    }
    
    if ((inlayer <= lay+1) && (lay+1 <= outlayer)){ 
      if (lay+1 != bps_cont.NumOfLayer){
        if (CheckBias(lay+1)){  
          if (dmode == BPS_TRACEMODE_COLOR){
            level = norm_color(1.0); 
            grainbow(level);
            gbox(0.0, gy_point+BoxWidth+YPlus/4,
                 BoxWidth, gy_point+BoxWidth*2+YPlus/4, 2, level);
            gnewpen(GpmCont.fColor);
            gbox(0.0, gy_point+BoxWidth+YPlus/4,
                 BoxWidth, gy_point+BoxWidth*2+YPlus/4, 1, -1.0);
          } else {
            level = norm_size(1.0);
            gnewpen(0);
            gbox(0.0, gy_point+BoxWidth+YPlus/4,
                 BoxWidth, gy_point+BoxWidth*2+YPlus/4, 2, -1.0);
            size(0.0, gy_point+BoxWidth+YPlus/4,
                 BoxWidth, gy_point+BoxWidth*2+YPlus/4, level);
            gnewpen(GpmCont.fColor);
            gbox(0.0, gy_point+BoxWidth+YPlus/4,
                 BoxWidth, gy_point+BoxWidth*2+YPlus/4, 1, -1.0);
          }
        }
      }
    }
    gy_point += BoxWidth + YPlus;
  }
  
  if (teach_flag)
    draw_teach(bps_cont.TeachData[ptrn_no], bps_cont.DisplayMode);
  
  gflush();
}

#ifdef __cplusplus
}
#endif
