/*
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/bps/command/sigmoid.cpp,v 1.8 2005/02/22 07:40:22 ninja Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libbps.h"
#include "libgpm.h"

/************************************************
*                                               *
*       Back Propergation Simurator(BPS)        *
*             subroutine package                *
*               Ver                             *
*         coded         in Nov.5 1990           *
*         coded by      K.Kuroda                *
*************************************************
*                                               *
*       filename sigmoid.c                      *
*                                               *
************************************************/

#ifdef __cplusplus
extern "C" {
#endif

#define	PLOT		'P'
#define	HISTGRAM	'H'

static char          plot_mode;
static int           lay_no, unit_no, plot_n;
static float        *net_data, *active_data;

/************************************************
 *	set parameters
 ************************************************/
int
set_param_sigmoid()
{
  char *str;

  lay_no    =   (int)GetScalar(0);
  unit_no   =   (int)GetScalar(1);
  str       =        GetString(2);
  
  if ((lay_no < 0) || ((bps_cont.NumOfLayer - 1) < lay_no))
		return 20; /* Illigal Layer No. */
  if (bps_cont.FuncBias[lay_no][0] != BPS_BIAS_SIGMOID)
		return 20; /* Illigal Layer No. */
  if ((unit_no < 1) || (bps_cont.NumOfCell[lay_no] < unit_no))
		return 24; /* Illigal Unit No. */
  if(str == NULL)
		return 2; /* illegal parameter */

  plot_mode = toupper(str[0]);
  switch (plot_mode) {
  case PLOT:     plot_n =   (int)GetScalar(3);  break;
  case HISTGRAM: plot_n =   (int)GetScalar(3);  break;
  default:
    return 22; /* Unknown Mode */
  }

	return 0;
}

/************************************************
 * allocate to array
 ************************************************/
int alloc_data()
{
  net_data    = (float*)emalloc(bps_cont.NumOfPtrn * sizeof(float));
  if (net_data == NULL)
		return 23; /* Can't Allocate To Array */

  active_data = (float*)emalloc(bps_cont.NumOfPtrn * sizeof(float));
  if (active_data == NULL)
		return 23; /* Can't Allocate To Array */

	return 0;
}

static void free_data()
{
  efree(net_data);
  efree(active_data);
}

/***********************************************
 * network test
 ***********************************************/
void
net_test()
{
  int     ptrn;
  bps_cel_t  *cur_cel;
  
  for (ptrn = 0; ptrn < bps_cont.NumOfPtrn; ptrn++) {
    test_network(bps_cont.InputData[ptrn]);
    cur_cel = bps_cont.BPNet[lay_no][unit_no].CellNode;
    net_data[ptrn] = (float)cur_cel->Net;
    active_data[ptrn] = (float)cur_cel->Active;
  }
}

/************************************************
 * draw sigmoid curve
 ************************************************/
int  SigmoidPlot()
{
  char     title[2][10];
  int      i, j, ptrn, datapoint, nn, jj;
  float    min_net, max_net, min_leng; 
  float    char_size, scale_width;
  float    act_sum, act_mean, act_vari, work, act_stand;
  float    delta, dat, hist_max;
  float   *data, *sigmoid_data;
  float   *histdata1, *histdata2;
  double   x, act_area;
  
  gpen(GpmCont.fLineWidth,GpmCont.gLineType, 0);
  gnewpen(GpmCont.fColor);
  gorigin(GpmCont.xOrigin, GpmCont.yOrigin);
  gfactor(GpmCont.factor);
  gfont(GpmCont.fontType);

  /* SET MINIMUM NET VALUE AND MAXIMUM NET VALUE */
  max_net = 5.0;
  for (ptrn = 0; ptrn < bps_cont.NumOfPtrn; ptrn++) {
    if (max_net < net_data[ptrn])
      max_net = net_data[ptrn];
    if (max_net < -net_data[ptrn])
      max_net = -net_data[ptrn];
  }
  max_net *= 10;
  max_net =  (float) (max_net + 0.5);
  max_net /= 10;
  min_net = -max_net;
  
  /* SET SIGMOID DATA */
  datapoint = (int) ((max_net - min_net) / 0.1);

  sigmoid_data = (float*)emalloc((datapoint + 1) * sizeof(float));
  if (sigmoid_data == NULL)
    return 46; /* Can't Allocate To \"sigmoid_data\" */

  data = (float*)emalloc((datapoint + 1) * sizeof(float));
  if (data == NULL)
    return 46; /* Can't Allocate To \"sigmoid_data\" */
  
  for (i = 0; i <= datapoint; i++) {
    x = min_net + i * 0.1;
    data[i] = (float)x;
    sigmoid_data[i] = (float)Sigmoid(x);
  }
  
  /* DRAW AXIS */
  strcpy(title[0], "NET");
  strcpy(title[1], "ACTIVATION");
  if (GpmCont.xSize > GpmCont.ySize)
    min_leng = GpmCont.ySize;
  else
    min_leng = GpmCont.xSize;
  char_size = (float)(min_leng * 0.04);
  scale_width = (-min_net < max_net) ? -min_net : max_net;
  
  gaxis(0.0, 0.0, GpmCont.xSize, 0.0, min_net, max_net, scale_width,
	1, 1, title[0], char_size, 0, GpmCont.ySize);
  gaxis(0.0, 0.0, GpmCont.ySize, (double)M_PI/2.0, 0.0, 1.0, 0.5,
	1, 0, title[1], char_size, 0, GpmCont.ySize);
  
  gfsize(GpmCont.xSize, GpmCont.ySize);
  gframe();
  
  /* DRAW SIGMOID CURVE */
  glimit(0.0, -0.1, GpmCont.xSize, GpmCont.ySize + 0.1);
  gpen(GpmCont.gLineWidth,GpmCont.gLineType, 0);
  gnewpen(GpmCont.gColor);
  gscale(min_net, 0.0, max_net, 1.0);
  ggraph(data, sigmoid_data, datapoint + 1, 1, 0, 0, 0.0);
  
  glimit(0.0, 0.0, 0.0, 0.0);
  gnewpen(GpmCont.fColor);
  gpen(GpmCont.fLineWidth, 1, 0);

  act_area = (double)(2.2 * GpmCont.xSize / (max_net * 2.0));
  act_area = (double)(GpmCont.xSize / 2.0 - act_area);
  gplot(act_area, 0.0, 0);
  gplot(act_area, GpmCont.ySize, 2);

  act_area = (double)GpmCont.xSize - act_area;
  gplot(act_area, 0.0, 0);
  gplot(act_area, GpmCont.ySize, 2);
  
  /* SET MEAN VALUE, VARIANCE AND STANDARD DEVIATION */
  act_sum = 0.0;
  for (i = 0; i < bps_cont.NumOfPtrn; i++)
    act_sum += active_data[i];
  act_mean = act_sum / (float) bps_cont.NumOfPtrn;
  
  act_sum = 0.0;
  for (i = 0; i < bps_cont.NumOfPtrn; i++) {
    work = active_data[i] - act_mean;
    act_sum += work * work;
  }
  
  act_vari = act_sum / (float) bps_cont.NumOfPtrn;
  act_stand = (float) sqrt((double) act_vari);
  
  printf("\n*** MEAN VALUE         = %15.5g\n", act_mean);
  printf("*** VARIANCE           = %15.5g\n", act_vari);
  printf("*** STANDARD DEVIATION = %15.5g\n", act_stand);
  
  if (plot_mode == PLOT) {
    /* PLOT */
    gnewpen(GpmCont.fColor);
    glimit(0.0, -0.1, GpmCont.xSize, GpmCont.ySize + 0.1);
    gscale(min_net, 0.0, max_net, 1.0);
    ggraph(net_data, active_data, bps_cont.NumOfPtrn, 1, 1, plot_n, char_size / 2);
  } else {
    /* HISTOGRAM */
    delta = (max_net - min_net) / (float) plot_n;
    nn = plot_n * 2 + 2;
    plot_n++;
    
    histdata1 = (float*)emalloc(nn * sizeof(float));
    if (histdata1 == NULL)
      return 46; /* Can't Allocate To \"sigmoid_data\" */
    histdata2 = (float*)emalloc(nn * sizeof(float));
    if (histdata2 == NULL)
      return 46; /* Can't Allocate To \"sigmoid_data\" */
    
    for (i = 0; i < plot_n; i++) {
      j = i * 2;
      dat = delta * (float) i + min_net;
      histdata1[j] = dat;
      histdata1[j + 1] = dat;
    }
    
    for (i = 0; i < nn; histdata2[i++] = 0.0);
    
    for (ptrn = 0; ptrn < bps_cont.NumOfPtrn; ptrn++) {
      if (net_data[ptrn] < min_net || net_data[ptrn] > max_net)
	continue;
      jj = (int) ((net_data[ptrn] - min_net) / delta);
      jj = jj * 2 + 1;
      histdata2[jj]++;
      histdata2[jj + 1]++;
    }
    
    hist_max = 0;
    for (i = 0; i < nn; i = i + 2)
      hist_max = (hist_max > histdata2[i]) ? hist_max : histdata2[i];
    hist_max = (float) (hist_max * 1.5 + 0.5);
    
    gnewpen(GpmCont.fColor);
    gpen(GpmCont.fLineWidth, GpmCont.fLineType, 0);
    strcpy(title[1], "FREQUENCY");
    gscale(min_net, 0.0, max_net, hist_max);
    scale_width = (float) (hist_max / 2.0);
    gaxis(GpmCont.xSize, 0.0, GpmCont.ySize, (double)M_PI/2.0, 0.0,
	  hist_max, scale_width, 2, 1, title[1],
	  char_size, 0, GpmCont.ySize);
    ggraph(histdata1, histdata2, nn, 1, 0, 0, 0);
  }
  
  efree(data);
  efree(sigmoid_data);
  glimit(0.0, 0.0, 0.0, 0.0);
  gflush();
  efree(histdata1);
  efree(histdata2);

	return 0;
}


/************************************************
  sigmoid plot
  ************************************************/
DLLEXPORT int mod_bps_sigmoid()
{
  /* GET PARAMETERS */
  rebps();
  regpm();
  
  GetStructureParameters();
  GetTestingParameters();
  set_param_sigmoid();
  
  /* SYSTEM INITIALIZE */
  rec_system_initialize();
  MakeNetwork();

  ReadWeight2(bps_cont.TestWgtFile, bps_cont.WgtHistoryNo);

  alloc_data();
  
  /* NETWORK TEST */
  net_test();
  
  /* DRAW SIGMOID CURVE AND PLOT */
  /* if (checktodvi())
    writegpm(); */
  gopen(GpmCont.paper, GpmCont.orientation, GpmCont.device, 
		GpmCont.paperWidth, GpmCont.paperHeight);
  gnewpen(GpmCont.fColor);
  gpen(GpmCont.fLineWidth,GpmCont.fLineType, 0);
  gorigin(GpmCont.xOrigin, GpmCont.yOrigin);
  gfactor(GpmCont.factor);
  
  SigmoidPlot();
  
  gflush();
  gclose();
  
  /* SYSTEM END */
  BreakNetwork();
  rec_system_end();

  wrbps();
  free_data();
  return 0;
}

#ifdef __cplusplus
}
#endif
