/*
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/bps/command/correlat.cpp,v 1.7 2005/02/22 07:38:50 ninja Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libbps.h"

/*************************************************
 *                                               *
 *       Back Propergation Simurator(BPS)        *
 *             subroutine package                *
 *               Ver                             *
 *         coded         in Nov.5  1990          *
 *         last modified in Nov.15 1990          *
 *         coded by      K.Kuroda                *
 *                                               *
 *************************************************
 *                                               *
 *       filename correlat.c			 *
 *                     				 *
 *************************************************/

#ifdef __cplusplus
extern "C" {
#endif

#define	CORRELAT	'C'
#define	DISTANCE	'D'

static char     corr_mode;
static int      start_buff, num_buff;
static int      num_data;
static float  **corre_data, **data;
static Buffer  *dum;

/************************************************
  set_parameters
  ************************************************/
int
set_param_correlat()
{
  char *str;

  start_buff = GetBufferID(0);
  str        = GetString(1);
  
  if (start_buff <= 0)
    return 17; /* Illigal Buffer No. */
  if(str == NULL)
		return 2; /* illegal parameter */

  corr_mode = toupper(str[0]);
  if ((corr_mode != CORRELAT) && (corr_mode != DISTANCE))
    return 22; /* Unknown Mode */

	return 0;
}


/************************************************
  allocate to array
  ************************************************/
int
alloc_array()
{
  corre_data = (float**)bps_malloc2D(num_buff, num_buff, sizeof(float));
  if (corre_data == NULL)
    return 23; /* Can't Allocate To Array */

	return 0;
}


/************************************************
  read data from buffer to array
  ************************************************/
int
read_data()
{
  int  i, j, k;
  int  dim, idx[MAX_INDEX];
  
  dum = ReadBuffer(start_buff, &dim, idx);
  
  num_data = idx[1];
  num_buff = idx[0];

  data = (float**)bps_malloc2D(num_buff, num_data, sizeof(float));
  if (data == NULL)
    return 23; /* Can't Allocate To Array */

  for(i = 0, k = 0; i < num_buff; i++)
    for(j = 0; j < num_data; j++, k++)
      data[i][j] = (float)dum[k];

  efree(dum);

	return 0;
}


/************************************************
  calculate covariance matrix
  ************************************************/
int
cal_covariance()
{
  int   i, j, k;
  float temp, *mean;
  
  mean = (float *)emalloc(num_buff * sizeof(float));
  if (mean == NULL)
    return 23; /* Can't Allocate To Array */
  
  /* CALCULATE MEAN */
  for (i = 0; i < num_buff; i++) {
    temp = 0.0;
    for (j = 0; j < num_data; j++)
      temp += data[i][j];
    mean[i] = temp / (float)num_data;
  }
  
  /* CALCULATE COVARIANCE MATRIX */
  for (i = 0; i < num_buff; i++)
    for (j = 0; j < num_buff; j++)
      corre_data[i][j] = 0.0;

  for (i = 0; i < num_buff; i++) {
    for (j = i; j < num_buff; j++) {
      temp = 0.0;
      for (k = 0; k < num_data; k++) {
	temp += data[i][k] * data[j][k];
      }
      corre_data[j][i] = (temp - num_data*mean[i]*mean[j])/(float)(num_data-1);
    }
  }

  efree(mean);
	return 0;
}


/************************************************
  calculate correlation matrix
  ************************************************/
int
cal_correlation()
{
  int             i, j;
  float          *stand_div;
  
  stand_div = (float *)emalloc(num_buff * sizeof(float));
  if (stand_div == NULL)
    return 23; /* Can't Allocate To Array */
  
  for (i = 0; i < num_buff; i++)
    stand_div[i] = (float) sqrt((double) corre_data[i][i]);
  
  for (i = 0; i < num_buff - 1; i++) {
    for (j = i + 1; j < num_buff; j++) {
      corre_data[i][j] = corre_data[j][i] / (stand_div[i] * stand_div[j]);
    }
  }

  efree(stand_div);
	return 0;
}


/************************************************
  calculate distance
  ************************************************/
void
cal_distance()
{
  int    i, j, k;
  float  tmp;
  
  for (i = 0; i < num_buff - 1; i++) {
    for (j = i + 1; j < num_buff; j++) {
      for (k = 0; k < num_data; k++) {
	tmp               = data[i][k] - data[j][k];
	corre_data[i][j] += tmp * tmp;
      }
    }
  }
}


/************************************************
  store data 
  ************************************************/
void
stordata()
{
  int      i,j;
  int	   idx[MAX_BUFF_DIM];
  double  *dum;
  
  idx[0] = num_buff;
  idx[1] = num_buff;
  
  dum = AllocBuffer(IndexSize(2, idx));
  
  printf("  ### Snapshot : 2d_Dimension ");

  for (i = 0; i < 2; i++)
    printf("[%d]", idx[i]);

  printf(" ###\n");
  printf("  ### CORRELATION ==> 2D Snapshot ###\n");
  
  for (i = 0; i < idx[0]; i++) {
    for(j = 0; j < idx[1]; j++) {
      *(dum+j+i*idx[1]) = (double)corre_data[i][j];
    }
    printf("     CORRELATION ==> INDEX[%d] \r", i );
    fflush(stdout);
  }

  printf("\n");

  ReturnSnapshot(dum, 2, idx);
  efree(dum);
}


/************************************************
  correlation analysys of weight
  ************************************************/
DLLEXPORT int mod_bps_correlat()
{
  /* GET PARAMETERS */
  set_param_correlat();
  
  read_data();
  alloc_array();
  
  printf("read_data is OK\n");
  
  cal_covariance();

  switch (corr_mode) {
  case CORRELAT:  cal_correlation(); break;
  case DISTANCE:  cal_distance();    break;
  }

  stordata();
  
  bps_free2D((char**)corre_data);
  bps_free2D((char**)data);

  return 0;
}

#ifdef __cplusplus
}
#endif