/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: Shell.cpp,v 1.5 2005/02/18 13:51:13 orrisroot Exp $ */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif
#ifdef HAVE_DIRECT_H
# include <direct.h>
#endif
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_PARAM_H
# include <sys/param.h>
#endif

#include "SL_header.h"
#include <cctype>

#include <libsatellite.h>

using namespace std;

#define  __IMPORTSYMBOL__
#include "SL_exception.h"
#include "history.h"
#include "module.h"
#include "tty_console.h"
#include "SL_Index.h"
#include "Base_Buffer.h"
#include "Series_Buffer.h"
#include "Snapshot_Buffer.h"
#include "String_Buffer.h"
#include "Scalar_Buffer.h"
#include "SL_Tool.h"
#include "SL_Object.h"
#include "SymbolList.h"
#include "SystemCommon.h"
#undef   __IMPORTSYMBOL__
#include "parse.h"
#include "Program.h"
#include "Datum.h"
#include "Inline.h"
#include "CommandAlias.h"
#include "LexicalAnalizer.h"
#include "pipe.h"
#include "Builtin.h"
#include "Builtin_Buffer.h"
#include "vmstat.h"
#define  __EXPORTSYMBOL__
#include "StackMachine.h"
#include "Shell.h"
#include "mathfunc.h"

#ifndef _MAX_PATH
# ifdef  MAXPATHLEN
#  define _MAX_PATH  MAXPATHLEN
# else
#  define _MAX_PATH  1024
# endif
#endif


static struct {         /* Constans */
  char           *name;
  double          cval;
}  consts[] = {
  {"+1",            1.0},
  {"BigEndian",     0.0},
  {"LittleEndian",  1.0},
  {"false",         0.0},
  {"true",          1.0},
  {"PHI",           1.61803398874989484820},
  {"GAMMA",         0.57721566490153286060},
  {"DEG",          57.29577951308232087680},
  {"E",             2.71828182845904523536},
  {"PI",            3.14159265358979323846},
  {"SATELLITE_VERSION", 4.0},
  {0,               0.0}
};

static struct {
  char           *name;
  BFUNC           func;
} builtins[] = {
  /*    Utilitys                */      
  {"history",    &BuiltinFunctions::history},
  {"alias",      &BuiltinFunctions::alias},
  {"symbols",    &BuiltinFunctions::symbols},
  {"length",     &BuiltinFunctions::lengthof},
  {"index",      &BuiltinFunctions::indexof},
  {"typeof",     &BuiltinFunctions::Typeof},
  {"printf",     &BuiltinFunctions::Printf},
  {"abort",      &BuiltinFunctions::Abort},
  {"eval",       &BuiltinFunctions::eval_program},
  {"welcome",    &BuiltinFunctions::welcome},
  {"goodbye",    &BuiltinFunctions::goodbye},
  {"strlen",     &BuiltinFunctions::Strlen},
  {"strict_grammar",  &BuiltinFunctions::strict_grammar},
  {"sl4vars",    &BuiltinFunctions::sl4vars},
#ifdef WIN32
  //{"pwd",        &BuiltinFunctions::Pwd},
#endif
  {0,           0}
};


static struct {
  char  *name;
  int    func;
} opcode[] = {
  // Mathematical Library double (*)(double,double)
  {"mod",       MATH_MOD},
  {"int",       MATH_INT},
  {"floor",     MATH_FLOOR},
  {"ceil",      MATH_CEIL},
  {"round",     MATH_ROUND},
  {"abs",       MATH_ABS},
  {"sgn",       MATH_SIGN},
  {"pow",       MATH_POW},
  {"exp",       MATH_EXP},
  {"log",       MATH_LOG},
  {"log2",      MATH_LOG2},
  {"log10",     MATH_LOG10},
  {"sqrt",      MATH_SQRT},
  {"sin",       MATH_SIN},
  {"cos",       MATH_COS},
  {"tan",       MATH_TAN},
  {"asin",      MATH_ASIN},
  {"acos",      MATH_ACOS},
  {"atan",      MATH_ATAN},
  {"atan2",     MATH_ATAN2},
  {0,           0}
};


static struct {
  char           *name;
  int             type;
} klasses[] = {
  {"series",   CLASS_TYPE_SERIES},
  {"snapshot", CLASS_TYPE_SNAPSHOT},
  {"scalar",   CLASS_TYPE_SCALAR},
  {"string",   CLASS_TYPE_STRING},
  {NULL, 0}
};

sl_shell::sl_shell(tty_console *con) : console(con), stackmachine(con),
  system_common(con){
}

sl_shell::~sl_shell(){
  wait_all_process();
}

/* install constants and built-ins in table */
void sl_shell::install_builtin_symbols(){
  symbol_t *sym;
  register int  i;
  Base_Buffer *buf;
  SL_Object *obj;
  /*** constants ***/
  for (i = 0; consts[i].name; i++){
    buf=new Scalar_Buffer;
    buf->InitBuffer();
    ((Scalar_Buffer*)buf)->SetScalar(consts[i].cval);
    obj=new_SL_Object(SL_OBJ::SCALAR_O,buf);
    sym=symbol_new(consts[i].name,obj,SYMBOL_TYPE_CONST);
    symbol_table_install(syscom->gl_symtab,sym);
  }
  /*** builtin functions ***/
  /* mathmatical functions */
  for (i = 0; opcode[i].name; i++){
    buf=new Builtin_Buffer;
    ((Builtin_Buffer*)buf)->SetOFunction(opcode[i].func);
    obj=new_SL_Object(SL_OBJ::BUILTIN_O,buf);
    sym=symbol_new(opcode[i].name, obj, SYMBOL_TYPE_OPCODE);
    symbol_table_install(syscom->gl_symtab,sym);
  }
  /* builtin functions */
  for (i = 0; builtins[i].name; i++){
    buf=new Builtin_Buffer;
    ((Builtin_Buffer*)buf)->SetBFunction(builtins[i].func);
    obj=new_SL_Object(SL_OBJ::BUILTIN_O,buf);
    sym=symbol_new(builtins[i].name, obj, SYMBOL_TYPE_BLTIN);
    symbol_table_install(syscom->gl_symtab,sym);
  }
  /*** class names ***/
  for (i = 0; klasses[i].name; i++){
    buf=new Builtin_Buffer;
    ((Builtin_Buffer*)buf)->SetClassType(klasses[i].type);
    obj=new_SL_Object(SL_OBJ::BUILTIN_O,buf);
    sym=symbol_new(klasses[i].name, obj, SYMBOL_TYPE_CLASS);
    symbol_table_install(syscom->gl_symtab,sym);
  }
}

bool sl_shell::initialize(int argc, char *argv[]){
  const char *sep;
  char tmp[_MAX_PATH];
  bool need_setup=true,need_rc=true,need_clean=true;
  bool ignore_options=false;
  string sysfile_path,userfile_path;

  stackmachine.lex.keep_alive=false;
  for(int i=1; i<argc; i++){
    if(ignore_options){
      if(!strcmp(argv[i],"-")) stackmachine.lex.keep_alive=true;
      else stackmachine.lex.infiles.push_back(string(argv[i]));
    }
    if     (!strcmp(argv[i],"-rc")){
      need_rc=false;
    }else if(!strcmp(argv[i],"-setup")){
      need_setup=false;
    }else if(!strcmp(argv[i],"-clean")){
      need_clean=false;
    }else if(!strcmp(argv[i],"-h")){
      print_usage(argv[0]); return false;
    }else if(!strcmp(argv[i],"-help")){
      print_usage(argv[0]); return false;
    }else if(!strcmp(argv[i],"--help")){
      print_usage(argv[0]); return false;
    }else if(!strcmp(argv[i],"--")){
      ignore_options=true;
    }else{
      if(!strcmp(argv[i],"-")) stackmachine.lex.keep_alive=true;
      else stackmachine.lex.infiles.push_back(string(argv[i]));
    }
  }
  if(!stackmachine.lex.keep_alive && stackmachine.lex.infiles.empty()) 
    stackmachine.lex.keep_alive=true;

  // change directory to home 
#ifdef WIN32
 {
   char home[_MAX_PATH];
   if(GetHomeDirectory(home,_MAX_PATH)!=NULL){
     if(_chdir(home)==0){
       // chdir() command success
       char buf[512];
       char *p=getenv("PWD");
       if(p!=0){
         if(getcwd(buf,512)!=0){
#ifdef HAVE_PUTENV
           string str("PWD=");
           str+=buf;
           putenv(str.c_str());
#endif
         }
       }
     }
   }
 }
#endif
  if(GetSystemResourceDirectory(tmp,_MAX_PATH)==NULL){
    strcpy(tmp,"/");
  }
  sysfile_path = tmp;
  if(GetUserResourceDirectory(tmp,_MAX_PATH)==NULL){
    strcpy(tmp,"/");
  }
  userfile_path = tmp;

  sep = "/"; /* TODO check (IsWindows())? "\\" : "/" ; */
  if(need_rc){
    stackmachine.lex.rc_file=sysfile_path;
    stackmachine.lex.rc_file+=sep;
    stackmachine.lex.rc_file+=DEFAULT_RC_FILE;
    if(Access(stackmachine.lex.rc_file.c_str(),SL_FATTR_ROK)!=0)
      stackmachine.lex.rc_file.erase();
  }
  if(need_setup){
    if(!userfile_path.empty()){
      stackmachine.lex.setup_file=userfile_path;
      stackmachine.lex.setup_file+=sep;
      stackmachine.lex.setup_file+=DEFAULT_SETUP_FILE;
      if(Access(stackmachine.lex.setup_file.c_str(),SL_FATTR_ROK)!=0)
        stackmachine.lex.setup_file.erase();
    }
    if(stackmachine.lex.setup_file.empty()){
      stackmachine.lex.setup_file=sysfile_path;
      stackmachine.lex.setup_file+=sep;
      stackmachine.lex.setup_file+=DEFAULT_SETUP_FILE;
      if(Access(stackmachine.lex.setup_file.c_str(),SL_FATTR_ROK)!=0)
        stackmachine.lex.setup_file.erase();
    }
  }
  if(need_clean){
    if(!userfile_path.empty()){
      stackmachine.lex.clean_file=userfile_path;
      stackmachine.lex.clean_file+=sep;
      stackmachine.lex.clean_file+=DEFAULT_CLEAN_FILE;
      if(Access(stackmachine.lex.clean_file.c_str(),SL_FATTR_ROK)!=0)
        stackmachine.lex.clean_file.erase();
    }
    if(stackmachine.lex.clean_file.empty()){
      stackmachine.lex.clean_file=sysfile_path;
      stackmachine.lex.clean_file+=sep;
      stackmachine.lex.clean_file+=DEFAULT_CLEAN_FILE;
      if(Access(stackmachine.lex.clean_file.c_str(),SL_FATTR_ROK)!=0)
        stackmachine.lex.clean_file.erase();
    }
  }
  if(stackmachine.lex.rc_file.empty())
    stackmachine.lex.rc_finished=true;
  else stackmachine.lex.rc_finished=false;
  if(stackmachine.lex.setup_file.empty())
    stackmachine.lex.setup_finished=true;
  else stackmachine.lex.setup_finished=false;
  if(stackmachine.lex.clean_file.empty())
    stackmachine.lex.clean_need=false;
  else stackmachine.lex.clean_need=true;

  if(init_syscom()!=0) return false;
  install_builtin_symbols();
  return true;
}

void sl_shell::print_usage(const char *comname){
  console->tty_printf("\nSATELLITE Version %s\n", PACKAGE_VERSION);
  console->tty_printf("\n%s [options] [files ...]\n", comname);
  console->tty_printf("\nOPTIONS:\n");
  console->tty_printf("  -clean         : don't read ~/.sl4rc/clean.sl\n");
  console->tty_printf("  -rc            : doesn't read system resouce file\n");
  console->tty_printf("  -setup         : doesn't read ~/.sl4rc/setup.sl\n");
  console->tty_printf("  -help or -h    : print this message\n");
  console->tty_printf("  -              : read instructions from stdin\n");
  printf("\n");
}

bool sl_shell::shell_main(){
  stackmachine.MainLoop();
  return 0;
}
