/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: util.c,v 1.3 2004/07/31 22:27:14 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#define   __EXPORTSYMBOL__

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif

#include  <stdio.h>
#include  <stdlib.h>
#include  <string.h>

#ifdef HAVE_PROCESS_H
# include <process.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_DIRENT_H
# include <dirent.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_DIRECT_H
# include <direct.h>
#endif

#include "libsatellite.h"
#include "path.h"

#ifndef _MAX_PATH
# ifdef  MAXPATHLEN
#  define _MAX_PATH  MAXPATHLEN
# else
#  define _MAX_PATH  1024
# endif
#endif

#define FACMP(m,a) ( ( (m) & (a) ) == (a) )

DLLEXPORT int IsWindows(){
#ifdef HAVE_WINDOWS_H
  return 1;
#else
  return 0;
#endif
}

DLLEXPORT int Mkdir(const char *path){
  char   *dir;
  size_t  i;
  if(path == NULL || *path == '\0') return -1;
  dir = strdup(path);
  if(dir == NULL) return -1;
  for(i=strlen(dir)-1; dir[i] == '/'; i--) dir[i] = '\0';
#ifdef HAVE_WINDOWS_H
  {
    char *p;
    for(p=dir; *p!='\0'; p++) if(*p == '/') *p='\\';
  }
  if(CreateDirectory(dir,NULL) == 0){
     free(dir);
     return -1;
  }
#else
# ifdef HAVE_MKDIR
  if(mkdir(dir, 0755) != 0){
    free(dir);
    return -1;
  }
# else
#  error "mkdir() required"
# endif
#endif
  free(dir);
  return 0;
}

DLLEXPORT int RmdirRecursive(const char *path){
  char buf[_MAX_PATH];
  DIR *dp;
  struct dirent *entry;
 retry:
  if((dp = opendir(path)) == NULL){
    return -1;
  }
  while((entry = readdir(dp)) != NULL){
    if(!strcmp(entry->d_name,".") || !strcmp(entry->d_name,".."))continue;
    if(strlen(path)+strlen(entry->d_name)+2 > _MAX_PATH){
      /* 2 means '/' + '\0' */
      closedir(dp);
      return -1;
    }
    strcpy(buf,path);
    strcat(buf,"/");
    strcat(buf,entry->d_name);
    if(IsDirectory(buf)){
      /* delete directory */
      closedir(dp);
      if(RmdirRecursive(buf) == 0)
        goto retry; /* retry again */
      else
        return -1;  /* error */
    }else{
      /* delete file */
      if(unlink(buf) != 0){
        closedir(dp); /* error */
        return -1;
      }
    }
  }
  closedir(dp);
#ifdef HAVE_RMDIR
  return rmdir(path);
#else
# error "rmdir() required"
#endif
}

DLLEXPORT int IsDirectory(const char *path){
#ifdef HAVE_WINDOWS_H
  DWORD attr;
  attr = GetFileAttributes(path);
  if(attr == -1) return 0;
  if((attr & FILE_ATTRIBUTE_DIRECTORY) == FILE_ATTRIBUTE_DIRECTORY) return 1;
#else
  struct stat sb;
  if(stat(path,&sb) !=0 ) return 0;
  if((sb.st_mode & S_IFDIR) == S_IFDIR) return 1;
#endif
  return 0;
}

DLLEXPORT int Access(const char *path, int mode){
#ifdef HAVE_WINDOWS_H
  DWORD attr, bintype;
  int   tries;
  char *tmp;
  size_t  len;
  int ret;
  len = strlen(path);
  tmp = (char*)malloc(len + 5); /* '5' means ".exe" or ".com" + '\0' */
  if(tmp == NULL) return -1;
  strcpy(tmp,path);
  tries = 0;
 retry:
  attr = GetFileAttributes(tmp);
  tries++;
  if(attr == (DWORD) -1){
    if((GetLastError() == ERROR_FILE_NOT_FOUND) && 
       FACMP(mode,SL_FATTR_XOK)){
      switch(tries){
      case 1: strcat(tmp,".exe"); break;
      case 2: strcpy(tmp,path); strcat(tmp,".com"); break;
      default: goto giveup; break;
      }
      goto retry;
    }
  }
 giveup:
  ret = 0;
  if(attr == (DWORD) -1) ret= -1;
  if(ret == 0){
    if(FACMP(mode, SL_FATTR_WOK) && FACMP(attr, FILE_ATTRIBUTE_READONLY))
      ret = -1;
  }
  if(ret == 0){
    if(FACMP(mode, SL_FATTR_XOK) && !GetBinaryType(tmp, &bintype))
      ret = -1;
  }
  free(tmp);
  return ret;
#else
  int am = 0;
  if(FACMP(mode,SL_FATTR_XOK)){ am |= X_OK; }
  if(FACMP(mode,SL_FATTR_WOK)){ am |= W_OK; }
  if(FACMP(mode,SL_FATTR_ROK)){ am |= R_OK; }
  return access(path, am);
#endif
}

DLLEXPORT int sl4_chdir(const char *path){
#ifdef WIN32
  static const char *separator = "\\";
#else
  static const char *separator = "/";
#endif
  char *path2, *next;
  char  buf[_MAX_PATH];
  if(path == NULL || *path == '\0'){
    if(GetHomeDirectory(buf, _MAX_PATH) == 0){
      path2 = strdup(separator);
    }else{
      path2 = strdup(buf);
    }
  }else{
    /* parse and split first path token */
    path2 = pathname_get_token(path, &next);
    if(path2 != NULL){
      /* convert escaped string to normal string */
      next  = pathname_convert_normal(path2);
      free(path2);
      if(next == NULL) return -1;
      path2 = next;
    }
  }
  if(path2 != NULL){
    if(chdir(path2) == -1){ /* do change directory */
      free(path2); return -1;
    }else{
      /* chdir() command success */
      if(getcwd(buf, 512)!=0){
#ifdef HAVE_SETENV
        setenv("PWD", buf,1);
#else
        char *tmp;
        tmp = (char*)malloc(sizeof(char)*(strlen(buf)+5)); 
        /* 5 means "PWD=" + '\0' */
        if(tmp){
          strcpy(tmp, "PWD=");
          strcat(tmp, buf);
          putenv(tmp);
          free(tmp);
        }
#endif
      }
    }
    free(path2);
  }
  return 0;
}
