/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: tty_console.h,v 1.2 2004/09/24 12:16:23 orrisroot Exp $ */
#ifndef __TTY_CONSOLE_H__
#define __TTY_CONSOLE_H__
#include "declspec.h"

namespace SL_TTY {
  enum COLOR { BLACK, RED, GREEN, YELLOW, BLUE, MAGENTA, CYAN, WHITE };
  enum ATTRIBUTE { 
    ATTR_NORMAL    = (0x00), ATTR_BOLD      = (0x01), 
    ATTR_UNDERLINE = (0x02), ATTR_BLINK     = (0x04),
    ATTR_REVERSE   = (0x08)
  };
  enum CONTROLKEYS {
    CTRL_A = 0x01, CTRL_B = 0x02, CTRL_C = 0x03, CTRL_D = 0x04,
    CTRL_E = 0x05, CTRL_F = 0x06, CTRL_G = 0x07, CTRL_H = 0x08,
    CTRL_I = 0x09, CTRL_J = 0x0a, CTRL_K = 0x0b, CTRL_L = 0x0c,
    CTRL_M = 0x0d, CTRL_N = 0x0e, CTRL_O = 0x0f, CTRL_P = 0x10,
    CTRL_Q = 0x11, CTRL_R = 0x12, CTRL_S = 0x13, CTRL_T = 0x14,
    CTRL_U = 0x15, CTRL_V = 0x16, CTRL_W = 0x17, CTRL_X = 0x18,
    CTRL_Y = 0x19, CTRL_Z = 0x1a, ESCKEY = 0x1b, DELKEY = 0x7f,
  };
  enum FUNCTIONKEYS {
    F1KEY,    F2KEY,    F3KEY,    F4KEY,    F5KEY,    F6KEY,    F7KEY,
    F8KEY,    F9KEY,    F10KEY,   F11KEY,   F12KEY,   HOMEKEY,  ENDKEY,
    PGUPKEY,  PGDNKEY,  UPKEY,    DOWNKEY,  RIGHTKEY, LEFTKEY,  SCRLKKEY, 
    _FUNCTIONKEYS_SIZE
  };
  enum KEYEVENT {
    KEYEVENT_IGNORE,
    KEYEVENT_NEWLINE,
    KEYEVENT_NEWLINE_OR_HELP,
    KEYEVENT_INSERT,
    KEYEVENT_DELETE,
    KEYEVENT_DELETE_OR_EOF,
    KEYEVENT_DELETE_OR_LIST_OR_EOF,
    KEYEVENT_MOVE_BOL,
    KEYEVENT_MOVE_EOL,
    KEYEVENT_MOVE_FORWARD,
    KEYEVENT_MOVE_BACKWARD,
    KEYEVENT_CANCEL,
    KEYEVENT_BACKSPACE,
    KEYEVENT_MARK,
    KEYEVENT_CUT_REGION,
    KEYEVENT_YANK,
    KEYEVENT_KILL_EOL,
    KEYEVENT_TRANS_CHAR,
    KEYEVENT_QUOTE_INPUT,
    KEYEVENT_HISTORY_PREV,
    KEYEVENT_HISTORY_NEXT,
    KEYEVENT_HISTORY_SEARCH_PREV,
    KEYEVENT_HISTORY_SEARCH_NEXT,
    KEYEVENT_COMPLETE_WORD,
    KEYEVENT_LIST_SATCOM,
    _KEYEVENT_SIZE
  };
}

class DECLSPEC tty_console;
typedef int (tty_console::*KEYEVENTFUNC)(int);

class DECLSPEC tty_console {
private:
  bool in_pipe;
  bool use_term_vbell;
  bool input_lock;
  size_t input_lock_len,input_lock_less;
  FILE *current_input;

  // symbol lists
  hash_table_t *commands;

  // input file stack
  sl4_stack_t *input_filept;
  sl4_stack_t *input_lineno;
  sl4_stack_t *input_fname;

  KEYEVENTFUNC kfuncs[SL_TTY::_KEYEVENT_SIZE];
  size_t pos,mark,prompt_length;
  int         col, lineno, cur_prompt_no;
  std::string ebuf, cutbuf, prompt[2];
  char       *file_name;
  history_t  *hist;
  unsigned char keymap[256];
  unsigned char fkeymap[SL_TTY::_FUNCTIONKEYS_SIZE];
  void   editbuf_standby();
  void   editbuf_trim();
  void   editbuf_addnewline();
  void   editbuf_insert(int c, size_t p);
  void   editbuf_delete(size_t p);
  size_t editbuf_length(size_t p);
  void   editbuf_print();
  int    editbuf_is_kanji2(size_t p);
  void cursor_goto_pos(size_t p);
  void _keyevent_ndelete(int n);
  int  keyevent_ignore(int c);
  int  keyevent_newline(int c);
  int  keyevent_newline_or_help(int c);
  int  keyevent_insert(int c);
  int  keyevent_delete(int c);
  int  keyevent_delete_or_eof(int c);
  int  keyevent_delete_or_list_or_eof(int c);
  int  keyevent_move_bol(int c);
  int  keyevent_move_eol(int c);
  int  keyevent_move_forward(int c);
  int  keyevent_move_backward(int c);
  int  keyevent_cancel(int c);
  int  keyevent_backspace(int c);
  int  keyevent_mark(int c);
  int  keyevent_cut_region(int c);
  int  keyevent_yank(int c);
  int  keyevent_kill_eol(int c);
  int  keyevent_trans_char(int c);
  int  keyevent_quote_input(int c);
  int  keyevent_history_prev(int c);
  int  keyevent_history_next(int c);
  int  keyevent_history_search_prev(int c);
  int  keyevent_history_search_next(int c);
  int  keyevent_complete_word(int c);
  int  keyevent_list_satcom(int c);
  void list_print(std::list<std::string> &lst);
  void prompt_print();
  char *edit_readline(char *buf, int size);

protected:
  char *hint_buf;
  SL_TTY::COLOR fg,bg;
  /* terminal control */
  virtual void  term_move_down(int y)=0;
  virtual void  term_move_up(int y)=0;
  virtual void  term_move_right(int x)=0;
  virtual void  term_move_left(int x)=0;
  virtual void  term_move_bol()=0;
  virtual void  term_move_newline()=0;
  virtual void  term_bell(){}
  virtual void  term_vbell(){}
  virtual void  term_clear_screen();
  virtual void  term_clear_eol(int x)=0;
  virtual void  term_set_attr_normal(){}
  virtual void  term_set_attr_bold(){}
  virtual void  term_set_attr_reverse(){}
  virtual void  term_set_attr_underline(){}
  virtual bool  term_has_color(){ return false; }
  virtual void  term_putc(int c)=0;
  virtual void  term_print(const char *str);
  virtual int   term_getc()=0;
  virtual char* term_gets(char *buf, int size)=0;
  virtual void  term_edit_mode(){}
  virtual void  term_normal_mode(){}
  virtual int   term_keypad_getc(){ return term_getc(); }
  virtual void  term_show_help(const char *str);
  virtual void  term_hide_help();

public:
  sl4_fd_t ifd,ofd,efd;
  tty_console();
  virtual ~tty_console();
  bool pipe_status()const{return in_pipe;}
  void pipe_status_set(bool stat){in_pipe=stat;}
  void satcom_db_add(module_command_t* cmd);
  void satcom_db_del(module_command_t* cmd);
  char *readline(char *buf, int size, int pn);
  int  input_file_push(const char *fname);
  int  input_file_pop();

  bool is_tty()const  { return  (file_name==0); }
  bool is_file()const { return  (file_name!=0); }
  const char *get_filename()const { return file_name; }
  char *get_filepath(size_t len, char *path);
  int  get_lineno()const { return lineno; }

  /* for build-in functions */
  void ls_command(const char *path, const char *fname, const char *arg);
  void history_listup();
  void history_resize(int size);
  void prompt_set(const char *format, int n);
  void prompt_get(char *format, int size, int n);
  void set_keybind(unsigned char key, SL_TTY::KEYEVENT e){ keymap[key]=e; }
  void set_keybind_default();

  /* tty i/o method */
  void  tty_set_attr(SL_TTY::ATTRIBUTE a);
  void  tty_printf(const char *fmt, ...);
  void  tty_print(const char *mes);
  void  tty_putc(int c);
  char* tty_gets(char *buf, int size);
  int   tty_getc();
  void  tty_cursor_move_left(int x){
    term_edit_mode(); term_move_left(x); term_normal_mode();
  }
  void  tty_cursor_move_right(int x){
    term_edit_mode(); term_move_right(x); term_normal_mode();
  }
  void  tty_cursor_move_up(int y){
    term_edit_mode(); term_move_up(y); term_normal_mode();
  }
  void  tty_cursor_move_down(int y){
    term_edit_mode(); term_move_down(y); term_normal_mode();
  }

  /* terminal control */
  virtual void term_flush(sl4_fd_t fd){}
  virtual void term_set_fgcolor(SL_TTY::COLOR color){}
  virtual void term_set_bgcolor(SL_TTY::COLOR color){}
  virtual void term_getmaxyx(int *y, int* x)=0;

  /* code error */
  void execerror(const char *str1, const char *str2);
  void warning(const char *str1, const char *str2);

  /* run pager & editor */
  int run_pager(const char *file);
  int run_editor(const char *file);

};
#endif
