/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: syscom.c,v 1.1.1.1 2004/03/31 08:15:07 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
# define FOPEN_READ  "r"
# define FOPEN_WRITE "w"
#endif
#ifdef HAVE_IO_H
# include <io.h>
# define FOPEN_READ  "rb"
# define FOPEN_WRITE "wb"
#endif

#include "libsatellite.h"

#ifndef _MAX_PATH
# ifdef MAXPATHLEN
#  define _MAX_PATH MAXPATHLEN
# else
#  define _MAX_PATH 1024
# endif
#endif

#define emalloc(x) malloc(x)
#define efree(x)   free(x)

#define SYSTEM_COMMON_FILE_NAME  "SYSTEM_COMMON.tmp"
#define SL4_TEMPDIR_ENV          "SATELLITE4_TEMPORARY_DIRECTORY"

/* initial values */
#define INITIAL_COMPAT2X_MODE    0 /* disable compat2x mode */
#define INITIAL_SL2_BUFF_LENG    2048
#define INITIAL_SL2_SAM_FREQ     1000.0
#define INITIAL_SL2_A_DATA_SIZE  4 /* sizeof(float) */
#define INITIAL_SL2_DATA_TYPE    SL2_DATAFILE_TYPE_BIGENDIAN /* SPARC, EWS */

extern char *GetTempDirectory(char *buf, int len);

/* system common data */
typedef struct _system_common_t {
  int    compat2x_mode;
  char   temp_dir[_MAX_PATH];
  long   sl2_buff_leng;
  double sl2_sam_freq;
  int    sl2_a_data_size;
  int    sl2_data_type;
} system_common_t;

static system_common_t *syscom = NULL;
static int              syscom_master = 1;

static void   fatal_error();
static int    set_syscom_filepath(char *path, size_t len, int is_child);

static int    init_syscom_sub();

DLLEXPORT int init_syscom(){
  if(init_syscom_sub() != 0) return -1;
  GetTempDirectory(syscom->temp_dir, _MAX_PATH);
  return 0;
}

DLLEXPORT int free_syscom(){
  if(syscom == NULL) return -1;
  if(syscom_master == 1)
    RmdirRecursive(syscom->temp_dir);
  efree(syscom);
  syscom=NULL;
  return 0;
}

DLLEXPORT int load_syscom_file(){
  /* call by child process */
  FILE *fp;
  char path[_MAX_PATH];
  size_t nread;
  if(syscom != NULL) return -1;
  if(init_syscom_sub() != 0) return -1;
  syscom_master = 0;
  if(set_syscom_filepath(path, _MAX_PATH, 1) != 0){
    free_syscom(); return -1;
  }
  fp=fopen(path, FOPEN_READ);
  if(fp == NULL){ 
    free_syscom(); return -1; 
  }
  nread = fread(syscom, sizeof(system_common_t), 1, fp);
  fclose(fp);
  if(nread != 1){
    free_syscom(); return -1; 
  }
  return 0;
}

DLLEXPORT int store_syscom_file(){
  /* call by parent process */
  FILE *fp;
  char path[_MAX_PATH];
  size_t nwrite;
  if(set_syscom_filepath(path, _MAX_PATH, 0) != 0) return -1;
  fp=fopen(path,FOPEN_WRITE);
  if(fp == NULL) return -1;
  nwrite = fwrite(syscom, sizeof(system_common_t), 1, fp);
  fclose(fp);
  if(nwrite != 1) return -1;
#ifdef HAVE_SETENV
  setenv(SL4_TEMPDIR_ENV, syscom->temp_dir, 1);
#else
  {
    /* 
       warning : 
        maybe, if putenv() specification is 4.3BSD like then it dose 
        not work.. because, the environment variable setted by putenv()
        is not influence for child process.
    */
    static char env[_MAX_PATH];
    if(strlen(syscom->temp_dir) + strlen(SL4_TEMPDIR_ENV) + 2 > _MAX_PATH)
       return -1;
    sprintf(env, "%s=%s", SL4_TEMPDIR_ENV, syscom->temp_dir); /* safe */
    putenv(env);
  }
#endif
  return 0;
}

DLLEXPORT int  unlink_syscom_file(){
  char path[_MAX_PATH];
  if(set_syscom_filepath(path, _MAX_PATH, 0) != 0) return -1;
  if(unlink(path) != 0) return -1;
#ifdef HAVE_SETENV
  unsetenv(SL4_TEMPDIR_ENV);
#else
  {
    /* 
       warning : 
        maybe, if putenv() specification is 4.3BSD like then it dose 
        not work.. because, the environment variable setted by putenv()
        is not influence for child process.
    */
    static char env[_MAX_PATH];
    if(strlen(SL4_TEMPDIR_ENV) + 2 > _MAX_PATH) return -1;
    sprintf(env, "%s=", SL4_TEMPDIR_ENV); /* safe */
    putenv(env);
  }
#endif
  return 0;
}

/* SATELLITE 2.9x compatible mode */
DLLEXPORT void set_compat2x_mode(int f){
  if(syscom == NULL) fatal_error();
  if(f != 0) f = 1;
  syscom->compat2x_mode = f;
}

DLLEXPORT int is_compat2x_mode(){
  if(syscom == NULL) fatal_error();
  return syscom->compat2x_mode;
}

DLLEXPORT int set_sampling(double freq){
  if(syscom == NULL) fatal_error();
  syscom->sl2_sam_freq = freq;
  return 0;
}

DLLEXPORT double get_sampling(){
  if(syscom == NULL) fatal_error();
  return syscom->sl2_sam_freq;
}

DLLEXPORT int ChangeDataType(int type){
  if(syscom == NULL) fatal_error();
  switch(type) {
  case 0: syscom->sl2_data_type = SL2_DATAFILE_TYPE_BIGENDIAN;    break;
  case 1: syscom->sl2_data_type = SL2_DATAFILE_TYPE_LITTLEENDIAN; break;
  case 2: syscom->sl2_data_type = SL2_DATAFILE_TYPE_BIGENDIAN;    break;
  case 3: syscom->sl2_data_type = SL2_DATAFILE_TYPE_LITTLEENDIAN; break;
  default: break;
  }
  return syscom->sl2_data_type;
}

DLLEXPORT int ChangeDataSize(int siz){
  if(syscom == NULL) fatal_error();
  if (siz == 2 || siz == 4 || siz == 8)
    syscom->sl2_a_data_size = siz;
  return syscom->sl2_a_data_size;
}

DLLEXPORT int sl2_get_data_size(){
  if(syscom == NULL) fatal_error();
  return syscom->sl2_a_data_size;
}

DLLEXPORT int sl2_get_data_type(){
  if(syscom == NULL) fatal_error();
  return syscom->sl2_data_type;
}

/*
int set_tmpdir(char *dir){
  strcpy(syscom->temp_dir, dir);
  return 0;
}
*/

const char *get_tmpdir(){
  if(syscom == NULL) return NULL;
  return syscom->temp_dir;
}

static void fatal_error(){
  fprintf(stderr, "fatal : system common area is not initialized\n");
  if(syscom) RmdirRecursive(syscom->temp_dir);
  exit(1);
}

static int init_syscom_sub(){
  if(syscom != NULL) return -1;
  syscom = (system_common_t*)emalloc(sizeof(system_common_t));
  if(syscom == NULL) return -1;
  memset(syscom, 0, sizeof(system_common_t));
  syscom->compat2x_mode   = INITIAL_COMPAT2X_MODE;
  syscom->sl2_buff_leng   = INITIAL_SL2_BUFF_LENG;
  syscom->sl2_sam_freq    = INITIAL_SL2_SAM_FREQ;
  syscom->sl2_a_data_size = INITIAL_SL2_A_DATA_SIZE;
  syscom->sl2_data_type   = INITIAL_SL2_DATA_TYPE;
  return 0;
}

static int set_syscom_filepath(char *path, size_t len, int is_child){
  static size_t syscom_file_name_len;
  size_t tmpdir_len;
  char *tmp;
  syscom_file_name_len = strlen(SYSTEM_COMMON_FILE_NAME);
  if(is_child == 0){
    if(syscom == NULL) return -1;
    tmp = syscom->temp_dir;
  }else{
    tmp = getenv(SL4_TEMPDIR_ENV);
  }
  if(tmp == NULL) return -1;
  tmpdir_len = strlen(tmp);
  if(tmpdir_len >= len) return -1;
  if(tmpdir_len + syscom_file_name_len + 2 > _MAX_PATH) return -1;
  sprintf(path,"%s/%s", tmp, SYSTEM_COMMON_FILE_NAME); /* safe */
  return 0;
}
