/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: path.c,v 1.3 2005/01/25 12:28:39 orrisroot Exp $ */
#define LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_DIRENT_H
# include <dirent.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif

#include "libsatellite.h"
#include "path.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef struct _path_hash_node_t {
  char *full;
  char *fname;
  size_t      full_len;
  size_t      fname_len;
} path_hash_node_t;

#ifdef WIN32
static const char *delimiter = ";";
static const char *separator = "\\";
#define lstat(x,y) stat((x),(y))
#define S_IXUSR    S_IEXEC
#else
static const char *delimiter = ":";
static const char *separator = "/";
#endif

static path_hash_node_t *path_hash_node_new(char *full);
static void path_hash_node_del(void *void_data, void *void_arg);

/* commands of path environment hash */
static hash_table_t *path_hash = NULL;

/***********************************/
/* path environment hash operation */
/***********************************/
int path_hash_build(){
  DIR *dirp;
  struct dirent *dp;
  char *env, *env2;
  char *token, *last, *full;
  size_t len;
  path_hash_node_t *node;
  /* initialize */
  if(path_hash != NULL){
    hash_table_erase(path_hash, path_hash_node_del, NULL);
  }else{
    path_hash = hash_table_str_new();
    if(path_hash == NULL) return -1;
  }
  /* insert PATH environment to hash table */
  env = getenv("PATH");
  if(env == NULL) return -1;
  env2 = strdup(env);
  token = strutil_strtok_r(env2, delimiter, &last);
  while(token != NULL){
    dirp = opendir(token);
    len = strlen(token);
    if(dirp != 0){
      while((dp = readdir(dirp)) != 0){
        if(!strcmp(dp->d_name, ".") || !strcmp(dp->d_name, "..")) continue;
#ifdef WIN32
        {
          char *ext;
          ext = strrchr(dp->d_name,'.');
          if(ext == NULL)
            continue;
          if(!(!stricmp(ext,".exe") ||
               !stricmp(ext,".com") ||
               !stricmp(ext,".bat")))
            continue;
        }
#endif
        full = (char*)malloc(len+strlen(dp->d_name)+ 2);
        if(full == NULL) continue;
        strcpy(full, token); strcat(full, separator); strcat(full, dp->d_name);
//#ifdef WIN32
//        {
//          char *p;
//          for(p=full;*p!='\0';p++) if(*p=='\\') *p = '/';
//        }
//#endif
        if(IsDirectory(full)){ free(full); continue; }
        node = path_hash_node_new(full);
        if(node == NULL){ free(full); continue; }
        /* if(Access(full, SL_FATTR_XOK) == 0){ */
          if(hash_table_insert(path_hash,full,node) != 0){ free(full); }
        /* } */
      }
      closedir(dirp);
    }
    token = strutil_strtok_r(NULL, delimiter, &last);
  }
  free(env2);
  return 0;
}

void path_hash_delete(){
  if(path_hash != NULL){
    hash_table_delete(path_hash, path_hash_node_del, NULL);
    path_hash = NULL;
  }
}

char *path_hash_lookup(const char *bin){
  size_t  len;
  char   *full, *path_env, *path_env2, *token, *last;

  /* Is bin full path ? */
  full = strdup(bin);
  if(full == NULL) return NULL;
  if(hash_table_lookup(path_hash, full)!=NULL) return full;
  free(full);
  full = NULL;

  /* search "PATH" environment */
  len = strlen(bin);
  path_env = getenv("PATH");
  path_env2 = strdup(path_env);
  if(path_env2 == 0) return NULL;
  token = strutil_strtok_r(path_env2,delimiter,&last);
  while(token != NULL){
    full = (char*)malloc(sizeof(char) * (strlen(token) + len + 2));
    if(full == NULL) return NULL;
    /* build full length commands */
    strcpy(full,token); strcat(full,separator); strcat(full,bin);
//#ifdef WIN32
//    {
//      char *p;
//      for(p=full;*p!='\0';p++) if(*p=='\\') *p = '/';
//    }
//#endif
    if(hash_table_lookup(path_hash, full)!=NULL) break;
    free(full);
    full = NULL;
    token = strutil_strtok_r(NULL, delimiter, &last);
  }
  free(path_env2);
  return full;
}

static path_hash_node_t *path_hash_node_new(char *full){
  path_hash_node_t *node;
  node = (path_hash_node_t*)malloc(sizeof(path_hash_node_t));
  if(node == NULL) return NULL;
  node->full = full;
  node->full_len = strlen(node->full);
  node->fname = strrchr(full, *separator);
  //node->fname = strrchr(full, '/');
  node->fname_len = strlen(node->fname);
  return node;
}

static void path_hash_node_del(void *void_data, void *void_arg){
  path_hash_node_t *node;
  node = (path_hash_node_t*)void_data;
  if(node->full)
    free(node->full);
  free(node);
}


/*****************************/
/* pathname string operation */
/*****************************/

char *pathname_convert_escape(const char *str){
  const char *p;
  char *ret;
  size_t i,len,esc;
  len = strlen(str);
  /* count up escaped length */
  esc = 0;
  for(p=str;*p!='\0';p++)
    if(*p == '\\' || *p == ' ' || *p == '"' || *p == '\'') esc++;
  /* memory allocation */
  ret = (char*)malloc(sizeof(char)*(len+esc+1));
  if(ret == NULL) return NULL;
  /* do convert */
  i = 0;
  for(p=str;*p!='\0';p++){
    if(*p == '\\' || *p == ' ' || *p == '"' || *p == '\''){
      ret[i] = '\\';
      i++; 
    }
    ret[i] = *p;
    i++;
  }
  ret[len+esc]='\0';
  return ret;
}

char *pathname_convert_normal(const char *str){
  int shift;
  int in_squote  = 0;
  int in_dquote  = 0;
  int in_escape = 0;
  size_t total, i, j;
  char *ret;
  total = strlen(str);
  ret = strdup(str);
  if(ret == NULL) return NULL;
  for(i=0;i<total;i++){
    if(in_dquote && in_escape){
      /* inside of 'quote' and previous char is 'back slash' - "\ */
      in_escape = 0;
      if(ret[i] == '\\' || ret[i] == '"'){ shift = 1; }else{ shift = 2; }
      total -= shift; i -= 1;
      for(j=0;j<total-i+1;j++){ ret[i+j] = ret[i+j+shift]; }
    }else if(in_dquote){
      /* inside of 'quote' " */
      if(ret[i] == '"'){
        in_dquote = 0;
        shift = 1; total -= shift;
        for(j=0;j<total-i+1;j++){ ret[i+j] = ret[i+j+shift]; }
      }else if(ret[i] == '\\'){
#ifndef WIN32
        in_escape = 1;
#endif
      }
    }else if(in_escape){
      /* outside of 'quote' and previous char is 'back slash' - \ */
      in_escape = 0;
      if(ret[i] == '\\' || ret[i] == '\'' || ret[i] == '"' || ret[i] == ' '){
        shift = 1;
      }else{
        shift = 2;
      }
      total -= shift; i -= 1;
      for(j=0;j<total-i+1;j++){ ret[i+j] = ret[i+j+shift]; }
    }else if(in_squote){
      if(ret[i] == '\''){
        in_squote = 0;
        shift = 1; total -= shift;
        for(j=0;j<total-i+1;j++){ ret[i+j] = ret[i+j+shift]; }
      }
    }else{
      shift = 0;
      /* normal condition */
      if(ret[i] == '\\'){
#ifndef WIN32
        in_escape = 1;
#endif
      }else if(ret[i] == '\''){
        in_squote = 1;
        shift = 1;
      }else if(ret[i] == '"'){
        in_dquote = 1;
        shift = 1;
      }
      if(shift > 0){
        total -= shift;
        for(j=0;j<total-i+1;j++){ ret[i+j] = ret[i+j+shift]; }
      }
    }
  }
  if(in_escape || in_squote || in_dquote || total == 0 ){
    free(ret);
    return NULL;
  }
  return ret;
}

/*  c:/test/hoge\ files    */
/*  "c:/test/hoge files"   */
/*  c:\\test\\hoge\ files  */
char *pathname_get_token(const char *str, const char **next){
  int status;
  size_t i,len;
  const char *p;
  char *ret;
  /* skip space */
  for(p=str; *p==' '; p++);
  str = p;
  /* do calculation */
  len = strutil_check_quote(str,' ',&status);
  if(status != 0){ *next = NULL; return NULL; }
  /* memory allocation */
  ret = (char*)malloc(sizeof(char) * (len+1));
  if(ret == NULL){ *next = NULL; return NULL; }
  /* copy string */
  for(i=0; i<len; i++) ret[i] = str[i];
  ret[i] = '\0';
  /* set next char */
  *next = str + len;
  /* skip space */
  for(p=*next; *p==' '; p++);
  if(*p == '\0'){
    *next = NULL;
  }else{
    *next = p;
  }
  return ret;
}

char *pathname_get_executable(const char *bin){
#ifdef WIN32
  char *token, *ext, *last, *tmp_bin;
  const char *pathext = ".exe;.EXE;.com;.COM;.bat;.BAT";
  char *full;
  struct stat sb;
  if(bin == NULL) return 0;
  /* check full path */
  if(lstat(bin, &sb) == 0){
    if(sb.st_mode & S_IFDIR){
      return NULL; /* bin is directory */
    }
    if(sb.st_mode & S_IXUSR){
      full = strdup(bin);
      if(full != NULL){
        return full; /* OK! bin is executable command */
      }
    }
  }
   ext = strdup(pathext);
  if(ext == NULL){ return NULL; }
  token = strutil_strtok_r(ext, delimiter, &last);
  tmp_bin = (char*)malloc(sizeof(char)*(strlen(bin) + 5));
  /* 5 means strlen(ext) + '\0' */
  while(token!=NULL){
    strcpy(tmp_bin, bin);
    strcat(tmp_bin, token);
    if(lstat(tmp_bin, &sb) == 0){
      if(sb.st_mode & S_IFDIR){
        return NULL; /* bin is directory */
      }
      if(sb.st_mode & S_IXUSR){
        full = strdup(bin);
        if(full != NULL){
          return full; /* OK! bin is executable command */
        }
      }
    }
    token = strutil_strtok_r(NULL, delimiter, &last);
  }
  free(tmp_bin);
  free(ext);
  /* search path environment */
  full = path_hash_lookup(bin);
  if(full == NULL){
    ext = strdup(pathext);
    if(ext == NULL){ return NULL; }
    token = strutil_strtok_r(ext, delimiter, &last);
    tmp_bin = (char*)malloc(sizeof(char)*(strlen(bin) + 5));
    /* 5 means strlen(ext) + '\0' */
    while(token!=NULL){
      strcpy(tmp_bin, bin);
      strcat(tmp_bin, token);
      full = path_hash_lookup(tmp_bin);
      if(full != NULL){
        break;
      }
      token = strutil_strtok_r(NULL, delimiter, &last);
    }
    free(tmp_bin);
    free(ext);
  }
  return full;
#else
  char *full;
  struct stat sb;
  if(bin == NULL) return 0;
  /* check full path */
  if(lstat(bin, &sb) == 0){
    if(sb.st_mode & S_IFDIR){
      return NULL; /* bin is directory */
    }
    if(sb.st_mode & S_IXUSR){
      full = strdup(bin);
      if(full != NULL){
        return full; /* OK! bin is executable command */
      }
    }
  }
  /* search path environment */
  full = path_hash_lookup(bin);
  return full;
#endif
}

DLLEXPORT int is_executable_command(const char *bin, char *full, size_t len){
  char *bin2;
  char *bin3;
  /* convert bin to normal (no escaped mode) */
  bin2 = pathname_convert_normal(bin);
  if(bin2 == NULL) return 0;
  /* get executable name */
  bin3 = pathname_get_executable(bin2);
  free(bin2);
  if(bin3 == NULL) return 0;
  if(full != NULL){
    strncpy(full, bin3, len);
    full[len-1] = '\0';
  }
  free(bin3);
  return 1;
}

#ifdef __cplusplus
}
#endif
