/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: module.cpp,v 1.5 2004/07/29 18:07:53 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#ifdef HAVE_DLFCN_H
# include <dlfcn.h>
#endif

#include "libsatellite.h"

#define  __EXPORTSYMBOL__
#include "module.h"
#undef   __EXPORTSYMBOL__

#ifndef HAVE_SNPRINTF
# error "snprintf() required"
#endif
#ifndef HAVE_STRDUP
# error "strdup required"
#endif

#define emalloc(x) malloc(x)
#define efree(x)   free(x)
#define estrdup(x) strdup(x)

#define TMPBUFSIZE 256

#define PARSE_STAT_COMMENT  1
#define PARSE_STAT_SUCCESS  0
#define PARSE_STAT_ERROR   -1
#define PARSE_STAT_OUTMEM  -2
#define PARSE_STAT_INSMIS  -3

#define SETFUNC_STAT_SUCCESS    0
#define SETFUNC_STAT_NOLIBRARY -1
#define SETFUNC_STAT_NOSYMBOL  -2

#define INIFILE_SECTION_FATAL   -1
#define INIFILE_SECTION_NOTHING  0
#define INIFILE_SECTION_COMMAND  1
#define INIFILE_SECTION_MESSAGE  2
#define INIFILE_SECTION_ERROR    3

typedef struct _module_command_finalchk_args_t {
  module_t *module;
  int status;
} module_command_finalchk_args_t;

typedef struct _module_command_setfunc_args_t {
  void *library;
  int   status;
  const char *mes;
} module_command_setfunc_args_t;

/* private function of module parse utilities */
static void   space_skip(const char *str, size_t len, size_t *pos);
static int    comma_skip(const char *str, size_t len, size_t *pos);
static int    get_number(const char *str, size_t len, size_t *pos, int *ret);
static int    get_string(const char *str, size_t len, size_t *pos, 
                         char *ret, size_t retlen);
/* private function of module_message_t */
static module_message_t *_module_message_new();
static void _module_message_free(module_message_t *mes);
static int  _module_message_parse(hash_table_t *mess, const char *str);
static void _module_message_free_for(void *void_mes, void *void_arg);

/* pravate function of module_command_t */
static module_command_t *_module_command_new();
static void _module_command_free(module_command_t *com);
static int  _module_command_parse(hash_table_t *coms, const char *str);
static void _module_command_free_for(void *void_com, void *void_arg);
static void _module_command_finalchk_for(void *void_cmd, void *void_arg);
static void _module_command_setfunc_for(void *void_cmd, void *void_arg);

/* private function of module_t */
typedef struct _module_parse_cond_t {
  int cmd_in;
  int mes_in;
  int err_in;
} module_parse_cond_t;

static int _module_parse(const char *str, module_parse_cond_t *cond);
static int _module_section_begin_parse(const char *str);
static int _module_section_end_parse(const char *str);
static int _module_set_command_func(module_t *mod);
static int _module_set_setup_func(module_t *mod);
static int _module_inifile_read(module_t *mod);

/* module callback event */
static module_event_callback_t module_cb_func = (module_event_callback_t)NULL;

DLLEXPORT module_t *module_new(const char *name){
  module_t *mod;
  char     *mod_name;
  mod = (module_t*)emalloc(sizeof(module_t));
  if(mod == NULL) return NULL;
  mod_name = estrdup(name);
  if(mod_name == NULL){
    efree(mod); return NULL;
  }
  mod->messages = hash_table_int_new();
  mod->errors   = hash_table_int_new();
  mod->commands = hash_table_str_new();
  if(mod->messages == NULL || mod->errors == NULL || mod->commands == NULL){
    if(mod->messages)
      hash_table_delete(mod->messages,_module_message_free_for,NULL);
    if(mod->errors) 
      hash_table_delete(mod->errors,  _module_message_free_for,NULL);
    if(mod->commands)
      hash_table_delete(mod->commands,_module_command_free_for,NULL);
    efree(mod); return NULL;
  }
  mod->module_name = mod_name;
  mod->err_mesg = NULL;
  mod->dll_file = NULL;
  mod->ini_file = NULL;
  mod->main_function  = NULL;
  mod->init_function  = NULL;
  mod->clean_function = NULL;
  mod->library   = NULL;
  mod->is_loaded = 0;
  if(module_cb_func)
    module_cb_func(MODULE_EVENT_NEW, mod);
  return mod;
}

DLLEXPORT void module_delete(module_t *mod){
  if(mod->is_loaded) module_unload(mod);
  if(module_cb_func)
    module_cb_func(MODULE_EVENT_DELETE, mod);
  hash_table_delete(mod->messages, _module_message_free_for, NULL);
  hash_table_delete(mod->errors, _module_message_free_for, NULL);
  hash_table_delete(mod->commands,_module_command_free_for,NULL);
  if(mod->dll_file) efree(mod->dll_file);
  if(mod->ini_file) efree(mod->ini_file);
  if(mod->err_mesg) efree(mod->err_mesg);
  if(mod->module_name) efree(mod->module_name);
  efree(mod);
}

DLLEXPORT int module_unload(module_t *mod){
  int unload_ok = 1;
  if(mod->err_mesg){ efree(mod->err_mesg); mod->err_mesg = NULL; }
  if(mod->is_loaded == 0) return 0; /* ok */
  if(module_cb_func)
    module_cb_func(MODULE_EVENT_UNLOAD, mod);
  /* TODO remove satellite command name from symbol table */
  hash_table_erase(mod->messages,_module_message_free_for, NULL);
  hash_table_erase(mod->errors,_module_message_free_for, NULL);
  hash_table_erase(mod->commands,_module_command_free_for, NULL);
  if(mod->clean_function) mod->clean_function();
  if(mod->main_function)
    unload_ok = mod->main_function(SL4_MODULE_DETACH);
  if(mod->library && unload_ok ){
    dlclose(mod->library); mod->library = NULL; 
  }
  mod->is_loaded = 0;
  mod->main_function  = NULL;
  mod->init_function  = NULL;
  mod->clean_function = NULL;
  return 0;
}

DLLEXPORT int module_load(module_t *mod){
  if(mod->err_mesg){ efree(mod->err_mesg); mod->err_mesg = NULL; }
  /* check. is module already loaded ? */
  if(mod->is_loaded){ return 0; /* OK */ }
  /* check. is ini file defined */
  if(mod->ini_file == NULL){
    mod->err_mesg = estrdup("not defined ini file."); return -1;
  }
  /* read command definition from ini file */
  if(_module_inifile_read(mod) != 0){ return -1; }
  /* load dynamic load library */
  if(mod->dll_file != NULL){
    mod->library = dlopen(mod->dll_file, RTLD_LAZY);
    if(mod->library == NULL){ /* fail to open dynamic library */
      mod->err_mesg = estrdup(dlerror()); return -1;
    }
  }
  /* set function pointer */
  if(_module_set_command_func(mod)!=0) return -1;
  if(_module_set_setup_func(mod)!=0)   return -1;

  /* do setup */
  mod->is_loaded = 1;
  if(mod->main_function)
    if(mod->main_function(SL4_MODULE_ATTACH) == 0){
      module_unload(mod); 
      mod->err_mesg = estrdup("FALSE returned from module main function.");
      return -1;
    }
  if(mod->init_function)
    if(mod->init_function()){
      module_unload(mod);
      mod->err_mesg = estrdup("FALSE returned from module setup function.");
      return -1;
    }
  
  if(module_cb_func)
    module_cb_func(MODULE_EVENT_LOAD, mod);
  return 0;
}

DLLEXPORT module_command_t *module_lookup(module_t *mod, const char *str){
  return (module_command_t*)hash_table_lookup(mod->commands, str);
}

DLLEXPORT int module_set_dll(module_t *mod, const char *file){
  char *name = NULL;
  if(file != NULL)
    name = estrdup(file);
  if(name == NULL) return -1;
  if(mod->dll_file) efree(mod->dll_file);
  mod->dll_file = name;
  return 0;
}

DLLEXPORT int module_set_ini(module_t *mod, const char *file){
  char *name = NULL;
  if(file != NULL)
    name = estrdup(file);
  if(name == NULL) return -1;
  if(mod->ini_file) efree(mod->ini_file);
  mod->ini_file = name;
  return 0;
}

/* private function of module parse utilities */
static void space_skip(const char *str, size_t len, size_t *pos){
  char c;
  int  in_cmt;
  size_t i;
  in_cmt=0;
  for(i=*pos;i<len;i++){
    c=str[i];
    /* some script language style comment */
    if(in_cmt==0 && ( c=='#' || c=='!' )){ i=len; break; }
    /* end of C style comment */
    if(in_cmt==1 && c=='*' && i+1<len && str[i+1]=='/'){
      i++; in_cmt=0; continue;
    }
    if( in_cmt==0 && c=='/' ){
      /* begin of C style comment */
      if(i+1<len && str[i+1]=='*'){ i++; in_cmt=1; continue; }
      /* c++ style commnent */
      if(i+1<len && str[i+1]=='/'){ i=len; break; }
    }
    /* break other word */
    if(in_cmt==0 && c!=' ' && c!='\t' && c!='\r' && c!='\n') break;
  }
  *pos = i;
  return;
}

static int comma_skip(const char *str, size_t len, size_t *pos){
  space_skip(str,len,pos);
  if(str[(*pos)]!=',') return -1;
  (*pos)++;
  space_skip(str,len,pos);
  return 0;
}

static int get_number(const char *str, size_t len, size_t *pos, int *ret){
  long  num;
  const char *begin;
  char *end;
  begin = &str[*pos];
  num = strtol(begin,&end,10);
  if(errno == ERANGE) return -1;
  (*ret) = (int)num;
  (*pos)  += end-begin;
  return 0;
}

static int get_string(const char *str, size_t len, size_t *pos, char *ret,
                      size_t retlen){
  size_t i=0;
  char c;
  if(len==*pos || str[*pos]!='"') return -1;
  for((*pos)++;(*pos)<len;(*pos)++){
    if(i >= retlen) return -1;
    c=str[*pos];
    if(c=='\\'){
      if((*pos)+1==len) return -1;
      switch(str[(*pos)+1]){
      case '\"' : ret[i++]='\"'; break;
      case 'n'  : ret[i++]='\n'; break;
      case 'r'  : ret[i++]='\r'; break;
      case 't'  : ret[i++]='\t'; break;
      case '\\' : ret[i++]='\\'; break;
      case '\'' : ret[i++]='\''; break;
      }
      (*pos)++;
      continue;
    }
    if(c=='\"'){ (*pos)++; ret[i]='\0'; return 0; }
    ret[i++]=c;
  }
  return -1;
}

/* private function of module_message_t */
static module_message_t *_module_message_new(){
  module_message_t *mes;
  mes = (module_message_t*)emalloc(sizeof(module_message_t));
  if(mes == NULL) return NULL;
  mes->message = NULL;
  mes->number  = 0;
  return mes;
}

static void _module_message_free(module_message_t *mes){
  if(mes->message != NULL) efree(mes->message);
  efree(mes);
}

static int _module_message_parse(hash_table_t *mess, const char *str){
  char buf[TMPBUFSIZE],*tmp;
  int num;
  size_t i,len;
  module_message_t *mes;
  i=0;
  len=strlen(str);
  space_skip(str,len,&i);
  if(i==len) return PARSE_STAT_COMMENT;
  if(get_number(str,len,&i,&num) != 0) return PARSE_STAT_ERROR;
  if(comma_skip(str,len,&i) != 0) return PARSE_STAT_ERROR;
  if(get_string(str,len,&i,buf,TMPBUFSIZE) != 0) return PARSE_STAT_ERROR;
  tmp = (char*)emalloc(strlen(buf)+1);
  if(tmp == NULL) return PARSE_STAT_OUTMEM;
  strcpy(tmp, buf); /* safe */
  mes = _module_message_new();
  if(mes == NULL){ efree(tmp); return PARSE_STAT_OUTMEM; }
  mes->message = tmp;
  mes->number  = num;
  if(hash_table_insert(mess, &(mes->number), mes) == -1){
    _module_message_free(mes);
    return PARSE_STAT_INSMIS;
  }
  return PARSE_STAT_SUCCESS;
}

static void _module_message_free_for(void *void_mes, void *void_arg){
  _module_message_free((module_message_t*)void_mes);
}

/* private function of module_command_t */
static module_command_t *_module_command_new(){
  module_command_t *com;
  com = (module_command_t*)emalloc(sizeof(module_command_t));
  if(com == NULL) return NULL;
  com->command_name  = NULL;
  com->function_name = NULL;
  com->argc = 0;
  com->argv = NULL;
  com->message_number = NULL;
  com->function = NULL;
  com->messages = NULL;
  return com;
}

static void _module_command_free(module_command_t *com){
  int i;
  if(com->command_name != NULL) efree(com->command_name);
  if(com->function_name != NULL) efree(com->function_name);
  if(com->argv != NULL){
    for(i=0;i<com->argc;i++)
      if(com->argv[i] != NULL) efree(com->argv[i]);
    efree(com->argv);
  }
  if(com->message_number != NULL) efree(com->message_number);
  if(com->messages != NULL) efree(com->messages);
  efree(com);
}

static int  _module_command_parse(hash_table_t *coms, const char *str){
  module_command_t *com;
  char   buf[TMPBUFSIZE];
  size_t i,len;
  int    j;
  i=0;
  len=strlen(str);
  space_skip(str,len,&i);
  if(i==len) return PARSE_STAT_COMMENT;
  com = _module_command_new();
  if(com == NULL) return PARSE_STAT_OUTMEM;
  /* parse command name */
  if(get_string(str,len,&i,buf,TMPBUFSIZE) != 0){
    _module_command_free(com); return PARSE_STAT_ERROR;
  }
  com->command_name = (char*)emalloc(strlen(buf)+1);
  if(com->command_name == NULL){
    _module_command_free(com); return PARSE_STAT_OUTMEM;
  }
  strcpy(com->command_name, buf); /* safe */

  /* seek to next parameter and parse function name */
  if(comma_skip(str,len,&i) != 0 || 
     get_string(str,len,&i,buf,TMPBUFSIZE) != 0){
    _module_command_free(com); return PARSE_STAT_ERROR;
  }
  com->function_name = (char*)emalloc(strlen(buf)+1);
  if(com->function_name == NULL){
    _module_command_free(com); return PARSE_STAT_OUTMEM;
  }
  strcpy(com->function_name, buf); /* safe */

  /* seek to next parameter and parse number of arguments */
  if(comma_skip(str,len,&i) != 0 ||
     get_number(str,len,&i,&com->argc) != 0){
    _module_command_free(com); return PARSE_STAT_ERROR;
  }

  if(com->argc != 0){
    com->argv = (char**)emalloc(sizeof(char*)*com->argc);
    if(com->argv == NULL){
      _module_command_free(com); return PARSE_STAT_OUTMEM;
    }
    for(j = 0; j < com->argc; j++) com->argv[j] = NULL;
    com->message_number = (int*)emalloc(sizeof(int)*com->argc);
    if(com->message_number == NULL){
      _module_command_free(com); return PARSE_STAT_OUTMEM;
    }
    com->messages = (char**)emalloc(sizeof(char*)*com->argc);
    if(com->messages == NULL){
      _module_command_free(com); return PARSE_STAT_OUTMEM;
    }
    /* parse arguments */
    for(j = 0; j < com->argc; j++){
      /* seek to next parapeter and parse a artument's string */
      if(comma_skip(str,len,&i) != 0 ||
         get_string(str,len,&i,buf,TMPBUFSIZE) != 0){
        _module_command_free(com); return PARSE_STAT_ERROR;
      }
      com->argv[j] = (char*)emalloc(strlen(buf)+1);
      if(com->argv[j] == NULL){
        _module_command_free(com); return PARSE_STAT_OUTMEM;
      }
      strcpy(com->argv[j], buf); /* safe */
    }
    /* parse number of messages */
    for(j = 0; j < com->argc; j++){
      /* seek to next parameter and parse number of message */
      if(comma_skip(str,len,&i) != 0 || 
         get_number(str,len,&i,&com->message_number[j]) != 0){
        _module_command_free(com); return PARSE_STAT_ERROR;
      }
    }
  }

  /* insert to hash table of commands */
  if(hash_table_insert(coms, com->command_name, com) != 0){
    _module_command_free(com); return PARSE_STAT_INSMIS;
  }
  return PARSE_STAT_SUCCESS;
}

static void _module_command_free_for(void *void_com, void *void_arg){
  _module_command_free((module_command_t*)void_com);
}

static void _module_command_finalchk_for(void *void_com, void *void_arg){
  int i;
  module_t *mod;
  module_message_t *mes;
  module_command_t *com;
  module_command_finalchk_args_t *arg;
  com = (module_command_t*)void_com;
  arg = (module_command_finalchk_args_t*)void_arg;
  mod = arg->module;
  if(arg->status != 0) return;
  for(i = 0; i < com->argc; i++){
    mes = (module_message_t*)
      hash_table_lookup(mod->messages,&com->message_number[i]);
    if(mes == NULL){ arg->status = -1; return; }
    com->messages[i] = mes->message;
  }
  com->module = mod;
}

static void _module_command_setfunc_for(void *void_com, void *void_arg){
  module_command_t *com;
  module_command_setfunc_args_t *arg;
  com = (module_command_t*)void_com;
  arg = (module_command_setfunc_args_t*)void_arg;
  if(arg->status != SETFUNC_STAT_SUCCESS) return;
  if(com->function_name[0] == '*'){
    com->function = NULL;
  } else{
    if(arg->library==0){
      arg->status = SETFUNC_STAT_NOLIBRARY;
    }else{
      com->function=(module_function_t)dlsym(arg->library, com->function_name);
      if(com->function == NULL){
        arg->status = SETFUNC_STAT_NOSYMBOL;
        arg->mes = com->function_name;
      }
    }
  }
}

/* private function of module_t */
static int _module_section_begin_parse(const char *str){
  size_t i,len;
  int    j,sec;
  const char *c;
  struct module_section_t {
    const char *str; size_t len; int ret; 
  } section[] ={
    {"Command", 7, INIFILE_SECTION_COMMAND},
    {"Message", 7, INIFILE_SECTION_MESSAGE},
    {"Error",   5, INIFILE_SECTION_ERROR}
  };
  len=strlen(str);
  i=0; sec = INIFILE_SECTION_FATAL;
  /* skip space */
  space_skip(str,len,&i);
  if(len==i) return INIFILE_SECTION_NOTHING;
  /*  get section of ini file */
  c=&str[i];
  for(j=0;j<3;j++)
    if(strncmp(c,section[j].str,section[j].len) == 0){
      i+=section[j].len; sec = section[j].ret; break;
    }
  if(j == 3) return INIFILE_SECTION_FATAL;
  /* skip space and check begin brace */
  space_skip(str,len,&i);
  if(str[i]!='{')return INIFILE_SECTION_FATAL;
  i++;
  space_skip(str,len,&i);
  if(i!=len) return INIFILE_SECTION_FATAL;
  return sec;
}

static int _module_section_end_parse(const char *str){
  size_t len,i;
  len=strlen(str);
  i=0;
  space_skip(str,len,&i);
  if(len==i) return 0;
  if(str[i]!='}') return -1;
  i++;
  space_skip(str,len,&i);
  if(len!=i) return -1;
  return 1;
}

static int _module_parse(module_t *mod, const char *str, 
                         module_parse_cond_t *pcond){
  int stat;
  if(pcond->cmd_in == 0 && pcond->mes_in == 0 && pcond->err_in == 0){
    stat = _module_section_begin_parse(str);
    switch(stat){
    case INIFILE_SECTION_FATAL:   return PARSE_STAT_ERROR; /* fatal error */
    case INIFILE_SECTION_NOTHING: return PARSE_STAT_COMMENT;
    case INIFILE_SECTION_COMMAND: pcond->cmd_in = 1; break;
    case INIFILE_SECTION_MESSAGE: pcond->mes_in = 1; break;
    case INIFILE_SECTION_ERROR:   pcond->err_in = 1; break;
    }
    return PARSE_STAT_SUCCESS;
  }else{
    stat = _module_section_end_parse(str);
    switch(stat){
    case 0: 
      return PARSE_STAT_COMMENT; /* comment */
    case 1: /* end of brace */
      if(pcond->cmd_in == 1){ pcond->cmd_in = 0; }
      if(pcond->mes_in == 1){ pcond->mes_in = 0; }
      if(pcond->err_in == 1){ pcond->err_in = 0; }
      return PARSE_STAT_SUCCESS;
    }
  }
  if(pcond->cmd_in == 1)
    return _module_command_parse(mod->commands, str);
  if(pcond->mes_in == 1)
    return _module_message_parse(mod->messages, str);
  if(pcond->err_in == 1)
    return _module_message_parse(mod->errors, str);
  return PARSE_STAT_SUCCESS;
}

static int _module_set_command_func(module_t *mod){
  module_command_setfunc_args_t arg;
  arg.library = mod->library;
  arg.status  = SETFUNC_STAT_SUCCESS;
  hash_table_foreach(mod->commands, _module_command_setfunc_for, &arg);
  switch(arg.status){
  case SETFUNC_STAT_SUCCESS:
    break;
  case SETFUNC_STAT_NOLIBRARY:
    mod->err_mesg = estrdup("does not open library."); return -1;
  case SETFUNC_STAT_NOSYMBOL:
    char buf[512];
    snprintf(buf,512,"could not find symbol : %s", arg.mes);
    mod->err_mesg = estrdup(buf); 
    dlclose(mod->library); mod->library = NULL;
    return -1;
  }
  return 0;
}

static int _module_set_setup_func(module_t *mod){
  if(mod->library){
    const char *(*command_func)();
    const char *command;

    /* main function  */
    mod->main_function = (module_function_main_t)dlsym(mod->library,
                                                       "sl4_module_main");
    /* setup command */
    command_func=(const char *(*)())dlsym(mod->library,"SetupCommand");
    if(command_func!=0){
      command=command_func();
      if(command!=0 && command[0]!='\0'){
	mod->init_function=(module_function_t)dlsym(mod->library,command);
      }
    }
    // cleanup command
    command_func=(const char *(*)())dlsym(mod->library,"CleanCommand");
    if(command_func != 0){
      command=command_func();
      if(command!=0 && command[0]!='\0')
	mod->clean_function=(module_function_t)dlsym(mod->library,command);
    }
  }
  return 0;
}

static int _module_inifile_read(module_t *mod){
  int lineno;
  int stat;
  FILE *fp;
  char stream[1024];
  module_parse_cond_t pcond;
  module_command_finalchk_args_t arg;
  pcond.cmd_in = 0;  pcond.mes_in = 0;  pcond.err_in = 0;
  lineno=1;
  if(mod->ini_file==0 || *mod->ini_file=='\0'){
    snprintf(stream,1024,"invalid name of ini file. (Module_Ini)");
    mod->err_mesg = estrdup(stream);
    return -1; // invalid file name
  }
  fp = fopen(mod->ini_file,"r");
  if(fp == NULL){
    snprintf(stream,1024,"can't open file. (%s)", mod->ini_file);
    mod->err_mesg = estrdup(stream);
    return -1;
  }
  while(fgets(stream,1024,fp)){
    if(feof(fp)!=0) break;
    stat = _module_parse(mod, stream, &pcond);
    switch(stat){
    case PARSE_STAT_COMMENT: break;
    case PARSE_STAT_SUCCESS: break;
    case PARSE_STAT_ERROR:
      snprintf(stream,1024,"parse error. (%s:%d)", mod->ini_file,lineno);
      mod->err_mesg = estrdup(stream);
      fclose(fp); return -1;
    case PARSE_STAT_OUTMEM:
      snprintf(stream,1024,"out of memory. (%s:%d)",
               mod->ini_file,lineno);
      mod->err_mesg = estrdup(stream);
      fclose(fp); return -1;
    case PARSE_STAT_INSMIS:
      snprintf(stream,1024,"illegal hash key. (%s:%d)", 
               mod->ini_file, lineno);
      mod->err_mesg = estrdup(stream);
      fclose(fp); return -1;
    }
    lineno++;
  }
  fclose(fp);

  /* final check of ini file parsing */
  if(pcond.cmd_in == 1 || pcond.mes_in == 1 || pcond.err_in == 1){
    snprintf(stream,1024,"missing quote. (%s:%d)",mod->ini_file, lineno);
    mod->err_mesg = estrdup(stream);
    return -1; /* end of brace missing */
  }
  arg.module = mod; arg.status = 0;
  hash_table_foreach(mod->commands, _module_command_finalchk_for, &arg);
  if(arg.status != 0){
    snprintf(stream,1024,"not enough message. (%s:%d)",mod->ini_file,lineno);
    return -1; /* less messages */
  }
  return 0; /* success */
}

DLLEXPORT void module_set_callback(module_event_callback_t cb){
  module_cb_func = cb;
}
