/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: mmapfile.c,v 1.2 2004/07/29 18:07:53 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_MMAP
# include <sys/mman.h>
#endif

#include "libsatellite.h"

#define emalloc(x) malloc(x)
#define efree(x)   free(x)

typedef struct _sl4_mmapfile_handle_t {
#ifdef WIN32
  HANDLE hmf;
  HANDLE fd;
#else
  int    fd;
#endif
} sl4_mmapfile_handle_t;

#ifdef HAVE_MMAP
# ifndef MAP_FILE
#  define MAP_FILE 0
# endif
#endif

#define MMF_HANDLE(mf) ((sl4_mmapfile_handle_t*)(mf)->handle)

DLLEXPORT sl4_mmapfile_t *sl4_mmapfile_new(){
  sl4_mmapfile_t *mf;
  mf = (sl4_mmapfile_t*)emalloc(sizeof(sl4_mmapfile_t));
  if(mf == NULL) return NULL;
  mf->mapped = 0;
  mf->size   = 0;
  mf->mem    = NULL;
  mf->handle = emalloc(sizeof(sl4_mmapfile_handle_t));
  if(mf->handle == NULL){ efree(mf); return NULL; }
#ifdef WIN32
  MMF_HANDLE(mf)->fd  = INVALID_HANDLE_VALUE;
  MMF_HANDLE(mf)->hmf = INVALID_HANDLE_VALUE;
#else
  MMF_HANDLE(mf)->fd   = -1;
#endif
  return mf;
}

DLLEXPORT void sl4_mmapfile_delete(sl4_mmapfile_t *mf){
  if(mf->mapped != 0)
    sl4_mmapfile_unmap(mf);
  efree(mf->handle);
  efree(mf);
}

/* do memory file mapping */
/*   file size >=  offset + mapping size  */
DLLEXPORT int sl4_mmapfile_map(sl4_mmapfile_t *mf, const char *file, int mode,
                               unsigned long size, unsigned long offset){
#ifdef WIN32
  DWORD cflag, oflag, cmflag, mvflag;
#else
  struct stat stbuf;
  int oflag, mflag;
#endif
  if(mf->mapped != 0) return -1;
  offset = 0;
#ifdef WIN32
  switch(mode){
  case SL4_MMAPFILE_RD:
    cflag  = GENERIC_READ;
    oflag  = OPEN_EXISTING; /* if file not exist then error */
    cmflag = PAGE_READONLY;
    mvflag = FILE_MAP_READ;
    break;
  case SL4_MMAPFILE_RDWR:
    cflag  = GENERIC_READ | GENERIC_WRITE;
    oflag  = OPEN_ALWAYS; /* if file not exist then create new file */
    cmflag = PAGE_READWRITE;
    mvflag = FILE_MAP_ALL_ACCESS;
    break;
  default:
    cflag = 0; oflag = 0; cmflag = 0; mvflag = 0; break;
  }
  MMF_HANDLE(mf)->fd = CreateFile(file, cflag, 0, NULL, oflag, 0, NULL);
  if(MMF_HANDLE(mf)->fd == INVALID_HANDLE_VALUE) goto map_error;
  /* need size of input file */
  if(size == 0){
    size = (unsigned long)GetFileSize(MMF_HANDLE(mf)->fd, NULL);
    size -= offset;
  }
  MMF_HANDLE(mf)->hmf = CreateFileMapping(MMF_HANDLE(mf)->fd, NULL, cmflag, 
                                          0, size, file);
  if(MMF_HANDLE(mf)->hmf == NULL) goto map_error;
  mf->mem = MapViewOfFile(MMF_HANDLE(mf)->hmf, mvflag, 0, offset, 0);
  if(mf->mem == NULL) goto map_error;
  mf->size = size;
  mf->mapped = 1;
  return 0;
 map_error:
  if(MMF_HANDLE(mf)->fd != INVALID_HANDLE_VALUE)
    CloseHandle(MMF_HANDLE(mf)->fd);
  MMF_HANDLE(mf)->fd  = INVALID_HANDLE_VALUE;
  MMF_HANDLE(mf)->hmf = INVALID_HANDLE_VALUE;
  mf->mem = NULL;
  return -1;
#else
  switch(mode){
  case SL4_MMAPFILE_RD:
    oflag = O_RDONLY;   /* if file not exist then error */
    mflag = PROT_READ;
    break;
  case SL4_MMAPFILE_RDWR:
    oflag = O_RDWR;     /* if file not exist then create new file? */
    mflag = PROT_READ | PROT_WRITE;
    break;
  default:
    oflag = 0; mflag = 0; break;
  }

  MMF_HANDLE(mf)->fd = open(file, oflag);
  if(MMF_HANDLE(mf)->fd < 0) goto map_error;
  /* need size of input file */
  if(fstat(MMF_HANDLE(mf)->fd, &stbuf) < 0) goto map_error;
  if(size == 0){ size = stbuf.st_size; size -= offset; }
  /* glow size of file */
  if(size > stbuf.st_size){
    if(lseek(MMF_HANDLE(mf)->fd, size-1, SEEK_SET) == -1)
      goto map_error;
    if(write(MMF_HANDLE(mf)->fd, "", 1) != 1) goto map_error;
  }
  mf->mem = mmap(NULL, size, mflag, MAP_FILE | MAP_SHARED, 
                 MMF_HANDLE(mf)->fd, offset);
  if(mf->mem == (caddr_t *)-1) goto map_error;
  mf->size =  size;
  mf->mapped = 1;
  return 0;
 map_error:
  if(MMF_HANDLE(mf)->fd >= 0)
    close(MMF_HANDLE(mf)->fd);
  MMF_HANDLE(mf)->fd   = -1;
  mf->mem = NULL;
  return -1;
#endif
}

DLLEXPORT void sl4_mmapfile_unmap(sl4_mmapfile_t *mf){
  if(mf->mapped == 0) return;
#ifdef WIN32
  UnmapViewOfFile(mf->mem);
  CloseHandle(MMF_HANDLE(mf)->hmf);
  CloseHandle(MMF_HANDLE(mf)->fd);
  MMF_HANDLE(mf)->hmf = INVALID_HANDLE_VALUE;
  MMF_HANDLE(mf)->fd  = INVALID_HANDLE_VALUE;
#else
  munmap(mf->mem, mf->size);
  close(MMF_HANDLE(mf)->fd);
  MMF_HANDLE(mf)->fd   = -1;
#endif
  mf->mapped = 0;
  mf->size =  0;
  mf->mem = NULL;
}
