/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/lib/common/libsatellite.h,v 1.4 2004/06/16 11:22:14 orrisroot Exp $ */

#ifndef __LIBSATELLITE_H__
#define __LIBSATELLITE_H__

#ifdef DLLEXPORT
# undef DLLEXPORT
#endif
#ifdef DLLIMPORT
# undef DLLIMPORT
#endif
#ifdef DECLSPEC
# undef DECLSPEC
#endif
#ifdef WIN32
#  define DLLEXPORT __declspec(dllexport)
#  define DLLIMPORT __declspec(dllimport)
# ifdef LIBSATELLITE_EXPORTS
#  define DECLSPEC DLLEXPORT
# else
#  define DECLSPEC DLLIMPORT
# endif
#else
# define DLLEXPORT
# define DLLIMPORT
# define DECLSPEC
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifndef Max
#define Max(a,b) ((a) > (b) ? (a) : (b))
#endif
#ifndef Min
#define Min(a,b) ((a) < (b) ? (a) : (b))
#endif

#ifndef MAX_INDEX
# define MAX_INDEX 10
#endif

/* file attributes */
#define SL_FATTR_FOK 0x00
#define SL_FATTR_XOK 0x01
#define SL_FATTR_WOK 0x02
#define SL_FATTR_ROK 0x04

/* data file type */
#define SL1_DATAFILE_TYPE_BIGENDIAN     0
#define SL1_DATAFILE_TYPE_LITTLEENDIAN  1
#define SL2_DATAFILE_TYPE_BIGENDIAN     2
#define SL2_DATAFILE_TYPE_LITTLEENDIAN  3

/* hash.c */
typedef void*        (*hash_key_copy_t)(const void *key);
typedef void         (*hash_key_free_t)(void *key);
typedef unsigned int (*hash_func_t)(const void *key);
typedef int          (*hash_comp_t)(const void *key1, const void *key2);
typedef void         (*hash_foreach_t)(void *data, void *arg);

typedef struct _hash_bucket_t {
  struct _hash_bucket_t *next;
  void                  *key;
  unsigned int           val;
  void                  *data;
} hash_bucket_t;

typedef struct _hash_table_t {
  hash_bucket_t **buckets;
  unsigned int    nbucks;
  unsigned int    nkeys;
  int             dyn_resize;
  hash_func_t     hash_func;
  hash_comp_t     hash_comp;
  hash_key_copy_t hash_key_copy;
  hash_key_free_t hash_key_free;
} hash_table_t;

/* list.c */
typedef struct _sl4_list_node_t {
  void *data;
  struct _sl4_list_node_t *next;
  struct _sl4_list_node_t *prev;
} sl4_list_node_t;

typedef struct _sl4_list_t {
  sl4_list_node_t *top;
  sl4_list_node_t *tail;
  unsigned int size;
} sl4_list_t;

typedef struct _sl4_list_iterator_t {
  int          status;
  sl4_list_t      *lst;
  sl4_list_node_t *cur;
} sl4_list_iterator_t;

/* if return value is 0 then no match data, else match data */
typedef int (*sl4_list_eq_func_t)(void*,void*);
/* stack type */
typedef sl4_list_t sl4_stack_t;

/* mmapfile.c */
#define SL4_MMAPFILE_RD    0
#define SL4_MMAPFILE_RDWR  1

typedef struct _sl4_mmapfile_t {
  int            mapped;
  unsigned long  size;
  void          *mem;
  void          *handle;
} sl4_mmapfile_t;

/***** SATELITE Data File Header (rwfile1.c) *****/
typedef struct _h_type {
  short           type_flag;    /* File Format Flag            */
  short           data_flag;    /* Data Type Flag              */
  char            opr_name[20]; /* Operator Name               */
  char            comment[200]; /* Comment Area                */
  char            con_file[10]; /* Relational File             */
  short           date[3];      /* Operation Date              */
  float           samf;         /* Sampling Freq               */
  unsigned short  rec_leng;     /* Block / 1 Record            */
  unsigned short  rec_num;      /* Total Records               */
  unsigned short  data_num;     /* Number of Data / 1 Records  */
  short           dummy[3];
} h_type;

/***** SATELLITE 2.9x Data File Header (rwfile2.c) *****/
typedef struct _Header {
  char            type_flag;    /* File Format Flag            */
  char            data_size;    /* Size of a data              */
  char            opr_name[30]; /* Operator Name               */
  char            comment[128]; /* Comment Area                */
  char            date[3];      /* Date of Making              */
  char            dim;          /* Dimension                   */
  int             index[10];    /* Number of Data              */
  float           samf;         /* Sampling Frequency          */
  char            free[48];     /* free area                   */
} Header;

/* hash.c */
DECLSPEC hash_table_t *hash_table_new(hash_func_t hash_func, 
                                      hash_comp_t hash_comp, 
                                      hash_key_copy_t hash_key_copy,
                                      hash_key_free_t hash_key_free);
DECLSPEC void          hash_table_delete(hash_table_t *table, 
                                         hash_foreach_t hash_del, 
                                         void *arg);
DECLSPEC hash_table_t *hash_table_str_new();
DECLSPEC hash_table_t *hash_table_int_new();
DECLSPEC hash_table_t *hash_table_ptr_new();
DECLSPEC void         *hash_table_lookup(hash_table_t *table, const void *key);
DECLSPEC void         *hash_table_remove(hash_table_t *table, const void *key);
DECLSPEC void          hash_table_erase(hash_table_t *table, 
                                        hash_foreach_t hash_del, 
                                        void *arg);
DECLSPEC int           hash_table_insert(hash_table_t *table, 
                                         void *key, void *data);
DECLSPEC void          hash_table_foreach(hash_table_t *table,
                                          hash_foreach_t hash_foreach, 
                                          void *arg);

/* list.c */
#define SL4_LIST_ITERATOR_STAT_FINE 0
#define SL4_LIST_ITERATOR_STAT_END  1
#define SL4_LIST_ITERATOR_STAT_ILL  2

DECLSPEC sl4_list_t *sl4_list_new();
#define sl4_list_empty(lst) (((lst)->size==0) ? 1: 0)
#define sl4_list_top(lst)   (((lst)->top==NULL) ? NULL : (lst)->top->data)
DECLSPEC int   sl4_list_push_front(sl4_list_t *lst, void *dat);
DECLSPEC int   sl4_list_push_back(sl4_list_t *lst, void *dat);
DECLSPEC int   sl4_list_pop_front(sl4_list_t *lst);
DECLSPEC int   sl4_list_pop_back(sl4_list_t *lst);
DECLSPEC int   sl4_list_erase(sl4_list_t *lst, sl4_list_iterator_t *it);
DECLSPEC int   sl4_list_begin(sl4_list_t *lst, sl4_list_iterator_t *it);
DECLSPEC int   sl4_list_end(sl4_list_t *lst, sl4_list_iterator_t *it);
DECLSPEC int   sl4_list_find(sl4_list_t *lst, void *dat, 
                             sl4_list_iterator_t *it, 
                             sl4_list_eq_func_t func);
DECLSPEC int   sl4_list_rfind(sl4_list_t *lst, void *dat, 
                              sl4_list_iterator_t *it, 
                              sl4_list_eq_func_t func);
DECLSPEC void  sl4_list_it_next(sl4_list_iterator_t *it);
DECLSPEC void  sl4_list_it_prev(sl4_list_iterator_t *it);
DECLSPEC void *sl4_list_it_data(sl4_list_iterator_t *it);
#define sl4_list_it_is_end(it)  \
                 (((it)->status == SL4_LIST_ITERATOR_STAT_END) ? 1: 0)
#define sl4_stack_new()         sl4_list_new()
#define sl4_stack_empty(st)     (((st)->size==0) ? 1: 0)
#define sl4_stack_push(st, dat) sl4_list_push_front(st,dat)
#define sl4_stack_pop(st)       sl4_list_pop_front(st)
#define sl4_stack_top(st)       (((st)->top==NULL) ? NULL : (st)->top->data)


/* mmapfile.c */
DECLSPEC sl4_mmapfile_t *sl4_mmapfile_new();
DECLSPEC void sl4_mmapfile_delete(sl4_mmapfile_t *mf);
DECLSPEC int  sl4_mmapfile_map(sl4_mmapfile_t *mf, const char *file, int mode,
                               unsigned long size, unsigned long offset);
DECLSPEC void sl4_mmapfile_unmap(sl4_mmapfile_t *mf);

/* strutil.c */
DECLSPEC size_t  strutil_check_quote(const char *str, char stop, int *status);
DECLSPEC char   *strutil_substr(const char *str, unsigned int offset, 
                                unsigned int len);
DECLSPEC char   *strutil_strtok_r(char *s, const char *delim, char **last);
DECLSPEC char   *strutil_trim(const char *str);
DECLSPEC char   *strutil_ltrim(const char *str);
DECLSPEC char   *strutil_rtrim(const char *str);

/* syscom.c */
DECLSPEC int     init_syscom();
DECLSPEC int     free_syscom();
DECLSPEC int     load_syscom_file();
DECLSPEC int     store_syscom_file();
DECLSPEC int     unlink_syscom_file();
DECLSPEC void    set_compat2x_mode(int f);
DECLSPEC int     is_compat2x_mode();
DECLSPEC int     set_sampling(double freq);
DECLSPEC double  get_sampling();
DECLSPEC int     ChangeDataType(int type);
DECLSPEC int     ChangeDataSize(int siz);
DECLSPEC int     sl2_get_data_type();
DECLSPEC int     sl2_get_data_size();
DECLSPEC const char *get_tmpdir();

/* index.c */
DECLSPEC int   RegularIndex(int *idx2, int *idx, int n);
DECLSPEC int   CopyIndex(int *to, int *from, int n);
DECLSPEC int   EqualIndex(int *to, int *from, int n);
DECLSPEC int   IndexSize(int dim, int *idx);
DECLSPEC int  *SubIndex(int *idx);
DECLSPEC int   _Index(int *idx2, int dim, int *idx);
DECLSPEC int  *rIndex(int n, int *idx2, int dim, int *idx);
DECLSPEC void  MaxIndex(int *a, int *b, int *idx, int dim);
DECLSPEC void  MinIndex(int *a, int *b, int *idx, int dim);

/* rwfile1.c */
DECLSPEC void  m_file(char *f_name);
DECLSPEC int   r_file(int s_blk, int num_blk, char *f_name, char *data);
DECLSPEC int   w_file(int s_blk, int num_blk, char *f_name, char *data);
DECLSPEC void  stor_init(char *f_name, int type);
DECLSPEC int   stor_data(float *data, int d_point, char *f_name, int numb);
DECLSPEC int   load_data(float *data, char *f_name, int numb);
DECLSPEC void  stor_header(char *f_name, h_type *head);
DECLSPEC int   load_header(char *f_name, h_type *head);

/* rwfile2.c */
DECLSPEC char *_ReadFile(const char *fname, Header *head);
  /* auto compat mode check */
DECLSPEC int   _WriteFile(const char *fname, int dim, int *index, char *data);
  /* compat2x mode : enable to fix index */
DECLSPEC int   _WriteFile2(const char *fname, int dim, int *index, char *data);
  /* satellite4 mode : disable to fix index */
DECLSPEC int   _WriteFile4(const char *fname, int dim, int *index, char *data);
DECLSPEC char *LoadData(const char *fname, int numb, Header *head);
DECLSPEC int   StoreData(const char *fname, int numb, int dim, 
                           int *index, char *data);
/* DECLSPEC void PrintHeader(Header *head); */
DECLSPEC void  InitHeader(Header *head);
DECLSPEC int   LoadHeader(const char *fname, Header *head);
DECLSPEC int   StoreHeader(const char *fname, Header *head);
DECLSPEC int   StoreTimeData(const char *fname, int sub_dim, int *sub_index,
                             char *data, int length);
DECLSPEC void  FreeData(void *mem);

/* util.c */
/* system call wrapper functions and other utility functions */
/* IsWindows()      --  1: WIN32, 0: other OS */ 
/* Mkdir()          --  0: success, -1: error */
/* RmdirRecursive() --  0: success, -1: error */
/* IsDirectory()    --  1: directory, 0: not dir */
/* Access()         --  0: success, -1: error */

DECLSPEC int   IsWindows();
DECLSPEC int   Mkdir(const char *path);
DECLSPEC int   RmdirRecursive(const char *path);
DECLSPEC int   IsDirectory(const char *path);
DECLSPEC int   Access(const char *path, int mode);
DECLSPEC int   sl4_chdir(const char *path);

/* userinfo.c */
DECLSPEC char *GetCurrentUserName(char *buf, int len);
DECLSPEC char *GetHomeDirectory(char *buf, int len);
/* DECLSPEC char *GetTempDirectory(char *buf, int len); */
DECLSPEC char *GetUserResourceDirectory(char *buf, int len);
DECLSPEC char *GetSystemResourceDirectory(char *buf, int len);
DECLSPEC char *GetModuleDirectory(char *buf, int len);
DECLSPEC char *GetShareDirectory(char *buf, int len);
DECLSPEC int   get_date(int *year, int *month, int *day);
DECLSPEC int   get_time(int *hour, int *minute, int *second);
#define getusername(x,y) GetCurrentUserName((x),(y))
DECLSPEC  int  WriteRegistInt(const char *key, const char* param, long value);
DECLSPEC  int  ReadRegistInt(const char *key, const char* param, long *value);

/* process.c */
typedef union _sl4_fd_t {
  void *handle;
  int   fd;
} sl4_fd_t;

DECLSPEC int create_child_process(const char *bin, sl4_fd_t ifd, sl4_fd_t ofd,
                                  sl4_fd_t efd, const char *libpath);
DECLSPEC int wait_child_process(int pid, int *status);
DECLSPEC int wait_all_process();

/* path.c */
DECLSPEC int  is_executable_command(const char *bin, char *full, size_t len);

/* signal.c */
/*
DECLSPEC void  setsigint();
DECLSPEC int   isintcatch();
DECLSPEC void  unsetsigint();
DECLSPEC void  sigintblock(int t);
*/

#ifdef __cplusplus
}
#endif

#endif /* __LIBSATELLITE_H__ */
