/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: Series_Buffer.cpp,v 1.2 2004/07/29 18:07:53 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "SL_header.h"

#include "libsatellite.h"

#define  __EXPORTSYMBOL__
#include "SL_exception.h"
#include "SL_Index.h"
#include "Base_Buffer.h"
#include "Series_Buffer.h"
#undef   __EXPORTSYMBOL__

using namespace std;

// static int sb_cnt=0;

// DLLEXPORT void print_series_buffer_num(){
//   printf("Series_Buffer res:%d\n",sb_cnt);
// }

Series_Buffer::Series_Buffer(){
//  sb_cnt++;
  type=SERIES_B;
}

Series_Buffer::~Series_Buffer(){
//  sb_cnt--;
  FreeBuffer();
}

Base_Buffer *Series_Buffer::duplicate(){
  Base_Buffer *ret;
  try{
    ret = new Series_Buffer;
  }catch(bad_alloc){
    return 0;
  }
  ret->CopyBuffer(this);
  return ret;
}

bool Series_Buffer::InitBuffer(){
  int i;
  FreeBuffer();
  if(index[0]<1)throw ill_dim();
  for(i=0;i<dim;i++)
    if(index[i]==0)throw ill_index();
  ClAllocBuffer(IndexSize());
  return true;
}

bool Series_Buffer::FreeBuffer(){
  bool stat;
  if(data_size==0)
    stat=false;
  else{
    delete [] (double*)data;
    stat=true;
    data=0;
  }
  data_size=0;
  return stat;
}

bool Series_Buffer::AllocBuffer(unsigned size){
  FreeBuffer();
  data_size=size;
  data=new double[size];
  return true;
}

bool Series_Buffer::ClAllocBuffer(unsigned size){
  AllocBuffer(size);
  memset(data,0,size*sizeof(double));
  return true;
}

bool Series_Buffer::ReAllocBuffer(unsigned size){
  double *tmp;
  int     old_size;
  tmp=(double *)data;
  old_size=data_size;
  data_size=size;
  try {
    data=new double[size];
  } catch(bad_alloc) {
    data=tmp;
    data_size=old_size;
    throw;
  }
  if(old_size!=0){
    memcpy(data,tmp,old_size*sizeof(double));
    delete [] tmp;
  }
  return true;
}

bool Series_Buffer::CopyBuffer(Base_Buffer *from){
  int size;
  if(from->GetType()!=SERIES_B)throw bad_type();
  FreeBuffer();
  CopyIndex(from->GetBufferIndex());
  size=from->IndexSize();
  AllocBuffer(size);
  memcpy(data,from->GetDataPointer(),size*sizeof(double));
  return true;
}

bool Series_Buffer::SetData(Index point, double val){
  int pt;
  Index sub;
  if(data==0)InitBuffer();
  pt=DataPoint(point);
  if(!RegularIndex(point)){
    sub=SubIndex();
    if(sub.RegularIndex(point.SubIndex()) && index[0]==point.GetIndex(0)){
      ReAllocBuffer(IndexSize()+SubIndexSize());
      index[0]++;
      ((double*)data)[pt]=val;
    }else{
      throw ill_index();
    }
  }else{
    ((double *)data)[pt]=val;
  }
  return true;
}

double Series_Buffer::GetData(Index point){
  int pt;
  if(!RegularIndex(point)) throw ill_index();
  pt=DataPoint(point);
  return ((double *)data)[pt];
}

// added sub buffer to tail - success: index[0], fail: -1
int Series_Buffer::AppendBuffer(Series_Buffer *sub){
  int time;
  if(dim==0)throw ill_dim();
  time=sub->GetIndex(0);
  PutSubBuffer(time,sub);
  return index[0];
}

// get sub buffer of time position
Series_Buffer *Series_Buffer::GetSubBuffer(int time){
  int  size,block;
  long offset;
  Series_Buffer *ret=0;
  size=SubIndexSize();
  if(size==0) throw ill_index();
  if(time>=index[0]) throw ill_index();
  block=size*sizeof(double);
  offset=(long)size*(long)time;
  ret=new Series_Buffer;
  try { ret->AllocBuffer(size); } catch(bad_alloc){
    delete ret;
    throw;
  }
  memcpy(ret->GetDataPointer(),(double *)data+offset,block);
  ret->CopyIndex(SubIndex());
  return ret;
}

// insert sub buffer to time position
int Series_Buffer::PutSubBuffer(int time, Series_Buffer *sub){
  int i, size, sub_size, offset;
  size=SubIndexSize();
  if(size==0)throw ill_index();
  sub_size=sub->IndexSize();
  if(size!=sub_size)throw bsize_mis();
  if(index[0]<=time){ 
    // if time is larger than current index[0] then realloc data size.
    ReAllocBuffer(size*(time+1));
    for(i=0;i<time-index[0];i++) // zero clear to added memory
      memset((double *)data+((index[0]+i)*size),0,size*sizeof(double));
    index[0]=time+1;
  }
  // ready.. data coping.
  offset=time*size;
  memcpy((double *)data+offset,sub->GetDataPointer(),size*sizeof(double));
  return sub_size;
}

// get time series of 'point' position
Series_Buffer *Series_Buffer::GetTimeSeries(Index point){
  Series_Buffer *ret=0;
  int i, offset, size;
  double *ret_data;
  if(dim!=point.GetDim())throw dim_mis();
  if(!RegularIndex(point))throw ill_index();
  ret=new Series_Buffer;
  try { ret->AllocBuffer(index[0]); } catch(bad_alloc) {
    delete ret;
    throw;
  }
  // set the offset at a point of zero [time]
  // get data pointer of writing data
  ret_data=(double *)ret->GetDataPointer();
  offset=DataPoint(point);  // get offset 
  size=SubIndexSize();      // get data size
  for (i=0;i<index[0];i++){
    memcpy(ret_data+i,((double *)data)+offset,sizeof(double));
    offset+=size;
  }
  ret->SetIndex(0,index[0]);
  ret->SetDim(1);
  return ret;
}

// put series buffer to 'point' position.
int Series_Buffer::PutTimeSeries(Index point, Series_Buffer *buf){
  int  i,size,length,offset;
  double *buf_pointer;
  // checking for location pointer 'point'
  if(dim!=point.GetDim()) throw ill_dim();
  if(point.GetIndex(0) != 0) throw ill_index();
  if(!RegularIndex(point)) throw ill_index();
  // checking for source buffer, source buffer 'buf' must be 1 dimension.
  if(buf->GetDim()!=1)throw ill_dim();
  // Let's write source buffer 'buf' to current buffer
  offset = DataPoint(point);
  size=SubIndexSize();
  length=buf->GetIndex(0);
  buf_pointer=(double *)buf->GetDataPointer();
  // if source buffer size larger then dest buffer size, then resize.
  if(index[0]<length){
    ReAllocBuffer(length*size);
    for(i=0;i<length-index[0];i++) // zero clear to added memory
      memset((double *)data+((index[0]+i)*size),0,size*sizeof(double));
    index[0]=length; // reset data size
  }
  for(i=0;i<length;i++){
    memcpy((double *)data+offset,&(buf_pointer[i]),sizeof(double));
    offset+=size;
  }
  return length;
}
