/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: SL_Object.cpp,v 1.11 2005/03/25 05:28:05 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "SL_header.h"

#include "libsatellite.h"

#define  __EXPORTSYMBOL__
#include "SL_exception.h"
#include "history.h"
#include "module.h"
#include "tty_console.h"
#include "SL_Index.h"
#include "Base_Buffer.h"
#include "Series_Buffer.h"
#include "Snapshot_Buffer.h"
#include "String_Buffer.h"
#include "Scalar_Buffer.h"
#include "SL_Tool.h"
#include "SL_Object.h"
#include "SymbolList.h"
#include "SystemCommon.h"
#undef   __EXPORTSYMBOL__
#include "mathfunc.h"

using namespace std;

typedef double(*OPEFUNC)(double,double);
typedef Base_Buffer *(*SOPEFUNC)(String_Buffer*, String_Buffer*);

static Base_Buffer *string_add(String_Buffer*, String_Buffer*);
static Base_Buffer *string_sub(String_Buffer*, String_Buffer*);
static Base_Buffer *string_mul(String_Buffer*, String_Buffer*);
static Base_Buffer *string_div(String_Buffer*, String_Buffer*);
static Base_Buffer *string_comp(bool, String_Buffer *, String_Buffer *);
static Base_Buffer *string_eq(String_Buffer*, String_Buffer*);
static Base_Buffer *string_ne(String_Buffer*, String_Buffer*);

static OPEFUNC operators[] = {
  math_add,    /* MATH_ADD     0 */
  math_sub,    /* MATH_SUB     1 */
  math_mul,    /* MATH_MUL     2 */
  math_div,    /* MATH_DIV     3 */
  math_mod,    /* MATH_MOD     4 */
  math_gt,     /* MATH_GT      5 */
  math_lt,     /* MATH_LT      6 */
  math_ge,     /* MATH_GE      7 */
  math_le,     /* MATH_LE      8 */
  math_eq,     /* MATH_EQ      9 */
  math_ne,     /* MATH_NE     10 */
  math_and,    /* MATH_AND    11 */
  math_or,     /* MATH_OR     12 */
  math_not,    /* MATH_NOT    13 */
  math_negate, /* MATH_NEGATE 14 */
  math_int,    /* MATH_INT    15 */
  math_floor,  /* MATH_FLOOR  16 */
  math_ceil,   /* MATH_CEIL   17 */
  math_round,  /* MATH_ROUND  18 */
  math_abs,    /* MATH_ABS    19 */
  math_sign,   /* MATH_SIGN   20 */
  math_pow,    /* MATH_POW    21 */
  math_exp,    /* MATH_EXP    22 */
  math_log,    /* MATH_LOG    23 */
  math_log2,   /* MATH_LOG2   24 */
  math_log10,  /* MATH_LOG10  25 */
  math_sqrt,   /* MATH_SQRT   26 */
  math_sin,    /* MATH_SIN    27 */
  math_cos,    /* MATH_COS    28 */
  math_tan,    /* MATH_TAN    29 */
  math_asin,   /* MATH_ASIN   30 */
  math_acos,   /* MATH_ACOS   31 */
  math_atan,   /* MATH_ATAN   32 */
  math_atan2,  /* MATH_ATAN2  33 */
  0            /* MATH_SIZE   34 */
};

static Base_Buffer *getbuffer(Base_Buffer *src, SL_OBJ::TYPE type);
static Base_Buffer *getstring(Base_Buffer *src, SL_OBJ::TYPE type);

// temporary implement for windows memory error !
DLLEXPORT SL_Object *new_SL_Object(SL_OBJ::TYPE type, Base_Buffer *src){
  return new SL_Object(type, src);
}
DLLEXPORT void delete_SL_Object(SL_Object *obj){
  delete obj;
}

SL_Object::SL_Object(SL_OBJ::TYPE type, Base_Buffer *src)
  : objtype(type), link(0), buffer(src){
  src->buf_ref();
}

SL_Object::~SL_Object(){
  //  if((size_t)this == (size_t)0x080ff3d0)
  //    abort();
  if(link!=0){
#ifndef WIN32
    fprintf(stderr, "Fatal Error: object ref is not zero\n");
#endif
    abort();
  }
  if(buffer){
    buffer->buf_unref();
    if(buffer->empty()) delete buffer;
  }
}

void SL_Object::obj_unref(){ 
  if(link==0){
#ifndef WIN32
    fprintf(stderr, "Fatal Error: object ref is already zero\n");
#endif
    abort();
  }else{
    link--;
  }
}

bool SL_Object::empty(){
  if(link==0)return true;
  return false;
}

SL_Object *SL_Object::obj_dup(){
  SL_Object *obj;
  Base_Buffer *buf;
  buf = buffer;
  if(buf){
    buf = buffer->duplicate();
    if(buf == 0) return 0;
  }
  obj = new SL_Object(objtype, buf);
  return obj;
}


int SL_Object::IsTrue(){
  int   i,n,ret;
  const char *s;
  Index      idx;
  n=buffer->IndexSize();
  switch(objtype){
  case SL_OBJ::SCALAR_O:
    ret = (int)((Scalar_Buffer*)buffer)->GetScalar();
    break;
  case SL_OBJ::STRING_O:
    for(i = 0; i < n; i++){
      idx=buffer->rIndex(i);
      s = ((String_Buffer*)buffer)->GetString(idx);
      if(s == 0 || s[0]=='\0'){ ret=0; break; }
    }
    ret=1;
    break;
  case SL_OBJ::SERIES_O:
  case SL_OBJ::SNAPSHOT_O:
    for(i = 0; i < n; i++){
      idx=buffer->rIndex(i);
      if(((Series_Buffer*)buffer)->GetData(idx)==0.0)
        break;
    }
    ret= (i == n) ? 1 : 0;
    break;
  default:
    ret= 0;
  }
  return ret;
}

bool SL_Object::Init(SL_OBJ::TYPE type, Index idx){
  objtype=type;
  if(buffer!=0){
    buffer->buf_unref();
    delete buffer;
  }
  switch(type){
  case SL_OBJ::SERIES_O:   buffer=new Series_Buffer;   break;
  case SL_OBJ::SNAPSHOT_O: buffer=new Snapshot_Buffer; break;
  case SL_OBJ::STRING_O:   buffer=new String_Buffer;   break;
  case SL_OBJ::SCALAR_O:   buffer=new Scalar_Buffer;   break;
  default: buffer=0; return false;
  }
  buffer->CopyIndex(idx);
  buffer->buf_ref();
  return buffer->InitBuffer();
}

SL_Object *SL_Object::Array(Index point){
  static char *where="(object array)";
  Base_Buffer *bufp=0;
  SL_Object *ret=0;
  SL_OBJ::TYPE ret_type=SL_OBJ::UNDEF_O;
  Index idx;
  double tmp=0.0;
  const char *str=0;
  switch(objtype){
  case SL_OBJ::SERIES_O:
    try {
      bufp=((Series_Buffer *)buffer)->GetTimeSeries(point);
    }catch(buffer_exception){
      syscom->console->execerror("failed to evaluate time-series",where);
    }
    ret_type=SL_OBJ::SERIES_O;
    break;
  case SL_OBJ::SNAPSHOT_O:
    if(!buffer->RegularIndex(point))
      syscom->console->execerror("illegal index",where);
    tmp=((Snapshot_Buffer*)buffer)->GetData(point);
    bufp=new Scalar_Buffer;
    try{
      bufp->CopyIndex(point);
      bufp->InitBuffer();
    }catch(buffer_exception err){
      delete bufp;
      syscom->console->execerror(err.what(),where);
    }catch(bad_alloc){
      delete bufp;
      throw;
    }
    ((Scalar_Buffer*)bufp)->SetScalar(tmp);
    ret_type=SL_OBJ::SCALAR_O;
    break;
  case SL_OBJ::STRING_O:
    if(!buffer->RegularIndex(point))
      syscom->console->execerror("illegal index",where); 
    str=((String_Buffer*)buffer)->GetString(point);
    bufp=new String_Buffer;
    try{
      bufp->SetDim(1);
      bufp->SetIndex(0,1);
      bufp->InitBuffer();
    }catch(buffer_exception err){
      delete bufp;
      syscom->console->execerror(err.what(),where);
    }catch(bad_alloc){
      delete bufp;
      throw;
    }
    idx.SetDim(1);
    try{
      ((String_Buffer*)bufp)->SetString(idx,str);
    }catch(buffer_exception err){
      delete bufp;
      syscom->console->execerror(err.what(),where); 
    }catch(bad_alloc){
      delete bufp;
      throw;
    }
    ret_type=SL_OBJ::STRING_O;
    break;
  case SL_OBJ::SCALAR_O:
    syscom->console->execerror("object type mismatch [scalar]",where); 
    break;
  default:
    syscom->console->execerror("object type mismatch [unknown]",where); 
    break;
  }
  try{
    ret=new SL_Object(ret_type,bufp);
  }catch(bad_alloc){
    delete bufp;
    throw;
  }
  return ret;
}

void SL_Object::ArrayAsgn(Index point, SL_Object *from){
  static char *where="(array assign)";
  int length=0,wpt=0;
  double tmp;
  const char *str=0;
  Base_Buffer *bufp=0;
  Index idx;
  if(buffer==0)
    syscom->console->execerror("failed to read a internal buffer",0);
  switch(objtype){
  case SL_OBJ::SERIES_O:
    length=buffer->GetIndex(0);
    bufp=new Series_Buffer;
    switch(from->TypeofOBJ()){
    case SL_OBJ::SERIES_O:
    case SL_OBJ::SNAPSHOT_O:
      if(from->GetBufferPointer()->GetDim()!=1)
        syscom->console->execerror("illegal dememsion of series",0);
      try {
        bufp->CopyBuffer(from->GetBufferPointer());
      }catch(buffer_exception err){
        delete bufp;
        syscom->console->execerror(err.what(),where);
      }catch(bad_alloc){
        delete bufp;
        throw;
      }
      break;
    case SL_OBJ::SCALAR_O:
      tmp=((Scalar_Buffer *)from->GetBufferPointer())->GetScalar();
      bufp->SetDim(1); bufp->SetIndex(0,1);
      try{
        bufp->InitBuffer();
        idx.SetDim(1);
        ((Series_Buffer *)bufp)->SetData(idx,tmp);
      }catch(buffer_exception err){
        delete bufp;
        syscom->console->execerror(err.what(),where);
      }catch(bad_alloc){
        delete bufp;
        throw;
      }
      break;
    default:
      syscom->console->execerror("object type mismatch",where);
      break;
    }
    try{
      wpt=((Series_Buffer*)buffer)->PutTimeSeries(point,(Series_Buffer *)bufp);
    }catch(buffer_exception){
      delete bufp;
      syscom->console->execerror("failed to set a time series",where);
    }catch(bad_alloc){
      delete bufp;
      throw;
    }
    if(wpt < length)
      syscom->console->warning("Warning ... too short time series",where);
    delete bufp;
    break;
  case SL_OBJ::SNAPSHOT_O:
    if(buffer->GetDim()!=point.GetDim())
      syscom->console->execerror("dimension mismatch [snapshot]",where);
    if(!buffer->RegularIndex(point))
      syscom->console->execerror("illegal index [snapshot]",where);
    if(from->TypeofOBJ()!=SL_OBJ::SCALAR_O)
      syscom->console->execerror("not scalar type object",where);
    bufp=from->GetBufferPointer();
    tmp=((Scalar_Buffer*)bufp)->GetScalar();
    ((Snapshot_Buffer*)buffer)->SetData(point,tmp);
    break;
  case SL_OBJ::STRING_O:
    if(from->TypeofOBJ()!=SL_OBJ::STRING_O)
      syscom->console->execerror("illegal object type [string]",where);
    if(buffer->GetDim()!=point.GetDim())
      syscom->console->execerror("dimension mismatch [string]",where);
    if((buffer->GetDim()!=1 && !buffer->RegularIndex(point)))
      syscom->console->execerror("illegal index [string]",where);
    length=buffer->IndexSize();
    wpt=buffer->DataPoint(point);
    bufp=from->GetBufferPointer();
    idx.SetDim(1);
    str=((String_Buffer*)bufp)->GetString(idx);
    if(length<=wpt){
      if(is_compat2x_mode())
        ((String_Buffer*)buffer)->ReSize(point.GetIndex(0));
      else
        syscom->console->execerror("illegal index [string]",where);
    }
    ((String_Buffer*)buffer)->SetString(point,str);
    break;
  default:
    syscom->console->execerror("illegal object type",where);
    break;
  }
}

SL_Object *SL_Object::Snap(int time){
  static char *where="(snapshot)";
  Base_Buffer *snap=0;
  SL_Object *ret=0;
  switch(objtype){
  case SL_OBJ::SERIES_O:
    try{
      snap=((Series_Buffer*)buffer)->GetSubBuffer(time);
    }catch(buffer_exception err){
      syscom->console->execerror("failed to evalute snapshot",where);
    }catch(bad_alloc){
      throw;
    }
    try{
      if(snap->GetDim()==0)
        ret=new SL_Object(SL_OBJ::SCALAR_O,snap);
      else
        ret=new SL_Object(SL_OBJ::SNAPSHOT_O,snap);
    }catch(bad_alloc){
      delete snap;
      throw;
    }
    break;
  default: //  STRING_O SNAPSHOT_O SCALAR_O
    syscom->console->execerror("object type mismatch",where);
    break;
  }
  return ret;
}
    
void SL_Object::SnapAsgn(int time, SL_Object *from){
  static char *where="(snapshot assgnment)";
  SL_OBJ::TYPE from_type;
  Base_Buffer *bufp=0;
  Index idx;
  double tmp;
  from_type=from->TypeofOBJ();
  if(objtype==SL_OBJ::SERIES_O){
    bufp=new Series_Buffer;
    try{
      switch(from_type){
      case SL_OBJ::SERIES_O:
      case SL_OBJ::SNAPSHOT_O:
        bufp->CopyBuffer(from->GetBufferPointer());
        break;
      case SL_OBJ::SCALAR_O:
        tmp=((Scalar_Buffer *)from->GetBufferPointer())->GetScalar();
        bufp->SetDim(1); bufp->SetIndex(0,1);
        bufp->InitBuffer();
        idx.SetDim(1);
        ((Series_Buffer *)bufp)->SetData(idx,tmp);
        break;
      default:
        syscom->console->execerror("object type mismatch",where);
        break;
      }
      ((Series_Buffer*)buffer)->PutSubBuffer(time,(Series_Buffer *)bufp);
    }catch(buffer_exception err){
      delete bufp;
      syscom->console->execerror("failed to set a shapshot",where);
    }catch(bad_alloc){
      delete bufp;
      throw;
    }
    delete bufp;
  }else{ // SNAPSHOT_O SCALAR_O STRING_O
    syscom->console->execerror("object type mismatch",where);
  }
}

SL_Object *SL_Object::Opecode(int dptr, SL_Object *obj){
  static char *where="(opecode)";
  SL_Object *ret_obj=0;
  switch(objtype){
  case SL_OBJ::SCALAR_O:
    ret_obj=scalar_op(dptr,obj);
    break;
  case SL_OBJ::SERIES_O:
    ret_obj=series_op(dptr,obj);
    break;
  case SL_OBJ::SNAPSHOT_O:
    ret_obj=snapshot_op(dptr,obj);
    break;
  case SL_OBJ::STRING_O:
    ret_obj=string_op(dptr,obj);
    break;
  default:
    syscom->console->execerror("unsupported object type",where);
  }
  return ret_obj;
}

void SL_Object::Read(){
  static char *where="(read)";
  size_t i,length;
  char str[ONELINE],*p=str;
  double buf[ONELINE];
  int dim = 1;
  Base_Buffer *bufp=0;
  Index idx;
  memset(str,0,ONELINE*sizeof(char));
  if(objtype == SL_OBJ::SERIES_O || objtype == SL_OBJ::SNAPSHOT_O ||
     objtype == SL_OBJ::STRING_O || objtype == SL_OBJ::SCALAR_O){
    if(syscom->console->tty_gets(str,ONELINE)==0)
      syscom->console->execerror("fail to read from console",where);
  }else{
    syscom->console->execerror("fatal: illegal object type",where);
  }
  buffer->FreeBuffer();
  switch(objtype){
  case SL_OBJ::SERIES_O:
  case SL_OBJ::SNAPSHOT_O:
    for(i=0;*p!='\0';i++){
      if(i==ONELINE){
        syscom->console->warning("too long input string",where);
        break;
      }
      sscanf(p, "%lf", &buf[i]);
      while(*p != ',' && *p != '\0') p++;
      if(*p == ',') p++;
    }
    length=i;
    buffer->SetDim(dim);
    buffer->SetIndex(0,(int)length); /* TODO: remove cast */
    buffer->InitBuffer();
    idx.SetDim(1);
    for(i=0;i<length;i++){
      idx.SetIndex(0,(int)i); /* TODO: remove cast */
      ((Series_Buffer*)buffer)->SetData(idx,buf[i]);
    }
    break;
  case SL_OBJ::STRING_O:
    if((length=strlen(str)) > 0)
      if(str[length-1] == '\n')str[length-1] = '\0';
    buffer->SetDim(dim);
    buffer->SetIndex(0,1);
    buffer->InitBuffer();
    idx.SetDim(dim);
    ((String_Buffer*)buffer)->SetString(idx,str);
    break;
  case SL_OBJ::SCALAR_O:
    if((length=strlen(str)) > 0)
      if(str[length-1] == '\n') str[length-1] = '\0';
    sscanf(str,"%lf",&buf[0]);
    buffer->InitBuffer();
    ((Scalar_Buffer *)buffer)->SetScalar(buf[0]);
    break;
  }
}

void SL_Object::Print(char *fmt,int win_c){
  static char *where="(print)";
  size_t i,n,length,columns;
  bool isTTY=true;
  string *argv;
  char deffmt[10];
  if(buffer==0) syscom->console->execerror("buffer is null",where);
  length=buffer->IndexSize();
  switch(objtype){
  case SL_OBJ::SERIES_O:
  case SL_OBJ::SNAPSHOT_O:
    if(buffer->GetDim()==0){
      syscom->console->execerror("zero point series",where);
      break;
    }
    for(i=0;i<length;i++){
      if(isintcatch())break;
      if(isTTY==true){
        index_header((int)i,fmt,win_c); /* TODO: check cast */
        if(fmt != 0)syscom->console->tty_printf(" ");
      }else
        syscom->console->tty_printf("\n");
      print_number(fmt,"%12.4g",((double*)buffer->GetDataPointer())[i]);
    }
    break;
  case SL_OBJ::STRING_O:
    n=length;
    argv=(string*)buffer->GetDataPointer();
    if(argv==0){
      syscom->console->tty_printf("(nil)");
      break;
    }
    if(n==1){
      char *format;
      sprintf(deffmt,"%%s");
      format = (fmt==0) ?  deffmt : fmt;
      if(argv[0].size()==0)
        syscom->console->tty_printf(format,"(nil)");
      else
        syscom->console->tty_printf(format,argv[0].c_str());
      break;
    }
    length=5;
    for(i=0;i<n;i++){
      if(isintcatch())break;
      size_t len=argv[i].size();
      if(len!=0){
        if(argv[i].c_str()[len-1]=='\n')
          length=win_c;
        if(length<len)
          length=len;
      }
    }
    sprintf(deffmt, "%%%ds", length);
    columns=win_c - (4*buffer->GetDim()+3);
    if(length>columns)
      length=columns;
    for(i=0;i<n;i++){
      if(isintcatch())break;
      if(isTTY)index_header((int)i,deffmt,win_c); /* TODO: check cast */
      else syscom->console->tty_printf("\n");
      syscom->console->tty_printf("%-*s ",length,
                (argv[i].size()!=0) ? argv[i].c_str() : "(nil)");
    }
    break;
  case SL_OBJ::SCALAR_O:
    print_number(fmt,"%.8g",((Scalar_Buffer*)buffer)->GetScalar());
    break;
  default:
    syscom->console->execerror("illegal object type",where);
    break;
  }
  // syscom->console->tty_printf("\n");
}

// Private Method
void SL_Object::print_number(char *fmt, char *def, double d){
  char  c=(fmt == 0) ? '\0' : fmt[strlen(fmt)-1];
  switch(c){
  case 'e' :
  case 'f' :
  case 'g' :
    syscom->console->tty_printf(fmt, d);
    break;
  case 'd' :
  case 'c' :
    syscom->console->tty_printf(fmt, (int)d);
    break;
  default:
    syscom->console->tty_printf(def, d);
    break;
  }
}

void SL_Object::index_header(int i, char *fmt, int win_c){
  int j;
  static int preindex, items;
  static size_t width[MAX_INDEX];
  Index idx;
  if(i==0){ // initialize
    char tmp[32];
    size_t  index_width=0;
    for(j=0; j<buffer->GetDim(); j++){
      sprintf(tmp, "%d", buffer->GetIndex(j)-1);
      width[j] = strlen(tmp);
      index_width+=width[j]+2; // '[' and ']'
    }
    items=formatting(fmt, index_width, win_c);
    preindex=0;
  }
  idx=buffer->rIndex(i); // convert i to Index
  if(objtype==SL_OBJ::SERIES_O && buffer->GetDim()!=1 && idx.GetIndex(0)!=preindex)
    syscom->console->tty_printf("\n");
  if(idx.GetIndex(buffer->GetDim()-1)%items==0){
    syscom->console->tty_printf("\n");
    for(j=0; j<buffer->GetDim(); j++){
      syscom->console->tty_printf("[%*d]", width[j], idx.GetIndex(j));
      if(j==0 && objtype==SL_OBJ::SERIES_O)
        syscom->console->tty_printf(":");
    }
    syscom->console->tty_printf("%% ");
  }
  preindex = idx.GetIndex(0);
}

int SL_Object::formatting(char *fmt, size_t index_width, int win_c){
  int m,columns;
  int obj_width = 0;
  if(fmt!=0)
    sscanf(fmt+1, "%d", &obj_width);
  if(obj_width == 0)
    obj_width=12;
  else if(obj_width < 0)
    obj_width = -obj_width;
  columns=(int)(win_c-index_width-3); /* TODO: remove cast */
  m=columns / (obj_width + 1);
  m = (m > 5 && m%2 == 1) ? m-1 : m; // odd --> even number 
  m = (m == 0) ? 1 : m;              // 0   --> 1
  return m;
}

// scalar opecode
SL_Object *SL_Object::scalar_op(int dptr_n, SL_Object *obj){
  static char *where="(scalar opration)";
  SL_Object   *ret_obj;
  Base_Buffer *ret_buf,*src_buf;
  double *d, *dst;
  int i,size;
  SL_OBJ::TYPE type;
  OPEFUNC dptr = operators[dptr_n];

  // error check
  d=(double*)buffer->GetDataPointer();
  if(d==0)
    syscom->console->execerror("null argument or not scalar type", where);

  // if obj is null then single operator
  if(obj==0){ 
    ret_buf = new Scalar_Buffer;
    try{
      ret_buf->InitBuffer();
      ((Scalar_Buffer*)ret_buf)->SetScalar((*dptr)(*d,0.0));
      ret_obj=new SL_Object(SL_OBJ::SCALAR_O,ret_buf);
    }catch(buffer_exception err){
      delete ret_buf;
      syscom->console->execerror(err.what(),where);
    }catch(bad_alloc){
      delete ret_buf;
      throw;
    }
    return ret_obj;
  }

  // else...
  type=obj->TypeofOBJ();
  if(type!=SL_OBJ::SCALAR_O && type!=SL_OBJ::STRING_O &&
     type!=SL_OBJ::SERIES_O && type!=SL_OBJ::SNAPSHOT_O)
    syscom->console->execerror("illegal object type",where);

  src_buf = obj->GetBufferPointer();
  try{
    ret_buf = getbuffer(src_buf, type);
  }catch(buffer_exception err){
    syscom->console->execerror(err.what(),where);
  }
  size    = ret_buf->IndexSize();
  dst     = (double*)ret_buf->GetDataPointer();
  for(i=0;i<size;i++)
    dst[i]=(*dptr)(*d,dst[i]);
  if(size==1){
    ret_obj=new SL_Object(SL_OBJ::SCALAR_O,ret_buf);
  }else{
    if(type == SL_OBJ::SNAPSHOT_O)
      ret_obj=new SL_Object(SL_OBJ::SNAPSHOT_O,ret_buf);
    else
      ret_obj=new SL_Object(SL_OBJ::SERIES_O,ret_buf);
  }
  return ret_obj;
}

// operation of series
SL_Object *SL_Object::series_op(int dptr_n, SL_Object *obj){
  static char *where="(series opration)";
  SL_Object *ret_obj;
  Base_Buffer *ret_buf=0,*buffer2=0;
  int    i,j,t,time=0,size=0, dimx, dimy;;
  bool   is_single, is_scalar, is_series;
  double *bufp=0,*bufx=0,*bufy=0;
  OPEFUNC dptr = operators[dptr_n];
  is_single=(obj == 0);
  is_scalar=(is_single) ? false : (obj->TypeofOBJ() == SL_OBJ::SCALAR_O);
  is_series=(is_single) ? false :
    ((obj->TypeofOBJ() == SL_OBJ::SERIES_O) || 
     (obj->TypeofOBJ() == SL_OBJ::SNAPSHOT_O));

  if(!is_single && obj->TypeofOBJ() == SL_OBJ::STRING_O && 
     obj->GetBufferPointer()!=0){
    buffer2=new Series_Buffer;
    buffer2->CopyIndex(obj->GetBufferPointer()->GetBufferIndex());
    try{
      buffer2->InitBuffer();
    }catch(bad_alloc){
      delete buffer2;
      throw;
    }catch(buffer_exception err){
      delete buffer2;
      syscom->console->execerror(err.what(),where);
    }
    for(i=0;i<obj->GetBufferPointer()->IndexSize();i++){
      ((double *)buffer2->GetDataPointer())[i]=
        atof(((string*)obj->GetBufferPointer()->GetDataPointer())[i].c_str());
    }
    if(obj->GetBufferPointer()->IndexSize()==1)is_scalar=true;
    if(obj->GetBufferPointer()->IndexSize()>1)is_series=true;
  }else{
    buffer2 = (is_single) ? buffer : obj->GetBufferPointer();
  }
  dimx = dimy = buffer->GetDim();
  if(!is_single && !is_scalar)
    dimy = buffer2->GetDim();
  if(dimx!=dimy)
    syscom->console->execerror("dimension mismatch", where);
  bufx=(double*)buffer->GetDataPointer();
  bufy=(double*)buffer2->GetDataPointer();
  try{
    if(buffer->GetIndex(0)>=buffer2->GetIndex(0)){
      ret_buf = buffer->duplicate();
    }else{
      ret_buf = buffer2->duplicate();
    }
  }catch(buffer_exception err){
    delete ret_buf;
    syscom->console->execerror(err.what(),where);
  }catch(bad_alloc){
    delete ret_buf;
    throw;
  }
  time=Min(buffer->GetIndex(0),
           (is_scalar)?buffer->GetIndex(0):buffer2->GetIndex(0));
  size=ret_buf->SubIndexSize();
  if(size==0)size=1;
  bufp=(double*)ret_buf->GetDataPointer();
  if(is_series || is_scalar || is_single){
    double d=0.0;
    if(is_scalar)
      d=bufy[0];
    for(t=0; t<time; t++){
      for(i=0; i<size; i++){
        j=t*size+i;
        if(!is_single){
          if(!is_scalar) d=bufy[j];
          bufp[j]=(*dptr)(bufx[j], d);
        }else{
          bufp[j]=(*dptr)(bufx[j],0.0);
        }
      }
    }
  }else{
    delete ret_buf;
    syscom->console->execerror("illegal object type", where);
  }
  try{
    if(objtype == SL_OBJ::SNAPSHOT_O)
      ret_obj=new SL_Object(SL_OBJ::SNAPSHOT_O,ret_buf);
    else
      ret_obj=new SL_Object(SL_OBJ::SERIES_O,ret_buf);
  }catch(buffer_exception err){
    delete ret_buf;
    syscom->console->execerror(err.what(),where);
  }catch(bad_alloc){
    delete ret_buf;
    throw;
  }
  if(!is_single && obj->TypeofOBJ() == SL_OBJ::STRING_O && 
     obj->GetBufferPointer()!=0){
    delete buffer2;
  }
  return ret_obj;
}

// operation of snapshot
SL_Object *SL_Object::snapshot_op(int dptr_n, SL_Object *obj){
  static char *where = "(snapshot operation)";
  int is_single;//, is_scalar, is_series;
  is_single=(obj == 0);
  // is_scalar=(is_single) ? 0 : (obj->TypeofOBJ() == SL_OBJ::SCALAR_O);
  // is_series=(is_single) ? 0 : 
  //   ((obj->TypeofOBJ() == SL_OBJ::SERIES_O) || (obj->TypeofOBJ() == SL_OBJ::SNAPSHOT_O));
  if(!is_single && (obj->TypeofOBJ()==SL_OBJ::SNAPSHOT_O || 
                    obj->TypeofOBJ()==SL_OBJ::SERIES_O)){
    if(buffer->GetDim()!=obj->GetBufferPointer()->GetDim())
      syscom->console->execerror("dimension mismatch",where);
    if(!buffer->EqualIndex(obj->GetBufferPointer()->GetBufferIndex()))
      syscom->console->execerror("index mismatch",where);
  }
  return series_op(dptr_n,obj);
}

// operation of string
SL_Object *SL_Object::string_op(int dptr_n, SL_Object *obj){
  static char *where = "(string operation)";
  SL_Object *ret_obj;
  Base_Buffer *ret_buf,*src_buf,*dst_buf;
  int  a_word;
  SL_OBJ::TYPE ret_type,src_type;
  SOPEFUNC method;
  /* search for method */
  switch(dptr_n){
  case MATH_ADD: method = string_add; break;
  case MATH_SUB: method = string_sub; break;
  case MATH_MUL: method = string_mul; break;
  case MATH_DIV: method = string_div; break;
  case MATH_EQ:  method = string_eq;  break;
  case MATH_NE:  method = string_ne;  break;
  default: 
    syscom->console->execerror("not supported method",where);
    break;
  }

  src_type=obj->TypeofOBJ();
  if(src_type!=SL_OBJ::SCALAR_O && src_type!=SL_OBJ::STRING_O &&
     src_type!=SL_OBJ::SERIES_O && src_type!=SL_OBJ::SNAPSHOT_O)
    syscom->console->execerror("illegal object type",where);
  dst_buf = buffer;
  try{
    src_buf = getstring(obj->GetBufferPointer(),src_type);
  }catch(buffer_exception err){
    syscom->console->execerror(err.what(),where);
  }catch(execerr_exception){
    throw;
  }catch(bad_alloc){
    throw;
  }
  a_word=(Min(src_buf->IndexSize(),dst_buf->IndexSize()) == 1); // one word ?
  if(!a_word && src_buf->GetDim() != dst_buf->GetDim()){
    delete src_buf;
    syscom->console->execerror("illegal index",where);
  }
  if(!a_word && src_buf->EqualIndex(dst_buf->GetBufferIndex())){
    delete src_buf;
    syscom->console->execerror("illegal index",where);
  }
  try{
    ret_buf=method((String_Buffer*)dst_buf,(String_Buffer*)src_buf);
  }catch(buffer_exception err){
    delete src_buf;
    syscom->console->execerror(err.what(),where);
  }catch(execerr_exception){
    delete src_buf;
    throw;
  }catch(bad_alloc){
    delete src_buf;
    throw;
  }
  delete src_buf;
  if(dptr_n == MATH_EQ || dptr_n == MATH_NE){ // comp
    if(ret_buf->IndexSize()==1) ret_type=SL_OBJ::SCALAR_O;
    else ret_type=SL_OBJ::SERIES_O;
  }else{
    ret_type = SL_OBJ::STRING_O;
  }
  ret_obj=new SL_Object(ret_type,ret_buf);
  return ret_obj;
}

char *SL_Object::TypeName(){
  static char *types[]={
    "series","snapshot","string","scalar","builtin","unknown"};
  char *ret;
  switch(objtype){
  case SL_OBJ::SERIES_O:
    ret=types[0];break;
  case SL_OBJ::SNAPSHOT_O:
    ret=types[1];break;
  case SL_OBJ::STRING_O:
    ret=types[2];break;
  case SL_OBJ::SCALAR_O:
    ret=types[3];break;
  case SL_OBJ::BUILTIN_O:
    ret=types[4];break;
  default:
    ret=types[5];break;
  }
  return ret;
}

// private functions
static Base_Buffer *getbuffer(Base_Buffer *src, SL_OBJ::TYPE type){
  int    i,size;
  double *buf;
  Base_Buffer *ret;
  size = src->IndexSize();
  if(size == 1){
    ret = new Scalar_Buffer;
  }else{
    ret = new Series_Buffer;
    ret->CopyIndex(src->GetBufferIndex());
  }
  try{
    ret->InitBuffer();
  }catch(buffer_exception err){
    delete ret;
    throw;
  }catch(bad_alloc){
    delete ret;
    throw;
  }
  buf = (double*)ret->GetDataPointer();
  if(type == SL_OBJ::STRING_O){
    string *str = (string*)src->GetDataPointer();
    for(i=0; i<size; i++)
      buf[i] = atof(str[i].c_str());
  }else{
    double *d = (double*)src->GetDataPointer();
    for(i=0; i<size; i++)
      buf[i] = d[i];
  }
  return ret;
}

static Base_Buffer *getstring(Base_Buffer *src, SL_OBJ::TYPE type){
  String_Buffer *ret;
  Index point;
  char sbuf[100];
  ret = new String_Buffer;
  try{
    ret->CopyIndex(src->GetBufferIndex());
    ret->InitBuffer();
    switch(type){
    case SL_OBJ::STRING_O:
      ret->CopyBuffer(src);
      break;
    case SL_OBJ::SCALAR_O:
      point.SetDim(1);
      sprintf(sbuf,"%g",((Scalar_Buffer*)src)->GetScalar());
      ret->SetString(point,sbuf);
    case SL_OBJ::SERIES_O:
    case SL_OBJ::SNAPSHOT_O:
      {
        int i,size;
        size=ret->IndexSize();
        for(i=0;i<size;i++){
          point=src->rIndex(i);
          sprintf(sbuf,"%g",((Series_Buffer*)src)->GetData(point));
          ret->SetString(point,sbuf);
        }
      }
      break;
    }
  }catch(buffer_exception err){
    delete ret;
    throw;
  }catch(bad_alloc){
    delete ret;
    throw;
  }
  return ret;
}


// for string operation

static Base_Buffer *string_add(String_Buffer *s1, String_Buffer *s2){
  static char *where="(string add)";
  int   i=0,size=0,u1=0,u2=0;
  Index point;
  String_Buffer *buf=0;
  const char *str1=0,*str2=0;
  char *work=0;
  u1=s1->IndexSize();
  u2=s2->IndexSize();
  buf=new String_Buffer;
  // Calculate large value
  size=Max(u1,u2);
  try{
    if(u1>u2) buf->CopyIndex(s1->GetBufferIndex());
    else      buf->CopyIndex(s2->GetBufferIndex());
    buf->InitBuffer();
    for(i=0;i<size;i++){
      point=buf->rIndex(i);
      if(i==0){
        str1=s1->GetString(point);
        str2=s2->GetString(point);
      }else{
        if(u1!=1)str1=s1->GetString(point);
        if(u2!=1)str2=s2->GetString(point);
      }
      if(str1!=0 && str2!=0){
        work=new char[strlen(str1)+strlen(str2)+1];
        strcpy(work,str1);
        try{
          buf->SetString(point,strcat(work,str2));
        }catch(bad_alloc){ 
          delete [] work; delete buf; throw; 
        }catch(buffer_exception err){
          delete [] work; delete buf;
          syscom->console->execerror(err.what(),where);
        }
        delete [] work;
      }
      else if(str1!=0)
        buf->SetString(point,str2);
      else
        buf->SetString(point,str1);
    }
  }catch(buffer_exception err){
    delete buf;
    syscom->console->execerror(err.what(),where);
  }catch(bad_alloc){
    delete buf;
    throw;
  }
  return buf;
}

static Base_Buffer *string_sub(String_Buffer *s1, String_Buffer *s2){
  static char *where="(string sub)";
  size_t j;
  int   i,size,u1,u2;
  Index point;
  String_Buffer *buf=0;
  const char *str1=0,*str2=0;
  char *work;
  u1=s1->IndexSize();
  u2=s2->IndexSize();
  buf = new String_Buffer;
  // Calculate large value
  size=Max(u1,u2); 
  try{
    if(u1>u2) buf->CopyIndex(s1->GetBufferIndex());
    else      buf->CopyIndex(s2->GetBufferIndex());
    buf->InitBuffer();
    for(i=0; i<size; i++){
      point=buf->rIndex(i);
      if(i==0){
        str1=s1->GetString(point);
        str2=s2->GetString(point);
      }else{
        if(u1!=1)str1=s1->GetString(point);
        if(u2!=1)str2=s2->GetString(point);
      }
      if(str1!=0){
        work=new char[strlen(str1)+1];
        strcpy(work,str1);
      }else{
        work="";
      }
      if(str1!=0 && str2!=0){
        for(j = strlen(work); j > 0; j--){
          if(!strcmp(&work[j], str2)){
            work[j] = '\0';
            break;
          }
        }
        if(!strcmp(&work[j], str2)){ work[j] = '\0'; }
      }
      buf->SetString(point,work);
      delete work;
    }
  }catch(buffer_exception err){
    delete buf;
    syscom->console->execerror(err.what(),where);
  }catch(bad_alloc){
    delete buf;
    throw;
  }
  return buf;
}

static Base_Buffer *string_mul(String_Buffer *s1, String_Buffer *s2){
  static char *where="(string mul)";
  int   j,i,size,u1,u2,n=0;
  Index point;
  String_Buffer *buf=0;
  const char *str1=0,*str2=0;
  char *work=0;
  u1=s1->IndexSize();
  u2=s2->IndexSize();
  buf=new String_Buffer;
  // Calculate large value
  size=Max(u1,u2); 
  try{
    if(u1>u2) buf->CopyIndex(s1->GetBufferIndex());
    else      buf->CopyIndex(s2->GetBufferIndex());
    buf->InitBuffer();
    for(i=0; i<size; i++){
      point=buf->rIndex(i);
      if(i==0){
        str1=s1->GetString(point);
        str2=s2->GetString(point);
        n=atoi(str2);
      }else{
        if(u1!=1)str1=s1->GetString(point);
        if(u2!=1){
          str2=s2->GetString(point);
          n=atoi(str2);
        }
      }
      if(n!=0 && str1!=0){
        work= new char[strlen(str1)*n+1];
        strcpy(work,"");
        for(j=0;j<n;j++)
        strcat(work,str1);
        try{
          buf->SetString(point,work);
        }catch(bad_alloc){ delete work; throw; }
        delete work;
      }
    }
  }catch(buffer_exception err){
    delete buf;
    syscom->console->execerror(err.what(),where);
  }catch(bad_alloc){
    delete buf;
    throw;
  }
  return buf;
}

static Base_Buffer *string_div(String_Buffer *s1, String_Buffer *s2){
  static char *where="(string mul)";
  int   n,i,size,siz1,siz2,d1,d2,u1,u2,dim;
  Index idx,point, pt1, pt2;
  String_Buffer *buf=0;
  const char *str1=0,*str2=0;
  char *work, *p;
  d1=s1->GetDim();
  d2=s2->GetDim();
  siz1=s1->IndexSize();
  siz2=s2->IndexSize();
  dim = (siz1 == 1) ? d2 : d1;
  if(siz1 == 1){
    idx = s2->GetBufferIndex();
  }else{
    idx = s1->GetBufferIndex();
  }
  u1 = (siz1 == 1) ? 0 : 1;
  u2 = (siz2 == 1) ? 0 : 1;

  buf=new String_Buffer;
  // calcurate large value
  try {
    size = idx.IndexSize();
    buf->CopyIndex(idx);
    if(dim != 1){
      char tmp[20];
      sprintf(tmp, "(%d) -> (1)", dim);
      syscom->console->warning("change dimension", tmp);
      buf->SetDim(1);
      buf->SetIndex(0,size);
    }
    buf->InitBuffer();
    for(n=0,i=0; i<size; i++){
      pt1 = s1->rIndex(i * u1);
      pt2 = s2->rIndex(i * u2);
      str1 = s1->GetString(pt1);
      str2 = s2->GetString(pt2);
      work=new char[strlen(str1)+1];
      strcpy(work, str1);
      p=strtok(work, str2);
      while(p!=0){
        point.SetDim(1);
        point.SetIndex(0,n);
        if(n > size) buf->ReSize(n);
        buf->SetString(point,work);
        n++;
        p=strtok(0,str2);
      }
      delete [] work;
    }
    if(i == 0){
      point.SetDim(1);
      point.SetIndex(0,1);
      buf->ReSize(1);
      buf->SetString(point,"");
    }
  }catch(buffer_exception err){
    delete buf;
    syscom->console->execerror(err.what(),where);
  }catch(bad_alloc){
    delete buf;
    throw;
  }
  return buf;
}

static Base_Buffer *string_comp(bool is_eq, String_Buffer *s1, 
                                String_Buffer *s2){
  static char *where="(string compare)";
  Index point;
  int   i,size,stat,u1,u2;
  Series_Buffer *buf=0;
  const char *str1=0,*str2=0;
  buf=new Series_Buffer;
  u1=s1->IndexSize();
  u2=s2->IndexSize();
  // Calculate large value
  size=Max(u1,u2); 
  if(u1>u2) buf->CopyIndex(s1->GetBufferIndex());
  else      buf->CopyIndex(s2->GetBufferIndex());
  try{
    buf->InitBuffer();
    for(i=0;i<size;i++){
      point=buf->rIndex(i);
      if(i==0){
        str1=s1->GetString(point);
        str2=s2->GetString(point);
      }else{
        if(u1!=1)str1=s1->GetString(point);
        if(u2!=1)str2=s2->GetString(point);
      }
      if(str1!=0 && str2!=0)
        stat=strcmp(str1,str2);
      else
        stat=1;
      if(is_eq){
        if(stat==0) buf->SetData(point,1.0);
        else buf->SetData(point,0.0);
      }else{
        if(stat==0) buf->SetData(point,0.0);
        else buf->SetData(point,1.0);
      }
    }
  }catch(buffer_exception err){
    delete buf;
    syscom->console->execerror(err.what(),where);
  }catch(bad_alloc){
    delete buf;
    throw;
  }
  return buf;
}

static Base_Buffer *string_eq(String_Buffer *s1, String_Buffer *s2){
  return string_comp(true, s1,s2);
}

static Base_Buffer *string_ne(String_Buffer *s1, String_Buffer *s2){
  return string_comp(false, s1,s2);
}

