﻿// //////////////////////////////////////////////////////////////////////////
//
// SoundGraph.cs
//              Sarbo Project
//          Powerd by CACTUS SOFTWARE <http://www.cactussoft.co.jp/>
//
//                                              ver 1.0  2009/08/28
// This file is part of Sarbo.
//
// Sarbo is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Sarbo is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Sarbo.  If not, see <http://www.gnu.org/licenses/>.
//
// //////////////////////////////////////////////////////////////////////////
using System;
using System.IO;
using System.Runtime.InteropServices;
using DShowCAC;

namespace DShowPlay
{
	public class SoundGraph
	{
		private IGraphBuilder		_iGraphBuilder = null;
		private IMediaControl		_iMediaControl = null;
		private IMediaSeeking		_iMediaSeeking = null;
		private IMediaEventEx		_iMediaEventEx = null;
		private IBasicAudio			_iBasicAudio = null;

		private FilterState			_filterState = FilterState.Stopped;
		private bool				_renderFile = false;
		private int					_register = 0;

		// //////////////////////////////////////////////////////////////////
		// Init

		public void InitInstance()
		{
			Type comType = Type.GetTypeFromCLSID(CLSID.FilterGraph);
			object comObj = Activator.CreateInstance(comType);
			_iGraphBuilder = (IGraphBuilder)comObj; comObj = null;

			_iMediaControl = (IMediaControl)_iGraphBuilder;
			_iMediaSeeking = (IMediaSeeking)_iGraphBuilder;
			_iMediaEventEx = (IMediaEventEx)_iGraphBuilder;
			_iBasicAudio = (IBasicAudio)_iGraphBuilder;
#if DEBUG
			int result = ROT.AddToRot(_iGraphBuilder, out _register);
			if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);
#endif
		}

		// //////////////////////////////////////////////////////////////////
		// Exit

		public void ExitInstance()
		{
#if DEBUG
			ROT.RemoveFromRot(ref _register);
#endif
			if (_iGraphBuilder != null)
				Marshal.ReleaseComObject(_iGraphBuilder); _iGraphBuilder = null;
			if (_iMediaControl != null)
				Marshal.ReleaseComObject(_iMediaControl); _iMediaControl = null;
			if (_iMediaSeeking != null)
				Marshal.ReleaseComObject(_iMediaSeeking); _iMediaSeeking = null;
			if (_iMediaEventEx != null)
				Marshal.ReleaseComObject(_iMediaEventEx); _iMediaEventEx = null;
			if (_iBasicAudio != null)
				Marshal.ReleaseComObject(_iBasicAudio); _iBasicAudio = null;

			_renderFile = false;
		}

		// //////////////////////////////////////////////////////////////////
		// RenderSongFile

		public void RenderSongFile(string strFilePath)
		{
			int result = HResult.OK;

			if (_iGraphBuilder == null)
				Marshal.ThrowExceptionForHR(HResult.NoInterface);

			if (IsWindowsMediaFile(strFilePath))
			{
				result = RenderWindowsMediaFile(strFilePath);
				if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);
			}
			else
			{
				result = _iGraphBuilder.RenderFile(strFilePath, null);
				if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);
			}
			_renderFile = true;
		}

		// //////////////////////////////////////////////////////////////////
		// RenderWindowsMediaFile

		private int RenderWindowsMediaFile(string strFilePath)
		{
			int result = HResult.OK;
			IBaseFilter iReader = null;
			IFileSourceFilter iFS = null;

			try
			{
				result = CreateFilter(CLSID.WMAsfReader, out iReader);
				if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);

				result = _iGraphBuilder.AddFilter(iReader, "ASF Reader");
				if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);

				iFS = (IFileSourceFilter)iReader;

				result = iFS.Load(strFilePath, null);
				if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);

				result = RenderOutputPins(_iGraphBuilder, iReader);
				if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);
			}
			catch (Exception)
			{
			}
			finally
			{
				if (iReader != null)
					Marshal.ReleaseComObject(iReader); iReader = null;
				if (iFS != null)
					Marshal.ReleaseComObject(iFS); iFS = null;
			}
			return result;
		}

		// //////////////////////////////////////////////////////////////////
		// CreateFilter

		public int CreateFilter(
			Guid				clsid,
			out IBaseFilter		ppFilter)
		{
			Type comType = Type.GetTypeFromCLSID(clsid);
			object comObj = Activator.CreateInstance(comType);
			ppFilter = (IBaseFilter)comObj; comObj = null;

			return HResult.OK;
		}

		// //////////////////////////////////////////////////////////////////
		// RenderOutputPins

		private int RenderOutputPins(
			IGraphBuilder		iGB,
			IBaseFilter			iFilter)
		{
			IEnumPins iEnumPin = null;
			int result = iFilter.EnumPins(out iEnumPin);

			if (HResult.Succeeded(result))
			{
				IPin[] iPins = new IPin[1];
				int pcFetched = 0;

				while (true)
				{
					result = iEnumPin.Next(1, iPins, out pcFetched);
					if (result != HResult.OK)		break;
					if (iPins[0] == null)			break;

					IPin iConnectedPin = null;
					result = iPins[0].ConnectedTo(out iConnectedPin);

					if (iConnectedPin != null)
					{
						Marshal.ReleaseComObject(iConnectedPin); iConnectedPin = null;
					}
					if (result == HResult.VfwENotConnected)
					{
						PinDirection pinDir;
						result = iPins[0].QueryDirection(out pinDir);

						if ((result == HResult.OK)&&(pinDir == PinDirection.Output))
						{
							result = iGB.Render(iPins[0]);
						}
					}
					Marshal.ReleaseComObject(iPins[0]); iPins[0] = null;

					if (HResult.Failed(result))		break;
				}
			}
			Marshal.ReleaseComObject(iEnumPin); iEnumPin = null;

			return result;
		}

		// //////////////////////////////////////////////////////////////////
		// NotifyWindow

		public void SetNotifyWindow(IntPtr hWnd)
		{
			if (_iMediaEventEx == null)
				Marshal.ThrowExceptionForHR(HResult.NoInterface);

			int result = _iMediaEventEx.SetNotifyWindow(hWnd, WM.GraphNotify, 0);
			if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);
		}

		public void ClearNotifyWindow()
		{
			if (_iMediaEventEx == null)	return;

			int result = _iMediaEventEx.SetNotifyWindow(IntPtr.Zero, 0, 0);
			if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);
		}

		// //////////////////////////////////////////////////////////////////
		// StartPlay/StopPlay

		public void StartPlay()
		{
			if (_filterState == FilterState.Running)		return;

			if (_iMediaControl == null)
				Marshal.ThrowExceptionForHR(HResult.NoInterface);

			int result = _iMediaControl.Run();
			if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);

			_filterState = FilterState.Running;
		}

		// //////////////////////////////////////////////////////////////////
		public void StopPlay()
		{
			if (_filterState == FilterState.Stopped)		return;

			if (_iMediaControl == null)
				Marshal.ThrowExceptionForHR(HResult.NoInterface);

			int result = _iMediaControl.Stop();
			if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);

			_filterState = FilterState.Stopped;
		}

		// //////////////////////////////////////////////////////////////////
		// WaitForStop

		public void WaitForStop()
		{
			if (_iMediaControl == null)	return;

			FilterState fs;
			int result = _iMediaControl.GetState(500, out fs);
			if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);
		}

		// //////////////////////////////////////////////////////////////////
		// SeekPosition

		public int SeekPosition(int position)
		{
			if (_iMediaSeeking == null)	return HResult.OK;

			int result = HResult.OK;

			try
			{
				ReferenceTime current = new ReferenceTime(position * TimeSpan.TicksPerSecond);
				result = _iMediaSeeking.SetPositions(current, SeekingFlags.AbsolutePositioning, null, 0);
				if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);
			}
			catch (Exception)
			{
			}
			return result;
		}

		// //////////////////////////////////////////////////////////////////
		// IsComplete

		public bool IsComplete()
		{
			EvCode lEventCode;
			int lParam1, lParam2;

			if (_iMediaEventEx == null)	return false;

			while (HResult.Succeeded(_iMediaEventEx.GetEvent(out lEventCode, out lParam1, out lParam2, 0)))
			{
				_iMediaEventEx.FreeEventParams(lEventCode, lParam1, lParam2);

				switch (lEventCode)
				{
				case EvCode.Complete:
				case EvCode.UserAbort:
				case EvCode.ErrorAbort:
					return true;
				}
			}
			return false;
		}

		// //////////////////////////////////////////////////////////////////
		// IsComplete

		public bool IsRenderFile()
		{
			return _renderFile;
		}

		// //////////////////////////////////////////////////////////////////
		// GetState

		public FilterState GetState()
		{
			return _filterState;
		}

		// //////////////////////////////////////////////////////////////////
		// Stop And Exit

		public void StopAndExit()
		{
			try
			{
				StopPlay();
				WaitForStop();
				ClearNotifyWindow();
				ExitInstance();
			}
			catch (Exception)
			{
			}
		}

		// //////////////////////////////////////////////////////////////////
		// IsWindowsMediaFile

		public bool IsWindowsMediaFile(string strFile)
		{
			switch (Path.GetExtension(strFile))
			{
			case ".asf":
			case ".wma":
			case ".wmv":
				return true;
			default:
				return false;
			}
		}
	}
}

// //////////////////////////////////////////////////////////////////////////
