/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 * 
 *    Project              : SagCAD
 *    Description          : CAD/CAM
 *    Source               : PostScript.c
 * 
 *    ----------------------------------
 * 
 *    License              : GNU General Public License (GPL)
 *    Copyright            : (C) 1998-2007 by Yutaka Sagiya
 *    email                : kappa@a6s.highway.ne.jp
 *                         : yutaka@sagiya.com
 *    Begin                : 2003/04/01
 *    Last                 : 2007/10/08
 * ====================================================================
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "intl.h"

#include "MemoryLeak.h"
#include "List_cad.h"
#include "List_Dimension.h"
#include "List_PolyLine.h"
#include "List_Ellipse.h"
#include "List_insert.h"
#include "List_Block.h"
#include "List_Undo.h"
#include "List_Select.h"
#include "MsgBox.h"
#include "sagcad.h"
#include "spline.h"
#include "B_spline.h"
#include "InputBox.h"

#include "global.h"
#include "culcfunc.h"
#include "Ellipse.h"
#include "etc.h"
#include "font.h"
#include "Assistance.h"

#define _POSTSCRIPT_
#include "PostScript.h"





/* -------------------------------------------------------------------
 * PostScript 形式で保存
 * 

         cm          points
a3  29.7 x 42.0    842 x 1190
a4  21.0 x 29.7    595 x  842
a5  14.9 x 21.0    421 x  595
b5  17.7 x 25.0    501 x  709
1 dict dup /PageSize [842 595] put setpagedevice /ImagingBBox [20 20 802 555] put setpagedevice

 */
int PostScript(char *SaveFileName)
{
//#define POSTSCRIPT_DEBUG
//	char str[256];
	char FileName[256];
	FILE *stream;

	CAD_LIST *p;
	DIMENSION_LIST *pd;
	POLYLINE_LIST *pld;
	ELLIPSE_LIST *pe;
	int ps = 5, debug = 0;


#ifdef POSTSCRIPT_DEBUG
	debug = 1;
#endif


	if (debug > 0) g_print("SaveFileName = %s\n", SaveFileName);


	/* ファイルを確認 */
	sprintf(FileName, "%s", SaveFileName);
	//cutExtName(FileName);
	if (strstr(SaveFileName, ".ps") != NULL) {
		//strcat(FileName, ".ps");
		ps = 1;
	}
	else if (strstr(SaveFileName, ".eps") != NULL) {
		//strcat(FileName, ".eps");
		ps = 0;
	}
	else {
		return 0;
	}

	/* Open */
	if ((stream = fopen( FileName, "w")) == NULL) {
		printf(_("The file could not be opened. [%s]\n"), FileName);
	}


	if (debug > 0) g_print("FileName = %s\n", FileName);


	/* PostScript の記述 */
	set_start_up(stream, ps);


	/* ViewPort の設定 */
	set_paper_size_ViewPort (printer, ps, stream);
	/* WorldWindow の設定 */
	set_world_size_Window (printer);


	if (debug > 0) g_print("ViewPort(%f,%f)-(%f,%f)  ViewPortAspect = %f\n", 
						   ViewPortX1, ViewPortY1, ViewPortX2, ViewPortY2, ViewPortAspect);
	if (debug > 0) g_print("WorldWindow (%f,%f)-(%f,%f)\n",WorldWindowX1, WorldWindowY1, WorldWindowX2, WorldWindowY2);


	fputs("newpath\n", stream );

	/* -----------------------------------------------------------------
	 * 図形描画 
	 */
	p = cad_list_info.head;
	while(p != NULL) {
		if(Layer[p->cad->layer].draw == 1) {
			switch (p->cad->code) {
				case 1: /* 直線のとき */
					LineDraw_PS(stream,	
							 p->cad->sx, p->cad->sy, p->cad->ex, p->cad->ey, 
							 p->cad->style, p->cad->color);
					break;
				case 2: /* 円弧のとき */
					ArcDraw_PS(stream, 
							p->cad->cx, p->cad->cy, p->cad->r, 
							p->cad->sx, p->cad->sy, p->cad->ex, p->cad->ey, 
							p->cad->style, p->cad->color);
					break;
				case 4: /* 円のとき */
					CircleDraw_PS(stream,	
							p->cad->cx, p->cad->cy, p->cad->r, 
							p->cad->style, p->cad->color);
					break;
			}
		}
		p = p->next;	/* ポインタを次のデータに移す */
	}



	/* -----------------------------------------------------------------
	 * 寸法描画
	 */
	/* 各寸法要素 */
	pd = dimension_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(pd != NULL) {
		if(Layer[pd->dimension->Layer].draw == 1) {
			Dimension_Draw_Line_PS (stream, pd->dimension);
			DrawTextVector_PS (stream, pd->dimension);
			
		}
		pd = pd->next;	/* ポインタを次のデータに移す */
	}



	/* -----------------------------------------------------------------
	 * ポリライン描画
	 */
	/* 各ポリライン要素 */
	pld = polyline_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(pld != NULL) {
		if(Layer[pld->polyline->layer].draw == 1) {
			PolyLineDraw_PS (stream, pld->polyline);
		}
		pld = pld->next;	/* ポインタを次のデータに移す */
	}



	/* -----------------------------------------------------------------
	 * 楕円描画
	 */
	pe = ellipse_list_info.head;
	while (pe != NULL) {
		if(Layer[pe->ellipse->layer].draw == 1) {
			EllipseDraw_PS(stream, pe->ellipse);
		}
		pe = pe->next;	/* ポインタを次のデータに移す */
	}



	fputs("stroke\n", stream );
	/* eps では、showpage を出力してはいけないらしい */
	if (ps == 1) {
		fputs("showpage\n", stream );
	}
	/* Close */
	fclose(stream);
	printf(_("The file saved. [%s]\n"), FileName);
	return 1;
}










/* ===================================================================
 * ----- < PostScript セット > -----
 * ===================================================================
 */


/* -------------------------------------------------------------------
 * 開始処理
 * 
 */
int set_start_up(FILE *stream, int ps)
{
	char str[256];

	/* PostScript の記述 */
	if (ps == 1) {
		sprintf(str, "%%!PS-Adobe-3.0\n");
	}
	else {
		sprintf(str, "%%!PS-Adobe-2.0 EPSF-1.2\n");
	}
	fputs(str, stream );
	return 1;
}


/* -------------------------------------------------------------------
 * 線幅設定
 * 
 */
int set_line_width(FILE *stream, double width)
{
	char str[256];

	sprintf(str, "closepath\nstroke\nnewpath\n%f setlinewidth\n", width);
	fputs(str, stream );
	return 1;
}


/* -------------------------------------------------------------------
 * 線色設定
 * 
 */
int set_line_color(FILE *stream, long color)
{
	char str[256];
	struct ColorDat coldat;

	coldat.rgb = color;
	color_split(&coldat);

	sprintf(str, "%f %f %f setrgbcolor\n", 
				 ((double)coldat.red)/256, 
				 ((double)coldat.green)/256, 
				 ((double)coldat.blue)/256);
	fputs(str, stream );
	return 1;
}

/* -------------------------------------------------------------------
 * 線種設定
 * 
 */
int set_line_style(FILE *stream, int style)
{
//	char str[256];

//[15 1 1 1] 0 setdash


//	sprintf(str, "[%f] 0 setdash\n", width);
//	fputs(str, stream );
	return 1;
}










/* ===================================================================
 * ----- < 座標変換セット > -----
 * ===================================================================
 */


/* -------------------------------------------------------------------
 * ビューポート(デバイス座標系)の設定
 * 
 *                     (WX2,WY2)                 (SX2,SY2)
 *      ┌────────┐             ┌─────┐
 *      │   ・           │             │  ・      │
 *      │ (WXp,WYp)      │             │ (SXp,SYp)│
 *      │                │             │          │
 *      │                │             │          │
 *      └────────┘             └─────┘
 *   (WX1,WY1)                       (SX1,SY1)         
 *           ウインドウ                    ビューポート
 * 
 */
int set_ViewPort(double SX1, double SY1, double SX2, double SY2)
{
//#define SET_VIEWPORT

	/* ViewPort Aspect */
	ViewPortAspect = (SX2-SX1) / (SY2-SY1);

	ViewPortX1 = SX1;
	ViewPortY1 = SY1;
	ViewPortX2 = SX2;
	ViewPortY2 = SY2;

#ifdef SET_VIEWPORT
g_print("ViewPort(%f,%f)-(%f,%f)  ViewPortAspect = %f\n", SX1, SY1, SX2, SY2, ViewPortAspect);
#endif

	return 1;
}



/* -------------------------------------------------------------------
 * ウインドウ(ワールド座標系)の設定
 * 
 * ViewPort を設定してから設定する
 * 
 *                     (WX2,WY2)                 (SX2,SY2)
 *      ┌────────┐             ┌─────┐
 *      │   ・           │             │  ・      │
 *      │ (WXp,WYp)      │             │ (SXp,SYp)│
 *      │                │             │          │
 *      │                │             │          │
 *      └────────┘             └─────┘
 *   (WX1,WY1)                       (SX1,SY1)         
 *           ウインドウ                    ビューポート
 * 
 */
int set_WorldWindow(double WX1, double WY1, double WX2, double WY2)
{
//#define SET_WORLDWINDOW
	double WXcenter, WYcenter, WXlength, WYlength;
	double WorldWindowAspect;
	int debug = 0;


#ifdef SET_WORLDWINDOW
	debug = 1;
#endif


	if (debug > 1) g_print("input window (%f,%f)-(%f,%f)\n", WX1, WY1, WX2, WY2);
	if (debug > 1) g_print("ViewPortAspect = %f\n", ViewPortAspect);

	/* WorldWindow Aspect */
	WorldWindowAspect = (WX2-WX1) / (WY2-WY1);


	/* 縦を基準 */
	if (ViewPortAspect > WorldWindowAspect) {
		if (debug > 0) g_print("縦を基準\n");
		WYlength = WY2 - WY1;
		WXlength = WYlength * ViewPortAspect;
	}
	/* 横を基準 */
	else {
		if (debug > 0) g_print("横を基準\n");
		WXlength = WX2 - WX1;
		WYlength = WXlength / ViewPortAspect;
	}


	WXcenter = (WX1 + WX2) / 2;
	WYcenter = (WY1 + WY2) / 2;

	WorldWindowX1 = WXcenter - (WXlength / 2);
	WorldWindowY1 = WYcenter - (WYlength / 2);
	WorldWindowX2 = WXcenter + (WXlength / 2);
	WorldWindowY2 = WYcenter + (WYlength / 2);

	if (debug > 1) g_print("output window (%f,%f)-(%f,%f)\n", WorldWindowX1, WorldWindowY1, WorldWindowX2, WorldWindowY2);

	return 1;
}



/* -------------------------------------------------------------------
 * Ｘをワールド座標からデバイス座標に変換
 */
double WorldToViewX(double x)
{
	return (x - WorldWindowX1) * (ViewPortX2 - ViewPortX1) / (WorldWindowX2 - WorldWindowX1) + ViewPortX1;
}

/* -------------------------------------------------------------------
 * Ｙをワールド座標からデバイス座標に変換
 */
double WorldToViewY(double y)
{
	return (y - WorldWindowY1) * (ViewPortY2 - ViewPortY1) / (WorldWindowY2 - WorldWindowY1) + ViewPortY1;
}





/* -------------------------------------------------------------------
 * 紙のサイズによるビューポート設定
 * 
 * A3 : 842 x 1191   297.039 x 420.159
 * A4 : 595 x  842   209.903 x 297.039
 * A5 : 420 x  595   148.167 x 209.903
 * B4 : 729 x 1032   257.175 x 364.067
 * B5 : 516 x  729   182.033 x 257.175
 * 
 * 1 dict dup /PageSize [842 595] put setpagedevice /ImagingBBox [20 20 802 555] put setpagedevice
 */
int set_paper_size_ViewPort (struct _printer lo_printer, int ps, FILE *stream)
{
//#define SET_PAPER_SIZE_VIEWPORT
	char str[256];
	int debug = 0;

#ifdef SET_PAPER_SIZE_VIEWPORT
	debug = 1;
	if (debug > 1) g_print ("set_paper_size_ViewPort() : in\n");
#endif


	if (debug > 1) g_print ("set_paper_size_ViewPort() : Paper = %d\n", lo_printer.Paper);


	switch (lo_printer.Paper) {
		/* User Setting Size */
		case USER_PAPER :
			if (debug > 0) g_print ("User Setting Size\n");
			break;

		/* A3 */
		case A3_PAPER:
			lo_printer.paper_right = 297.039;
			lo_printer.paper_top = 420.159;
			if (printer.Orientation == LANDSCAPE) {
				if (debug > 0) g_print ("Paper Size A3 LANDSCAPE(横)\n");
				Swap_double (&lo_printer.paper_right, &lo_printer.paper_top);
			}
			else {
				if (debug > 0) g_print ("Paper Size A3 PORTRAIT(縦)\n");
			}
			break;

		/* A4 */
		case A4_PAPER:
			lo_printer.paper_right = 209.903;
			lo_printer.paper_top = 297.039;
			if (printer.Orientation == LANDSCAPE) {
				if (debug > 0) g_print ("Paper Size A4 LANDSCAPE(横)\n");
				Swap_double (&lo_printer.paper_right, &lo_printer.paper_top);
			}
			else {
				if (debug > 0) g_print ("Paper Size A4 PORTRAIT(縦)\n");
			}
			break;

		/* A5 */
		case A5_PAPER:
			lo_printer.paper_right = 148.167;
			lo_printer.paper_top = 209.903;
			if (printer.Orientation == LANDSCAPE) {
				if (debug > 0) g_print ("Paper Size A5 LANDSCAPE(横)\n");
				Swap_double (&lo_printer.paper_right, &lo_printer.paper_top);
			}
			else {
				if (debug > 0) g_print ("Paper Size A5 PORTRAIT(縦)\n");
			}
			break;

		/* B5 */
		case B5_PAPER:
			lo_printer.paper_right = 182.033;
			lo_printer.paper_top = 257.175;
			if (printer.Orientation == LANDSCAPE) {
				if (debug > 0) g_print ("Paper Size B5 LANDSCAPE(横)\n");
				Swap_double (&lo_printer.paper_right, &lo_printer.paper_top);
			}
			else {
				if (debug > 0) g_print ("Paper Size B5 PORTRAIT(縦)\n");
			}
			break;

		/* B4 */
		case B4_PAPER:
			lo_printer.paper_right = 257.175;
			lo_printer.paper_top = 364.067;
			if (printer.Orientation == LANDSCAPE) {
				if (debug > 0) g_print ("Paper Size B4 LANDSCAPE(横)\n");
				Swap_double (&lo_printer.paper_right, &lo_printer.paper_top);
			}
			else {
				if (debug > 0) g_print ("Paper Size B4 PORTRAIT(縦)\n");
			}
			break;

		default:
			if (debug > 0) g_print ("紙の指定が間違っています。\n");
			lo_printer.paper_right = 297;
			lo_printer.paper_top = 210;
			break;
	}
	if (lo_printer.Paper != USER_PAPER) {
		lo_printer.paper_left = 0;
		lo_printer.paper_bottom = 0;
	}



	/* 余白をいれて ViewPort を設定 */
	set_ViewPort(
				 Millimeter_To_Point(lo_printer.paper_left + lo_printer.blank_left), 
				 Millimeter_To_Point(lo_printer.paper_bottom + lo_printer.blank_bottom), 
				 Millimeter_To_Point(lo_printer.paper_right - lo_printer.blank_right), 
				 Millimeter_To_Point(lo_printer.paper_top - lo_printer.blank_top) 
				 );



	/* ps */
	if (ps == 1) {
		/* 縦置き */
		if (lo_printer.Orientation == PORTRAIT) {
			sprintf(str, "%%%%Orientation: Portrait\n");
			fputs(str, stream );
			sprintf(str, "%%%%Pages: 1\n");
			fputs(str, stream );

			/* ps に Paper Size を出力 */
			sprintf(str, "2 dict dup /PageSize [%d %d] put dup /ImagingBBox [%d %d %d %d] put setpagedevice\n", 
						 (int) Millimeter_To_Point(lo_printer.paper_right), 
						 (int) Millimeter_To_Point(lo_printer.paper_top), 
						 (int) Millimeter_To_Point(lo_printer.paper_left + lo_printer.blank_left), 
						 (int) Millimeter_To_Point(lo_printer.paper_bottom + lo_printer.blank_bottom), 
						 (int) Millimeter_To_Point(lo_printer.paper_right - lo_printer.blank_right), 
						 (int) Millimeter_To_Point(lo_printer.paper_top - lo_printer.blank_top) 
						 );
			fputs(str, stream );
		}


		/* 横置き */
		if (lo_printer.Orientation == LANDSCAPE) {
			sprintf(str, "%%%%Orientation: Landscape\n");
			fputs(str, stream );
			sprintf(str, "%%%%Pages: 1\n");
			fputs(str, stream );

#ifdef TEST
			/* ps に Paper Size を出力 */
			sprintf(str, "2 dict dup /PageSize [%d %d] put dup /ImagingBBox [%d %d %d %d] put setpagedevice\n", 
						 (int) Millimeter_To_Point(lo_printer.paper_right), 
						 (int) Millimeter_To_Point(lo_printer.paper_top), 
						 (int) Millimeter_To_Point(lo_printer.paper_left + lo_printer.blank_left), 
						 (int) Millimeter_To_Point(lo_printer.paper_bottom + lo_printer.blank_bottom), 
						 (int) Millimeter_To_Point(lo_printer.paper_right - lo_printer.blank_right), 
						 (int) Millimeter_To_Point(lo_printer.paper_top - lo_printer.blank_top) 
						 );
			fputs(str, stream );

			g_print("[%d,%d]  (%d,%d)-(%d,%d)\n", 
						 (int) Millimeter_To_Point(lo_printer.paper_right), 
						 (int) Millimeter_To_Point(lo_printer.paper_top), 
						 (int) Millimeter_To_Point(lo_printer.paper_left + lo_printer.blank_left), 
						 (int) Millimeter_To_Point(lo_printer.paper_bottom + lo_printer.blank_bottom), 
						 (int) Millimeter_To_Point(lo_printer.paper_right - lo_printer.blank_right), 
						 (int) Millimeter_To_Point(lo_printer.paper_top - lo_printer.blank_top) 
						);
#endif

			sprintf(str, "%d 0 translate\n", (int) Millimeter_To_Point(lo_printer.paper_top));
			fputs(str, stream );

			sprintf(str, "90 rotate\n");
			fputs(str, stream );

		}
	}

	/* eps */
	else if (ps == 0) {
		sprintf(str, "%%%%Title: drawing\n");
		fputs(str, stream );
		sprintf(str, "%%Creator: SagCAD ");
		strcat(str, VERSION);
		strcat(str, "\n");
		fputs(str, stream );
		sprintf(str, "%%%%Pages: 1\n");
		fputs(str, stream );

		/* ps に Paper Size を出力 */
		sprintf(str, "%%%%BoundingBox: %d %d %d %d\n", 
					 (int) Millimeter_To_Point(lo_printer.paper_left), 
					 (int) Millimeter_To_Point(lo_printer.paper_bottom), 
					 (int) Millimeter_To_Point(lo_printer.paper_right), 
					 (int) Millimeter_To_Point(lo_printer.paper_top) 
					 );
		fputs(str, stream );

		sprintf(str, "%%%%EndComments\n");
		fputs(str, stream );
		sprintf(str, "%%%%EndProlog\n");
		fputs(str, stream );
	}


	return 1;
}



/* ミリメートル ---> ポイント */
double Millimeter_To_Point(double millimeter)
{
	return ((72 / 25.4) * millimeter);
}



/* ポイント ---> ミリメートル */
double Point_To_Millimeter(double point)
{
	return ((25.4 / 72) * point);
}



/* -------------------------------------------------------------------
 * 出力するワールド座標系を設定
 * 
 * ViewPort を設定してから設定する
 * 
 */
int set_world_size_Window (struct _printer lo_printer)
{
//#define SET_WORLD_SIZE_WINDOW
	double WXlength, WYlength;
	int debug = 0;


#ifdef SET_WORLD_SIZE_WINDOW
	debug = 1;
#endif


	if (printer.Draw_View == DRAWING_VIEW) {
		if (debug > 0) {
			g_print ("Drawing View Setting\n");
		}
		WYlength = Point_To_Millimeter((ViewPortY2 - ViewPortY1) / lo_printer.scale);
		WXlength = Point_To_Millimeter((ViewPortX2 - ViewPortX1) / lo_printer.scale);
		
		WorldWindowX1 = MagX - (WXlength / 2);
		WorldWindowY1 = MagY - (WYlength / 2);
		WorldWindowX2 = MagX + (WXlength / 2);
		WorldWindowY2 = MagY + (WYlength / 2);

		return 1;
	}

	else { // DISPLAY_VIEW
		if (debug > 0) g_print ("Display View Setting\n");
		if (debug > 1) g_print("window (%f,%f)-(%f,%f)\n", window_left, window_bottom, window_right, window_top);

		set_WorldWindow(MagX - ((sagcad_system.ViewX/Mag)/2), 
						MagY - (((sagcad_system.ViewX/Mag)/2)*sagcad_system.Aspect), 
						MagX + ((sagcad_system.ViewX/Mag)/2), 
						MagY + (((sagcad_system.ViewX/Mag)/2)*sagcad_system.Aspect));

		return 1;
	}

	return 0;
}












/* ===================================================================
 * ----- < 描画セット > -----
 * ===================================================================
 */


/* -------------------------------------------------------------------
 * 線を描画
 */
int LineDraw_PS(FILE *stream, double sx, double sy, double ex, double ey, int style, long color)
{
	int Ret;


	Ret = LineCheckDraw_PS (&sx, &sy, &ex, &ey);
	if(Ret == 0) return 0;
	else if(Ret == 2 || Ret == 1) {
		LineK_PS(stream, sx, sy, ex, ey, style, color);
	}

	return 1;
}


/* ---------------------------------------------------------------------
 * 円を描画
 */
int CircleDraw_PS(FILE *stream, double CX, double CY, double R, int Style, long color)
{
	CircleK_PS(stream, CX, CY, R, 0, 360, Style, color);

	return 1;
}


/* ---------------------------------------------------------------------
 * 円弧を描画
 */
int ArcDraw_PS (FILE *stream, double CX, double CY, double R, double SX, double SY, double EX, double EY, int Style, long color)
{
	struct RtnDat LA;
	double SA, EA;
	
	LA.sx[1] = CX; LA.sy[1] = CY; LA.ex[1] = SX; LA.ey[1] = SY;
	la(&LA);
	SA = LA.angle;

	LA.sx[1] = CX; LA.sy[1] = CY; LA.ex[1] = EX; LA.ey[1] = EY;
	la(&LA);
	EA = LA.angle;

	CircleK_PS(stream, CX, CY, R, SA, EA, Style, color);

	return 1;
}


/* -------------------------------------------------------------------
 * 楕円描画 (ELLIPSE)
 *	
 * 選択		： DrawType = 1  SCD_SELECT
 * 元の色	： DrawType = 0  SCD_ORG
 * 消		： DrawType = 2  SCD_HIDE
 */
int EllipseDraw_PS(FILE *stream, ELLIPSE *ellipse)
{
	long color = ellipse->color;


	if (Layer[ellipse->layer].draw == 1) {
		EllipseK_PS(stream, 
				 ellipse->cx, ellipse->cy, 
				 get_ellipse_a(*ellipse), get_ellipse_b(*ellipse), 
				 ellipse->sa, ellipse->ea, 
				 get_ellipse_angle(*ellipse), 
				 ellipse->style, color);
	}
	return 1;
}










/* ===================================================================
 * ----- < 基本描画セット > -----
 * ===================================================================
 */


/* -------------------------------------------------------------------
 * 直線を style で描く
 *	
 */
int LineK_PS(FILE *stream, double SX, double SY, double EX, double EY, int style, long color)
{
	char str[256];
	double Total_Len, Angle;
	double Now_Len = 0, Now_SX, Now_SY, Now_EX, Now_EY;
	struct RtnDat PPH, LAH, PAPH;
	double mm;
//	double mm, min = 1, mid = 3, max = 10;


	/* 線幅出力 */
	set_line_width(stream, ((double)sagcad_line_style[style].PrinterWidth) / 2);


	Now_SX = SX;
	Now_SY = SY;
	mm = 1 / Mag;

	/* ２点間の距離 */
	PPH.sx[1] = SX;
	PPH.sy[1] = SY;
	PPH.ex[1] = EX;
	PPH.ey[1] = EY;
	pp(&PPH);
	Total_Len = PPH.l;

	/* 直線の角度 */
	LAH.sx[1] = SX;
	LAH.sy[1] = SY;
	LAH.ex[1] = EX;
	LAH.ey[1] = EY;
	la(&LAH);
	Angle = LAH.angle;


	//MoveToEx(gc, ToDeviceX(SX), ToDeviceY(SY), NULL);

	/* 実線 */
	if (sagcad_line_style[style].style == 1) {
		sprintf(str, "%f %f moveto\n", WorldToViewX(SX), WorldToViewY(SY));
		fputs(str, stream );
		sprintf(str, "%f %f lineto\n", WorldToViewX(EX), WorldToViewY(EY));
		fputs(str, stream );
		return 1;
	}



	/* 点線 */
	else if (sagcad_line_style[style].style == 2) {
		while (1) {
			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].max * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].max * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				return 1;
			}
			else {
				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}


			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].min * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].min * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				return 1;
			}
			else {

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}
		}
		return 1;
	}



	/* １点鎖線 */
	else if (sagcad_line_style[style].style == 3) {
		while (1) {
			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].max * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].max * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				return 1;
			}
			else {
				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}


			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].min * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].min * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				return 1;
			}
			else {

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}


			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].min * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].min * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				return 1;
			}
			else {
				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}


			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].min * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].min * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				return 1;
			}
			else {

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}


		}
		return 1;
	}


	/* 二点鎖線 */
	else if (sagcad_line_style[style].style == 4) {
		while (1) {
			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].max * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].max * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				return 1;
			}
			else {
				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}


			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].min * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].min * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				return 1;
			}
			else {

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}


			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].min * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].min * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				return 1;
			}
			else {
				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}


			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].min * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].min * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				return 1;
			}
			else {

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}


			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].min * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].min * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				return 1;
			}
			else {

				sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				fputs(str, stream );
				sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}


			PAPH.sx[1] = Now_SX;
			PAPH.sy[1] = Now_SY;
			PAPH.angle = Angle;
			PAPH.l = sagcad_line_style[style].min * mm;
			pap(&PAPH);
			Now_EX = sg(PAPH.ex[1], calcu_digits);
			Now_EY = sg(PAPH.ey[1], calcu_digits);
			Now_Len = Now_Len + (sagcad_line_style[style].min * mm);
			if (Total_Len <= Now_Len) {
				Now_EX = EX;
				Now_EY = EY;

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				return 1;
			}
			else {

				//sprintf(str, "%f %f moveto\n", WorldToViewX(Now_SX), WorldToViewY(Now_SY));
				//fputs(str, stream );
				//sprintf(str, "%f %f lineto\n", WorldToViewX(Now_EX), WorldToViewY(Now_EY));
				//fputs(str, stream );

				Now_SX = Now_EX;
				Now_SY = Now_EY;
			}
		}
		return 1;
	}
	return 0;
}





/* -------------------------------------------------------------------
 * 円の表示
 * 
 */
int CircleK_PS(FILE *stream, double CX, double CY, double R, double SA, double EA, int style, long color)
{
	double mm;
	double X, Y, oX = 0, oY = 0, i, step;
	int k;

	if (R <= 0) return 0;

	mm = 1 / Mag;
	step = (mm * 360) / (2 * PI * R);

	if(SA < EA) {
		/* 実線 */
		if (sagcad_line_style[style].style == 1) {
			for(i = SA ; i <= EA ; i = i + step) {
				X = R * cos(degrad(i)) + CX;
				Y = R * sin(degrad(i)) + CY;


				if(i != SA) LineDraw_PS(stream, oX, oY, X, Y, style, color);
				oX = X;
				oY = Y;
			}
			X = R * cos(degrad(EA)) + CX;
			Y = R * sin(degrad(EA)) + CY;
			LineDraw_PS(stream, oX, oY, X, Y, style, color);
		}



		/* 点線 */
		if (sagcad_line_style[style].style == 2) {
			k = 0;
			for(i = SA ; i < EA ; i = i + step ) {
				k++;
				X = R * cos(degrad(i)) + CX;
				Y = R * sin(degrad(i)) + CY;
				if(i != SA && k<= sagcad_line_style[style].max) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if(k < sagcad_line_style[style].max + sagcad_line_style[style].min) {
					// 
				}
				else if(k == sagcad_line_style[style].max + sagcad_line_style[style].min) {
					k = 0;
				}
				oX = X;
				oY = Y;
			}
			X = R * cos(degrad(EA)) + CX;
			Y = R * sin(degrad(EA)) + CY;
			LineDraw_PS(stream, oX, oY, X, Y, style, color);
		}



		/* 一点鎖線 */
		if (sagcad_line_style[style].style == 3) {
			k = 0;
			for(i = SA ; i < EA ; i = i + step ) {
				k++;
				X = R * cos(degrad(i)) + CX;
				Y = R * sin(degrad(i)) + CY;
				if(i != SA && k<= sagcad_line_style[style].max) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if(k <= sagcad_line_style[style].max + sagcad_line_style[style].min) {
					//
				}
				else if(k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 2)) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if(k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 3)) {
					if(k == sagcad_line_style[style].max + (sagcad_line_style[style].min * 3)) k=0;
				}
				oX = X;
				oY = Y;
			}
			X = R * cos(degrad(EA)) + CX;
			Y = R * sin(degrad(EA)) + CY;
			LineDraw_PS(stream, oX, oY, X, Y, style, color);
		}



		/* 二点鎖線 */
		if (sagcad_line_style[style].style == 4) {
			k = 0;
			for(i = SA ; i < EA ; i = i + step ) {
				k++;
				X = R * cos(degrad(i)) + CX;
				Y = R * sin(degrad(i)) + CY;
				/* max */
				if(i != SA && k <= sagcad_line_style[style].max) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if(k <= sagcad_line_style[style].max + sagcad_line_style[style].min) {
					// 
				}
				else if(k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 2)) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if(k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 3)) {
					// 
				}
				else if(k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 4)) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if(k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 5)) {
					if(k == sagcad_line_style[style].max + (sagcad_line_style[style].min * 5)) k = 0;
				}
				oX = X;
				oY = Y;
			}
			X = R * cos(degrad(EA)) + CX;
			Y = R * sin(degrad(EA)) + CY;
			LineDraw_PS(stream, oX, oY, X, Y, style, color);
		}
	}



	if(SA > EA) {
		CircleK_PS(stream, CX, CY, R, SA, 360, style, color);
		CircleK_PS(stream, CX, CY, R, 0, EA, style, color);
	}
	return 1;
}





/* -------------------------------------------------------------------
 * 楕円の表示
 * 
 */
int EllipseK_PS(FILE *stream, 
			 double cx, double cy, double a, double b, 
			 double sa, double ea, double angle , 
			 int style, long color)
{

	double mm;
	double X, Y, oX = 0, oY = 0, i, step;
	int k;
	double RAD, DX, DY;


	if (a < 0 || b < 0) return 0;

	mm = 1/Mag;
	step = (mm * 360) / (2 * PI * a);
	/* 移動ベクトル */
	RAD = degrad(angle);

	if (sa < ea) {
		/* 実線 */
		if (sagcad_line_style[style].style == 1) {
			for (i = sa ; i <= ea ; i = i + step) {
				X = a * cos(degrad(i)) + cx;
				Y = b * sin(degrad(i)) + cy;
				
				DX = X - cx;
				DY = Y - cy;
				X = sg((DX * cos(RAD) - DY * sin(RAD)) + cx, calcu_digits);
				Y = sg((DY * cos(RAD) + DX * sin(RAD)) + cy, calcu_digits);
				
				if (i != sa) LineDraw_PS(stream, oX, oY, X, Y, style, color);
				oX = X;
				oY = Y;
			}
			X = a * cos(degrad(ea)) + cx;
			Y = b * sin(degrad(ea)) + cy;
			
			DX = X - cx;
			DY = Y - cy;
			X = sg((DX * cos(RAD) - DY * sin(RAD)) + cx, calcu_digits);
			Y = sg((DY * cos(RAD) + DX * sin(RAD)) + cy, calcu_digits);
			LineDraw_PS(stream, oX, oY, X, Y, style, color);
		}



		/* 点線 */
		if (sagcad_line_style[style].style == 2) {
			k = 0;
			for(i = sa ; i < ea ; i = i + step ) {
				k++;
				X = a * cos(degrad(i)) + cx;
				Y = b * sin(degrad(i)) + cy;
				
				DX = X - cx;
				DY = Y - cy;
				X = sg((DX * cos(RAD) - DY * sin(RAD)) + cx, calcu_digits);
				Y = sg((DY * cos(RAD) + DX * sin(RAD)) + cy, calcu_digits);

				if (i != sa && k <= sagcad_line_style[style].max) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if (k < sagcad_line_style[style].max + sagcad_line_style[style].min) {
					// 
				}
				else if (k == sagcad_line_style[style].max + sagcad_line_style[style].min) {
					k = 0;
				}
				oX = X;
				oY = Y;
			}

			X = a * cos(degrad(ea)) + cx;
			Y = b * sin(degrad(ea)) + cy;
			
			DX = X - cx;
			DY = Y - cy;
			X = sg((DX * cos(RAD) - DY * sin(RAD)) + cx, calcu_digits);
			Y = sg((DY * cos(RAD) + DX * sin(RAD)) + cy, calcu_digits);
			LineDraw_PS(stream, oX, oY, X, Y, style, color);
		}



		/* 一点鎖線 */
		if (sagcad_line_style[style].style == 3) {
			k = 0;
			for(i = sa ; i < ea ; i = i + step ) {
				k++;

				X = a * cos(degrad(i)) + cx;
				Y = b * sin(degrad(i)) + cy;
				
				DX = X - cx;
				DY = Y - cy;
				X = sg((DX * cos(RAD) - DY * sin(RAD)) + cx, calcu_digits);
				Y = sg((DY * cos(RAD) + DX * sin(RAD)) + cy, calcu_digits);

				if (i != sa && k <= sagcad_line_style[style].max) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if (k <= sagcad_line_style[style].max + sagcad_line_style[style].min) {
					//
				}
				else if (k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 2)) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if (k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 3)) {
					if (k == sagcad_line_style[style].max + (sagcad_line_style[style].min * 3)) k=0;
				}
				oX = X;
				oY = Y;
			}

			X = a * cos(degrad(ea)) + cx;
			Y = b * sin(degrad(ea)) + cy;
			
			DX = X - cx;
			DY = Y - cy;
			X = sg((DX * cos(RAD) - DY * sin(RAD)) + cx, calcu_digits);
			Y = sg((DY * cos(RAD) + DX * sin(RAD)) + cy, calcu_digits);
			LineDraw_PS(stream, oX, oY, X, Y, style, color);
		}



		/* 二点鎖線 */
		if (sagcad_line_style[style].style == 4) {
			k = 0;
			for (i = sa ; i < ea ; i = i + step ) {
				k++;

				X = a * cos(degrad(i)) + cx;
				Y = b * sin(degrad(i)) + cy;
				
				DX = X - cx;
				DY = Y - cy;
				X = sg((DX * cos(RAD) - DY * sin(RAD)) + cx, calcu_digits);
				Y = sg((DY * cos(RAD) + DX * sin(RAD)) + cy, calcu_digits);

				/* max */
				if (i != sa && k <= sagcad_line_style[style].max) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if (k <= sagcad_line_style[style].max + sagcad_line_style[style].min) {
					// 
				}
				else if (k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 2)) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if (k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 3)) {
					// 
				}
				else if (k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 4)) {
					LineDraw_PS(stream, oX, oY, X, Y, style, color);
				}
				else if (k <= sagcad_line_style[style].max + (sagcad_line_style[style].min * 5)) {
					if (k == sagcad_line_style[style].max + (sagcad_line_style[style].min * 5)) k = 0;
				}
				oX = X;
				oY = Y;
			}

			X = a * cos(degrad(ea)) + cx;
			Y = b * sin(degrad(ea)) + cy;
			
			DX = X - cx;
			DY = Y - cy;
			X = sg((DX * cos(RAD) - DY * sin(RAD)) + cx, calcu_digits);
			Y = sg((DY * cos(RAD) + DX * sin(RAD)) + cy, calcu_digits);
			LineDraw_PS(stream, oX, oY, X, Y, style, color);
		}
	}



	if (sa > ea) {
		EllipseK_PS(stream, cx, cy, a, b, sa, 360, angle, style, color);
		EllipseK_PS(stream, cx, cy, a, b, 0, ea, angle, style, color);
	}
	return 1;
}





/* -------------------------------------------------------------------
 * 直線を枠の中だけの座標を返す
 *	
 *	
 * 戻値　： int    0:不可視    1:変換 OK	2:変換無し可視
 *	
 */
int LineCheckDraw_PS(double *SX, double *SY, double *EX, double *EY)
{
	double left = 0, top = 0, right = 0, bottom = 0;
	double sx, sy, ex, ey;
	struct RtnDat LLPH;
	int s_bit = 0, e_bit = 0;
	int sx_Frag = 0, ex_Frag = 0;
	int CP;


	sx = *SX;
	sy = *SY;
	ex = *EX;
	ey = *EY;

	left   = WorldWindowX1;
	top    = WorldWindowY2;
	right  = WorldWindowX2;
	bottom = WorldWindowY1;

	/* 始点 */
	if(sy > top)	s_bit = s_bit + 8;
	if(sy < bottom) s_bit = s_bit + 4;
	if(sx > right)	s_bit = s_bit + 2;
	if(sx < left)	s_bit = s_bit + 1;

	/* 終点 */
	if(ey > top)	e_bit = e_bit + 8;
	if(ey < bottom) e_bit = e_bit + 4;
	if(ex > right)	e_bit = e_bit + 2;
	if(ex < left)	e_bit = e_bit + 1;



	/* ---------------------------------------------------------------
	 * 完全に可視
	 */
	if(s_bit == 0 && e_bit == 0) {
		return 2;
	}



	/* ---------------------------------------------------------------
	 * 完全に不可視
	 */
	if ((s_bit & e_bit) != 0) {
		return 0;
	}



	/* ---------------------------------------------------------------
	 * 可視の可能性がある (1)
	 */
	if ( ((s_bit && e_bit) == 0) 
		 || 
		 ( s_bit == 0x8 && e_bit == 0x4 )
		 || 
		 ( s_bit == 0x4 && e_bit == 0x8 )
		 || 
		 ( s_bit == 0x2 && e_bit == 0x1 )
		 || 
		 ( s_bit == 0x1 && e_bit == 0x2 ) ) 
	{
		/* -----------------------------------------------------------
		 * 上の境界線と交点があるか
		 */
		if ( (s_bit & 8) == 8 || (e_bit & 8) == 8 ) {
			/* 直線と上の境界線との交点を求める */
			LLPH.sx[1] = sx;	LLPH.sy[1] = sy;	LLPH.ex[1] = ex;	LLPH.ey[1] = ey;
			LLPH.sx[2] = left;	LLPH.sy[2] = top;	LLPH.ex[2] = right;	LLPH.ey[2] = top;
			llp(&LLPH);
			/* 交点がある */
			if (LLPH.sx[3] > left && LLPH.sx[3] < right) {
				if ((s_bit & 8) == 8) {
					*SX = LLPH.sx[3];
					*SY = LLPH.sy[3];
					sx_Frag = 1;
				}
				else if ((e_bit & 8) == 8) {
					*EX = LLPH.sx[3];
					*EY = LLPH.sy[3];
					ex_Frag = 1;
				}
			}
		}

		/* -----------------------------------------------------------
		 * 下の境界線と交点があるか
		 */
		if ( (s_bit & 4) == 4 || (e_bit & 4) == 4 ) {
			/* 直線と下の境界線との交点を求める */
			LLPH.sx[1] = sx;	LLPH.sy[1] = sy;	LLPH.ex[1] = ex;	LLPH.ey[1] = ey;
			LLPH.sx[2] = left;	LLPH.sy[2] = bottom;	LLPH.ex[2] = right;	LLPH.ey[2] = bottom;
			llp(&LLPH);
			/* 交点がある */
			if (LLPH.sx[3] > left && LLPH.sx[3] < right) {
				if ( (s_bit & 4) == 4 ) {
					*SX = LLPH.sx[3];
					*SY = LLPH.sy[3];
					sx_Frag = 1;
				}
				else if ( (e_bit & 4) == 4 ) {
					*EX = LLPH.sx[3];
					*EY = LLPH.sy[3];
					ex_Frag = 1;
				}
			}
		}

		/* -----------------------------------------------------------
		 * 右の境界線と交点があるか
		 */
		if ( (s_bit & 2) == 2 || (e_bit & 2) == 2 ) {
			/* 直線と右の境界線との交点を求める */
			LLPH.sx[1] = sx;	LLPH.sy[1] = sy;	LLPH.ex[1] = ex;	LLPH.ey[1] = ey;
			LLPH.sx[2] = right;	LLPH.sy[2] = top;	LLPH.ex[2] = right;	LLPH.ey[2] = bottom;
			llp(&LLPH);
			/* 交点がある */
			if (LLPH.sy[3] > bottom && LLPH.sy[3] < top) {
				if ( ( s_bit & 2) == 2 ) {
					*SX = LLPH.sx[3];
					*SY = LLPH.sy[3];
					sx_Frag = 1;
				}
				else if ( (e_bit & 2) == 2 ) {
					*EX = LLPH.sx[3];
					*EY = LLPH.sy[3];
					ex_Frag = 1;
				}
			}
		}

		/* -----------------------------------------------------------
		 * 左の境界線と交点があるか
		 */
		if ( (s_bit & 1) == 1 || (e_bit & 1) == 1 ) {
			/* 直線と左の境界線との交点を求める */
			LLPH.sx[1] = sx;	LLPH.sy[1] = sy;	LLPH.ex[1] = ex;	LLPH.ey[1] = ey;
			LLPH.sx[2] = left;	LLPH.sy[2] = top;	LLPH.ex[2] = left;	LLPH.ey[2] = bottom;
			llp(&LLPH);
			/* 交点がある */
			if (LLPH.sy[3] > bottom && LLPH.sy[3] < top) {
				if ( (s_bit & 1) == 1 ) {
					*SX = LLPH.sx[3];
					*SY = LLPH.sy[3];
					sx_Frag = 1;
				}
				else if ( (e_bit & 1) == 1 ) {
					*EX = LLPH.sx[3];
					*EY = LLPH.sy[3];
					ex_Frag = 1;
				}
			}
		}

		/* -----------------------------------------------------------
		 * 
		 */
		if(sx_Frag == 1 || ex_Frag == 1) {
			return 1;
		}
		else return 0;
	}



	/* ---------------------------------------------------------------
	 * 可視の可能性がある (2)
	 * 始点終点とも枠外なので、交点が２個有れば可視
	 */
	if(	( (s_bit & 8) == 8 && (e_bit & 1) == 1 ) 
		|| 
		( (s_bit & 8) == 8 && (e_bit & 2) == 2 ) 
		|| 
		( (s_bit & 4) == 4 && (e_bit & 1) == 1 ) 
		|| 
		( (s_bit & 4) == 4 && (e_bit & 2) == 2 ) 
		|| 
		( (s_bit & 1) == 1 && (e_bit & 8) == 8 ) 
		|| 
		( (s_bit & 2) == 2 && (e_bit & 8) == 8 ) 
		|| 
		( (s_bit & 1) == 1 && (e_bit & 4) == 4 ) 
		|| 
		( (s_bit & 2) == 2 && (e_bit & 4) == 4 ) )
	{
		CP = 0;

		/* -------------------------------------------------
		 * 上の境界線と交点があるか
		 */
		LLPH.sx[1] = sx;	LLPH.sy[1] = sy;	LLPH.ex[1] = ex;	LLPH.ey[1] = ey;
		LLPH.sx[2] = left;	LLPH.sy[2] = top;	LLPH.ex[2] = right;	LLPH.ey[2] = top;
		/* 交点がある */
		if (Act_llp(&LLPH) == 1) {
			CP++;
			if(CP == 1) {
				*SX = LLPH.sx[3]; *SY = LLPH.sy[3];
			}
			else if(CP == 2) {
				*EX = LLPH.sx[3]; *EY = LLPH.sy[3];
			}
		}

		/* -------------------------------------------------
		 * 下の境界線と交点があるか
		 */
		LLPH.sx[1] = sx;	LLPH.sy[1] = sy;	LLPH.ex[1] = ex;	LLPH.ey[1] = ey;
		LLPH.sx[2] = left;	LLPH.sy[2] = bottom;	LLPH.ex[2] = right;	LLPH.ey[2] = bottom;
		/* 交点がある */
		if (Act_llp(&LLPH) == 1) {
			CP++;
			if(CP == 1) {
				*SX = LLPH.sx[3]; *SY = LLPH.sy[3];
			}
			else if(CP == 2) {
				*EX = LLPH.sx[3]; *EY = LLPH.sy[3];
			}
		}

		/* -------------------------------------------------
		 * 右の境界線と交点があるか
		 */
		LLPH.sx[1] = sx;	LLPH.sy[1] = sy;	LLPH.ex[1] = ex;	LLPH.ey[1] = ey;
		LLPH.sx[2] = right;	LLPH.sy[2] = top;	LLPH.ex[2] = right;	LLPH.ey[2] = bottom;
		/* 交点がある */
		if (Act_llp(&LLPH) == 1) {
			CP++;
			if(CP == 1) {
				*SX = LLPH.sx[3]; *SY = LLPH.sy[3];
			}
			else if(CP == 2) {
				*EX = LLPH.sx[3]; *EY = LLPH.sy[3];
			}
		}

		/* -------------------------------------------------
		 * 左の境界線と交点があるか
		 */
		LLPH.sx[1] = sx;	LLPH.sy[1] = sy;	LLPH.ex[1] = ex;	LLPH.ey[1] = ey;
		LLPH.sx[2] = left;	LLPH.sy[2] = top;	LLPH.ex[2] = left;	LLPH.ey[2] = bottom;
		/* 交点がある */
		if (Act_llp(&LLPH) == 1) {
			CP++;
			if(CP == 1) {
				*SX = LLPH.sx[3]; *SY = LLPH.sy[3];
			}
			else if(CP == 2) {
				*EX = LLPH.sx[3]; *EY = LLPH.sy[3];
			}
		}


		if (CP == 2) return 1;
	}



	return 0;
}




#ifdef TEST
void DUMY_______________________________________01(void){}
#endif




/* ===================================================================
 * ----- < 寸法セット > -----
 * ===================================================================
 */


/* -------------------------------------------------------------------
 * 終点に矢印 (defin = 10)
 *	
 * LineFrag
 *	   0	  : 矢印のみ
 *	   0 以外 : 矢印と直線
 *	
 */
int LineEndArrow_PS (FILE *stream, double SX, double SY, double EX, double EY, int LineFrag, int DrawType, long color)
{
	struct RtnDat LAH, PAPH;
	double EX1, EY1, EX2, EY2;

	/* LA	直線の角度 */
	LAH.sx[1] = EX;
	LAH.sy[1] = EY;
	LAH.ex[1] = SX;
	LAH.ey[1] = SY;
	la(&LAH);
	/* PAP	始点と角度と距離で直線の終点を求める */
	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	/* (EX1 , EY1) */
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	/* (EX2 , EY2) */
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

	if (LineFrag != 0) LineDraw_PS(stream, SX, SY, EX, EY, 5, color);
	LineDraw_PS(stream, EX, EY, EX1, EY1, 5, color);
	LineDraw_PS(stream, EX, EY, EX2, EY2, 5, color);
	return 0;
}


/* -------------------------------------------------------------------
 * 両端に矢印 (defin = 20)
 *	
 * LineFrag
 *	   0	  : 矢印のみ
 *	   0 以外 : 矢印と直線
 */
int LineBothArrow_PS (FILE *stream, double SX, double SY, double EX, double EY, int LineFrag, int DrawType, long color)
{
	struct RtnDat LAH, PAPH;
	double EX1, EY1, EX2, EY2;


	/* LA	直線の角度 */
	LAH.sx[1] = EX;
	LAH.sy[1] = EY;
	LAH.ex[1] = SX;
	LAH.ey[1] = SY;
	la(&LAH);
	/* PAP	始点と角度と距離で直線の終点を求める */
	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

	LineDraw_PS(stream, EX, EY, EX1, EY1, 5, color);
	LineDraw_PS(stream, EX, EY, EX2, EY2, 5, color);

	/* LA	直線の角度 */
	LAH.sx[1] = SX;
	LAH.sy[1] = SY;
	LAH.ex[1] = EX;
	LAH.ey[1] = EY;
	la(&LAH);

	/* PAP	始点と角度と距離で直線の終点を求める */
	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

	LineDraw_PS(stream, SX, SY, EX1, EY1, 5, color);
	LineDraw_PS(stream, SX, SY, EX2, EY2, 5, color);

	if (LineFrag != 0) LineDraw_PS(stream, SX, SY, EX, EY, 5, color);

	return 0;
}


/* -------------------------------------------------------------------
 * 両端に逆矢印 (defin = 30)
 *	
 * LineFrag
 *	   0	  : 矢印のみ
 *	   0 以外 : 矢印と直線
 */
int LineBothConverseArrow_PS (FILE *stream, double SX, double SY, double EX, double EY, int LineFrag, int DrawType, long color)
{
	struct RtnDat LAH,PAPH;
	double EX1,EY1,EX2,EY2;
	
	/* LA	直線の角度 */
	LAH.sx[1] = EX;
	LAH.sy[1] = EY;
	LAH.ex[1] = SX;
	LAH.ey[1] = SY;
	la(&LAH);

	/* PAP	始点と角度と距離で直線の終点を求める */
	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

	LineDraw_PS(stream, SX, SY, EX1, EY1, 5, color);
	LineDraw_PS(stream, SX, SY, EX2, EY2, 5, color);

	/* LA	直線の角度 */
	LAH.sx[1] = SX;
	LAH.sy[1] = SY;
	LAH.ex[1] = EX;
	LAH.ey[1] = EY;
	la(&LAH);
	/* PAP	始点と角度と距離で直線の終点を求める */
	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

	LineDraw_PS(stream, EX, EY, EX1, EY1, 5, color);
	LineDraw_PS(stream, EX, EY, EX2, EY2, 5, color);

	if (LineFrag != 0) LineDraw_PS(stream, SX, SY, EX, EY, 5, color);

	return 0;
}


/* -------------------------------------------------------------------
 * 両端に矢印(円弧) (defin = 60)
 *	
 * ArcFrag
 *	   0	  : 矢印のみ
 *	   0 以外 : 矢印と直線
 */
int ArcBothArrow_PS (FILE *stream, double CX, double CY, double R, double SX, double SY, double EX, double EY, int ArcFrag, int DrawType, long color)
{
	/* Dumy = ArcBow(CX, CY, R, SA,EA, INDEX) */

	struct RtnDat PAPH, LAH;
	double SA, EA;
	double SX1, SY1, SX2, SY2;
	double EX1, EY1, EX2, EY2;


	/* 中心点から始点の線の角度 SA を求める。 */
	/* LA	  直線の角度 */
	LAH.sx[1] = CX;
	LAH.sy[1] = CY;
	LAH.ex[1] = SX;
	LAH.ey[1] = SY;
	la(&LAH);
	SA = LAH.angle;
	if (SA > 360) SA = SA - 360;

	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = SA + 90 - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	SX1 = PAPH.ex[1];
	SY1 = PAPH.ey[1];

	PAPH.sx[1] = SX;
	PAPH.sy[1] = SY;
	PAPH.angle = SA + 90 + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	SX2 = PAPH.ex[1];
	SY2 = PAPH.ey[1];

	LineDraw_PS(stream, SX, SY, SX1, SY1, 5, color);
	LineDraw_PS(stream, SX, SY, SX2, SY2, 5, color);



	/* 中心点から終点の線の角度 EA を求める。 */
	/* LA	  直線の角度 */
	LAH.sx[1] = CX;
	LAH.sy[1] = CY;
	LAH.ex[1] = EX;
	LAH.ey[1] = EY;
	la(&LAH);
	EA = LAH.angle;
	if(EA > 360) EA = EA - 360;

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = EA - 90 - sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = EA - 90 + sagcad_dimension.arrow_angle;
	PAPH.l = ArrowLength(DrawType);
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

	LineDraw_PS(stream, EX, EY, EX1, EY1, 5, color);
	LineDraw_PS(stream, EX, EY, EX2, EY2, 5, color);

	if (ArcFrag != 0) ArcDraw_PS (stream, CX, CY, R, SX, SY, EX, EY, 5, color);

	return 0;
}


/* -------------------------------------------------------------------
 * 補助線を描画   始点側:引出隙間	終点側:引出延長 (defin = 70)
 *	
 * 1 線の角度(Angle) を求める。
 * 2 線の長さ(LineLen) を求める。
 * 3 start_point を求める。
 * 4 end_point を求める。
 * 5 補助線を描画
 *	
 */
int AssistanceLine_PS (FILE *stream, double SX, double SY, double EX, double EY, int DrawType, long color)
{
	double Angle, LineLen, start_pointX, start_pointY, end_pointX, end_pointY;
	struct RtnDat a;
	int Ret;

	/* 1 線の角度(Angle) を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.ex[1] = EX;
	a.ey[1] = EY;
	Ret = la(&a);
	Angle = a.angle;

	/* 2 線の長さ(LineLen) を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.ex[1] = EX;
	a.ey[1] = EY;
	Ret = pp(&a);
	LineLen = a.l;

	/* 3 start_point を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.angle = Angle;

	/* 図面の縮尺表示 (プリンターイメージ) */
	if (DrawType == DRAW_DISP)
		a.l = sagcad_dimension.assistance_line_space / printer.scale;
	/* 文字サイズ固定表示 */
	else if (DrawType == DRAW_CONST) 
		a.l = (sagcad_dimension.assistance_line_space / Mag);
	/* 設定した大きさをそのまま表示 */
	else if (DrawType == DRAW_REAL) 
		a.l = sagcad_dimension.assistance_line_space;
	else 
		a.l = sagcad_dimension.assistance_line_space;
	Ret = pap(&a);
	start_pointX = a.ex[1];
	start_pointY = a.ey[1];

	/* 4 end_point を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.angle = Angle;

	/* 図面の縮尺表示 (プリンターイメージ) */
	if (DrawType == DRAW_DISP)
		a.l = LineLen + sagcad_dimension.assistance_line_extension / printer.scale;
	/* 文字サイズ固定表示 */
	else if (DrawType == DRAW_CONST) 
		a.l = LineLen + (sagcad_dimension.assistance_line_extension / Mag);
	/* 設定した大きさをそのまま表示 */
	else if (DrawType == DRAW_REAL) 
		a.l = LineLen + sagcad_dimension.assistance_line_extension;
	else 
		a.l = LineLen + sagcad_dimension.assistance_line_extension;
	Ret = pap(&a);
	end_pointX = a.ex[1];
	end_pointY = a.ey[1];

	/* 5 補助線を描画*/
	LineDraw_PS(stream, start_pointX, start_pointY, end_pointX, end_pointY, 5, color);
	return 1;
}


/* -------------------------------------------------------------------
 * (矢印)補助線の延長を描画 (defin = 80)
 *	
 * (SX,SY)-(EX,EY) で指定された直線を sagcad_dimension.arrow_line_extension 
 * で指定された長さだけ終点を延長する。
 *	
 * 1 線の角度(Angle) を求める。
 * 2 線の長さ(LineLen) を求める。
 * 3 end_point を求める。
 * 4 補助線を描画
 *	
 */
int LineEndExtension_PS (FILE *stream, double SX, double SY, double EX, double EY, int DrawType, long color)
{
	double Angle, LineLen, end_pointX, end_pointY;
	struct RtnDat a;
	int Ret;

	/* 1 線の角度(Angle) を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.ex[1] = EX;
	a.ey[1] = EY;
	Ret = la(&a);
	Angle = a.angle;

	/* 2 線の長さ(LineLen) を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.ex[1] = EX;
	a.ey[1] = EY;
	Ret = pp(&a);
	LineLen = a.l;

	/* 3 end_point を求める。*/
	a.sx[1] = SX;
	a.sy[1] = SY;
	a.angle = Angle;

	/* 図面の縮尺表示 (プリンターイメージ) */
	if (DrawType == DRAW_DISP)
		a.l = LineLen + (sagcad_dimension.arrow_line_extension / printer.scale);
	/* 文字サイズ固定表示 */
	else if (DrawType == DRAW_CONST) 
		a.l = LineLen + (sagcad_dimension.arrow_line_extension / Mag);
	/* 設定した大きさをそのまま表示 */
	else if (DrawType == DRAW_REAL) 
		a.l = LineLen + sagcad_dimension.arrow_line_extension;
	else 
		a.l = LineLen + sagcad_dimension.arrow_line_extension;

	Ret = pap(&a);
	end_pointX = a.ex[1];
	end_pointY = a.ey[1];

	/* 4 補助線を描画*/
	LineDraw_PS(stream, SX, SY, end_pointX, end_pointY, 5, color);
	return 1;
}





/* -------------------------------------------------------------------
 * 寸法描画 線 (HDC)
 *	
 * TextFrag = 0:描かない	1:描く	  2:セレクトカラー
 *	
 */
int Dimension_Draw_Line_PS (FILE *stream, DIMENSION *a)
{
	int i;
	long color = a->Assist_color;


	if (a->index > 0 && a->AssistLine != NULL) {
		for (i = 0 ; i < a->index ; i++) {
			if (&a->AssistLine[i] == NULL) {
				g_print("AssistLine is null.\n");
				return 0;
			}
			
			switch (a->AssistLine[i].defin) {
				/* 点 */
				case 0:
					//PointDraw(widget, a->AssistLine[i].sx, a->AssistLine[i].sy, 1, color);
					break;
				/* 線 */
				case 1:
					LineDraw_PS (stream, a->AssistLine[i].sx, a->AssistLine[i].sy, 
								 a->AssistLine[i].ex, a->AssistLine[i].ey, 5, color);
					break;
				/* 円弧 */
				case 2:
					ArcDraw_PS (stream, a->AssistLine[i].cx, a->AssistLine[i].cy, a->AssistLine[i].r, 
								a->AssistLine[i].sx, a->AssistLine[i].sy, 
								a->AssistLine[i].ex, a->AssistLine[i].ey, 5, color);
					break;
				/* 円 */
				case 4:
					CircleDraw_PS (stream, a->AssistLine[i].cx, a->AssistLine[i].cy, a->AssistLine[i].r, 5, color);
					break;
				/* 終点に矢印(線) */
				case 10:
					LineEndArrow_PS (stream, a->AssistLine[i].sx, a->AssistLine[i].sy, 
									 a->AssistLine[i].ex, a->AssistLine[i].ey, 0, a->DrawType, color);
					break;
				/* 終点に矢印(線) */
				case 11:
					LineEndArrow_PS (stream, a->AssistLine[i].sx, a->AssistLine[i].sy, 
									 a->AssistLine[i].ex, a->AssistLine[i].ey, 1, a->DrawType, color);
					break;
				/* 両端に矢印(線) */
				case 20:
					LineBothArrow_PS (stream, a->AssistLine[i].sx, a->AssistLine[i].sy, 
								  a->AssistLine[i].ex, a->AssistLine[i].ey, 0, a->DrawType, color);
					break;
				/* 両端に矢印(線) */
				case 21:
					LineBothArrow_PS (stream, a->AssistLine[i].sx, a->AssistLine[i].sy, 
									  a->AssistLine[i].ex, a->AssistLine[i].ey, 1, a->DrawType, color);
					break;
				/* 両端に逆矢印(線) */
				case 30:
					LineBothConverseArrow_PS (stream, a->AssistLine[i].sx, a->AssistLine[i].sy, 
											  a->AssistLine[i].ex, a->AssistLine[i].ey, 0, a->DrawType, color);
					break;
				/* 両端に逆矢印(線) */
				case 31:
					LineBothConverseArrow_PS (stream, a->AssistLine[i].sx, a->AssistLine[i].sy, 
											  a->AssistLine[i].ex, a->AssistLine[i].ey, 1, a->DrawType, color);
					break;
				/* 始点に矢印(円弧) */
	//			case 40:
				/* 終点に矢印(円弧) */
	//			case 50:
				/* 両端に矢印(円弧) */
				case 60:
					ArcBothArrow_PS (stream, a->AssistLine[i].cx, a->AssistLine[i].cy, a->AssistLine[i].r, 
									 a->AssistLine[i].sx, a->AssistLine[i].sy, 
									 a->AssistLine[i].ex, a->AssistLine[i].ey, 0, a->DrawType, color);
					break;
				/* 両端に矢印(円弧) */
				case 61:
					ArcBothArrow_PS (stream, a->AssistLine[i].cx, a->AssistLine[i].cy, a->AssistLine[i].r, 
								 a->AssistLine[i].sx, a->AssistLine[i].sy, 
								 a->AssistLine[i].ex, a->AssistLine[i].ey, 1, a->DrawType, color);
					break;
				/* 補助線 */
				case 70:
					AssistanceLine_PS (stream, a->AssistLine[i].sx, a->AssistLine[i].sy, 
									   a->AssistLine[i].ex, a->AssistLine[i].ey, a->DrawType, color);
					break;
				/* (矢印)補助線の延長 */
				case 80:
					LineEndExtension_PS (stream, a->AssistLine[i].sx, a->AssistLine[i].sy, 
										 a->AssistLine[i].ex, a->AssistLine[i].ey, a->DrawType, color);
					break;
			}
		}
	}
	else if (a->index > 0 && a->AssistLine == NULL) {
		OneShotLog("寸法線で index > 0 なのに AssistLine が NULL です。\n");
	}

//g_print("Dimension.c : Dimension_Line_Basic() : out\n");

	return 1;
}





/* -------------------------------------------------------------------
 * 文字列を表示する
 * -------------------------------------------------------------------
 *	1 : n 文字目の データを 原点に読み込んでスケーリングしたデータを関数(char_scale)で作成
 *	2 : 1 のデータを now_char_X オフセットして取り込む。
 *	3 : now_char_X を取り込んだ 文字幅 + distance オフセットさせる。
 *	4 : 1,2,3 を文字数繰り返す。
 *	
 *	5 : できたデータを angle で回転させ、(x､y) オフセットさせ表示。
 *	
 * text	 : テキスト
 * height: 高さ
 * space : 文字と文字の間隔
 * angle : 角度
 * x	 : 基準Ｘ
 * y	 : 基準Ｙ
 * color : 色
 * point : 0,1,2 書き出し位置
 *		   0 :	．TEXT
 *		   1 :	TE．XT
 *		   2 :	TEXT
 * frag  : 0:文字列長さ計算だけ   1:表示
 *	
 *	
 * hDC も渡して、LineBasic で描いた方がいいのでは？
 * -------------------------------------------------------------------
 */
int DrawTextVector_PS (FILE *stream, DIMENSION *a)
{
	int i, Ret;
	
	double SX, SY, EX, EY;


	if (a->diagram_of_char == NULL) {
		CulcTextVector (drawing_area, a);
	}


	/* -------------------------------------------------------------------
	 * 直線を枠の中だけの座標を返す
	 *	
	 *	
	 * 戻値　： int    0:不可視    1:変換 OK	2:変換無し可視
	 *	
	 */
	SX = a->start.x;
	SY = a->start.y;
	EX = a->end.x;
	EY = a->end.y;
	Ret = LineCheckDraw_PS(&SX, &SY, &EX, &EY);
	if (Ret == 0) {
		return 0;
	}


//	if (a->FontHeight < ((sagcad_system.ViewX / Mag) / 400)) {
//		LineDraw(widget, SX, SY, EX, EY, 1, color);
//		return 0;
//	}


	for (i = 0 ; i < a->diagram_of_char_index ; i++) {
		if (a->diagram_of_char[i].defin == 0) {
			//
		}
		if (a->diagram_of_char[i].defin == 1) {
			LineDraw_PS (stream,
						 a->diagram_of_char[i].sx, a->diagram_of_char[i].sy, 
						 a->diagram_of_char[i].ex, a->diagram_of_char[i].ey, 6, a->text_color);
		}
		else if (a->diagram_of_char[i].defin == 2) {
			ArcDraw_PS (stream,	
						a->diagram_of_char[i].cx, a->diagram_of_char[i].cy,  a->diagram_of_char[i].r,
						a->diagram_of_char[i].sx, a->diagram_of_char[i].sy, 
						a->diagram_of_char[i].ex, a->diagram_of_char[i].ey, 
						6, a->text_color);	
		}
		else if (a->diagram_of_char[i].defin == 4) {
			CircleDraw_PS (stream, 
						   a->diagram_of_char[i].cx, 
						   a->diagram_of_char[i].cy, 
						   a->diagram_of_char[i].r, 6, a->text_color);
		}
	}

	return 1;
}





/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 *    Project              : SagCAD
 *    Source               : PostScript.c
 * ====================================================================
 */





#ifdef TEST
void DUMY_______________________________________01(void){}
#endif





/* -------------------------------------------------------------------
 * スプライン補間
 * 
 * 開曲線
 * 
 */
int open_spline_PS (FILE *stream, int n, VERTEX *vertex, int split, int style, long color)
{
	int i, Ret;
	double pitch;
	static double old_x = 0, old_y = 0;
	double SX, SY, EX, EY;

	double u, v;
	double *x, *y, *p, *a, *b;
	double *h, *d;


	x = (double *) xmalloc( n * sizeof(double) );
	y = (double *) xmalloc( n * sizeof(double) );
	p = (double *) xmalloc( n * sizeof(double) );
	a = (double *) xmalloc( n * sizeof(double) );
	b = (double *) xmalloc( n * sizeof(double) );

	h = (double *) xmalloc( n * sizeof(double) );
	d = (double *) xmalloc( n * sizeof(double) );


	for (i = 0 ; i < n ; i++) {
		x[i] = vertex[i].x;
		y[i] = vertex[i].y;
		p[i] = 0;
		a[i] = 0;
		b[i] = 0;

		h[i] = 0;
		d[i] = 0;
	}


	open_maketable2(n, p, x, y, a, b, h, d);

	pitch = 1.0/(double)split;
	for (i = 0; i <= split ; i++) {
		open_spline2(n, pitch * i, &u, &v, p, x, y, a, b);
		if (i == 0) {
			// pset
		}
		else {
			SX = old_x;
			SY = old_y;
			EX = u;
			EY = v;

			Ret = LineCheckDraw_PS (&SX, &SY, &EX, &EY);
			if (Ret == 2 || Ret == 1) {
				LineK_PS (stream, SX, SY, EX, EY, style, color);
			}
		}
		old_x = u;
		old_y = v;
	}


	xfree(x);
	xfree(y);
	xfree(p);
	xfree(a);
	xfree(b);

	xfree(h);
	xfree(d);

	return 1;
}





/* -------------------------------------------------------------------
 * スプライン補間
 * 
 * 閉曲線
 * 
 */
int close_spline_PS (FILE *stream, int n, VERTEX *vertex, int split, int style, long color)
{
	int i, Ret;
	double pitch;
	static double old_x = 0, old_y = 0;
	double SX, SY, EX, EY;

	double u, v;
	double *x, *y, *p, *a, *b;
	double *h, *d, *w;


	x = (double *) xmalloc( (n+1) * sizeof(double) );
	y = (double *) xmalloc( (n+1) * sizeof(double) );
	p = (double *) xmalloc( (n+1) * sizeof(double) );
	a = (double *) xmalloc( (n+1) * sizeof(double) );
	b = (double *) xmalloc( (n+1) * sizeof(double) );

	h = (double *) xmalloc( (n+1) * sizeof(double) );
	d = (double *) xmalloc( (n+1) * sizeof(double) );
	w = (double *) xmalloc( (n+1) * sizeof(double) );

	for (i = 0 ; i < (n+1) ; i++) {
		x[i] = vertex[i].x;
		y[i] = vertex[i].y;
		p[i] = 0;
		a[i] = 0;
		b[i] = 0;

		h[i] = 0;
		d[i] = 0;
		w[i] = 0;
	}


	close_maketable2(n, p, x, y, a, b, h, d, w);
	pitch = 1.0/(double)split;
	for (i = 0; i <= split ; i++) {
		close_spline2(n, pitch * i, &u, &v, p, x, y, a, b);
		if (i == 0) {
			// pset
		}
		else {
			SX = old_x;
			SY = old_y;
			EX = u;
			EY = v;
			Ret = LineCheckDraw_PS(&SX, &SY, &EX, &EY);
			if (Ret == 2 || Ret == 1) {
				LineK_PS(stream, SX, SY, EX, EY, style, color);
			}
		}
		old_x = u;
		old_y = v;
	}


	xfree(x);
	xfree(y);
	xfree(p);
	xfree(a);
	xfree(b);

	xfree(h);
	xfree(d);
	xfree(w);

	return 1;
}





/* -------------------------------------------------------------------
 * ３次Ｂスプライン
 * 
 * データの数が最低７個は必要となる。
 *     データの数が３個の場合、中間点を３分割
 *     データの数が４個から６個の場合、中間点を２分割
 * 
 */
int b_spline_PS (FILE *stream, int n, VERTEX *vertex, double pitch, int style, long color)
{
	int i, j, Ret;
	double u, s1, s2, s3, s4;
	double x, y;
	double SX, SY, EX, EY;
	static double old_x = 0, old_y = 0;
	VERTEX *dummy_vertex = NULL;

	VERTEX sp = {0,0};
	VERTEX ep = {0,0};
	VERTEX ap1 = {0,0};
	VERTEX ap2 = {0,0};


	if (n == 3) {
		dummy_vertex = (VERTEX *)xmalloc( 7 * sizeof(VERTEX) );
		j = 0;
		dummy_vertex[j].x = vertex[0].x;
		dummy_vertex[j].y = vertex[0].y;
		for (i = 1 ; i < n ; i++) {
			sp.x = vertex[i-1].x;
			sp.y = vertex[i-1].y;
			ep.x = vertex[i].x;
			ep.y = vertex[i].y;
			split3(sp, ep, &ap1, &ap2);
			j++;
			dummy_vertex[j].x = ap1.x;
			dummy_vertex[j].y = ap1.y;
			j++;
			dummy_vertex[j].x = ap2.x;
			dummy_vertex[j].y = ap2.y;
			j++;
			dummy_vertex[j].x = vertex[i].x;
			dummy_vertex[j].y = vertex[i].y;
		}
		n = 7;
	}
	else if (n > 3 && n < 7) {
		dummy_vertex = (VERTEX *)xmalloc(((n * 2) - 1) * sizeof(VERTEX));
		j = 0;
		dummy_vertex[j].x = vertex[0].x;
		dummy_vertex[j].y = vertex[0].y;
		for (i = 1 ; i < n ; i++) {
			sp.x = vertex[i-1].x;
			sp.y = vertex[i-1].y;
			ep.x = vertex[i].x;
			ep.y = vertex[i].y;
			split2(sp, ep, &ap1);
			j++;
			dummy_vertex[j].x = ap1.x;
			dummy_vertex[j].y = ap1.y;
			j++;
			dummy_vertex[j].x = vertex[i].x;
			dummy_vertex[j].y = vertex[i].y;
		}
		n = (n * 2) - 1;
	}
	else if (n > 6) {
		dummy_vertex = (VERTEX *)xmalloc( n * sizeof(VERTEX) );
		for (i = 0 ; i < n ; i++) {
			dummy_vertex[i].x = vertex[i].x;
			dummy_vertex[i].y = vertex[i].y;
		}
	}



	for (i = 0 ; i < n-3 ; i++) {
		for (u = 0.0 ; u <= 1.0 + pitch/2 ; u = u + pitch) {
			/* 最初の区間 */
			if (i == 0) {
				spl1(u, &s1, &s2, &s3, &s4);
			}
			/* ２番目の区間 */
			else if (i == 1) {
				spl2(u, &s1, &s2, &s3, &s4);
			}
			/* 中間の区間 */
			else if (i > 1 && i < n-5) {
				spl3(u, &s1, &s2, &s3, &s4);
			}
			/* 最後から２番目の区間 */
			else if (i > 1 && i == n-5) {
				spl4(u, &s1, &s2, &s3, &s4);
			}
			/* 最後の区間 */
			else if (i > 2 && i == n-4) {
				spl5(u, &s1, &s2, &s3, &s4);
			}
			x = s1*dummy_vertex[i].x + s2*dummy_vertex[i+1].x + s3*dummy_vertex[i+2].x + s4*dummy_vertex[i+3].x;
			y = s1*dummy_vertex[i].y + s2*dummy_vertex[i+1].y + s3*dummy_vertex[i+2].y + s4*dummy_vertex[i+3].y;
			if (u == 0) {
				//pset;
			}
			else {
				SX = old_x;
				SY = old_y;
				EX = x;
				EY = y;
				Ret = LineCheckDraw_PS (&SX, &SY, &EX, &EY);
				if (Ret == 2 || Ret == 1) {
					LineK_PS(stream, SX, SY, EX, EY, style, color);
				}
			}
			old_x = x;
			old_y = y;
		}
	}

	xfree(dummy_vertex);
	return 1;
}





/* -------------------------------------------------------------------
 * ポリライン描画
 * 
 */
int PolyLineDraw_PS (FILE *stream, POLYLINE *a)
{
	long i;
	VERTEX_LIST *v;
	VERTEX first, old;
	VERTEX *vertex;


	if (Layer[a->layer].draw == 1) {

		/* ポリライン(折れ線) */
		if (a->code == 8) {
			v = a->vertex_list_info.head;
			i = 0;
			while (v != NULL) {
				if (i == 0) {
					first = *v->vertex;
					old = *v->vertex;
					i++;
				}
				else {
					LineDraw_PS (stream, old.x, old.y, v->vertex->x, v->vertex->y, a->style, a->color);
					old = *v->vertex;
				}
				v = v->next;
			}

			/* 閉じているとき */
			if ((a->frag & 1) == 1) {
				LineDraw_PS (stream, first.x, first.y, old.x, old.y, a->style, a->color);
			}
		}




		/* スプライン補間 */
		else if (a->code == 16) {
			if (a->index > 0) {
				vertex = (VERTEX *) xmalloc( (a->index + 1) * sizeof(VERTEX) );

				v = a->vertex_list_info.head;
				i = 0;
				while (v != NULL) {
					vertex[i] = *v->vertex;
					i++;
					v = v->next;
				}

				/* 閉じているとき(閉曲線) */
				if ((a->frag & 1) == 1) {
					vertex[i] = vertex[0];
					close_spline_PS(stream, a->index, vertex, a->index * a->split, a->style, a->color);
				}
				/* (開曲線) */
				else {
					open_spline_PS(stream, a->index, vertex, a->index * a->split, a->style, a->color);
				}

				xfree(vertex);
			}
		}


		/* Ｂスプライン曲線 */
		else if (a->code == 32) {
			if (a->index > 0) {
				vertex = (VERTEX *) xmalloc( a->index * sizeof(VERTEX) );

				v = a->vertex_list_info.head;
				i = 0;
				while (v != NULL) {
					vertex[i] = *v->vertex;
					i++;
					v = v->next;
				}
				b_spline_PS (stream, a->index, vertex, a->pitch, a->style, a->color);
				xfree(vertex);
			}
		}

	}
	return 1;
}





