require 'gtk2'
require 'gtkhtml2'
require 'open-uri'
require 'iconv'
require 'gettext'
require 'ruggregator/filter_dialog'
require 'ruggregator/filter'
require 'ruggregator/rss_dialog'
require 'ruggregator/preferences_dialog'
require 'ruggregator/configure'
require 'ruggregator/about_dialog'

include GetText
bindtextdomain("ruggregator")
GetText.charset = "UTF-8"

TITLE_COLUMN = 0
TITLE_COLOR_COLUMN = 1
TITLE_BACKGROUND_COLOR_COLUMN = 2
TITLE_WEIGHT_COLUMN = 3
DATE_COLUMN = 4
DATE_COLOR_COLUMN = 5
DATE_BACKGROUND_COLOR_COLUMN = 6
DATE_WEIGHT_COLUMN = 7
RSS_COLUMN = 8
RSS_COLOR_COLUMN = 9
RSS_BACKGROUND_COLOR_COLUMN = 10
RSS_WEIGHT_COLUMN = 11

# see the ruby reference: "strftime" method for Time object

$read_items = Array.new
$logo_link = String.new

def show_preferences_dialog
  pref_dialog = PreferencesDialog.new
  pref_dialog.run{ |response|
    if response == Gtk::Dialog::RESPONSE_ACCEPT
      pref_dialog.update_config
      $filter_selection.signal_emit("changed")
    end
    pref_dialog.destroy
  }
end

def show_properties_dialog
  name = selected_filter_name()
  if name
    filter = nil
    filter = get_filter_by_name(name)
    filter_dialog = FilterDialog.new
    filter_dialog.name = filter.name
    filter_dialog.select_rsses(filter.rss_urls)
    filter_dialog.key = filter.key
    filter_dialog.rule = filter.rule
    filter_dialog.run{ |response|
      if response == Gtk::Dialog::RESPONSE_ACCEPT
        filter.name = filter_dialog.name
        filter.key = filter_dialog.key
        filter.rule = filter_dialog.rule
        filter.clear_rss()
        filter_dialog.selected_rss_urls.each{ |url|
          filter.add_rss(url)
        }
        $filter_selection.signal_emit("changed")
      end
      filter_dialog.destroy
    }
  end
end

def remove_selected_filter
  name = selected_filter_name()
  remove_filter_by_name(name) if name
end

def show_filter_dialog_for_adding
    filter_dialog = FilterDialog.new
    filter_dialog.run{ |response|
      if response == Gtk::Dialog::RESPONSE_ACCEPT
        new_filter = Filter.new(filter_dialog.name,
                                filter_dialog.selected_rss_urls,
                                filter_dialog.key,
                                filter_dialog.rule)
        add_filter(new_filter)
      end
      filter_dialog.destroy
    }
end

def create_icon(stock_id,style,size=Gtk::IconSize::LARGE_TOOLBAR)
  icon_set = style.lookup_icon_set(stock_id)
  pixbuf = icon_set.render_icon(style,
                                Gtk::Widget::TEXT_DIR_NONE,
                                Gtk::STATE_NORMAL,
                                size)
  icon = Gtk::Image.new(pixbuf)
  return icon
end

def add_html_info(body,encoding)
  source = "<html><head>"
  source += "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=#{encoding}\">"
  source += "</head><body>"
  source += body
  source += "</body></html>"
end

def set_read_title_style(list,iter)
  list.set_value(iter,TITLE_COLOR_COLUMN,"black")
  list.set_value(iter,TITLE_BACKGROUND_COLOR_COLUMN,"white")
  list.set_value(iter,TITLE_WEIGHT_COLUMN,500)
  list.set_value(iter,DATE_COLOR_COLUMN,"black")
  list.set_value(iter,DATE_BACKGROUND_COLOR_COLUMN,"white")
  list.set_value(iter,DATE_WEIGHT_COLUMN,500)
  list.set_value(iter,RSS_COLOR_COLUMN,"black")
  list.set_value(iter,RSS_BACKGROUND_COLOR_COLUMN,"white")
  list.set_value(iter,RSS_WEIGHT_COLUMN,500)
end

def set_unread_title_style(list,iter)
  list.set_value(iter,TITLE_COLOR_COLUMN,"red")
  list.set_value(iter,TITLE_BACKGROUND_COLOR_COLUMN,"snow2")
  list.set_value(iter,TITLE_WEIGHT_COLUMN,700)
  list.set_value(iter,DATE_COLOR_COLUMN,"red")
  list.set_value(iter,DATE_BACKGROUND_COLOR_COLUMN,"snow2")
  list.set_value(iter,DATE_WEIGHT_COLUMN,700)
  list.set_value(iter,RSS_COLOR_COLUMN,"red")
  list.set_value(iter,RSS_BACKGROUND_COLOR_COLUMN,"snow2")
  list.set_value(iter,RSS_WEIGHT_COLUMN,700)
end

def create_window
  Gtk.init
  hpane=nil
  vpane=nil

  # load a file storing read items to find out read item
  load_read_items
  load_cached_items

  # making new window
  window = Gtk::Window.new
  window.signal_connect("delete_event"){
    $config["window/x"] = window.position[0]
    $config["window/y"] = window.position[1]
    $config["window/width"] = window.size[0]
    $config["window/height"] = window.size[1]
    $config["window/hpos"] = hpane.position
    $config["window/vpos"] = vpane.position

    output_configuration
    output_read_items
    output_cached_items
    false
  }
  window.signal_connect("destroy"){
    Gtk.main_quit
  }
  window.border_width = 1
  window.title = "Ruggregator"

  vbox = Gtk::VBox.new
  window.add(vbox)

  handlebox = Gtk::HandleBox.new
  vbox.pack_start(handlebox,false,false,0)
  
  toolbar = Gtk::Toolbar.new
  toolbar.toolbar_style = Gtk::Toolbar::BOTH_HORIZ
  handlebox.add(toolbar)

  icon = create_icon(Gtk::Stock::QUIT,toolbar.style)
  toolbar.append(_("Quit"),_("Quit this application"), "Private",icon){
    window.destroy unless window.signal_emit("delete_event",nil)
  }
  icon = create_icon(Gtk::Stock::INDEX,toolbar.style)
  toolbar.append(_("RSS"),_("Add or remove RSS feeds"), "Private",icon){
    rss_dialog = RSSDialog.new
    rss_dialog.run{ |response|
      rss_dialog.destroy
    }
  }
  icon = create_icon(Gtk::Stock::PREFERENCES,toolbar.style)
  toolbar.append(_("Preferences"),_("Configure this application"), "Private",icon){
    show_preferences_dialog
  }
  icon = create_icon(Gtk::Stock::DIALOG_INFO,toolbar.style)
  toolbar.append(_("About"),_("About this application"), "Private",icon){
    show_about_window
  }

  hpane = Gtk::HPaned.new
  vbox.pack_start(hpane,true,true,0)

  # creating sub vbox
  filter_vbox = Gtk::VBox.new
  hpane.add(filter_vbox)

  # creating filter list
  filter_list_view = Gtk::TreeView.new($filter_list)
  renderer = Gtk::CellRendererText.new
  column = Gtk::TreeViewColumn.new(_("Filter"),renderer, :text => 0)
  filter_list_view.append_column(column)
  $filter_selection = filter_list_view.selection
  $filter_selection.mode = Gtk::SELECTION_SINGLE
  filter_vbox.pack_start(filter_list_view,true,true,0)

  # action for double click event
  filter_list_view.signal_connect("row-activated"){
    show_properties_dialog
  }

  # context menu for right clicking in a filter view
  filter_menu = Gtk::Menu.new

  menu_item = Gtk::ImageMenuItem.new(_("Remove"))
  menu_item.image = create_icon(Gtk::Stock::REMOVE,filter_menu.style,Gtk::IconSize::MENU)
  menu_item.signal_connect("activate"){
    remove_selected_filter
  }
  filter_menu.append(menu_item)

  menu_item = Gtk::ImageMenuItem.new(_("Properties"))
  menu_item.image = create_icon(Gtk::Stock::PROPERTIES,filter_menu.style,Gtk::IconSize::MENU)
  menu_item.signal_connect("activate"){
    show_properties_dialog
  }
  filter_menu.append(menu_item)

  filter_menu.append(Gtk::SeparatorMenuItem.new)

  menu_item = Gtk::ImageMenuItem.new(_("Add"))
  menu_item.image = create_icon(Gtk::Stock::ADD,filter_menu.style,Gtk::IconSize::MENU)
  menu_item.signal_connect("activate"){
    show_filter_dialog_for_adding
  }
  filter_menu.append(menu_item)

  filter_menu.show_all

  # handler for right clicking
  filter_list_view.add_events(Gdk::Event::BUTTON_PRESS_MASK)
  filter_list_view.signal_connect("button_press_event") do |widget, event|
	if event.kind_of? Gdk::EventButton
      if (event.button == 3)
        click_event = event.clone
        click_event.button = 1
        filter_list_view.signal_emit("button_press_event",click_event)
        filter_menu.popup(nil, nil, event.button, event.time)
      end
	end
  end

  # toolbar for manipulating filters
  filter_toolbar = Gtk::Toolbar.new
  filter_toolbar.toolbar_style = Gtk::Toolbar::ICONS
  filter_vbox.pack_start(filter_toolbar,false,false,0)

  icon = create_icon(Gtk::Stock::ADD,filter_toolbar.style)
  filter_toolbar.append(_("Add"),_("Add a new filter"), "Private",icon){
    show_filter_dialog_for_adding
  }

  icon = create_icon(Gtk::Stock::REMOVE,filter_toolbar.style)
  filter_toolbar.append(_("Remove"),_("Remove the selected filter"), "Private",icon){
    remove_selected_filter
  }

  icon = create_icon(Gtk::Stock::PROPERTIES,filter_toolbar.style)
  filter_toolbar.append(_("Properties"),_("Configure the selected filter"), "Private",icon){
    show_properties_dialog
  }


  vpane = Gtk::VPaned.new
  hpane.add(vpane)

  # creating item list
  item_list = Gtk::ListStore.new(String,String,String,Integer,String,String,String,Integer,String,String,String,Integer)

  scrolled_window = Gtk::ScrolledWindow.new
  scrolled_window.set_policy(Gtk::POLICY_AUTOMATIC,Gtk::POLICY_AUTOMATIC);

  item_list_view = Gtk::TreeView.new(item_list)

  renderer = Gtk::CellRendererText.new
  column = Gtk::TreeViewColumn.new(_("Title"),renderer, :text => TITLE_COLUMN, :foreground => TITLE_COLOR_COLUMN, :background => TITLE_BACKGROUND_COLOR_COLUMN, :weight => TITLE_WEIGHT_COLUMN)
  column.resizable = true
  item_list_view.append_column(column)

  renderer = Gtk::CellRendererText.new
  column = Gtk::TreeViewColumn.new(_("Date"),renderer, :text => DATE_COLUMN, :foreground => DATE_COLOR_COLUMN, :background => DATE_BACKGROUND_COLOR_COLUMN, :weight => DATE_WEIGHT_COLUMN)
  column.resizable = true
  item_list_view.append_column(column)

  renderer = Gtk::CellRendererText.new
  column = Gtk::TreeViewColumn.new(_("RSS Title"),renderer, :text => RSS_COLUMN, :foreground => RSS_COLOR_COLUMN, :background => RSS_BACKGROUND_COLOR_COLUMN, :weight => RSS_WEIGHT_COLUMN)
  column.resizable = true
  item_list_view.append_column(column)

  scrolled_window.add_with_viewport(item_list_view)
  vpane.add(scrolled_window)

  # a handler for changed selection of filter list
  # to change item list in concern with the selected filter
  $filter_selection.signal_connect("changed"){
    item_list.clear
    filter_name = selected_filter_name()
    if filter_name
      filter = get_filter_by_name(filter_name)
      for i in 0..(filter.n_item-1)
        item = filter.item(i)
        iter = item_list.append
        item_list.set_value(iter,TITLE_COLUMN,item.title)

        date_str = item.dc_date.strftime($config["date-format"])
        if /\A.*\.(.*)\Z/ =~ ENV["LANG"]
          encoded_date_str = Iconv.iconv("UTF-8",$1.strip.upcase,date_str).to_s
        else
          encoded_date_str = date_str
        end
        item_list.set_value(iter,DATE_COLUMN,encoded_date_str)

        channel = selected_filter.channel(i)
        item_list.set_value(iter,RSS_COLUMN,channel.title)

        # check if it is a already-read item
        hash = "#{item.title}:#{item.dc_date}"
        if $read_items.include?(hash)
          set_read_title_style(item_list,iter)
        else
          set_unread_title_style(item_list,iter)
        end
      end
    else
      # nothing selected
      item_list.clear
    end
  }

  # creating title label above HTML view
  html_vbox = Gtk::VBox.new
  vpane.add(html_vbox)

  channel_box = Gtk::HBox.new
  html_vbox.pack_start(channel_box,false,false,0)

  channel_label = Gtk::Label.new()
  channel_label.markup = "<b>" + _("Channel") + ": </b>"
  channel_label.set_alignment(0,0.5)
  channel_box.pack_start(channel_label,false,false,0)

  logo_event_box = Gtk::EventBox.new()
  logo_event_box.signal_connect("button_press_event"){
    command = $config["url-handler"].gsub(/\%s/,$logo_link) + " &"
    system command
  }
  channel_box.pack_start(logo_event_box,false,false,0)

  title_label = Gtk::Label.new()
  title_label.markup = "<b>" + _("Title") + ": </b>"
  title_label.set_alignment(0,0.5)
  html_vbox.pack_start(title_label,false,false,0)

  link_event_box = Gtk::EventBox.new
  link_label = Gtk::Label.new()
  link_label.markup = "<b>" + _("Link") + ": </b> "
  link_label.set_alignment(0,0.5)
  link_event_box.add(link_label)
  link = ""
  link_event_box.signal_connect("button_press_event"){
    command = $config["url-handler"].gsub(/\%s/,link) + " &"
    system command
  }
  html_vbox.pack_start(link_event_box,false,false,0)

  # creating HTML view (require gtkhtml and ruby-gtkhtml)
  html_view = Gtk::HtmlView.new()
  scrolled_window2 = Gtk::ScrolledWindow.new(nil,nil)
  scrolled_window2.set_policy(Gtk::POLICY_ALWAYS,Gtk::POLICY_ALWAYS)
  scrolled_window2.add(html_view)

  # set default content for the HTML view
  doc = Gtk::HtmlDocument.new()
  doc.open_stream("text/html")
  doc.write_stream("<html><body><h1>RSS Reader</h1></body></html>")
  doc.close_stream()
  html_view.document = doc

  doc.signal_connect( "link_clicked" ) {  |doc,link|
    # launch WWW browser in background
    command = $config["url-handler"].gsub(/\%s/,link) + " &"
    system command
  }
  doc.signal_connect("request_url") { |html_doc, url, stream|
    open(url) { |f|
      stream.write f.read
      stream.close
    }
  }


  html_vbox.pack_start(scrolled_window2,true,true,0)

  # a handler for changed selection of item list
  # to change HTML view content in concern with the selected item
  item_list_selection = item_list_view.selection
  item_list_selection.signal_connect("changed"){
    html_view.document.clear
    iter = item_list_selection.selected
    if iter
      index = iter.path.indices[0]
      item = selected_filter.item(index)
      # compare with read item hashes
      # if read item hashes don't include this item, add it's item to them
      hash = "#{item.title}:#{item.dc_date}"
      unless $read_items.include?(hash)
        $read_items.push(hash)
        set_read_title_style(item_list,iter)
      end

      # change title label
      channel = selected_filter.channel(index)
      channel_label.markup = "<b>" + _("Channel") + ": </b> #{channel.title}"
      title_label.markup = "<b>" + _("Title") + ": </b> #{item.title}"

      # show title label
      if channel.image
        open(channel.image.resource){ |f|
          home = ENV["HOME"]
          o = open(tmplogofile,"w")
          o.puts f.read
          o.close
        }
        logo = Gtk::Image.new(tmplogofile)
        logo_event_box.each{ |child|
          logo_event_box.remove(child)
        }
        logo_event_box.add(logo)
        $logo_link = channel.link
        logo.show
      else
        logo_event_box.each{ |child|
          logo_event_box.remove(child)
        }
      end

      # change link label
      link = item.link
      link_label.markup = "<b>" + _("Link") + ": </b> <span foreground='blue' underline='single'>" + link + "</span>"

      # change HTML view
      if item.description
        html_view.document.open_stream("text/html")
        html_view.document.write_stream(add_html_info(item.description,"utf-8"))
        html_view.document.close_stream
      elsif item.content_encoded
        html_view.document.open_stream("text/html")
        html_view.document.write_stream(add_html_info(item.content_encoded,"utf-8"))
        html_view.document.close_stream
      elsif item.link
        source = ""
        open(item.link){ |f|
          source = f.gets(nil)
        }
        html_view.document.open_stream("text/html")
        html_view.document.write_stream(source)
        html_view.document.close_stream
      end
    end
  }

  # Accel Group for short cut keys
  ag = Gtk::AccelGroup.new
  a_key_config = $config["accel-key/Quit"]
  ag.connect(a_key_config["keyval"],a_key_config["mask"],Gtk::ACCEL_VISIBLE){
    window.destroy unless window.signal_emit("delete_event",nil)
  }
  a_key_config = $config["accel-key/Preferences"]
  ag.connect(a_key_config["keyval"],a_key_config["mask"],Gtk::ACCEL_VISIBLE){
    show_preferences_dialog
  }
  a_key_config = $config["accel-key/About"]
  ag.connect(a_key_config["keyval"],a_key_config["mask"],Gtk::ACCEL_VISIBLE){
    show_about_window
  }
  window.add_accel_group(ag)

  # set default window size
  window.set_default_size($config["window/width"],$config["window/height"])
  window.move($config["window/x"],$config["window/y"])
  hpane.position = $config["window/hpos"]
  vpane.position = $config["window/vpos"]

  window.show_all

  Gtk.main
end

