#!/usr/bin/ruby -Ku

require 'test/unit'
require 'punycode4r'
require 'iconv'

class TC_Punycode < Test::Unit::TestCase
  def encode(str)
    Punycode.encode(Iconv.iconv('UCS-4LE', 'UTF-8', str)[0])
  end
  def decode(str)
    Iconv.iconv('UTF-8', 'UCS-4LE', Punycode.decode(str))[0]
  end

  Example = [
    # 7.1 Sample strings

    # (G) Japanese (kanji and hiragana):
    ['なぜみんな日本語を話してくれないのか',
      'n8jok5ay5dzabd5bym9f0cm5685rrjetr6pdxa'],

    # (L) 3<nen>B<gumi><kinpachi><sensei>
    ['3年B組金八先生', '3B-ww4c5e180e575a65lsy2b'],

    # (M) <amuro><namie>-with-SUPER-MONKEYS
    ['安室奈美恵-with-SUPER-MONKEYS',
      '-with-SUPER-MONKEYS-pc58ag80a8qai00g7n9n'],

    # (N) Hello-Another-Way-<sorezore><no><basho>
    ['Hello-Another-Way-それぞれの場所',
      'Hello-Another-Way--fc4qua05auwb3674vfr0b'],

    # (O) <hitotsu><yane><no><shita>2
    ['ひとつ屋根の下2', '2-u9tlzr9756bt3uc0v'],

    # (P) Maji<de>Koi<suru>5<byou><mae>
    ['MajiでKoiする5秒前', 'MajiKoi5-783gue6qz075azm5e'],

    # (Q) <pafii>de<runba>
    ['パフィーdeルンバ', 'de-jg4avhby1noc0d'],

    # (R) <sono><supiido><de>
    ['そのスピードで', 'd9juau41awczczp'],

    # (S) -> $1.00 <-
    ['-> $1.00 <-', '-> $1.00 <--'],
  ]

  def test_encode
    Example.each do |input, excepted|
      assert_equal(excepted, encode(input))
    end
  end
  def test_decode
    Example.each do |excepted, input|
      assert_equal(excepted, decode(input))
    end
  end
end

if __FILE__ == $0
  begin
    require 'test/unit/ui/gtk/testrunner'
    Test::Unit::UI::GTK::TestRunner.run(TC_Punycode.suite)
  rescue LoadError
  end
end
