#!/usr/bin/ruby
#
# MultiRange
#
# Copyright (c) 2002- Kazuhiro NISHIYAMA.
# You can redistribute it and/or modify it under the same terms as Ruby.
#
# $Id: multirange.rb,v 1.6 2002/11/16 10:31:49 znz Exp $

class MultiRangeError < RangeError; end

class MultiRange
  def initialize(range, *ranges)
    @ranges = []
    add(range)
    ranges.each do |range|
      add(range)
    end
    @sorted = false
    @merged = false
  end
  def ranges
    normalize!
    @ranges
  end
  protected :ranges

  def add(arg)
    @sorted = false
    @merged = false
    case arg
    when Range
      if arg.exclude_end?
        @ranges.push arg
      else
        @ranges.push Range.new(arg.first, arg.last.succ, true)
      end
    when Array
      arg.each do |e|
        add(e)
      end
    else
      @ranges.push Range.new(arg, arg.succ, true)
    end
    self
  end
  alias << add
  alias push add

  def +(arg)
    mr = self.class.new(*(@ranges.dup))
    mr.add(arg)
    mr
  end

  def subtract!(arg)
    normalize!
    changed = false
    if arg.is_a? MultiRange
      arg.ranges.each do |range|
        changed |= ! subtract!(range).nil?
      end
    elsif arg.is_a? Range
      first = arg.first
      last = (arg.exclude_end? ? arg.last : arg.last.succ)
      new_ranges = []
      @ranges.each do |range|
        if range.last <= first
          new_ranges.push range
        elsif range.first <= first
          if range.last <= last
            changed = true
            new_ranges.push Range.new(range.first, first, true)
          else
            changed = true
            new_ranges.push Range.new(range.first, first, true)
            new_ranges.push Range.new(last, range.last, true)
          end
        elsif last <= range.first
          new_ranges.push range
        else
          changed = true
          new_ranges.push Range.new(last, range.last, true)
        end
      end
      if changed
        new_ranges.delete_if {|range| range.first == range.last }
        if new_ranges.empty?
          raise MultiRangeError, "MultiRange must not be empty."
        end
        @ranges = new_ranges
      end
    else
      return subtract!(Range.new(arg, arg.succ, true))
    end
    if changed
      return self
    else
      return nil
    end
  end

  def subtract(arg)
    mr = self.class.new(*(@ranges.dup))
    mr.subtract!(arg)
    mr
  end
  alias - subtract

  def to_s
    normalize!
    @ranges.collect{|r|
      first = r.first
      last = r.last - 1
      if first == last
        last.to_s
      else
        "#{first}-#{last}"
      end
    }.join(",")
  end

  def MultiRange.parse(str)
    raise ArgumentError, "#{str}" unless /\A(?:-?\d+(?:--?\d+)?)(?:,-?\d+(?:--?\d+)?)*\Z/ === str
    mr = MultiRange.new(str.to_i)
    str.scan(/(-?\d+)(?:-(-?\d+))?/) do |first, last|
      last ||= first
      mr.add(Range.new(first.to_i, last.to_i.succ, true))
    end
    mr.__send__(:normalize!)
    mr
  end

  # normalize method
  def sort!
    return if @sorted
    @ranges.sort!{|a,b|
      a.first <=> b.first
    }
    @sorted = true
  end
  private :sort!

  def merge!
    return if @merged
    sort! unless @sorted
    old_ranges = @ranges
    prev = old_ranges.shift
    @ranges = []
    old_ranges.each do |range|
      if $DEBUG # assertion
        raise "not sorted!" unless prev.first <= range.first
      end
      if range.last <= prev.last # range in prev
        next
      elsif range.first <= prev.last # overlap
        prev = Range.new(prev.first, range.last, true)
        next
      else
        @ranges.push prev
        prev = range
      end
    end
    @ranges.push prev
    @merged = true
  end
  private :merge!

  def normalize!
    sort!
    merge!
  end
  private :normalize!

  # for test
  def _inner_size_
    @ranges.size
  end
  private :_inner_size_

  # Range like methods
  def ==(other)
    normalize!
    if other.is_a?(Numeric) and /:in `==='\z/ === caller(1)[0] #`
      self === other
    elsif other.is_a? self.class
      @ranges == other.ranges
    else
      false
    end
  end

  def ===(other)
    if other.is_a? self.class
      return self == other
    end
    normalize!
    @ranges.each do |r|
      return true if r === other
    end
    false
  end

  include Enumerable
  def each(&block)
    normalize!
    @ranges.each do |range|
      range.each(&block)
    end
  end

  def first
    normalize!
    @ranges.first.first
  end
  alias begin first

  def last
    normalize!
    @ranges.last.last
  end
  alias end last
end


if __FILE__ == $0
  eval DATA.read
end

__END__
require 'runit/testcase'
require 'runit/cui/testrunner'

unless defined?(NoMethodError)
  NoMethodError = NameError
end

class MultiRange_Test < RUNIT::TestCase
  def test_new1
    mr = MultiRange.new(1..2)
    assert_equal('1-2', mr.to_s)
    assert_equal(1, mr.begin)
    assert_equal(1, mr.first)
    assert_equal(3, mr.end)
    assert_equal(3, mr.last)
    assert_equal(1, mr.send(:_inner_size_))
    assert_equal(false, mr === -1)
    assert_equal(false, mr === 0)
    assert_equal(true, mr === 1)
    assert_equal(true, mr === 2)
    assert_equal(false, mr === 3)
    assert_equal([1,2], mr.to_a)

    mr = MultiRange.new(1...2)
    assert_equal('1', mr.to_s)
    assert_equal(1, mr.begin)
    assert_equal(1, mr.first)
    assert_equal(2, mr.end)
    assert_equal(2, mr.last)
    assert_equal(1, mr.send(:_inner_size_))
    assert_equal(false, mr === -1)
    assert_equal(false, mr === 0)
    assert_equal(true, mr === 1)
    assert_equal(false, mr === 2)
    assert_equal(false, mr === 3)
    assert_equal([1], mr.to_a)

    mr = MultiRange.new(1)
    assert_equal('1', mr.to_s)
    assert_equal(1, mr.begin)
    assert_equal(1, mr.first)
    assert_equal(2, mr.end)
    assert_equal(2, mr.last)
    assert_equal(1, mr.send(:_inner_size_))
    assert_equal(false, mr === -1)
    assert_equal(false, mr === 0)
    assert_equal(true, mr === 1)
    assert_equal(false, mr === 2)
    assert_equal(false, mr === 3)
    assert_equal([1], mr.to_a)
  end

  def test_new2
    mr = MultiRange.new(1, 2)
    assert_equal('1-2', mr.to_s)
    assert_equal(1, mr.first)
    assert_equal(3, mr.last)
    assert_equal(1, mr.send(:_inner_size_))
    assert_equal(false, mr === -1)
    assert_equal(false, mr === 0)
    assert_equal(true, mr === 1)
    assert_equal(true, mr === 2)
    assert_equal(false, mr === 3)
    assert_equal([1,2], mr.to_a)

    mr = MultiRange.new(2, 1)
    assert_equal('1-2', mr.to_s)
    assert_equal(1, mr.first)
    assert_equal(3, mr.last)
    assert_equal(1, mr.send(:_inner_size_))
    assert_equal(false, mr === -1)
    assert_equal(false, mr === 0)
    assert_equal(true, mr === 1)
    assert_equal(true, mr === 2)
    assert_equal(false, mr === 3)
    assert_equal([1,2], mr.to_a)

    mr = MultiRange.new(1..2, 3..4)
    assert_equal('1-4', mr.to_s)
    assert_equal(1, mr.first)
    assert_equal(5, mr.last)
    assert_equal(1, mr.send(:_inner_size_))
    assert_equal(false, mr === -1)
    assert_equal(false, mr === 0)
    assert_equal(true, mr === 1)
    assert_equal(true, mr === 2)
    assert_equal(true, mr === 3)
    assert_equal(true, mr === 4)
    assert_equal(false, mr === 5)
    assert_equal([1,2,3,4], mr.to_a)

    mr = MultiRange.new(1, 3..4)
    assert_equal('1,3-4', mr.to_s)
    assert_equal(1, mr.first)
    assert_equal(5, mr.last)
    assert_equal(2, mr.send(:_inner_size_))
    assert_equal(false, mr === -1)
    assert_equal(false, mr === 0)
    assert_equal(true, mr === 1)
    assert_equal(false, mr === 2)
    assert_equal(true, mr === 3)
    assert_equal(true, mr === 4)
    assert_equal(false, mr === 5)
    assert_equal([1,3,4], mr.to_a)

    mr = MultiRange.new(1, 0..4)
    assert_equal('0-4', mr.to_s)
    assert_equal(0, mr.first)
    assert_equal(5, mr.last)
    assert_equal(1, mr.send(:_inner_size_))
    assert_equal(false, mr === -1)
    assert_equal(true, mr === 0)
    assert_equal(true, mr === 1)
    assert_equal(true, mr === 2)
    assert_equal(true, mr === 3)
    assert_equal(true, mr === 4)
    assert_equal(false, mr === 5)
    assert_equal([0,1,2,3,4], mr.to_a)
  end

  def test_new3
    mr = MultiRange.new(0, 2...3, 4..5)
    assert_equal('0,2,4-5', mr.to_s)
    assert_equal(0, mr.first)
    assert_equal(6, mr.last)
    assert_equal(3, mr.send(:_inner_size_))
    assert_equal(false, mr === -1)
    assert_equal(true, mr === 0)
    assert_equal(false, mr === 1)
    assert_equal(true, mr === 2)
    assert_equal(false, mr === 3)
    assert_equal(true, mr === 4)
    assert_equal(true, mr === 5)
    assert_equal(false, mr === 6)
    assert_equal([0,2,4,5], mr.to_a)
  end

  def test_private
    mr = MultiRange.new(1)
    assert_exception(NoMethodError) { mr.sort! }
    assert_exception(NoMethodError) { mr.merge! }
    assert_exception(NoMethodError) { mr.normalize! }
    assert_exception(NoMethodError) { mr._inner_size_ }
  end

  def test_eq
    assert_operator(MultiRange.new(1, 3..4), :==, MultiRange.new(1, 3..4))
    assert_operator(MultiRange.new(1, 2), :==, MultiRange.new(2, 1))
    assert_equal(true, MultiRange.new(1) != MultiRange.new(2))
    mr = MultiRange.new(0, 2...3, 4..5)
    assert_operator(mr, :==, mr)
  end

  def test_eqq
    assert_equal(false, 0 === MultiRange.new(1))
    assert_equal(true,  1 === MultiRange.new(1))
    assert_equal(false, 2 === MultiRange.new(1))
    assert_equal(true, MultiRange.new(1) === MultiRange.new(1))
    assert_equal(false, MultiRange.new(1) === MultiRange.new(2))
  end

  def test_add
    assert_equal(MultiRange.new(1..2), MultiRange.new(1).add(2))
    assert_equal(MultiRange.new(1..3), MultiRange.new(1).add([2,3]))
    assert_equal(MultiRange.new(1..3), MultiRange.new(1,2).add([2,3]))
    assert_equal(MultiRange.new(1..5), MultiRange.new(1).add([[2,4],[3,5]]))
    assert_equal(MultiRange.new(1..10), MultiRange.new(1).add([[2..3,[6,7,8]],[4..5,9...11]]))
  end

  def test_plus
    assert_equal(MultiRange.new(1..3), ((mr=MultiRange.new(1,2)) + 3))
    assert_equal(MultiRange.new(1,2), mr)
    assert_equal(MultiRange.new(1..4), (MultiRange.new(1,2) + (3..4)))
    assert_equal(MultiRange.new(1,2), mr)
  end

  def test_subtract
    assert_equal(nil, (mr=MultiRange.new(1)).subtract!(3))
    assert_equal(MultiRange.new(1), mr)
    assert_equal(nil, (mr=MultiRange.new(1,2)).subtract!(3))
    assert_equal(MultiRange.new(1,2), mr)
    assert_equal(MultiRange.new(1,2), MultiRange.new(1..3).subtract!(3))
    assert_equal(MultiRange.new(1,2,4), MultiRange.new(1..4).subtract!(3))
    assert_equal(MultiRange.new(1,2,4,5), MultiRange.new(1..5).subtract!(3))
    assert_equal(MultiRange.new(2), MultiRange.new(2,3).subtract!(3))
    assert_exception(MultiRangeError) { MultiRange.new(3).subtract!(3) }
    assert_equal(MultiRange.new(4), MultiRange.new(3,4).subtract!(3))
    assert_equal(MultiRange.new(4,5), MultiRange.new(3..5).subtract!(3))
    assert_equal(nil, (mr=MultiRange.new(4,5)).subtract!(3))
    assert_equal(MultiRange.new(4,5), mr)
    assert_equal(nil, (mr=MultiRange.new(5)).subtract!(3))
    assert_equal(MultiRange.new(5), mr)

    assert_equal(MultiRange.new(5), MultiRange.new(1,3,5).subtract!(MultiRange.new(1,3)))
    assert_equal(MultiRange.new(5), MultiRange.new(1,5,9).subtract!(MultiRange.new(1,9)))
    assert_equal(nil, (mr=MultiRange.new(1,3,5)).subtract!(MultiRange.new(2,4)))
    assert_equal(MultiRange.new(1,3,5), mr)

    assert_equal(MultiRange.new(1), MultiRange.new(1).subtract(3))
    assert_equal(MultiRange.new(1,2), MultiRange.new(1,2).subtract(3))
    assert_equal(MultiRange.new(1,2), (mr=MultiRange.new(1..3)).subtract(3))
    assert_equal(MultiRange.new(1..3), mr)

    assert_equal(MultiRange.new(1), (MultiRange.new(1) - 3))
    assert_equal(MultiRange.new(1,2), (MultiRange.new(1,2) - (3..4)))
  end

  def test_parse
    [
      '1',
      '1-2',
      ['1-2', '1,2'],
      ['1-2', '2,1'],
      ['1-4', '1-2,3-4'],
      '1,3-4',
      ['0-4', '1,0-4'],
      '0,2,4-5',

      '-1',
      '-1-2',
      ['-2--1', '-1,-2'],
      '-2--1',
      ['-3-4', '-2--1,-3-4'],
      '-1,3-4',
      ['-4--3,-1', '-1,-3,-4'],
      ['-4--1', '-4--3,-2--1'],
      ['-4,1', '1,-4'],

      '10',
      ['10-11', '10,11'],
      '10-11',
      '10-11,21',
      ['10-21', '10,11-21'],
      '10-11,21-22',
    ].each do |expected, str|
      str ||= expected
      assert_equal(expected, MultiRange.parse(str).to_s)
    end

    [ '--1', '-1-', '1--', ',', '-1,-' ].each do |str|
      assert_exception(ArgumentError) { MultiRange.parse(str) }
    end
  end
end

RUNIT::CUI::TestRunner.run(MultiRange_Test.suite)
