#!/usr/local/bin/ruby
# $Id: test_wpm_page_with_xmlparser.rb,v 1.1.1.1 2004/04/04 15:22:51 toki Exp $
# test of Page

require 'rubyunit'
require 'wpm'
require 'wpm/xmlparser'

module TestWebPageMaker
  class TestPageByXMLParser < RUNIT::TestCase
    def init_page(name)
      Dir.mkdir(name)
      nil
    end
    private :init_page

    def build_page(name='MainPage')
      src_name = File.join(name, name + '.rb')
      map_name = File.join(name, name + '.map')
      xml_name = File.join(name, name + '.xml')

      File.open(src_name, 'w') { |src_output|
	File.open(map_name, 'w') { |map_output|
	  File.open(xml_name, 'w') { |xml_output|
	    yield(src_output, map_output, xml_output)
	  }
	}
      }

      nil
    end
    private :build_page

    def run_page
      page = ''
      @page.compile
      @page.setup
      @page.run
      @page.follow
      @page.make_page do |page_text|
	page.concat(page_text)
      end
      page.concat("\n")

      page
    end
    private :run_page

    def clean_page(name)
      src_name = File.join(name, name + '.rb')
      map_name = File.join(name, name + '.map')
      xml_name = File.join(name, name + '.xml')

      File.delete(src_name) if (File.exist? src_name)
      File.delete(map_name) if (File.exist? map_name)
      File.delete(xml_name) if (File.exist? xml_name)
      Dir.delete(name)

      nil
    end
    private :clean_page

    def setup
      # for Rucy::Driver class
      @params_call = 0
      @params = Hash.new
      @page_path_call = 0
      @path_info = ''

      # setup Rucy::Page
      @driver = self
      @xml_assist = WPM::XMLAssistByXMLParser
      @page = WPM::Page.new('MainPage', @driver, @xml_assist)
      init_page('MainPage')
    end

    def params
      @params_call += 1
      @params.dup
    end

    def page_path(page_name='MainPage', page_path='/PageMaker')
      @page_path_call += 1
      page_path + '/' + page_name + @path_info
    end

    def teardown
      clean_page('MainPage')
    end

    def test_plain_html
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
	xml_output.print "<head><title>test</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<h1>Test</h1>\n"
	xml_output.print "<h2>Plain Text</h2>\n"
	xml_output.print "<p>Hello world.</p>\n"
	xml_output.print "<h2>Special Characters</h2>\n"
	xml_output.print "<p>&lt;&gt;&amp;</p>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test</title></head>\n"
      dst += "<body>\n"
      dst += "<h1>Test</h1>\n"
      dst += "<h2>Plain Text</h2>\n"
      dst += "<p>Hello world.</p>\n"
      dst += "<h2>Special Characters</h2>\n"
      dst += "<p>&lt;&gt;&amp;</p>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal(@page, @page.find('MainPage'))
    end

    def test_string_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "<string name=\"StringWidget\">\n"
	map_output.print "<value type=\"string\">Hello world.</value>\n"
	map_output.print "</string>\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of StringWidget</title></head>\n"
	xml_output.print "<body><p><pm:widget name=\"StringWidget\" /></p></body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of StringWidget</title></head>\n"
      dst += "<body><p>Hello world.</p></body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(0, @page_path_call)
    end

    def test_attribute_expansion
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<string name=\"Hello\">\n"
	map_output.print "<value type=\"string\">Hello world.</value>\n"
	map_output.print "</string>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of attribute expansion</title></head>\n"
	xml_output.print "<body comment=\"1. ${Hello} | 2. ${Hello}\">\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of attribute expansion</title></head>\n"
      dst += "<body comment=\"1. Hello world. | 2. Hello world.\">\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(0, @page_path_call)
    end

    def test_if_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @condition = true\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :condition\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "<if name=\"IfWidget\">\n"
	map_output.print "<condition type=\"accessor\">condition</condition>\n"
	map_output.print "</if>\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of IfWidget</title></head>\n"
	xml_output.print "<body><p><pm:widget name=\"IfWidget\">Hello world.</pm:widget></p></body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of IfWidget</title></head>\n"
      dst += "<body><p>Hello world.</p></body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal(true, @page.context.condition)

      @page.clear
      @page.context_hook{ |context|
	context.condition = false
      }
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of IfWidget</title></head>\n"
      dst += "<body><p></p></body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal(false, @page.context.condition)
    end

    def test_repeat_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "<repeat name=\"Repeat\">\n"
	map_output.print "<times type=\"number\">5</times>\n"
	map_output.print "</repeat>\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of RepeatWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<p>\n"
	xml_output.print "<pm:widget name=\"Repeat\"\n"
	xml_output.print ">Hello world.\n"
	xml_output.print "</pm:widget\n"
	xml_output.print "></p>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RepeatWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<p>\n"
      dst += "Hello world.\n"
      dst += "Hello world.\n"
      dst += "Hello world.\n"
      dst += "Hello world.\n"
      dst += "Hello world.\n"
      dst += "</p>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(0, @page_path_call)
    end

    def test_foreach_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  attr_accessor :item\n"
	src_output.print "  attr_accessor :index\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<foreach name=\"ForeachWidget\">\n"
	map_output.print "<list type=\"eval\">[ 'a', 'b', 'c' ]</list>\n"
	map_output.print "<item type=\"accessor\">item</item>\n"
	map_output.print "<index type=\"accessor\">index</index>\n"
	map_output.print "</foreach>\n"
	map_output.print "\n"
	map_output.print "<string name=\"Item\">\n"
	map_output.print "<value type=\"accessor\">item</value>\n"
	map_output.print "</string>\n"
	map_output.print "\n"
	map_output.print "<string name=\"Index\">\n"
	map_output.print "<value type=\"accessor\">index</value>\n"
	map_output.print "</string>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of ForeachWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<ul>\n"
	xml_output.print "<pm:widget name=\"ForeachWidget\"\n"
	xml_output.print "><li>(<pm:widget name=\"Index\" />) <pm:widget name=\"Item\" /></li>\n"
	xml_output.print "</pm:widget\n"
	xml_output.print "></ul>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of ForeachWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<ul>\n"
      dst += "<li>(1) a</li>\n"
      dst += "<li>(2) b</li>\n"
      dst += "<li>(3) c</li>\n"
      dst += "</ul>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(0, @page_path_call)
    end

    def test_import_widget
      init_page('AnotherPage')
      begin
	build_page{ |src_output, map_output, xml_output|
	  src_output.print "class MainPage < WPM::PageContext\n"
	  src_output.print "end\n"

	  map_output.print "<?xml version=\"1.0\"?>\n"
	  map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	  map_output.print "<import name=\"ImportWidget\" page=\"AnotherPage\">\n"
	  map_output.print "<title type=\"string\">Another Page</title>\n"
	  map_output.print "<body type=\"string\">Hello world.</body>\n"
	  map_output.print "</import>\n"
	  map_output.print "</map>\n"

	  xml_output.print "<?xml version=\"1.0\"?>\n"
	  xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	  xml_output.print "<head><title>test of ImportWidget</title></head>\n"
	  xml_output.print "<body>\n"
	  xml_output.print "<pm:widget name=\"ImportWidget\" />\n"
	  xml_output.print "</body>\n"
	  xml_output.print "</html>\n"
	}

	build_page('AnotherPage') { |src_output, map_output, xml_output|
	  src_output.print "class AnotherPage < WPM::PageContext\n"
	  src_output.print "  attr_accessor :title\n"
	  src_output.print "  attr_accessor :body\n"
	  src_output.print "end\n"

	  map_output.print "<?xml version=\"1.0\"?>\n"
	  map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	  map_output.print "\n"
	  map_output.print "<string name=\"Title\">\n"
	  map_output.print "<value type=\"accessor\">title</value>\n"
	  map_output.print "</string>\n"
	  map_output.print "\n"
	  map_output.print "<string name=\"Body\">\n"
	  map_output.print "<value type=\"accessor\">body</value>\n"
	  map_output.print "</string>\n"
	  map_output.print "\n"
	  map_output.print "</map>\n"

	  xml_output.print "<?xml version=\"1.0\"?>\n"
	  xml_output.print "<pm:import xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\"\n"
	  xml_output.print "><h1><pm:widget name=\"Title\" /></h1>\n"
	  xml_output.print "<p><pm:widget name=\"Body\" /></p\n"
	  xml_output.print "></pm:import>\n"
	}

	dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
	dst += "<head><title>test of ImportWidget</title></head>\n"
	dst += "<body>\n"
	dst += "<h1>Another Page</h1>\n"
	dst += "<p>Hello world.</p>\n"
	dst += "</body>\n"
	dst += "</html>\n"
	assert_equal(dst, run_page)
	assert_equal(0, @params_call)
	assert_equal(0, @page_path_call)
	assert_instance_of(WPM::Page, @page.find('MainPage.ImportWidget.AnotherPage'))
	assert_equal('MainPage.ImportWidget.AnotherPage', @page.find('MainPage.ImportWidget.AnotherPage').page_name)
      ensure
	clean_page('AnotherPage')
      end
    end

    def test_content_widget
      init_page('AnotherPage')
      begin
	build_page{ |src_output, map_output, xml_output|
	  src_output.print "class MainPage < WPM::PageContext\n"
	  src_output.print "end\n"

	  map_output.print "<?xml version=\"1.0\"?>\n"
	  map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	  map_output.print "<import name=\"ImportWidget\" page=\"AnotherPage\" />\n"
	  map_output.print "</map>\n"

	  xml_output.print "<?xml version=\"1.0\"?>\n"
	  xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	  xml_output.print "<head><title>test of ImportWidget</title></head>\n"
	  xml_output.print "<body>\n"
	  xml_output.print "<pm:widget name=\"ImportWidget\">Hello world.</pm:widget>\n"
	  xml_output.print "</body>\n"
	  xml_output.print "</html>\n"
	}

	build_page('AnotherPage') { |src_output, map_output, xml_output|
	  src_output.print "class AnotherPage < WPM::PageContext\n"
	  src_output.print "end\n"

	  map_output.print "<?xml version=\"1.0\"?>\n"
	  map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	  map_output.print "<content name=\"Content\" />\n"
	  map_output.print "</map>\n"

	  xml_output.print "<?xml version=\"1.0\"?>\n"
	  xml_output.print "<pm:import xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\"\n"
	  xml_output.print "><p><pm:widget name=\"Content\" /></p\n"
	  xml_output.print "></pm:import>\n"
	}

	dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
	dst += "<head><title>test of ImportWidget</title></head>\n"
	dst += "<body>\n"
	dst += "<p>Hello world.</p>\n"
	dst += "</body>\n"
	dst += "</html>\n"
	assert_equal(dst, run_page)
	assert_equal(0, @params_call)
	assert_equal(0, @page_path_call)
	assert_instance_of(WPM::Page, @page.find('MainPage.ImportWidget.AnotherPage'))
	assert_equal('MainPage.ImportWidget.AnotherPage', @page.find('MainPage.ImportWidget.AnotherPage').page_name)
      ensure
	clean_page('AnotherPage')
      end
    end

    def test_content_widget_without_import
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "<content name=\"Content\" />\n"
	map_output.print "</map>\n"

	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of ContentWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Content\" />\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      assert_exception(WPM::WidgetError) { run_page }
    end

    def test_hyperlink_widget_with_action
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @message = nil\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_reader :message\n"
	src_output.print "\n"
	src_output.print "  def hello\n"
	src_output.print "    @message = 'Hello world.'\n"
	src_output.print "  end\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<hyperlink name=\"Hello\">\n"
	map_output.print "<action type=\"method\">hello</action>\n"
	map_output.print "</hyperlink>\n"
	map_output.print "\n"
	map_output.print "<string name=\"Message\">\n"
	map_output.print "<value type=\"accessor\">message</value>\n"
	map_output.print "</string>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of HyperlinkWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<ul>\n"
	xml_output.print "<li><pm:widget name=\"Hello\">hello action</pm:widget></li>\n"
	xml_output.print "<li><pm:widget name=\"Message\" /></li>\n"
	xml_output.print "</ul>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<ul>\n"
      dst += "<li><a href=\"/PageMaker/MainPage?action=MainPage.Hello.0\">hello action</a></li>\n"
      dst += "<li></li>\n"
      dst += "</ul>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(1, @page_path_call)
      assert_nil(@page.context.message)

      @page.clear
      @params['action'] = 'MainPage.Hello.0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<ul>\n"
      dst += "<li><a href=\"/PageMaker/MainPage?action=MainPage.Hello.0\">hello action</a></li>\n"
      dst += "<li>Hello world.</li>\n"
      dst += "</ul>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(2, @page_path_call)
      assert_equal('Hello world.', @page.context.message)

      @page.clear
      @path_info = '/Foo/Bar'
      @params['action'] = 'MainPage.Hello.0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<ul>\n"
      dst += "<li><a href=\"/PageMaker/MainPage/Foo/Bar?action=MainPage.Hello.0\">hello action</a></li>\n"
      dst += "<li>Hello world.</li>\n"
      dst += "</ul>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(3, @params_call)
      assert_equal(3, @page_path_call)
      assert_equal('Hello world.', @page.context.message)
    end

    def test_hyperlink_widget_with_action_and_query
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @message = nil\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_reader :message\n"
	src_output.print "\n"
	src_output.print "  def hello\n"
	src_output.print "    @message = 'Hello world.'\n"
	src_output.print "  end\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<hyperlink name=\"Hello\">\n"
	map_output.print "<action type=\"method\">hello</action>\n"
	map_output.print "<query type=\"eval\">{ 'debug' =&gt; 'true' }</query>\n"
	map_output.print "</hyperlink>\n"
	map_output.print "\n"
	map_output.print "<string name=\"Message\">\n"
	map_output.print "<value type=\"accessor\">message</value>\n"
	map_output.print "</string>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of HyperlinkWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<ul>\n"
	xml_output.print "<li><pm:widget name=\"Hello\">hello action</pm:widget></li>\n"
	xml_output.print "<li><pm:widget name=\"Message\" /></li>\n"
	xml_output.print "</ul>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<ul>\n"
      dst += "<li><a href=\"/PageMaker/MainPage?action=MainPage.Hello.0&amp;debug=true\">hello action</a></li>\n"
      dst += "<li></li>\n"
      dst += "</ul>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(1, @page_path_call)
      assert_nil(@page.context.message)

      @page.clear
      @params['action'] = 'MainPage.Hello.0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<ul>\n"
      dst += "<li><a href=\"/PageMaker/MainPage?action=MainPage.Hello.0&amp;debug=true\">hello action</a></li>\n"
      dst += "<li>Hello world.</li>\n"
      dst += "</ul>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(2, @page_path_call)
      assert_equal('Hello world.', @page.context.message)

      @page.clear
      @path_info = '/Foo/Bar'
      @params['action'] = 'MainPage.Hello.0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<ul>\n"
      dst += "<li><a href=\"/PageMaker/MainPage/Foo/Bar?action=MainPage.Hello.0&amp;debug=true\">hello action</a></li>\n"
      dst += "<li>Hello world.</li>\n"
      dst += "</ul>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(3, @params_call)
      assert_equal(3, @page_path_call)
      assert_equal('Hello world.', @page.context.message)
    end

    def test_hyperlink_widget_with_page
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "<hyperlink name=\"AnotherPage\">\n"
	map_output.print "<page type=\"string\">AnotherPage</page>\n"
	map_output.print "</hyperlink>\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of HyperlinkWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<p><pm:widget name=\"AnotherPage\">click here.</pm:widget></p>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<p><a href=\"/PageMaker/AnotherPage\">click here.</a></p>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(1, @page_path_call)

      @page.clear
      @path_info = '/Foo/Bar'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<p><a href=\"/PageMaker/AnotherPage/Foo/Bar\">click here.</a></p>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(2, @page_path_call)
    end

    def test_hyperlink_widget_with_page_and_query
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "<hyperlink name=\"AnotherPage\">\n"
	map_output.print "<page type=\"string\">AnotherPage</page>\n"
	map_output.print "<query type=\"eval\">{ 'debug' =&gt; 'true' }</query>\n"
	map_output.print "</hyperlink>\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of HyperlinkWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<p><pm:widget name=\"AnotherPage\">click here.</pm:widget></p>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<p><a href=\"/PageMaker/AnotherPage?debug=true\">click here.</a></p>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(1, @page_path_call)

      @page.clear
      @path_info = '/Foo/Bar'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<p><a href=\"/PageMaker/AnotherPage/Foo/Bar?debug=true\">click here.</a></p>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(2, @page_path_call)
    end

    def test_hyperlink_widget_with_href
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "<hyperlink name=\"Link\">\n"
	map_output.print "<href type=\"string\">http://www.ruby-lang.org/</href>\n"
	map_output.print "</hyperlink>\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of HyperlinkWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<p><pm:widget name=\"Link\">click here.</pm:widget></p>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HyperlinkWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<p><a href=\"http://www.ruby-lang.org/\">click here.</a></p>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(0, @page_path_call)
    end

    def test_image_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "<image name=\"Image\">\n"
	map_output.print "<src type=\"string\">/icons/apache.gif</src>\n"
	map_output.print "<alt type=\"string\">Apache</alt>\n"
	map_output.print "</image>\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of ImageWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<p><pm:widget name=\"Image\" /></p>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of ImageWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<p><img src=\"/icons/apache.gif\" alt=\"Apache\" /></p>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(0, @page_path_call)
    end

    def test_frame_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<frame name=\"Navigation\">\n"
	map_output.print "<name type=\"string\">navi</name>\n"
	map_output.print "<page type=\"string\">NaviPage</page>\n"
	map_output.print "</frame>\n"
	map_output.print "\n"
	map_output.print "<frame name=\"Main\">\n"
	map_output.print "<name type=\"string\">main</name>\n"
	map_output.print "<page type=\"string\">MainPage</page>\n"
	map_output.print "</frame>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of FrameWidget</title></head>\n"
	xml_output.print "<frameset rows=\"10%, 90%\">\n"
	xml_output.print "<pm:widget name=\"Navigation\" />\n"
	xml_output.print "<pm:widget name=\"Main\" />\n"
	xml_output.print "</frameset>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of FrameWidget</title></head>\n"
      dst += "<frameset rows=\"10%, 90%\">\n"
      dst += "<frame name=\"navi\" src=\"/PageMaker/NaviPage\" />\n"
      dst += "<frame name=\"main\" src=\"/PageMaker/MainPage\" />\n"
      dst += "</frameset>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(0, @params_call)
      assert_equal(2, @page_path_call)
    end

    def test_text_field_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @textfield = \"none\"\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :textfield\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<textfield name=\"TextField\">\n"
	map_output.print "<value type=\"accessor\">textfield</value>\n"
	map_output.print "</textfield>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of TextFieldWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<p><pm:widget name=\"TextField\" /></p>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of TextFieldWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.TextField.0\" name=\"MainPage.TextField.0\" type=\"text\" value=\"none\" /></p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('none', @page.context.textfield)

      @page.clear
      @params['MainPage.TextField.0'] = 'Hello world.'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of TextFieldWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.TextField.0\" name=\"MainPage.TextField.0\" type=\"text\" value=\"Hello world.\" /></p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('Hello world.', @page.context.textfield)
    end

    def test_text_field_widget_with_label
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @textfield = \"none\"\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :textfield\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<textfield name=\"TextField\">\n"
	map_output.print "<value type=\"accessor\">textfield</value>\n"
	map_output.print "</textfield>\n"
	map_output.print "\n"
	map_output.print "<label name=\"Label\">\n"
	map_output.print "<for type=\"string\">TextField</for>\n"
	map_output.print "</label>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of TextFieldWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<p>\n"
	xml_output.print "<pm:widget name=\"Label\">text:</pm:widget>\n"
	xml_output.print "<pm:widget name=\"TextField\" />\n"
	xml_output.print "</p>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of TextFieldWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p>\n"
      dst += "<label for=\"MainPage.TextField.0\">text:</label>\n"
      dst += "<input id=\"MainPage.TextField.0\" name=\"MainPage.TextField.0\" type=\"text\" value=\"none\" />\n"
      dst += "</p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('none', @page.context.textfield)

      @page.clear
      @params['MainPage.TextField.0'] = 'Hello world.'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of TextFieldWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p>\n"
      dst += "<label for=\"MainPage.TextField.0\">text:</label>\n"
      dst += "<input id=\"MainPage.TextField.0\" name=\"MainPage.TextField.0\" type=\"text\" value=\"Hello world.\" />\n"
      dst += "</p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('Hello world.', @page.context.textfield)
    end

    def test_password_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @password = nil\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :password\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<password name=\"Password\">\n"
	map_output.print "<value type=\"accessor\">password</value>\n"
	map_output.print "</password>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of PasswordWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<p><pm:widget name=\"Password\" /></p>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of PasswordWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Password.0\" name=\"MainPage.Password.0\" type=\"password\" /></p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
      assert_nil(@page.context.password)

      @page.clear
      @params['MainPage.Password.0'] = 'Hello world.'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of PasswordWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Password.0\" name=\"MainPage.Password.0\" type=\"password\" /></p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('Hello world.', @page.context.password)
    end

    def test_password_widget_with_label
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @password = nil\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :password\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<password name=\"Password\">\n"
	map_output.print "<value type=\"accessor\">password</value>\n"
	map_output.print "</password>\n"
	map_output.print "\n"
	map_output.print "<label name=\"Label\">\n"
	map_output.print "<for type=\"string\">Password</for>\n"
	map_output.print "</label>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of PasswordWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<p>\n"
	xml_output.print "<pm:widget name=\"Label\">password:</pm:widget>\n"
	xml_output.print "<pm:widget name=\"Password\" />\n"
	xml_output.print "</p>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of PasswordWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p>\n"
      dst += "<label for=\"MainPage.Password.0\">password:</label>\n"
      dst += "<input id=\"MainPage.Password.0\" name=\"MainPage.Password.0\" type=\"password\" />\n"
      dst += "</p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
      assert_nil(@page.context.password)

      @page.clear
      @params['MainPage.Password.0'] = 'Hello world.'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of PasswordWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p>\n"
      dst += "<label for=\"MainPage.Password.0\">password:</label>\n"
      dst += "<input id=\"MainPage.Password.0\" name=\"MainPage.Password.0\" type=\"password\" />\n"
      dst += "</p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('Hello world.', @page.context.password)
    end

    def test_checkbox_widget_with_checked
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @checked = false\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :checked\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<checkbox name=\"Checkbox\">\n"
	map_output.print "<checked type=\"accessor\">checked</checked>\n"
	map_output.print "</checkbox>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of CheckboxWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<p><pm:widget name=\"Checkbox\" /></p>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of CheckboxWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Checkbox.0\" name=\"MainPage.Checkbox.0\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<div><input type=\"hidden\" name=\"call-range:MainPage.Checkbox\" value=\"0-0\" /></div></form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal(false, @page.context.checked)

      @page.clear
      @params['MainPage.Checkbox.0'] = ''
      @params['call-range:MainPage.Checkbox'] = '0-0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of CheckboxWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Checkbox.0\" name=\"MainPage.Checkbox.0\" type=\"checkbox\" value=\"\" checked=\"checked\" /></p>\n"
      dst += "<div><input type=\"hidden\" name=\"call-range:MainPage.Checkbox\" value=\"0-0\" /></div></form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal(true, @page.context.checked)
    end

    def test_checkbox_widget_with_selected
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @selected = nil\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :selected\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<checkbox name=\"Checkbox\">\n"
	map_output.print "<value type=\"string\">foo</value>\n"
	map_output.print "<selected type=\"accessor\">selected</selected>\n"
	map_output.print "</checkbox>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of CheckboxWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<p><pm:widget name=\"Checkbox\" /></p>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of CheckboxWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Checkbox.0\" name=\"MainPage.Checkbox.0\" type=\"checkbox\" value=\"foo\" /></p>\n"
      dst += "<div><input type=\"hidden\" name=\"call-range:MainPage.Checkbox\" value=\"0-0\" /></div></form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
      assert_nil(@page.context.selected)

      @page.clear
      @params['MainPage.Checkbox.0'] = 'foo'
      @params['call-range:MainPage.Checkbox'] = '0-0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of CheckboxWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Checkbox.0\" name=\"MainPage.Checkbox.0\" type=\"checkbox\" value=\"foo\" checked=\"checked\" /></p>\n"
      dst += "<div><input type=\"hidden\" name=\"call-range:MainPage.Checkbox\" value=\"0-0\" /></div></form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('foo', @page.context.selected)

      @page.clear
      @params['MainPage.Checkbox.0'] = 'bar'
      @params['call-range:MainPage.Checkbox'] = '0-0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of CheckboxWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Checkbox.0\" name=\"MainPage.Checkbox.0\" type=\"checkbox\" value=\"foo\" /></p>\n"
      dst += "<div><input type=\"hidden\" name=\"call-range:MainPage.Checkbox\" value=\"0-0\" /></div></form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(3, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('bar', @page.context.selected)
    end

    def test_listed_checkbox_widget_with_checked
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @call_index = nil\n"
	src_output.print "    @item = nil\n"
	src_output.print "    @checked = Hash.new\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :call_index\n"
	src_output.print "  attr_accessor :item\n"
	src_output.print "\n"
	src_output.print "  def checked\n"
	src_output.print "    if (@checked.include? @call_index) then\n"
	src_output.print "      return @checked[@call_index]\n"
	src_output.print "    elsif (@call_index == 7) then\n"
	src_output.print "      return true\n"
	src_output.print "    else\n"
	src_output.print "      return false\n"
	src_output.print "    end\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  def checked=(new_checked)\n"
	src_output.print "    @checked[@call_index] = new_checked\n"
	src_output.print "  end\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<repeat name=\"Repeat\">\n"
	map_output.print "<times type=\"number\">2</times>\n"
	map_output.print "</repeat>\n"
	map_output.print "\n"
	map_output.print "<foreach name=\"List\">\n"
	map_output.print "<list type=\"eval\">1..5</list>\n"
	map_output.print "<item type=\"accessor\">item</item>\n"
	map_output.print "</foreach>\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<checkbox name=\"Checkbox\">\n"
	map_output.print "<call-index type=\"accessor\">call_index</call-index>\n"
	map_output.print "<checked type=\"accessor\">checked</checked>\n"
	map_output.print "</checkbox>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of CheckboxWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Repeat\"\n"
	xml_output.print "><pm:widget name=\"Form\">\n"
	xml_output.print "<pm:widget name=\"List\"\n"
	xml_output.print "><p><pm:widget name=\"Checkbox\" /></p>\n"
	xml_output.print "</pm:widget\n"
	xml_output.print "></pm:widget>\n"
	xml_output.print "</pm:widget\n"
	xml_output.print "></body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of CheckboxWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Checkbox.0\" name=\"MainPage.Checkbox.0\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.1\" name=\"MainPage.Checkbox.1\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.2\" name=\"MainPage.Checkbox.2\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.3\" name=\"MainPage.Checkbox.3\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.4\" name=\"MainPage.Checkbox.4\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<div><input type=\"hidden\" name=\"call-range:MainPage.Checkbox\" value=\"0-4\" /></div></form>\n"
      dst += "<form id=\"MainPage.Form.1\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Checkbox.5\" name=\"MainPage.Checkbox.5\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.6\" name=\"MainPage.Checkbox.6\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.7\" name=\"MainPage.Checkbox.7\" type=\"checkbox\" value=\"\" checked=\"checked\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.8\" name=\"MainPage.Checkbox.8\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.9\" name=\"MainPage.Checkbox.9\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<div><input type=\"hidden\" name=\"call-range:MainPage.Checkbox\" value=\"5-9\" /></div></form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
      @page.context.call_index = 0
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 1
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 2
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 3
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 4
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 5
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 6
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 7
      assert_equal(true, @page.context.checked)
      @page.context.call_index = 8
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 9
      assert_equal(false, @page.context.checked)

      @page.clear
      @params['MainPage.Checkbox.5'] = ''
      @params['call-range:MainPage.Checkbox'] = '5-9'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of CheckboxWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Checkbox.0\" name=\"MainPage.Checkbox.0\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.1\" name=\"MainPage.Checkbox.1\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.2\" name=\"MainPage.Checkbox.2\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.3\" name=\"MainPage.Checkbox.3\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.4\" name=\"MainPage.Checkbox.4\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<div><input type=\"hidden\" name=\"call-range:MainPage.Checkbox\" value=\"0-4\" /></div></form>\n"
      dst += "<form id=\"MainPage.Form.1\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Checkbox.5\" name=\"MainPage.Checkbox.5\" type=\"checkbox\" value=\"\" checked=\"checked\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.6\" name=\"MainPage.Checkbox.6\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.7\" name=\"MainPage.Checkbox.7\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.8\" name=\"MainPage.Checkbox.8\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<p><input id=\"MainPage.Checkbox.9\" name=\"MainPage.Checkbox.9\" type=\"checkbox\" value=\"\" /></p>\n"
      dst += "<div><input type=\"hidden\" name=\"call-range:MainPage.Checkbox\" value=\"5-9\" /></div></form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      @page.context.call_index = 0
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 1
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 2
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 3
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 4
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 5
      assert_equal(true, @page.context.checked)
      @page.context.call_index = 6
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 7
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 8
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 9
      assert_equal(false, @page.context.checked)
    end

    def test_radio_button_widget_with_checked
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @foo_checked = true\n"
	src_output.print "    @bar_checked = false\n"
	src_output.print "    @baz_checked = false\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :foo_checked\n"
	src_output.print "  attr_accessor :bar_checked\n"
	src_output.print "  attr_accessor :baz_checked\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<radio name=\"Foo\">\n"
	map_output.print "<name type=\"string\">radio_button</name>\n"
	map_output.print "<checked type=\"accessor\">foo_checked</checked>\n"
	map_output.print "</radio>\n"
	map_output.print "\n"
	map_output.print "<radio name=\"Bar\">\n"
	map_output.print "<name type=\"string\">radio_button</name>\n"
	map_output.print "<checked type=\"accessor\">bar_checked</checked>\n"
	map_output.print "</radio>\n"
	map_output.print "\n"
	map_output.print "<radio name=\"Baz\">\n"
	map_output.print "<name type=\"string\">radio_button</name>\n"
	map_output.print "<checked type=\"accessor\">baz_checked</checked>\n"
	map_output.print "</radio>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of RadioButtonWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<ol>\n"
	xml_output.print "<li><pm:widget name=\"Foo\" /></li>\n"
	xml_output.print "<li><pm:widget name=\"Bar\" /></li>\n"
	xml_output.print "<li><pm:widget name=\"Baz\" /></li>\n"
	xml_output.print "</ol>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.Foo.0\" name=\"radio_button\" type=\"radio\" value=\"MainPage.Foo.0\" checked=\"checked\" /></li>\n"
      dst += "<li><input id=\"MainPage.Bar.0\" name=\"radio_button\" type=\"radio\" value=\"MainPage.Bar.0\" /></li>\n"
      dst += "<li><input id=\"MainPage.Baz.0\" name=\"radio_button\" type=\"radio\" value=\"MainPage.Baz.0\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Baz\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Foo\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Bar\" value=\"0-0\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(4, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal(true,  @page.context.foo_checked)
      assert_equal(false, @page.context.bar_checked)
      assert_equal(false, @page.context.baz_checked)

      @page.clear
      @params['radio_button'] = 'MainPage.Bar.0'
      @params['call-range:MainPage.Foo'] = '0-0'
      @params['call-range:MainPage.Bar'] = '0-0'
      @params['call-range:MainPage.Baz'] = '0-0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.Foo.0\" name=\"radio_button\" type=\"radio\" value=\"MainPage.Foo.0\" /></li>\n"
      dst += "<li><input id=\"MainPage.Bar.0\" name=\"radio_button\" type=\"radio\" value=\"MainPage.Bar.0\" checked=\"checked\" /></li>\n"
      dst += "<li><input id=\"MainPage.Baz.0\" name=\"radio_button\" type=\"radio\" value=\"MainPage.Baz.0\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Baz\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Foo\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Bar\" value=\"0-0\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(8, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal(false, @page.context.foo_checked)
      assert_equal(true,  @page.context.bar_checked)
      assert_equal(false, @page.context.baz_checked)

      @page.clear
      @params['radio_button'] = 'MainPage.Baz.0'
      @params['call-range:MainPage.Foo'] = '0-0'
      @params['call-range:MainPage.Bar'] = '0-0'
      @params['call-range:MainPage.Baz'] = '0-0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.Foo.0\" name=\"radio_button\" type=\"radio\" value=\"MainPage.Foo.0\" /></li>\n"
      dst += "<li><input id=\"MainPage.Bar.0\" name=\"radio_button\" type=\"radio\" value=\"MainPage.Bar.0\" /></li>\n"
      dst += "<li><input id=\"MainPage.Baz.0\" name=\"radio_button\" type=\"radio\" value=\"MainPage.Baz.0\" checked=\"checked\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Baz\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Foo\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Bar\" value=\"0-0\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(12, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal(false, @page.context.foo_checked)
      assert_equal(false, @page.context.bar_checked)
      assert_equal(true,  @page.context.baz_checked)
    end

    def test_radio_button_widget_with_selected
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @selected = 'foo'\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :selected\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<radio name=\"Foo\">\n"
	map_output.print "<name type=\"string\">radio_button</name>\n"
	map_output.print "<value type=\"string\">foo</value>\n"
	map_output.print "<selected type=\"accessor\">selected</selected>\n"
	map_output.print "</radio>\n"
	map_output.print "\n"
	map_output.print "<radio name=\"Bar\">\n"
	map_output.print "<name type=\"string\">radio_button</name>\n"
	map_output.print "<value type=\"string\">bar</value>\n"
	map_output.print "<selected type=\"accessor\">selected</selected>\n"
	map_output.print "</radio>\n"
	map_output.print "\n"
	map_output.print "<radio name=\"Baz\">\n"
	map_output.print "<name type=\"string\">radio_button</name>\n"
	map_output.print "<value type=\"string\">baz</value>\n"
	map_output.print "<selected type=\"accessor\">selected</selected>\n"
	map_output.print "</radio>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of RadioButtonWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<ol>\n"
	xml_output.print "<li><pm:widget name=\"Foo\" /></li>\n"
	xml_output.print "<li><pm:widget name=\"Bar\" /></li>\n"
	xml_output.print "<li><pm:widget name=\"Baz\" /></li>\n"
	xml_output.print "</ol>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.Foo.0\" name=\"radio_button\" type=\"radio\" value=\"foo\" checked=\"checked\" /></li>\n"
      dst += "<li><input id=\"MainPage.Bar.0\" name=\"radio_button\" type=\"radio\" value=\"bar\" /></li>\n"
      dst += "<li><input id=\"MainPage.Baz.0\" name=\"radio_button\" type=\"radio\" value=\"baz\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Baz\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Foo\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Bar\" value=\"0-0\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(4, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('foo', @page.context.selected)

      @page.clear
      @params['radio_button'] = 'bar'
      @params['call-range:MainPage.Foo'] = '0-0'
      @params['call-range:MainPage.Bar'] = '0-0'
      @params['call-range:MainPage.Baz'] = '0-0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.Foo.0\" name=\"radio_button\" type=\"radio\" value=\"foo\" /></li>\n"
      dst += "<li><input id=\"MainPage.Bar.0\" name=\"radio_button\" type=\"radio\" value=\"bar\" checked=\"checked\" /></li>\n"
      dst += "<li><input id=\"MainPage.Baz.0\" name=\"radio_button\" type=\"radio\" value=\"baz\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Baz\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Foo\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Bar\" value=\"0-0\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(8, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('bar', @page.context.selected)

      @page.clear
      @params['radio_button'] = 'baz'
      @params['call-range:MainPage.Foo'] = '0-0'
      @params['call-range:MainPage.Bar'] = '0-0'
      @params['call-range:MainPage.Baz'] = '0-0'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.Foo.0\" name=\"radio_button\" type=\"radio\" value=\"foo\" /></li>\n"
      dst += "<li><input id=\"MainPage.Bar.0\" name=\"radio_button\" type=\"radio\" value=\"bar\" /></li>\n"
      dst += "<li><input id=\"MainPage.Baz.0\" name=\"radio_button\" type=\"radio\" value=\"baz\" checked=\"checked\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Baz\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Foo\" value=\"0-0\" />"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.Bar\" value=\"0-0\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(12, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('baz', @page.context.selected)
    end

    def test_listed_radio_button_widget_with_checked
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @list = [ 'foo', 'bar', 'baz' ]\n"
	src_output.print "    @item = nil\n"
	src_output.print "    @call_index = nil\n"
	src_output.print "    @checked = Hash.new\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_reader :list\n"
	src_output.print "  attr_accessor :item\n"
	src_output.print "  attr_accessor :call_index\n"
	src_output.print "\n"
	src_output.print "  def checked\n"
	src_output.print "    if (@checked.include? @call_index) then\n"
	src_output.print "      return @checked[@call_index]\n"
	src_output.print "    elsif (@call_index == 0) then\n"
	src_output.print "      return true\n"
	src_output.print "    else\n"
	src_output.print "      return false\n"
	src_output.print "    end\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  def checked=(new_checked)\n"
	src_output.print "    @checked[@call_index] = new_checked\n"
	src_output.print "  end\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<foreach name=\"List\">\n"
	map_output.print "<list type=\"accessor\">list</list>\n"
	map_output.print "<item type=\"accessor\">item</item>\n"
	map_output.print "</foreach>\n"
	map_output.print "\n"
	map_output.print "<radio name=\"RadioButton\">\n"
	map_output.print "<call-index type=\"accessor\">call_index</call-index>\n"
	map_output.print "<checked type=\"accessor\">checked</checked>\n"
	map_output.print "</radio>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of RadioButtonWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<ol>\n"
	xml_output.print "<pm:widget name=\"List\"\n"
	xml_output.print "><li><pm:widget name=\"RadioButton\" /></li>\n"
	xml_output.print "</pm:widget\n"
	xml_output.print "></ol>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.RadioButton.0\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"MainPage.RadioButton.0\" checked=\"checked\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.1\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"MainPage.RadioButton.1\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.2\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"MainPage.RadioButton.2\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.RadioButton\" value=\"0-2\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      @page.context.call_index = 0
      assert_equal(true, @page.context.checked)
      @page.context.call_index = 1
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 2
      assert_equal(false, @page.context.checked)

      @page.clear
      @params['MainPage.RadioButton'] = 'MainPage.RadioButton.1'
      @params['call-range:MainPage.RadioButton'] = '0-2'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.RadioButton.0\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"MainPage.RadioButton.0\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.1\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"MainPage.RadioButton.1\" checked=\"checked\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.2\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"MainPage.RadioButton.2\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.RadioButton\" value=\"0-2\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(4, @params_call)
      assert_equal(0, @page_path_call)
      @page.context.call_index = 0
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 1
      assert_equal(true, @page.context.checked)
      @page.context.call_index = 2
      assert_equal(false, @page.context.checked)

      @page.clear
      @params['MainPage.RadioButton'] = 'MainPage.RadioButton.2'
      @params['call-range:MainPage.RadioButton'] = '0-2'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.RadioButton.0\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"MainPage.RadioButton.0\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.1\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"MainPage.RadioButton.1\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.2\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"MainPage.RadioButton.2\" checked=\"checked\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.RadioButton\" value=\"0-2\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(6, @params_call)
      assert_equal(0, @page_path_call)
      @page.context.call_index = 0
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 1
      assert_equal(false, @page.context.checked)
      @page.context.call_index = 2
      assert_equal(true, @page.context.checked)
    end

    def test_listed_radio_button_widget_with_selected
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @list = [ 'foo', 'bar', 'baz' ]\n"
	src_output.print "    @item = nil\n"
	src_output.print "    @call_index = nil\n"
	src_output.print "    @selected = 'foo'\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_reader :list\n"
	src_output.print "  attr_accessor :item\n"
	src_output.print "  attr_accessor :call_index\n"
	src_output.print "  attr_accessor :selected\n"
	src_output.print "\n"
	src_output.print "  def value\n"
	src_output.print "    @list[@call_index]\n"
        src_output.print "  end\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<foreach name=\"List\">\n"
	map_output.print "<list type=\"accessor\">list</list>\n"
	map_output.print "<item type=\"accessor\">item</item>\n"
	map_output.print "</foreach>\n"
	map_output.print "\n"
	map_output.print "<radio name=\"RadioButton\">\n"
	map_output.print "<call-index type=\"accessor\">call_index</call-index>\n"
	map_output.print "<value type=\"accessor\">value</value>\n"
	map_output.print "<selected type=\"accessor\">selected</selected>\n"
	map_output.print "</radio>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of RadioButtonWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<ol>\n"
	xml_output.print "<pm:widget name=\"List\"\n"
	xml_output.print "><li><pm:widget name=\"RadioButton\" /></li>\n"
	xml_output.print "</pm:widget\n"
	xml_output.print "></ol>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.RadioButton.0\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"foo\" checked=\"checked\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.1\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"bar\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.2\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"baz\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.RadioButton\" value=\"0-2\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('foo', @page.context.selected)

      @page.clear
      @params['MainPage.RadioButton'] = 'bar'
      @params['call-range:MainPage.RadioButton'] = '0-2'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.RadioButton.0\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"foo\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.1\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"bar\" checked=\"checked\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.2\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"baz\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.RadioButton\" value=\"0-2\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(4, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('bar', @page.context.selected)

      @page.clear
      @params['MainPage.RadioButton'] = 'baz'
      @params['call-range:MainPage.RadioButton'] = '0-2'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of RadioButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<ol>\n"
      dst += "<li><input id=\"MainPage.RadioButton.0\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"foo\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.1\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"bar\" /></li>\n"
      dst += "<li><input id=\"MainPage.RadioButton.2\" name=\"MainPage.RadioButton\" type=\"radio\" value=\"baz\" checked=\"checked\" /></li>\n"
      dst += "</ol>\n"
      dst += "<div>"
      dst += "<input type=\"hidden\" name=\"call-range:MainPage.RadioButton\" value=\"0-2\" />"
      dst += "</div>"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(6, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('baz', @page.context.selected)
    end

    def test_submit_button_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @hello = nil\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_reader :hello\n"
	src_output.print "\n"
	src_output.print "  def action\n"
	src_output.print "    @hello = 'Hello world.'\n"
	src_output.print "  end\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<submit name=\"SubmitButton\">\n"
	map_output.print "<action type=\"method\">action</action>\n"
	map_output.print "<value type=\"string\">hello</value>\n"
	map_output.print "</submit>\n"
	map_output.print "\n"
	map_output.print "<string name=\"Hello\">\n"
	map_output.print "<value type=\"accessor\">hello</value>\n"
	map_output.print "</string>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of SubmitButtonWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<p><pm:widget name=\"SubmitButton\" /></p>\n"
	xml_output.print "<p><pm:widget name=\"Hello\" /></p>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of SubmitButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.SubmitButton.0\" name=\"MainPage.SubmitButton.0\" type=\"submit\" value=\"hello\" /></p>\n"
      dst += "<p></p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal(nil, @page.context.hello)

      @page.clear
      @params['MainPage.SubmitButton.0'] = 'hello'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of SubmitButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.SubmitButton.0\" name=\"MainPage.SubmitButton.0\" type=\"submit\" value=\"hello\" /></p>\n"
      dst += "<p>Hello world.</p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('Hello world.', @page.context.hello)
    end

    def test_reset_button_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<reset name=\"Reset\">\n"
	map_output.print "<value type=\"string\">clear</value>\n"
	map_output.print "</reset>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of ResetButtonWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<p><pm:widget name=\"Reset\" /></p>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of ResetButtonWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p><input id=\"MainPage.Reset.0\" type=\"reset\" value=\"clear\" /></p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
    end

    def test_hidden_attribute_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @value = 'none'\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :value\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<hidden name=\"HiddenAttribute\">\n"
	map_output.print "<value type=\"accessor\">value</value>\n"
	map_output.print "</hidden>\n"
	map_output.print "\n"
	map_output.print "<string name=\"Value\">\n"
	map_output.print "<value type=\"accessor\">value</value>\n"
	map_output.print "</string>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of HiddenAttributeWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<div><pm:widget name=\"HiddenAttribute\" /></div>\n"
	xml_output.print "<p><pm:widget name=\"Value\" /></p>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HiddenAttributeWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<div><input id=\"MainPage.HiddenAttribute.0\" name=\"MainPage.HiddenAttribute.0\" type=\"hidden\" value=\"none\" /></div>\n"
      dst += "<p>none</p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('none', @page.context.value)

      @page.clear
      @params['MainPage.HiddenAttribute.0'] = 'Hello world.'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of HiddenAttributeWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<div><input id=\"MainPage.HiddenAttribute.0\" name=\"MainPage.HiddenAttribute.0\" type=\"hidden\" value=\"Hello world.\" /></div>\n"
      dst += "<p>Hello world.</p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('Hello world.', @page.context.value)
    end

    def test_text_area_widget
      build_page{ |src_output, map_output, xml_output|
	src_output.print "class MainPage < WPM::PageContext\n"
	src_output.print "  def init_context\n"
	src_output.print "    @value = 'none'\n"
	src_output.print "  end\n"
	src_output.print "\n"
	src_output.print "  attr_accessor :value\n"
	src_output.print "end\n"

	map_output.print "<?xml version=\"1.0\"?>\n"
	map_output.print "<map xmlns=\"http://www.freedom.ne.jp/toki/ruby/PageMaker/Map\">\n"
	map_output.print "\n"
	map_output.print "<form name=\"Form\">\n"
	map_output.print "</form>\n"
	map_output.print "\n"
	map_output.print "<textarea name=\"TextArea\">\n"
	map_output.print "<value type=\"accessor\">value</value>\n"
	map_output.print "<rows type=\"number\">25</rows>\n"
	map_output.print "<cols type=\"number\">80</cols>\n"
	map_output.print "</textarea>\n"
	map_output.print "\n"
	map_output.print "</map>\n"

	xml_output.print "<?xml version=\"1.0\"?>\n"
	xml_output.print "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:pm=\"http://www.freedom.ne.jp/toki/ruby/PageMaker\">\n"
	xml_output.print "<head><title>test of TextAreaWidget</title></head>\n"
	xml_output.print "<body>\n"
	xml_output.print "<pm:widget name=\"Form\">\n"
	xml_output.print "<p>\n"
	xml_output.print "<pm:widget name=\"TextArea\" />\n"
	xml_output.print "</p>\n"
	xml_output.print "</pm:widget>\n"
	xml_output.print "</body>\n"
	xml_output.print "</html>\n"
      }

      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of TextAreaWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p>\n"
      dst += "<textarea id=\"MainPage.TextArea.0\" name=\"MainPage.TextArea.0\" rows=\"25\" cols=\"80\">none</textarea>\n"
      dst += "</p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(1, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('none', @page.context.value)

      @page.clear
      @params['MainPage.TextArea.0'] = 'Hello world.'
      dst = "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      dst += "<head><title>test of TextAreaWidget</title></head>\n"
      dst += "<body>\n"
      dst += "<form id=\"MainPage.Form.0\" method=\"post\" enctype=\"application/x-www-form-urlencoded\">\n"
      dst += "<p>\n"
      dst += "<textarea id=\"MainPage.TextArea.0\" name=\"MainPage.TextArea.0\" rows=\"25\" cols=\"80\">Hello world.</textarea>\n"
      dst += "</p>\n"
      dst += "</form>\n"
      dst += "</body>\n"
      dst += "</html>\n"
      assert_equal(dst, run_page)
      assert_equal(2, @params_call)
      assert_equal(0, @page_path_call)
      assert_equal('Hello world.', @page.context.value)
    end
  end
end
