#!/usr/local/bin/ruby
# $Id: test_response.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

require 'rubyunit'
require 'pseudo_io'
require 'test_message'
require 'rucy/response'

module TestRucy
  class TestResponse < TestMessage
    def setup
      # for Rucy::Request class
      @host_call = 0

      @response = Rucy::Response.new
      @messg = @response
    end

    def host
      @host_call += 1
      'server:8080'
    end

    def test_status
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(200, @response.status)
      assert_equal('OK', @response.reason)

      assert_equal(500, @response.status = 500)
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(500, @response.status)
      assert_equal('Internal Server Error', @response.reason)

      assert_equal('Bug', @response.reason = 'Bug')
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(500, @response.status)
      assert_equal('Bug', @response.reason)
    end

    def test_parse_status
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(200, @response.status)
      assert_equal('OK', @response.reason)

      @response.parse_status
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(200, @response.status)
      assert_equal('OK', @response.reason)

      @response.set_header('Status', '404 Not Found')
      @response.parse_status
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(404, @response.status)
      assert_equal('Not Found', @response.reason)
      assert_nil(@response.header('Status'))

      @response.set_header('Status', '')
      assert_exception(Rucy::ParseError) { @response.parse_status }
      @response.set_header('Status', 'XXX Unknown')
      assert_exception(Rucy::ParseError) { @response.parse_status }
    end

    def test_parse_location
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(200, @response.status)
      assert_equal('OK', @response.reason)

      @response.set_header('Location', 'http://server:8080/foo/bar')
      @response.parse_status
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(302, @response.status)
      assert_equal('Found', @response.reason)
      assert_nil(@response.header('Status'))
      assert_equal('http://server:8080/foo/bar', @response.header('Location'))

      @response.set_header('Status', '307 Temporary Redirect')
      @response.parse_status
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(307, @response.status)
      assert_equal('Temporary Redirect', @response.reason)
      assert_nil(@response.header('Status'))
      assert_equal('http://server:8080/foo/bar', @response.header('Location'))
    end

    def test_parse_line
      @response.version = nil
      @response.status = 0
      assert_nil(@response.version)
      assert_equal(0, @response.status)
      assert_equal('Unknown', @response.reason)

      pio = PseudoIO.new
      pio << "HTTP/1.1 200 OK\r\n"
      pio << "Content-Type: text/html\r\n"

      @response.parse_line(pio)
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(200, @response.status)
      assert_equal('OK', @response.reason)
      assert_equal("Content-Type: text/html\r\n", pio.read)
    end

    def test_parse_line_with_LF
      @response.version = nil
      @response.status = 0
      assert_nil(@response.version)
      assert_equal(0, @response.status)
      assert_equal('Unknown', @response.reason)

      pio = PseudoIO.new
      pio << "HTTP/1.1 200 OK\n"
      pio << "Content-Type: text/html\n"

      @response.parse_line(pio)
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(200, @response.status)
      assert_equal('OK', @response.reason)
      assert_equal("Content-Type: text/html\n", pio.read)
    end

    def test_parse_line_with_preamble
      @response.version = nil
      @response.status = 0
      assert_nil(@response.version)
      assert_equal(0, @response.status)
      assert_equal('Unknown', @response.reason)

      pio = PseudoIO.new
      pio << "\r\n"
      pio << "\r\n"
      pio << "\r\n"
      pio << "HTTP/1.1 200 OK\r\n"
      pio << "Content-Type: text/html\r\n"

      @response.parse_line(pio)
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(200, @response.status)
      assert_equal('OK', @response.reason)
      assert_equal("Content-Type: text/html\r\n", pio.read)
    end

    def test_parse_line_without_status
      pio = PseudoIO.new
      pio = "HTTP/1.1 \r\n"

      assert_exception(Rucy::ParseError) {
	@response.parse_line(pio)
      }
    end

    def test_parse_line_with_invalid_version
      pio = PseudoIO.new
      pio << "HTTP/6.6.6 200 OK\r\n"
      pio << "Content-Type: text/html\r\n"

      assert_exception(Rucy::ParseError) {
	@response.parse_line(pio)
      }
    end

    def test_parse_line_with_invalid_version
      pio = PseudoIO.new
      pio << "HTTP/1.1 XXX Unknown\r\n"
      pio << "Content-Type: text/html\r\n"

      assert_exception(Rucy::ParseError) {
	@response.parse_line(pio)
      }
    end

    def test_parse
      @response.version = nil
      @response.status = 0
      assert_nil(@response.version)
      assert_equal(0, @response.status)
      assert_equal('Unknown', @response.reason)

      pio = PseudoIO.new
      pio << "HTTP/1.1 200 OK\r\n"
      pio << "Content-Type: text/html\r\n"
      pio << "\r\n"
      pio << "<html>Hello world.</html>\n"

      @response.parse(pio)
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(200, @response.status)
      assert_equal('OK', @response.reason)
      assert_equal('text/html', @response.header('Content-Type'))
      assert_equal("<html>Hello world.</html>\n", pio.read)
    end

    def test_absolute_location
      @response.set_header('Location', 'http://www.foo.bar/hello/world')
      @response.absolute_location(self)
      assert_equal(0, @host_call)
      assert_equal('http://www.foo.bar/hello/world', @response.header('Location'))

      @response.set_header('Location', '/hello/world')
      @response.absolute_location(self)
      assert_equal(1, @host_call)
      assert_equal('http://server:8080/hello/world', @response.header('Location'))
    end

    def test_conn_closed
      # HTTP/0.9
      @response.version = 'HTTP/0.9'
      @response.status = 200
      assert(@response.conn_closed?)

      # HTTP/1.0
      @response.version = 'HTTP/1.0'
      assert(! @response.conn_closed?)
      @response.set_header('Connection', 'close')
      assert(@response.conn_closed?)
      @response.delete_header('Connection')

      # HTTP/1.1
      @response.version = 'HTTP/1.1'
      assert(! @response.conn_closed?)
      @response.set_header('Connection', 'close')
      assert(@response.conn_closed?)
    end

    def test_conn_close
      @response.set_header('Connection', 'Keep-Alive')
      @response.set_header('Keep-Alive', 'timeout=8')
      @response.set_header('Content-Length', '128')
      assert(! @response.conn_closed?)
      @response.conn_close
      assert(@response.conn_closed?)
      assert(@response.headers('Connection').find{ |v| v =~ /close/i })
      assert(! @response.headers('Connection').find{ |v| v =~ /Keep-Alive/i })
      assert(! (@response.has_header? 'Keep-Alive'))
    end

    def test_conn_keep_alive_HTTP_1_1
      @response.version = 'HTTP/1.1'
      @response.set_header('Connection', 'close')
      assert(@response.conn_closed?)
      @response.conn_keep_alive('HTTP/1.1')
      assert(! @response.conn_closed?)
      assert(! @response.headers('Connection').find{ |v| v =~ /close/i })
      assert(! @response.headers('Connection').find{ |v| v =~ /Keep-Alive/i })
    end

    def test_conn_keep_alive_HTTP_1_0
      @response.version = 'HTTP/1.0'
      @response.set_header('Connection', 'close')
      assert(@response.conn_closed?)

      # failed to keep alive
      @response.conn_keep_alive('HTTP/1.0')
      assert(@response.conn_closed?)
      assert(@response.headers('Connection').find{ |v| v =~ /close/i })
      assert(! @response.headers('Connection').find{ |v| v =~ /Keep-Alive/i })

      # succeed to keep alive
      @response.set_header('Content-Length', '128')
      @response.conn_keep_alive('HTTP/1.0')
      assert(! @response.conn_closed?)
      assert(! @response.headers('Connection').find{ |v| v =~ /close/i })
      assert(@response.headers('Connection').find{ |v| v =~ /Keep-Alive/i })
    end

    def test_local_path
      assert_nil(@response.local_path)
      assert_equal('/home/foo/public_html/index.html', @response.local_path = '/home/foo/public_html/index.html')
      assert_equal('/home/foo/public_html/index.html', @response.local_path)
    end
  end
end
