#!/usr/local/bin/ruby
# $Id: test_redirect.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

require 'rubyunit'
require 'pseudo_req_res'
require 'rucy/redirect'

module TestRucy
  class TestRedirectDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse

    def test_URL
      redirect = Rucy::RedirectDocument.new('http://www.foo.bar/index.html')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_with_status
      redirect = Rucy::RedirectDocument.new('http://www.foo.bar/index.html', 307)
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(307, @messg_head.status)
      assert_equal('Temporary Redirect', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_and_HEAD
      redirect = Rucy::RedirectDocument.new('http://www.foo.bar/index.html')
      @request.method = 'HEAD'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(@messg_body.empty?)
    end

    def test_URL_and_method_not_allowed
      redirect = Rucy::RedirectDocument.new('http://www.foo.bar/index.html')
      @request.method = 'DELETE'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	redirect.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(405, $!.status) #  Method Not Allowed
	assert_equal('GET, HEAD', $!.header('Allow'))
	done = true
      end
      assert(done)
      assert(@messg_body.empty?)
    end

    def test_rel_URL_abs_path
      redirect = Rucy::RedirectDocument.new('/foo/bar/index.html')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://server:8080/foo/bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end
  end

  class TestRegexpRedirectDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse

    def test_URL
      redirect = Rucy::RegexpRedirectDocument.new(%r"^.*/bar*?$", 'http://www.foo.bar/index.html')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_with_replacing
      redirect = Rucy::RegexpRedirectDocument.new(%r"^.*/bar/(.*)$", 'http://www.foo.bar/\1/index.html')
      @request.method = 'GET'
      @request.path = '/foo/bar/baz'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/baz/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_with_query
      redirect = Rucy::RegexpRedirectDocument.new(%r"^.*/bar/(.*)\?(.*)?$", 'http://www.foo.bar/\1/index.html?\2')
      @request.method = 'GET'
      @request.path = '/foo/bar/baz'
      @request.query = 'alpha=1&beta=2'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/baz/index.html?alpha=1&beta=2', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_with_status
      redirect = Rucy::RegexpRedirectDocument.new(%r"^.*/bar*?$", 'http://www.foo.bar/index.html', 307)
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(307, @messg_head.status)
      assert_equal('Temporary Redirect', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end

    def test_URL_and_HEAD
      redirect = Rucy::RegexpRedirectDocument.new(%r"^.*/bar.*?$", 'http://www.foo.bar/index.html')
      @request.method = 'HEAD'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://www.foo.bar/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(@messg_body.empty?)
    end

    def test_URL_and_not_found
      redirect = Rucy::RegexpRedirectDocument.new(%r"^.*/bar*?$", 'http://www.foo.bar/index.html')
      @request.method = 'GET'
      @request.path = '/baz'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	redirect.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	done = true
      end
      assert(done)
      assert(@messg_body.empty?)
    end

    def test_URL_and_method_not_allowed
      redirect = Rucy::RegexpRedirectDocument.new(%r"^.*/bar*?$", 'http://www.foo.bar/index.html')
      @request.method = 'DELETE'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	redirect.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(405, $!.status) # Method Not Allowed
	assert_equal('GET, HEAD', $!.header('Allow'))
	done = true
      end
      assert(done)
      assert(@messg_body.empty?)
    end

    def test_rel_URL_abs_path
      redirect = Rucy::RegexpRedirectDocument.new(%r"^.*/bar*?$", '/index.html')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      redirect.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('http://server:8080/index.html', @messg_head.header('Location'))
      assert((@messg_head.has_header? 'Content-Type'))
      assert(! @messg_body.empty?)
    end
  end
end
