#!/usr/local/bin/ruby
# $Id: test_properties.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

require 'rubyunit'
require 'rucy/properties'

module TestRucy
  class TestModifiedCount < RUNIT::TestCase
    def setup
      @mc = Object.new
      @mc.extend(Rucy::ModifiedCount)
    end

    def test_modified_count
      assert_nil(@mc.modified_count)
      assert_equal(0, @mc.modified_count = 0)
      assert_equal(0, @mc.modified_count)
    end
  end

  class TestProperties < RUNIT::TestCase
    def setup
      Dir.mkdir('settings')
      @properties = load_properties
    end

    def load_properties
      Rucy::Properties.new('settings/properties.db')
    end

    def teardown
      File.delete('settings/properties.db~') if (File.exist? 'settings/properties.db~')
      File.delete('settings/properties.db') if (File.exist? 'settings/properties.db')
      Dir.delete('settings')
    end

    def test_list
      list = @properties.list('test')
      assert_equal([], list)
      assert_equal(0, list.modified_count)
    end

    def test_set_list
      list = @properties.list('test')
      list.push('HALO')
      @properties.set_list('test', list)
      assert_equal([ 'HALO' ], @properties.list('test'))
      assert_equal([ 'HALO' ], load_properties.list('test'))
    end

    def test_list_modified_check
      foo_list = @properties.list('foo')
      bar_list = @properties.list('bar')
      @properties.set_list('foo', foo_list)
      assert_exception(RuntimeError) {
	@properties.set_list('bar', bar_list)
      }
      bar_list.modified_count = foo_list.modified_count
      @properties.set_list('bar', bar_list)
    end

    def test_map
      map = @properties.map('test')
      assert_equal({}, map)
      assert_equal(0, map.modified_count)
    end

    def test_set_map
      map = @properties.map('test')
      map['message'] = 'Hello world.'
      @properties.set_map('test', map)
      assert_equal({ 'message' => 'Hello world.' }, @properties.map('test'))
      assert_equal({ 'message' => 'Hello world.' }, load_properties.map('test'))
    end

    def test_map_modified_check
      foo_map = @properties.map('foo')
      bar_map = @properties.map('bar')
      @properties.set_map('foo', foo_map)
      assert_exception(RuntimeError) {
	@properties.set_map('bar', bar_map)
      }
      bar_map.modified_count = foo_map.modified_count
      @properties.set_map('bar', bar_map)
    end

    def test_params
      params = @properties.params('foo', 'bar')
      assert((params.has_key? 'foo'))
      assert((params.has_key? 'bar'))
      assert_nil(params['foo'])
      assert_nil(params['bar'])
      assert_equal(0, params.modified_count)
    end

    def test_set_params
      params = @properties.params('foo', 'bar')
      params['foo'] = 'Alice'
      params['bar'] = 'Bob'
      @properties.set_params(params)
      assert_equal({ 'foo' => 'Alice', 'bar' => 'Bob' }, @properties.params('foo', 'bar'))
      assert_equal({ 'foo' => 'Alice', 'bar' => 'Bob' }, load_properties.params('foo', 'bar'))
    end

    def test_params_modified_check
      params1 = @properties.params('foo', 'bar')
      params2 = @properties.params('foo', 'bar')
      @properties.set_params(params1)
      assert_exception(RuntimeError) {
	@properties.set_params(params2)
      }
      params2.modified_count = params1.modified_count
      @properties.set_params(params2)
    end

    def test_list_type_check
      params = @properties.params('string')
      params['string'] = 'Hello world.'
      @properties.set_params(params)
      assert_exception(RuntimeError) {
	@properties.list('string')
      }
    end

    def test_map_type_check
      params = @properties.params('string')
      params['string'] = 'Hello world.'
      @properties.set_params(params)
      assert_exception(RuntimeError) {
	@properties.map('string')
      }
    end
  end
end
