#!/usr/local/bin/ruby
# $Id: test_multi_process.rb,v 1.2 2004/04/22 08:42:09 toki Exp $

require 'rubyunit'
require 'socket'
require 'sockutils'
require 'pseudo_io'
require 'rucy/logger'
require 'rucy/restart'
require 'rucy/document'
require 'rucy/messenger'
require 'rucy/server'

module TestRucy
  class TestSocketPair < RUNIT::TestCase
    def setup
      @s1, @s2 = UNIXSocket.socketpair
      @s1.sync = true
      @s2.sync = true
    end

    def teardown
      @s1.close unless @s1.closed?
      @s2.close unless @s2.closed?
    end

    def test_pipe
      if (pid = fork) then
	@s2.close
	@s1.write('HALO S2')
	assert_equal('HALO S1', @s1.read(7))
      else
	@s1.close
	assert_equal('HALO S2', @s2.read(7))
	@s2.write('HALO S1')
	exit!(0)
      end
      assert_equal(0, Process.waitpid2(pid)[1].exitstatus)
    end

    def test_io_passing
      if (pid = fork) then
	@s2.close
	io_passing_parent
      else
	@s1.close
	io_passing_child
	exit!(0)
      end
      assert_equal(0, Process.waitpid2(pid)[1].exitstatus)
    end

    def io_passing_parent
      pin, pout = IO.pipe
      @s1.write('send_io')
      @s1.send_io(pout)
      pout.close
      assert_equal('recv_io', @s1.read(7))
      assert_equal("Hello world.\n", pin.read)
      pin.close

      nil
    end
    private :io_passing_parent

    def io_passing_child
      assert_equal('send_io', @s2.read(7))
      pout = @s2.recv_io(IO, 'w')
      @s2.write('recv_io')
      pout.print "Hello world.\n"
      pout.close

      nil
    end
    private :io_passing_child
  end

  class TestMultiProcessRestartSignal < RUNIT::TestCase
    def setup
      @restart_call = 0
      @close_call = 0
      @restart_signal = Rucy::MultiProcessRestartSignal.new
      @restart_signal.server = self
    end

    def restart
      @restart_call += 1
      nil
    end

    def close
      @close_call += 1
      nil
    end

    def test_notify_restart
      pid_list = Array.new
      begin
	10.times do
	  pid_list.push fork{
	    @restart_signal.notify_restart # call on child process
	    exit!
	  }
	end
	@restart_signal.wait	# call on parent process
	assert_equal(1, @restart_call)
      ensure
	for pid in pid_list
	  Process.waitpid(pid, 0)
	end
      end
      assert_equal(1, @restart_call)
    end

    def test_notify_close
      pid_list = Array.new
      begin
	10.times do
	  pid_list.push fork{
	    @restart_signal.notify_close # call on child process
	    exit!
	  }
	end
	@restart_signal.wait	# call on parent process
	assert_equal(1, @close_call)
      ensure
	for pid in pid_list
	  Process.waitpid(pid, 0)
	end
      end
      assert_equal(1, @close_call)
    end

    def test_cancel
      pin, pout = IO.pipe
      pid = fork{
	pout.close
	pin.read(1)
	pin.close
	@restart_signal.notify_restart # call on child process
	exit!
      }
      pin.close

      begin
	th = Thread.new{
	  @restart_signal.wait	# call on parent process
	}
	@restart_signal.cancel	# call on parent process
	pout.write("\0")
	pout.close
      ensure
	th.join
      end
      Process.waitpid(pid, 0)
      assert_equal(0, @restart_call)
      assert_equal(0, @close_call)
    end
  end

  class TestMultiProcessMessenger < RUNIT::TestCase
    include SocketUtils

    def messenger_threads
      @messenger_threads_call += 1
      4
    end

    def messenger_queue_length
      @messenger_queue_length_call += 1
      2
    end

    def subprocess_user
      @subprocess_user_call += 1
      'nobody'
    end

    def subprocess_group
      @subprocess_group_call += 1
      'nogroup'
    end

    def setup
      @document = Rucy::PageDocument.new("<html>Hello world.</html>\n")
      @log_pio = PseudoIO.new
      @logger = Rucy::Logger.new(@log_pio)
      @messenger = Rucy::MultiProcessMessenger.new(@document, @logger)

      @messenger_threads_call = 0
      @messenger_queue_length_call = 0
      @subprocess_user_call = 0
      @subprocess_group_call = 0
      assert_nil(@messenger.messenger_threads)
      assert_nil(@messenger.messenger_queue_length)
      assert_nil(@messenger.subprocess_user)
      assert_nil(@messenger.subprocess_group)
      @messenger.open(self)
      assert_equal(1, @messenger_threads_call)
      assert_equal(1, @messenger_queue_length_call)
      assert_equal(1, @subprocess_user_call)
      assert_equal(1, @subprocess_group_call)
      assert_equal(4, @messenger.messenger_threads)
      assert_equal(2, @messenger.messenger_queue_length)
      assert_equal('nobody', @messenger.subprocess_user)
      assert_equal('nogroup', @messenger.subprocess_group)

      @queue = Rucy::SocketQueue.new(2)
      @thread = Thread.new{
	@messenger.accept(@queue)
      }
    end

    def teardown
      @queue.push_close
      @thread.join
    end

    def test_GET
      nconns = 16; nreqs = 4
      nconns.times do
	cli_sock, svr_sock = tcp_socketpair
	begin
	  @queue.push(svr_sock)
	  nreqs.times do |i|
	    cli_sock << "GET / HTTP/1.1\r\n"
	    cli_sock << "Host: localhost:8080\r\n"
	    cli_sock << "Connection: close\r\n" if (i + 1 == nreqs)
	    cli_sock << "\r\n"
	    cli_sock.flush

	    assert_equal("HTTP/1.1 200 OK\r\n", cli_sock.gets("\n"))
	    assert_equal("Connection: close\r\n", cli_sock.gets("\n")) if (i + 1 == nreqs)
	    assert_equal("Content-Length: 26\r\n", cli_sock.gets("\n"))
	    assert_equal("Content-Type: text/html\r\n", cli_sock.gets("\n"))
	    assert_match(cli_sock.gets("\n"), Regexp.compile(Regexp.quote("Date: ") + "(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \\d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \\d{4} \\d{2}:\\d{2}:\\d{2} GMT" + Regexp.quote("\r\n")))
	    assert_equal("Server: #{Rucy::SERVER_TOKEN_LIST}\r\n", cli_sock.gets("\n"))
	    assert_equal("\r\n", cli_sock.gets("\n"))
	    assert_equal("<html>Hello world.</html>\n", cli_sock.read(26))
	  end
	  assert_equal('', cli_sock.read)
	  assert(cli_sock.eof?)
	ensure
	  cli_sock.close
	end
      end
    end
  end

  class TestMultiProcessServer < RUNIT::TestCase
    include SocketUtils

    SERVER_PORT = 9876

    def setup
      #@log_io = File.open('test_multi_process_server.log', 'a')
      #@log_io.sync = true
      #@log_io.puts '<start>'
      @restart_called = 0
      @server_open_call = 0
      @server_close_call = 0
      @document_open_call = 0
      @document_close_call = 0
      @server = Rucy::Server.new
      @server.open_hook{ |s|
	@server_open_call += 1
	s.port = SERVER_PORT
	s.mount(Rucy::PageDocument.new("<html>Hello world.</html>\n"), '/')
	s.mount(self, '/test_document_open_close')
	s.messenger_factory = Rucy::MultiProcessMessenger
	#logger = Rucy::Logger.new(@log_io)
	#logger.log_debug = true
	#s.add_logger(logger)
      }
      @server.close_hook{ |s|
	@server_close_call += 1
      }
      @svr_pid = fork{
	trap(:USR1) { @server.close }
	begin
	  @server.accept
	  assert_equal(1 + @restart_called, @server_open_call)
	  assert_equal(1 + @restart_called, @server_close_call)
	  assert_equal(1 + @restart_called, @document_open_call)
	  assert_equal(1 + @restart_called, @document_close_call)
	rescue
	  STDERR.puts "server error: #{$!}"
	  for frame in $!.backtrace
	    puts "server error: #{frame}"
	  end
	end
	exit!
      }
      sleep(0.1)
    end

    def open
      @document_open_call += 1
      nil
    end

    def close
      @document_close_call += 1
      nil
    end

    def each(traversed={})
      yield(self)
      nil
    end

    def teardown
      Process.kill(:USR1, @svr_pid)
      Process.waitpid(@svr_pid)
      #@log_io.puts '<end>'
      #@log_io.close
    end

    def test_restart
      @server.restart
      @restart_called += 1
      sleep(1)
    end

    def test_GET
      socket = TCPSocket.new('localhost', SERVER_PORT)
      setsockopt(socket)
      begin
	socket.print "GET / HTTP/1.1\r\n"
	socket.print "Host: localhost:#{SERVER_PORT}\r\n"
	socket.print "\r\n"
	socket.flush

	response = Rucy::Response.new
	response.parse(socket)
	assert_equal('HTTP/1.1 200 OK', response.line)
	assert(! response.headers('Connection').find{ |v| v =~ /close/i })
	assert_equal('26', response.header('Content-Length'))
	assert_equal('text/html', response.header('Content-Type'))
	assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	assert_equal("<html>Hello world.</html>\n", socket.read(26))

	socket.print "GET / HTTP/1.1\r\n"
	socket.print "Host: localhost:#{SERVER_PORT}\r\n"
	socket.print "Connection: close\r\n"
	socket.print "\r\n"
	socket.flush

	response = Rucy::Response.new
	response.parse(socket)
	assert_equal('HTTP/1.1 200 OK', response.line)
	assert_equal('close', response.header('Connection'))
	assert_equal('26', response.header('Content-Length'))
	assert_equal('text/html', response.header('Content-Type'))
	assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	assert_equal("<html>Hello world.</html>\n", socket.read(26))
      ensure
	socket.close
      end
    end

    def make_access_thread(nreqs, wait)
      Thread.new{
	socket = TCPSocket.new('localhost', SERVER_PORT)
	setsockopt(socket)
	wait.call
	begin
	  (nreqs - 1).times do
	    socket.print "GET / HTTP/1.1\r\n"
	    socket.print "Host: localhost:#{SERVER_PORT}\r\n"
	    socket.print "\r\n"

	    response = Rucy::Response.new
	    response.parse(socket)
	    assert_equal('HTTP/1.1 200 OK', response.line)
	    assert(! response.headers('Connection').find{ |v| v =~ /close/i })
	    assert_equal('26', response.header('Content-Length'))
	    assert_equal('text/html', response.header('Content-Type'))
	    assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	    assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	    assert_equal("<html>Hello world.</html>\n", socket.read(26))
	  end

	  socket.print "GET / HTTP/1.1\r\n"
	  socket.print "Host: localhost:#{SERVER_PORT}\r\n"
	  socket.print "Connection: close\r\n"
	  socket.print "\r\n"

	  response = Rucy::Response.new
	  response.parse(socket)
	  assert_equal('HTTP/1.1 200 OK', response.line)
	  assert_equal('close', response.header('Connection'))
	  assert_equal('26', response.header('Content-Length'))
	  assert_equal('text/html', response.header('Content-Type'))
	  assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	  assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	  assert_equal("<html>Hello world.</html>\n", socket.read(26))
	ensure
	  socket.close
	end
      }
    end
    private :make_access_thread

    def test_multi_access
      start = false
      lock = Mutex.new
      cond = ConditionVariable.new
      wait = proc{
	lock.synchronize{
	  until (start)
	    cond.wait(lock)
	  end
	}
      }

      th_grp = ThreadGroup.new
      nconns = 64; nreqs = 32
      nconns.times do |i|
	th_grp.add(make_access_thread(nreqs, wait))
      end

      lock.synchronize{
	start = true
	cond.broadcast
      }

      for th in th_grp.list
	th.join
      end
    end
  end

  class TestMultiProcessServerTimeout < RUNIT::TestCase
    include SocketUtils

    SERVER_PORT = 9876

    def setup
      #@log_io = File.open('test_multi_process_server.log', 'a')
      #@log_io.sync = true
      #@log_io.puts '<start>'
      @server_open_call = 0
      @server_close_call = 0
      @document_open_call = 0
      @document_close_call = 0
      @server = Rucy::Server.new
      @server.open_hook{ |s|
	@server_open_call += 1
	s.port = SERVER_PORT
	s.mount(Rucy::PageDocument.new("<html>Hello world.</html>\n"), '/')
	s.mount(self, '/test_document_open_close')
	s.messenger_factory = Rucy::MultiProcessMessenger
	s.timeout = 0.01	# <- short timeout for test
	#logger = Rucy::Logger.new(@log_io)
	#logger.log_debug = true
	#s.add_logger(logger)
      }
      @server.close_hook{ |s|
	@server_close_call += 1
      }
      @svr_pid = fork{
	trap(:USR1) { @server.close }
	begin
	  @server.accept
	  assert_equal(1, @server_open_call)
	  assert_equal(1, @server_close_call)
	  assert_equal(1, @document_open_call)
	  assert_equal(1, @document_close_call)
	rescue
	  STDERR.puts "server error: #{$!}"
	  for frame in $!.backtrace
	    puts "server error: #{frame}"
	  end
	end
	exit!
      }
      sleep(0.1)
    end

    def open
      @document_open_call += 1
      nil
    end

    def close
      @document_close_call += 1
      nil
    end

    def each(traversed={})
      yield(self)
      nil
    end

    def teardown
      Process.kill(:USR1, @svr_pid)
      Process.waitpid(@svr_pid)
      #@log_io.puts '<end>'
      #@log_io.close
    end

    def test_timeout
      socket = TCPSocket.new('localhost', SERVER_PORT)
      setsockopt(socket)
      begin
	sleep(0.1)		# wait for timeout
	assert_nil(socket.gets)
      ensure
	socket.close
      end
    end
  end
end
