#!/usr/local/bin/ruby
# $Id: test_messenger.rb,v 1.2 2004/04/22 08:41:43 toki Exp $

require 'rubyunit'
require 'socket'
require 'sockutils'
require 'pseudo_io'
require 'rucy/logger'
require 'rucy/response'
require 'rucy/document'
require 'rucy/messenger'

Thread.abort_on_exception = true

module TestRucy
  class TestMultiThreadMessenger < RUNIT::TestCase
    include SocketUtils

    def setup
      @document = Rucy::PageDocument.new("<html>Hello world.</html>\n")
      @log_pio = PseudoIO.new
      @logger = Rucy::Logger.new(@log_pio)
      @messenger = Rucy::MultiThreadMessenger.new(@document, @logger)
      @messenger.open(self)
      @queue = Rucy::SocketQueue.new
      @thread = Thread.new{
	@messenger.accept(@queue)
      }
    end

    def teardown
      @queue.push_close
      @thread.join
    end

    def test_GET
      cli_sock, svr_sock = unix_socketpair
      begin
	@queue.push(svr_sock)

	cli_sock << "GET / HTTP/1.1\r\n"
	cli_sock << "Host: localhost:8080\r\n"
	cli_sock << "Connection: close\r\n"
	cli_sock << "\r\n"
	cli_sock.flush

	response = Rucy::Response.new
	response.parse(cli_sock)
	assert_equal('HTTP/1.1', response.version)
	assert_equal(200, response.status)
	assert_equal('OK', response.reason)
	assert_equal('close', response.header('Connection'))
	assert_equal('26', response.header('Content-Length'))
	assert_equal('text/html', response.header('Content-Type'))
	assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	assert_equal("<html>Hello world.</html>\n", cli_sock.read)
	assert(cli_sock.eof?)
      ensure
	cli_sock.close
      end
    end

    def test_HEAD
      cli_sock, svr_sock = unix_socketpair
      begin
	@queue.push(svr_sock)

	cli_sock << "HEAD / HTTP/1.1\r\n"
	cli_sock << "Host: localhost:8080\r\n"
	cli_sock << "Connection: close\r\n"
	cli_sock << "\r\n"
	cli_sock.flush

	response = Rucy::Response.new
	response.parse(cli_sock)
	assert_equal('HTTP/1.1', response.version)
	assert_equal(200, response.status)
	assert_equal('OK', response.reason)
	assert_equal('close', response.header('Connection'))
	assert_equal('26', response.header('Content-Length'))
	assert_equal('text/html', response.header('Content-Type'))
	assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	assert(cli_sock.eof?)
      ensure
	cli_sock.close
      end
    end

    def test_parse_error
      cli_sock, svr_sock = unix_socketpair
      begin
	@queue.push(svr_sock)

	cli_sock << "GET / HTTP/1.1\r\n"
	cli_sock << "Connection: close\r\n"
	cli_sock << "\r\n"
	cli_sock.flush

	response = Rucy::Response.new
	response.parse(cli_sock)
	assert_equal('HTTP/1.1', response.version)
	assert_equal(400, response.status)
	assert_equal('Bad Request', response.reason)
	assert_equal('close', response.header('Connection'))
	assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	assert(! cli_sock.read.empty?)
	assert(cli_sock.eof?)
      ensure
	cli_sock.close
      end
    end

    def test_http_error
      cli_sock, svr_sock = unix_socketpair
      begin
	@queue.push(svr_sock)

	cli_sock << "DELETE / HTTP/1.1\r\n"
	cli_sock << "Host: localhost:8080\r\n"
	cli_sock << "Connection: close\r\n"
	cli_sock << "\r\n"
	cli_sock.flush

	response = Rucy::Response.new
	response.parse(cli_sock)
	assert_equal('HTTP/1.1', response.version)
	assert_equal(405, response.status)
	assert_equal('Method Not Allowed', response.reason)
	assert_equal('GET, HEAD', response.header('Allow'))
	assert_equal('close', response.header('Connection'))
	assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	assert(! cli_sock.read.empty?)
	assert(cli_sock.eof?)
      ensure
	cli_sock.close
      end
    end

    def test_multi_access
      conns = 16
      reqs = 32

      th_grp = ThreadGroup.new
      conns.times do
	th_grp.add Thread.new{
	  cli_sock, svr_sock = unix_socketpair
	  begin
	    @queue.push(svr_sock)

	    (reqs - 1).times do
	      cli_sock.print "GET / HTTP/1.1\r\n"
	      cli_sock.print "Host: localhost\r\n"
	      cli_sock.print "\r\n"

	      response = Rucy::Response.new
	      response.parse(cli_sock)
	      assert_equal('HTTP/1.1 200 OK', response.line)
	      assert(! response.headers('Connection').find{ |v| v =~ /close/i })
	      assert_equal('26', response.header('Content-Length'))
	      assert_equal('text/html', response.header('Content-Type'))
	      assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	      assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	      assert_equal("<html>Hello world.</html>\n", cli_sock.read(26))
	    end

	    cli_sock.print "GET / HTTP/1.1\r\n"
	    cli_sock.print "Host: localhost\r\n"
	    cli_sock.print "Connection: close\r\n"
	    cli_sock.print "\r\n"

	    response = Rucy::Response.new
	    response.parse(cli_sock)
	    assert_equal('HTTP/1.1 200 OK', response.line)
	    assert_equal('close', response.header('Connection'))
	    assert_equal('26', response.header('Content-Length'))
	    assert_equal('text/html', response.header('Content-Type'))
	    assert_match(response.header('Date'), /^(Mon|Tue|Wed|Thu|Fri|Sat|Sun), \d{2} (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) \d{4} \d{2}:\d{2}:\d{2} GMT$/)
	    assert_equal(Rucy::SERVER_TOKEN_LIST, response.header('Server'))
	    assert_equal("<html>Hello world.</html>\n", cli_sock.read(26))
	  ensure
	    cli_sock.close
	  end
	}
      end

      for th in th_grp.list
	th.join
      end
    end
  end
end
