#!/usr/local/bin/ruby
# $Id: test_local.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

require 'rubyunit'
require 'time'
require 'pseudo_req_res'
require 'rucy/request'
require 'rucy/response'
require 'rucy/local'

module TestRucy
  class TestContentTypeResolver < RUNIT::TestCase
    def setup
      @resolver = Rucy::ContentTypeResolver.new
    end

    def test_content_type
      assert_equal('application/octet-stream', @resolver.content_type('/binary'))
      assert_equal('application/octet-stream', @resolver.content_type('/index.html'))
      assert_equal('application/octet-stream', @resolver.content_type('/index_ja.html'))
      @resolver.set_content_type('.html', 'text/html')
      assert_equal('application/octet-stream', @resolver.content_type('/binary'))
      assert_equal('text/html', @resolver.content_type('/index.html'))
      assert_equal('text/html', @resolver.content_type('/index_ja.html'))
      @resolver.set_content_type('ja.html', 'text/html; charset=euc-jp')
      assert_equal('application/octet-stream', @resolver.content_type('/binary'))
      assert_equal('text/html', @resolver.content_type('/index.html'))
      assert_equal('text/html', @resolver.content_type('/index_a.html'))
      assert_equal('text/html; charset=euc-jp', @resolver.content_type('/index_ja.html'))
    end

    def test_default_content_type
      assert_equal('application/octet-stream', @resolver.content_type('/detarame'))
      @resolver.set_default_content_type('text/plain')
      assert_equal('text/plain', @resolver.content_type('/detarame'))
    end

    def test_dup
      assert_equal('application/octet-stream', @resolver.content_type('.html'))
      resolver2 = @resolver.dup
      resolver2.set_content_type('.html', 'text/html')
      assert_equal('text/html', resolver2.content_type('.html'))
      assert_equal('application/octet-stream', @resolver.content_type('.html'))
    end
  end

  class TestLocalDirectoryDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse

    def setup
      super
      @dir_doc = Rucy::LocalDirectoryDocument.new('test')
      Dir.mkdir('test')
      Dir.mkdir('test/subdir')
      File.open('test/foo', 'w').close
    end

    def teardown
      Dir.delete('test/subdir')
      File.delete('test/foo')
      Dir.delete('test')
    end

    def test_GET
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @dir_doc.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_match(@messg_body, /foo/)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalDirectoryDocument/)
    end

    def test_HEAD
      @request.method = 'HEAD'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @dir_doc.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalDirectoryDocument/)
    end

    def test_not_allowed
      @request.method = 'POST'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@dir_doc.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(405, $!.status)
	assert_equal('GET, HEAD', $!.header('Allow'))
	done = true
      end
      assert(done)
      assert_nil(@messg_head)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalDirectoryDocument/)
    end

    def test_not_found
      @request.method = 'GET'
      @request.path = '/nothing/'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@dir_doc.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status)
	done = true
      end
      assert(done)
      assert_nil(@messg_head)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalDirectoryDocument/)
    end

    def test_subdir
      @request.method = 'GET'
      @request.path = '/subdir/'
      @request.version = 'HTTP/1.1'
      @dir_doc.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalDirectoryDocument/)
    end

    def test_file
      @request.method = 'GET'
      @request.path = '/foo/'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@dir_doc.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status)
	done = true
      end
      assert(done)
      assert_nil(@messg_head)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalDirectoryDocument/)
    end
  end

  class TestLocalFileDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse

    HTML = "<html>Hello world.</html>\n"
    DATA = '0123456789'

    def setup
      super
      Dir.mkdir('test')
      Dir.mkdir('test/subdir')
      File.open('test/index.html', 'w') { |output|
	output.binmode
	output.print HTML
      }
      File.open('test/subdir/foo', 'w') { |output|
	output.binmode
	output.print DATA
      }

      @local_doc = Rucy::LocalFileDocument.new('test')
    end

    def teardown
      File.delete('test/index.html') if (File.exist? 'test/index.html')
      File.delete('test/subdir/foo') if (File.exist? 'test/subdir/foo')
      Dir.delete('test/subdir')
      Dir.delete('test')
    end

    def test_GET_root_200_OK
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @local_doc.publish('', @request, @response, self)
      assert_equal('/', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal(HTML.length.to_s, @messg_head.header('Content-Length'))
      assert_equal(HTML, @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_HEAD_root_200_OK
      @request.method = 'HEAD'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @local_doc.publish('', @request, @response, self)
      assert_equal('/', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal(HTML.length.to_s, @messg_head.header('Content-Length'))
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_POST_root_405_Method_Not_Allowed
      @request.method = 'POST'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@local_doc.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(405, $!.status)
	assert_equal('GET, HEAD', $!.header('Allow'))
	done = true
      end
      assert(done)
      assert_nil(@messg_head)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_html_200_OK
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal(HTML.length.to_s, @messg_head.header('Content-Length'))
      assert_equal(HTML, @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_HEAD_html_200_OK
      @request.method = 'HEAD'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal(HTML.length.to_s, @messg_head.header('Content-Length'))
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_POST_html_405_Method_Not_Allowed
      @request.method = 'POST'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@local_doc.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(405, $!.status)
	assert_equal('GET, HEAD', $!.header('Allow'))
	done = true
      end
      assert(done)
      assert_nil(@messg_head)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_nothing_404_Not_Found
      @request.method = 'GET'
      @request.path = '/nothing'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@local_doc.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # 404 Not Found
	done = true
      end
      assert(done)
      assert_nil(@messg_head)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_subdir_301_Moved_Permanently
      @request.method = 'GET'
      @request.path = '/subdir'
      @request.version = 'HTTP/1.1'
      @local_doc.publish('', @request, @response, self)
      assert_equal('/subdir', @messg_head.doc_path)
      assert_match(@response.local_path, %r"test/subdir$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(301, @messg_head.status)
      assert_equal('Moved Permanently', @messg_head.reason)
      assert_equal('http://server:8080/subdir/', @messg_head.header('Location'))
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_subdir_200_OK
      @request.method = 'GET'
      @request.path = '/subdir/'
      @request.version = 'HTTP/1.1'
      @local_doc.publish('', @request, @response, self)
      assert_equal('/subdir/', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/subdir$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_match(@messg_body, /foo/)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_subdir_file_200_OK
      @request.method = 'GET'
      @request.path = '/subdir/foo'
      @request.version = 'HTTP/1.1'
      @local_doc.publish('', @request, @response, self)
      assert_equal('/subdir/foo', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/subdir/foo$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('application/octet-stream', @messg_head.header('Content-Type'))
      assert_equal(DATA, @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_illegal_path_403_Forbidden
      @request.method = 'GET'
      @request.path = '/../../..'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@local_doc.publish('', @request, @response, self)
      rescue Rucy::HTTPError
	assert_equal(403, $!.status) # Forbidden
	done = true
      end
      assert(done)
      assert_nil(@messg_head)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert_match(@log_info, /illegal path: /)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_if_modified_since_304_Not_Modified
      mtime = File.mtime('test/index.html')
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('If-Modified-Since', mtime.httpdate)
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(304, @messg_head.status)
      assert_equal('Not Modified', @messg_head.reason)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_if_modified_since_200_OK
      mtime = File.mtime('test/index.html')
      mtime -= 1
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('If-Modified-Since', mtime.httpdate)
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal(HTML.length.to_s, @messg_head.header('Content-Length'))
      assert_equal(HTML, @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_if_unmodified_since_412_Precondition_Failed
      mtime = File.mtime('test/index.html')
      mtime -= 1
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('If-Unmodified-Since', mtime.httpdate)
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(412, @messg_head.status)
      assert_equal('Precondition Failed', @messg_head.reason)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_if_unmodified_since_200_OK
      mtime = File.mtime('test/index.html')
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('If-Unmodified-Since', mtime.httpdate)
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal(HTML.length.to_s, @messg_head.header('Content-Length'))
      assert_equal(HTML, @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_range_206_Partial_Content
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('Range', 'bytes=7-')
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(206, @messg_head.status)
      assert_equal('Partial Content', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal("bytes 7-#{HTML.length - 1}/#{HTML.length}", @messg_head.header('Content-Range'))
      assert_equal((HTML.length - 7).to_s, @messg_head.header('Content-Length'))
      assert_equal(HTML[7..-1], @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_range_416_Requested_Range_Not_Satisfiable
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('Range', 'bytes=1024-')
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(416, @messg_head.status)
      assert_equal('Requested Range Not Satisfiable', @messg_head.reason)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_range_200_OK
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('Range', 'bytes=-1024') # invalid range spec.
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_nil(@messg_head.header('Content-Range'))
      assert_equal(HTML.length.to_s, @messg_head.header('Content-Length'))
      assert_equal(HTML, @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_range_if_modified_since_304_Not_Modified
      mtime = File.mtime('test/index.html')
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('Range', 'bytes=7-')
      @request.set_header('If-Modified-Since', mtime.httpdate)
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(304, @messg_head.status)
      assert_equal('Not Modified', @messg_head.reason)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_range_if_modified_since_206_Partial_Content
      mtime = File.mtime('test/index.html')
      mtime -= 1
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('Range', 'bytes=7-')
      @request.set_header('If-Modified-Since', mtime.httpdate)
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(206, @messg_head.status)
      assert_equal('Partial Content', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal("bytes 7-#{HTML.length - 1}/#{HTML.length}", @messg_head.header('Content-Range'))
      assert_equal((HTML.length - 7).to_s, @messg_head.header('Content-Length'))
      assert_equal(HTML[7..-1], @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_range_if_unmodified_since_412_Precondition_Failed
      mtime = File.mtime('test/index.html')
      mtime -= 1
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('Range', 'bytes=7-')
      @request.set_header('If-Unmodified-Since', mtime.httpdate)
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(412, @messg_head.status)
      assert_equal('Precondition Failed', @messg_head.reason)
      assert(@messg_body.empty?)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_range_if_unmodified_since_206_Partial_Content
      mtime = File.mtime('test/index.html')
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('Range', 'bytes=7-')
      @request.set_header('If-Unmodified-Since', mtime.httpdate)
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(206, @messg_head.status)
      assert_equal('Partial Content', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal("bytes 7-#{HTML.length - 1}/#{HTML.length}", @messg_head.header('Content-Range'))
      assert_equal((HTML.length - 7).to_s, @messg_head.header('Content-Length'))
      assert_equal(HTML[7..-1], @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_if_range_206_Partial_Content
      mtime = File.mtime('test/index.html')
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('Range', 'bytes=7-')
      @request.set_header('If-Range', mtime.httpdate)
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(206, @messg_head.status)
      assert_equal('Partial Content', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal("bytes 7-#{HTML.length - 1}/#{HTML.length}", @messg_head.header('Content-Range'))
      assert_equal((HTML.length - 7).to_s, @messg_head.header('Content-Length'))
      assert_equal(HTML[7..-1], @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end

    def test_GET_if_range_200_OK
      mtime = File.mtime('test/index.html')
      mtime -= 1
      @request.method = 'GET'
      @request.path = '/index.html'
      @request.version = 'HTTP/1.1'
      @request.set_header('Range', 'bytes=7-')
      @request.set_header('If-Range', mtime.httpdate)
      @local_doc.publish('', @request, @response, self)
      assert_equal('/index.html', @messg_head.doc_path)
      assert_match(@messg_head.local_path, %r"test/index\.html$")
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_nil(@messg_head.header('Content-Range'))
      assert_equal(HTML.length.to_s, @messg_head.header('Content-Length'))
      assert_equal(HTML, @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::LocalFileDocument/)
    end
  end
end
