#!/usr/local/bin/ruby
# $Id: test_loadlib.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

require 'rubyunit'
require 'rucy/loadlib'

module TestRucy
  class TestLoadLib < RUNIT::TestCase
    def setup
      Dir.mkdir('test_lib')
      File.open('test_lib/foo.rb', 'w') { |output|
	output.print "module Foo\n"
	output.print "  NAME = 'foo'\n"
	output.print "end\n"
      }
      File.open('test_lib/bar.rb', 'w') { |output|
	output.print "require 'bar.rb'\n"
	output.print "\n"
	output.print "module Foo\n"
	output.print "  module Bar\n"
	output.print "    NAME = 'bar'\n"
	output.print "  end\n"
	output.print "end\n"
      }
      File.open('test_lib/baz.rb', 'w') { |output|
	output.print "module Baz\n"
	output.print "  NAME = 'baz'\n"
	output.print "end\n"
      }
    end

    def teardown
      File.delete('test_lib/foo.rb')
      File.delete('test_lib/bar.rb')
      File.delete('test_lib/baz.rb')
      File.delete('test_lib/fail.rb') if (File.exist? 'test_lib/fail.rb')
      Dir.delete('test_lib')
    end

    def test_load_libraries
      Rucy::LoadLib.load_lib('test_lib')
      assert_equal('foo', Foo::NAME)
      assert_equal('bar', Foo::Bar::NAME)
      assert_equal('baz', Baz::NAME)
    end

    def test_load_errors
      File.open('test_lib/fail.rb', 'w') { |output|
	output.print 'Detarame\n'
      }
      Rucy::LoadLib.load_lib('test_lib')
      err_alist = Rucy::LoadLib.load_errors
      assert_equal(1, err_alist.length)
      assert_equal('fail.rb', err_alist[0][0])
      assert_instance_of(SyntaxError, err_alist[0][1])
    end
  end

  class TestLibraryLoader < RUNIT::TestCase
    def setup
      @load_lib_call = 0
      @load_lib_path_list = Array.new
      @load_errors_call = 0
      @loader = Rucy::LibraryLoader.new(self)
    end

    def load_lib(path)
      @load_lib_call += 1
      @load_lib_path_list.push(path)
      nil
    end

    def load_errors
      @load_errors_call += 1
      [ [ 'foo.rb', RuntimeError.new('HALO') ] ]
    end

    def test_load_lib
      @loader.add_load_path('foo')
      @loader.add_load_path('bar')
      @loader.load_lib
      assert_equal(2, @load_lib_call)
      assert_equal('foo', @load_lib_path_list[0])
      assert_equal('bar', @load_lib_path_list[1])
    end

    def test_load_errors
      err_alist = @loader.load_errors
      assert_equal(1, @load_errors_call)
      assert_equal(1, err_alist.length)
      assert_equal('foo.rb', err_alist[0][0])
      assert_instance_of(RuntimeError, err_alist[0][1])
      assert_equal('HALO', err_alist[0][1].message)
    end
  end
end
