#!/usr/local/bin/ruby
# $Id: test_lang.rb,v 1.6 2004/04/14 15:18:08 toki Exp $

require 'rubyunit'
require 'pseudo_req_res'
require 'rucy/lang'

module TestRucy
  class TestLanguageSelector < RUNIT::TestCase
    def setup
      @lang_sel = Rucy::LanguageSelector.new
    end

    def test_LanguageSelector_parse_accept_lang
      assert_equal([ { :language => 'da',
		       :country => nil,
		       :q => 1.0
		     },
		     { :language => 'en',
		       :country => 'gb',
		       :q => 0.8
		     },
		     { :language => 'en',
		       :country => nil,
		       :q => 0.7
		     }
		   ],
		   Rucy::LanguageSelector.parse_accept_lang('da, en-gb;q=0.8, en;q=0.7'))
      assert_equal([ # the first item has the most large qvalue.
		     { :language => 'da',
		       :country => nil,
		       :q => 1.0
		     },
		     { :language => 'en',
		       :country => 'gb',
		       :q => 0.8
		     },
		     { :language => 'en',
		       :country => nil,
		       :q => 0.7
		     }
		   ],
		   Rucy::LanguageSelector.parse_accept_lang('en;q=0.7, en-gb;q=0.8, da'))
      assert_equal([ # the first item has the longest language-range.
		     { :language => 'en',
		       :country => 'gb',
		       :q => 1.0
		     },
		     # the second and the third item have same length of language-range,
		     # and they are ordered by source order.
		     { :language => 'da',
		       :country => nil,
		       :q => 1.0
		     },
		     { :language => 'en',
		       :country => nil,
		       :q => 1.0
		     }
		   ],
		   Rucy::LanguageSelector.parse_accept_lang('da, en, en-gb'))
    end

    def test_select
      @lang_sel.add_lang('en-gb')
      @lang_sel.add_lang('en')
      assert_equal('en-gb', @lang_sel.select('da, en-gb;q=0.8, en;q=0.7'))
      assert_equal('en', @lang_sel.select('da, en-gb;q=0.7, en;q=0.8'))
      assert_equal('en-gb', @lang_sel.select('da, *;q=0.8, en;q=0.7'))
      assert_equal(nil, @lang_sel.select('ja, fr, da'))
    end
  end

  class TestLanguageRedirectFilter < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include PseudoDocument

    def setup
      super

      @lang = Rucy::LanguageRedirectFilter.new
      @lang.add_lang('ja', '/index_ja.html')
      @doc = Rucy::FilterDocument.new(self, @lang)
    end

    def test_GET_200_OK
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @doc.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
    end

    def test_GET_302_Found
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Accept-Language', 'ja, en;q=0.5')
      @doc.publish('', @request, @response, @logger)
      assert_equal(0, @publish_call)
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(302, @response.status)
      assert_equal('Found', @response.reason)
      assert_equal('http://server:8080/index_ja.html', @response.header('Location'))
      assert(! @messg_body.empty?)
    end

    def test_HEAD_200_OK
      @request.method = 'HEAD'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @doc.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert(@messg_body.empty?)
    end

    def test_HEAD_302_Found
      @request.method = 'HEAD'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Accept-Language', 'ja, en;q=0.5')
      @doc.publish('', @request, @response, @logger)
      assert_equal(0, @publish_call)
      assert_equal('HTTP/1.1', @response.version)
      assert_equal(302, @response.status)
      assert_equal('Found', @response.reason)
      assert_equal('http://server:8080/index_ja.html', @response.header('Location'))
      assert(@messg_body.empty?)
    end
  end
end
