#!/usr/local/bin/ruby
# $Id: test_document.rb,v 1.2 2004/04/16 03:24:05 toki Exp $

require 'rubyunit'
require 'pseudo_req_res'
require 'rucy/request'
require 'rucy/response'
require 'rucy/document'

module TestRucy
  class TestHTMLUtil < RUNIT::TestCase
    def test_escapeHTML
      assert_equal('', Rucy::HTMLUtil.escapeHTML(''))
      assert_equal('&lt;,&gt;,&amp;,&quot;', Rucy::HTMLUtil.escapeHTML('<,>,&,"'))
    end
  end

  class TestDocument < RUNIT::TestCase
    def setup
      @doc = Rucy::Document.new
    end

    def test_open
      @doc.open
    end

    def test_close
      @doc.close
    end

    def test_each
      for doc in @doc
	assert_equal(@doc, doc)
      end
    end
  end

  class TestFilter < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include PseudoDocument
    include PseudoFilter

    def setup
      super
      @filter_document = Rucy::FilterDocument.new(self, self)
    end

    def test_filter_init
      @filter_document.open
      assert_equal(1, @init_call)
    end

    def test_filter_final
      @filter_document.close
      assert_equal(1, @final_call)
    end

    def test_each
      doc_list = [ @filter_document, self ]
      for doc in @filter_document
	assert(doc_list.delete(doc))
      end
      assert(doc_list.empty?)
    end

    def test_GET
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @filter_document.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal(1, @filter_open_call)
      assert_equal(1, @filter_head_call)
      assert_equal(1, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('', @filter_script_name)
      assert_equal('/', @filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      log = "pseudo filter_open.\n"
      log += "pseudo publish.\n"
      log += "pseudo filter_head.\n"
      log += "pseudo filter_body.\n"
      log += "pseudo filter_close.\n"
      log_pat = Regexp.compile(Regexp.quote(log) + "\\z")
      assert_match(@log_debug, log_pat)
    end
  end

  class TestFilterTermination < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include PseudoDocument
    include PseudoFilter

    class FilterTermination < Rucy::Filter
      def initialize(parent)
	@parent = parent
      end

      def filter_open(context, script_name, request, response, logger)
	@parent.filter_open(context, script_name, request, response, logger)
	response.status = 404	# Not Found
	response.set_header('Content-Type', 'text/html')
	response.start_body
	response << "<html>bar</html>\n"
	terminate_filter	# skip to filter_close.
	nil
      end

      def filter_head(context, script_name, request, response, logger)
	@parent.filter_head(context, script_name, request, response, logger)
	nil
      end

      def filter_body(context, script_name, request, response, logger, messg_body)
	@parent.filter_body(context, script_name, request, response, logger, messg_body)
	nil
      end

      def filter_close(context, script_name, request, response, logger)
	@parent.filter_close(context, script_name, request, response, logger)
	nil
      end
    end

    def setup
      super
      @filter_document = Rucy::FilterDocument.new(self, FilterTermination.new(self))
    end

    def test_GET
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @filter_document.publish('', @request, @response, @logger)
      assert_equal(0, @publish_call)
      assert_equal(1, @filter_open_call)
      assert_equal(0, @filter_head_call)
      assert_equal(0, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(404, @messg_head.status)
      assert_equal('Not Found', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal("<html>bar</html>\n", @messg_body)
    end
  end

  class TestDocumentFactory < RUNIT::TestCase
    def setup
      @doc_name_call = 0
      @doc_args_call = 0
      @set_doc_option_call = 0
      @set_doc_option_last_name = nil
      @set_doc_option_last_value = nil
      @filter_name_call = 0
      @filter_args_call = 0
      @set_filter_option_call = 0
      @set_filter_option_last_name = nil
      @set_filter_option_last_value = nil
      @new_call = 0
      @new_string_value = nil
      @new_text_value = nil
      @new_number_value = nil
      @factory = Rucy::DocumentFactory.new
    end

    def doc_name
      @doc_name_call += 1
      'test_document'
    end

    def doc_args
      @doc_args_call += 1
      [ [ 'string',  :string,  nil              ],
	[ 'text',    :text,    "Hello world.\n" ],
	[ 'number',  :number,  0                ]
      ]
    end

    def set_doc_option(name, value)
      @set_doc_option_call += 1
      @set_doc_option_last_name = name
      @set_doc_option_last_value = value
      nil
    end

    def filter_name
      @filter_name_call += 1
      'test_filter'
    end

    def filter_args
      @filter_args_call += 1
      [ [ 'string',  :string,  nil              ],
	[ 'text',    :text,    "Hello world.\n" ],
	[ 'number',  :number,  0                ]
      ]
    end

    def set_filter_option(name, value)
      @set_filter_option_call += 1
      @set_filter_option_last_name = name
      @set_filter_option_last_value = value
      nil
    end

    def new(string, text, number)
      @new_call += 1
      @new_string_value = string
      @new_text_value = text
      @new_number_value = number
      :test
    end

    def test_has_document?
      assert(! (@factory.has_document? 'test_document'))
      @factory.add_document(self)
      assert((@factory.has_document? 'test_document'))
    end

    def test_doc_names
      @factory.add_document(self)
      assert_equal(1, @doc_name_call)
      doc_names = @factory.doc_names
      assert_equal(1, doc_names.length)
      assert_equal('test_document', doc_names[0])
    end

    def test_doc_args
      @factory.add_document(self)
      assert_equal(1, @doc_name_call)
      doc_args = @factory.doc_args('test_document')
      assert_equal(1, @doc_args_call)
      assert_equal(3, doc_args.length)
      arg_name, arg_type, arg_default = doc_args[0]
      assert_equal('string', arg_name)
      assert_equal(:string, arg_type)
      assert_nil(arg_default)
      arg_name, arg_type, arg_default = doc_args[1]
      assert_equal('text', arg_name)
      assert_equal(:text, arg_type)
      assert_equal("Hello world.\n", arg_default)
      arg_name, arg_type, arg_default = doc_args[2]
      assert_equal('number', arg_name)
      assert_equal(:number, arg_type)
      assert_equal(0, arg_default)
    end

    def test_doc_build
      @factory.add_document(self)
      assert_equal(1, @doc_name_call)
      dummy_doc = @factory.doc_build('test_document', [ 'string', 'text', 0 ])
      assert_equal(1, @new_call)
      assert_equal('string', @new_string_value)
      assert_equal('text', @new_text_value)
      assert_equal(0, @new_number_value)
      assert_equal(:test, dummy_doc)
    end

    def test_set_doc_option
      @factory.add_document(self)
      @factory.set_doc_option('test_document', :test_message, "Hello world.")
      assert_equal(1, @set_doc_option_call)
      assert_equal(:test_message, @set_doc_option_last_name)
      assert_equal("Hello world.", @set_doc_option_last_value)
    end

    def test_has_filter?
      assert(! (@factory.has_filter? 'test_filter'))
      @factory.add_filter(self)
      assert((@factory.has_filter? 'test_filter'))
    end

    def test_filter_names
      @factory.add_filter(self)
      assert_equal(1, @filter_name_call)
      filter_names = @factory.filter_names
      assert_equal(1, filter_names.length)
      assert_equal('test_filter', filter_names[0])
    end

    def test_filter_args
      @factory.add_filter(self)
      assert_equal(1, @filter_name_call)
      filter_args = @factory.filter_args('test_filter')
      assert_equal(1, @filter_args_call)
      assert_equal(3, filter_args.length)
      arg_name, arg_type, arg_default = filter_args[0]
      assert_equal('string', arg_name)
      assert_equal(:string, arg_type)
      assert_nil(arg_default)
      arg_name, arg_type, arg_default = filter_args[1]
      assert_equal('text', arg_name)
      assert_equal(:text, arg_type)
      assert_equal("Hello world.\n", arg_default)
      arg_name, arg_type, arg_default = filter_args[2]
      assert_equal('number', arg_name)
      assert_equal(:number, arg_type)
      assert_equal(0, arg_default)
    end

    def test_filter_build
      @factory.add_filter(self)
      assert_equal(1, @filter_name_call)
      dummy_filter = @factory.filter_build('test_filter', [ 'string', 'text', 0 ])
      assert_equal(1, @new_call)
      assert_equal('string', @new_string_value)
      assert_equal('text', @new_text_value)
      assert_equal(0, @new_number_value)
      assert_equal(:test, dummy_filter)
    end

    def test_set_filter_option
      @factory.add_filter(self)
      @factory.set_filter_option('test_filter', :test_message, "Hello world.")
      assert_equal(1, @set_filter_option_call)
      assert_equal(:test_message, @set_filter_option_last_name)
      assert_equal("Hello world.", @set_filter_option_last_value)
    end
  end

  class TestPageDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse

    def setup
      super
      @page = Rucy::PageDocument.new("<html>Hello world.</html>\n")
    end

    def test_GET
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @page.publish('', @request, @response, @logger)
      assert_nil(@messg_head.doc_path)
      assert_nil(@messg_head.local_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal('26', @messg_head.header('Content-Length'))
      assert_equal("<html>Hello world.</html>\n", @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::PageDocument/)
    end

    def test_HEAD
      @request.method = 'HEAD'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @page.publish('', @request, @response, @logger)
      assert_nil(@messg_head.doc_path)
      assert_nil(@messg_head.local_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal('26', @messg_head.header('Content-Length'))
      assert_equal('', @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::PageDocument/)
    end
  end

  class TestSubsetDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include PseudoDocument

    def setup
      super
      @doc = Rucy::SubsetDocument.new(self, '/foo')
    end

    def test_each
      doc_list = [ @doc, self ]
      for doc in @doc
	assert(doc_list.delete(doc))
      end
      assert(doc_list.empty?)
    end

    def test_publish_foo
      @request.method = 'GET'
      @request.path = '/foo'
      @request.version = 'HTTP/1.1'
      @doc.publish('/foo', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_publish_foo_bar
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @doc.publish('/foo', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/foo/bar', @publish_request_path)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_publish_bar_foo
      @request.method = 'GET'
      @request.path = '/bar/foo'
      @request.version = 'HTTP/1.1'
      @doc.publish('/bar/foo', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/bar', @publish_script_name)
      assert_equal('/bar/foo', @publish_request_path)
      assert_equal("pseudo publish.\n", @log_debug)
    end


    def test_publish_bar_foo_baz
      @request.method = 'GET'
      @request.path = '/bar/foo/baz'
      @request.version = 'HTTP/1.1'
      @doc.publish('/bar/foo', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/bar', @publish_script_name)
      assert_equal('/bar/foo/baz', @publish_request_path)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_publish_bar_foo2
      @request.method = 'GET'
      @request.path = '/bar/foo'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@doc.publish('/bar', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status)
	assert_equal(0, @publish_call)
	assert(@log_debug.empty?)
	done = true
      end
      assert(done)
    end
  end

  class TestFolderDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include PseudoDocument
    include PseudoFilter

    def setup
      super
      @folder = Rucy::FolderDocument.new
    end

    def test_umount
      @folder.mount(self, '/foo')
      assert_equal(self, @folder.find('/foo')[0])
      assert_equal('/foo', @folder.find('/foo')[1])
      assert_equal(self, @folder.find('/foo/bar')[0])
      assert_equal('/foo', @folder.find('/foo/bar')[1])
      assert_equal(self, @folder.umount('/foo'))
      assert_nil(@folder.find('/foo'))

      @folder.mount(self, '/foo', /\.cgi/)
      assert_nil(@folder.find('/foo/bar'))
      assert_equal(self, @folder.find('/foo/bar.cgi')[0])
      assert_equal('/foo', @folder.find('/foo/bar.cgi')[1])
      assert_equal(self, @folder.umount('/foo', /\.cgi/))
      assert_nil(@folder.find('/foo/bar.cgi'))
    end

    def test_each
      folder2 = Rucy::FolderDocument.new
      @folder.mount(folder2, '/foo')
      folder2.mount(@folder, '/bar')
      @folder.mount(self, '/')
      doc_list = [ @folder, folder2, self ]
      for doc in @folder
	assert(doc_list.delete(doc))
      end
      assert(doc_list.empty?)
    end

    def test_GET_root
      @folder.mount(self, '/')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_root_foo
      @folder.mount(self, '/')
      @request.method = 'GET'
      @request.path = '/foo'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_foo
      @folder.mount(self, '/foo')
      @request.method = 'GET'
      @request.path = '/foo'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_foo_bar
      @folder.mount(self, '/foo')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_foo_not_found
      @folder.mount(self, '/foo')
      @request.method = 'GET'
      @request.path = '/bar'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@folder.publish('', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	assert(0, @publish_call)
	assert_match(@log_debug, /not mounted/)
	done = true
      end
      assert(done)
    end

    def tset_GET_mask
      @folder.mount(self, '/', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo.cgi'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/foo.cgi', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_mask_not_found
      @folder.mount(self, '/', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo.html'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@folder.publish('', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	assert(0, @publish_call)
	assert_match(@log_debug, /not mounted/)
	done = true
      end
      assert(done)
    end

    def test_GET_filter
      @folder.mount(self, '/foo')
      @folder.attach(self, '/foo')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar', @publish_request_path)
      assert_equal(1, @filter_open_call)
      assert_equal(1, @filter_head_call)
      assert_equal(1, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('/foo', @filter_script_name)
      assert_equal('/foo/bar', @filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      log = "pseudo filter_open.\n"
      log += "pseudo publish.\n"
      log += "pseudo filter_head.\n"
      log += "pseudo filter_body.\n"
      log += "pseudo filter_close.\n"
      log_pat = Regexp.compile(Regexp.quote(log) + "\\z")
      assert_match(@log_debug, log_pat)
    end

    def test_GET_filter2
      @folder.mount(self, '/foo')
      @folder.attach(self, '/foo/bar')
      @request.method = 'GET'
      @request.path = '/foo/bar/baz'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar/baz', @publish_request_path)
      assert_equal(1, @filter_open_call)
      assert_equal(1, @filter_head_call)
      assert_equal(1, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('/foo/bar', @filter_script_name)
      assert_equal('/foo/bar/baz', @filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      log = "pseudo filter_open.\n"
      log += "pseudo publish.\n"
      log += "pseudo filter_head.\n"
      log += "pseudo filter_body.\n"
      log += "pseudo filter_close.\n"
      log_pat = Regexp.compile(Regexp.quote(log) + "\\z")
      assert_match(@log_debug, log_pat)
    end

    def test_GET_filter_mask
      @folder.mount(self, '/foo')
      @folder.attach(self, '/foo', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo/bar.cgi'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar.cgi', @publish_request_path)
      assert_equal(1, @filter_open_call)
      assert_equal(1, @filter_head_call)
      assert_equal(1, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('/foo', @filter_script_name)
      assert_equal('/foo/bar.cgi', @filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      log = "pseudo filter_open.\n"
      log += "pseudo publish.\n"
      log += "pseudo filter_head.\n"
      log += "pseudo filter_body.\n"
      log += "pseudo filter_close.\n"
      log_pat = Regexp.compile(Regexp.quote(log) + "\\z")
      assert_match(@log_debug, log_pat)
    end

    def test_GET_filter_mask2
      @folder.mount(self, '/foo')
      @folder.attach(self, '/foo', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar', @publish_request_path)
      assert_equal(0, @filter_open_call)
      assert_equal(0, @filter_head_call)
      assert_equal(0, @filter_body_call)
      assert_equal(0, @filter_close_call)
      assert_nil(@filter_script_name)
      assert_nil(@filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_virtual
      @folder.virtual_mount('foo', self, '/')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'foo:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_virtual_not_found
      @folder.virtual_mount('foo', self, '/')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'bar')
      begin
	@folder.publish('', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	assert(0, @publish_call)
	assert_match(@log_debug, /not mounted/)
	done = true
      end
      assert(done)
    end

    def test_GET_virtual_mask
      @folder.virtual_mount('foo', self, '/', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'foo:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/foo.cgi', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_virtual_mask_not_found
      @folder.virtual_mount('foo', self, '/', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'bar:80')
      begin
	@folder.publish('', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	assert(0, @publish_call)
	assert_match(@log_debug, /not mounted/)
	done = true
      end
      assert(done)
    end

    def test_GET_virtual_filter
      @folder.virtual_mount('foo', self, '/foo')
      @folder.virtual_attach('foo', self, '/foo')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'foo:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar', @publish_request_path)
      assert_equal(1, @filter_open_call)
      assert_equal(1, @filter_head_call)
      assert_equal(1, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('/foo', @filter_script_name)
      assert_equal('/foo/bar', @filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      log = "pseudo filter_open.\n"
      log += "pseudo publish.\n"
      log += "pseudo filter_head.\n"
      log += "pseudo filter_body.\n"
      log += "pseudo filter_close.\n"
      log_pat = Regexp.compile(Regexp.quote(log) + "\\z")
      assert_match(@log_debug, log_pat)
    end

    def test_GET_virtual_filter_not_found
      @folder.virtual_mount('foo', self, '/foo')
      @folder.virtual_attach('foo', self, '/foo')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'bar:80')
      begin
	@folder.publish('', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	assert(0, @publish_call)
	assert(0, @filter_open_call)
	assert(0, @filter_head_call)
	assert(0, @filter_body_call)
	assert(0, @filter_close_call)
	assert_match(@log_debug, /not mounted/)
	done = true
      end
      assert(done)
    end

    def test_alias_root
      @folder.set_alias('/foo', '/')	# /foo -> /
      @folder.mount(self, '/')
      @request.method = 'GET'
      @request.path = '/foo'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_alias_root2
      @folder.set_alias('/', '/foo')	# / -> /foo
      @folder.mount(self, '/foo')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_alias_foo
      @folder.set_alias('/bar', '/foo')	# /bar -> /foo
      @folder.mount(self, '/foo')
      @request.method = 'GET'
      @request.path = '/bar'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_virtual_alias_root
      @folder.set_virtual_alias('baz', '/foo', '/') # /foo -> /
      @folder.virtual_mount('baz', self, '/')
      @request.method = 'GET'
      @request.path = '/foo'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'baz:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_virtual_alias_root2
      @folder.set_virtual_alias('baz', '/', '/foo') # / -> /foo
      @folder.virtual_mount('baz', self, '/foo')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'baz:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_virtual_alias_foo
      @folder.set_virtual_alias('baz', '/bar', '/foo') # /bar -> /foo
      @folder.virtual_mount('baz', self, '/foo')
      @request.method = 'GET'
      @request.path = '/bar'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'baz:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end
  end
end
