#!/usr/local/bin/ruby
# $Id: test_control.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

require 'rubyunit'
require 'wpm'
require 'wpm/rucydriver'
require 'rucy/control'
require 'rucy/document'
require 'rucy/passwd'
require 'rucy/access'
require 'rucy/auth'

module TestRucy
  class TestControl < RUNIT::TestCase
    def setup
      # for Rucy::Properties class
      @properties = Hash.new
      @list_call = 0
      @list_last_name = nil
      @set_list_call = 0
      @set_list_last_name = nil
      @map_call = 0
      @map_last_name = nil
      @set_map_call = 0
      @set_map_last_name = nil
      @params_call = 0
      @params_last_names = nil
      @set_params_call = 0
      @set_params_last_names = nil

      # for Rucy::Server class
      @restart_call = 0
      @close_call = 0
      @restart_signal_call = 0
      @restart_signal = nil
      @port_call = 0
      @port = nil
      @timeout_call = 0
      @timeout = nil
      @keep_alive_call = 0
      @keep_alive = nil
      @max_requests_call = 0
      @max_requests = nil
      @queue_length_call = 0
      @queue_length = nil
      @messengers_call = 0
      @messengers = nil
      @messenger_factory_call = 0
      @messenger_factory = nil
      @messenger_threads_call = 0
      @messenger_threads = nil
      @messenger_queue_length_call = 0
      @messenger_queue_length = nil
      @subprocess_user_call = 0
      @subprocess_user = nil
      @subprocess_group_call = 0
      @subprocess_group = nil
      @add_logger_call = 0
      @set_alias_call = 0
      @set_alias_alist = Array.new
      @set_virtual_alias_call = 0
      @set_virtual_alias_alist = Array.new
      @mount_call = 0
      @virtual_mount_call = 0
      @attach_call = 0
      @virtual_attach_call = 0
      @folder = Rucy::FolderDocument.new
      @close_hook_call = 0

      # for Rucy::PasswordEncryptor
      @pw_enc = Rucy::PasswordEncryptor.new
      @crypt_call = 0

      # for Rucy::LibraryLoader class
      @load_lib_call = 0
      @load_errors_call = 0
      @factory = Rucy::DocumentFactory.new

      @control = nil
    end

    def list(name)
      @list_call += 1
      @list_last_name = name
      if (@properties[name]) then
	assert_instance_of(Array, @properties[name])
	return @properties[name].dup
      else
	return Array.new
      end
    end

    def set_list(name, list)
      @set_list_call += 1
      @set_list_last_name = name
      @properties[name] = list.dup
      nil
    end

    def map(name)
      @map_call += 1
      @map_last_name = name
      if (@properties[name]) then
	assert_instance_of(Hash, @properties[name])
	return @properties[name].dup
      else
	return Hash.new
      end
    end

    def set_map(name, map)
      @set_map_call += 1
      @set_map_last_name = name
      @properties[name] = map.dup
      nil
    end

    def params(*names)
      @params_call += 1
      @params_last_names = names
      params = Hash.new
      for name in names
	params[name] = @properties[name]
      end
      params
    end

    def set_params(params)
      @set_params_call += 1
      @set_params_last_names = params.keys
      for name, value in params
	@properties[name] = value
      end
      nil
    end

    def restart
      @restart_call += 1
      nil
    end

    def close
      @close_call += 1
      nil
    end

    def restart_signal=(restart_signal)
      @restart_signal_call += 1
      @restart_signal = restart_signal
      @restart_signal.server = self
      restart_signal
    end

    def port=(port)
      @port_call += 1
      @port = port
    end

    def timeout=(timeout)
      @timeout_call += 1
      @timeout = timeout
    end

    def keep_alive=(keep_alive)
      @keep_alive_call += 1
      @keep_alive = keep_alive
    end

    def max_requests=(max_requests)
      @max_requests_call += 1
      @max_requests = max_requests
    end

    def queue_length=(queue_length)
      @queue_length_call += 1
      @queue_length = queue_length
    end

    def messengers=(messengers)
      @messengers_call += 1
      @messengers = messengers
    end

    def messenger_factory=(messenger_factory)
      @messenger_factory_call += 1
      @messenger_factory = messenger_factory
    end

    def messenger_threads=(messenger_threads)
      @messenger_threads_call += 1
      @messenger_threads = messenger_threads
    end

    def messenger_queue_length=(messenger_queue_length)
      @messenger_queue_length_call += 1
      @messenger_queue_length = messenger_queue_length
    end

    def subprocess_user=(subprocess_user)
      @subprocess_user_call += 1
      @subprocess_user = subprocess_user
    end

    def subprocess_group=(subprocess_group)
      @subprocess_group_call += 1
      @subprocess_group = subprocess_group
    end

    def add_logger(logger)
      @add_logger_call += 1
      nil
    end

    def set_alias(alias_path, orig_path)
      @set_alias_call += 1
      @set_alias_alist.push([ alias_path, orig_path ])
      @folder.set_alias(alias_path, orig_path)
      nil
    end

    def set_virtual_alias(host, alias_path, orig_path)
      @set_virtual_alias_call += 1
      @set_virtual_alias_alist.push([ host, alias_path, orig_path ])
      @folder.set_virtual_alias(host, alias_path, orig_path)
      nil
    end

    def mount(document, path, mask=nil)
      @mount_call += 1
      @folder.mount(document, path, mask)
      nil
    end

    def virtual_mount(host, document, path, mask=nil)
      @virtual_mount_call += 1
      @folder.virtual_mount(host, document, path, mask)
      nil
    end

    def attach(filter, path, mask=nil)
      @attach_call += 1
      @folder.attach(filter, path, mask)
      nil
    end

    def virtual_attach(host, document, path, mask=nil)
      @virtual_attach_call += 1
      @folder.virtual_attach(host, document, path, mask)
      nil
    end

    def close_hook(&block)
      @close_hook_call += 1
      nil
    end

    def crypt(password)
      @crypt_call += 1
      @pw_enc.crypt(password)
    end

    def load_lib
      @load_lib_call += 1
      nil
    end

    def load_errors
      @load_errors_call += 1
      [ [ 'foo.rb', RuntimeError.new('HALO') ] ]
    end

    def build_control
      @control = Rucy::Control.new
      @control.lib_loader = self
      @control.pw_enc = self
      @control.doc_factory = @factory
      @control.properties = self
      @control.page_path = 'nothing'
    end
    private :build_control

    def test_load_errors
      build_control
      err_alist = @control.load_errors
      assert_equal(1, @load_errors_call)
      assert_equal(1, err_alist.length)
      assert_equal('foo.rb', err_alist[0][0])
      assert_instance_of(RuntimeError, err_alist[0][1])
      assert_equal('HALO', err_alist[0][1].message)
    end

    def test_server_restart
      build_control
      @control.server_setup(self)
      assert_equal(0, @restart_call)
      @control.server_restart
      @restart_signal.wait
      assert_equal(1, @restart_call)
    end

    def test_server_close
      build_control
      @control.server_setup(self)
      assert_equal(0, @close_call)
      @control.server_close
      @restart_signal.wait
      assert_equal(1, @close_call)
    end

    def test_server_params
      build_control
      server_params = @control.server_params
      assert_equal(1, @params_call)
      assert_equal([ 'server_type',
		     'port',
		     'timeout',
		     'keep_alive',
		     'max_requests',
		     'queue_length',
		     'messengers',
		     'messenger_threads',
		     'messenger_queue_length',
		     'subprocess_user',
		     'subprocess_group'
		   ].sort, @params_last_names.sort)
      assert_equal('multithread', server_params['server_type'])
      assert_equal(8888, server_params['port'])
      assert_equal(300, server_params['timeout'])
      assert_equal(8, server_params['keep_alive'])
      assert_equal(32, server_params['max_requests'])
      assert_equal(16, server_params['queue_length'])
      assert_equal(8, server_params['messengers'])
      assert_equal(4, server_params['messenger_threads'])
      assert_equal(4, server_params['messenger_queue_length'])
      assert_equal('nobody', server_params['subprocess_user'])
      assert_equal('nobody', server_params['subprocess_group'])

      server_params['server_type'] = 'multiprocess'
      server_params['port'] = 80
      server_params['timeout'] = 60
      server_params['keep_alive'] = 4
      server_params['max_requests'] = 24
      server_params['queue_length'] = 32
      server_params['messengers'] = 16
      server_params['messenger_threads'] = 8
      server_params['messenger_queue_length'] = 8
      server_params['subprocess_user'] = 'httpd'
      server_params['subprocess_group'] = 'httpd'
      @control.set_server_params(server_params)
      assert_equal(1, @set_params_call)
      assert_equal([ 'server_type',
		     'port',
		     'timeout',
		     'keep_alive',
		     'max_requests',
		     'queue_length',
		     'messengers',
		     'messenger_threads',
		     'messenger_queue_length',
		     'subprocess_user',
		     'subprocess_group'
		   ].sort, @set_params_last_names.sort)
      assert_equal('multiprocess', @properties['server_type'])
      assert_equal(80, @properties['port'])
      assert_equal(60, @properties['timeout'])
      assert_equal(4, @properties['keep_alive'])
      assert_equal(24, @properties['max_requests'])
      assert_equal(32, @properties['queue_length'])
      assert_equal(16, @properties['messengers'])
      assert_equal(8, @properties['messenger_threads'])
      assert_equal(8, @properties['messenger_queue_length'])
      assert_equal('httpd', @properties['subprocess_user'])
      assert_equal('httpd', @properties['subprocess_group'])
    end

    def test_logging_params
      build_control
      logging_params = @control.logging_params
      assert_equal(1, @params_call)
      assert_equal([ 'stdout_emerg',
		     'stdout_alert',
		     'stdout_crit',
		     'stdout_err',
		     'stdout_warning',
		     'stdout_notice',
		     'stdout_info',
		     'stdout_debug',
		     'logfiles'
		   ].sort, @params_last_names.sort)
      assert_equal(true, logging_params['stdout_emerg'])
      assert_equal(true, logging_params['stdout_alert'])
      assert_equal(true, logging_params['stdout_crit'])
      assert_equal(true, logging_params['stdout_err'])
      assert_equal(true, logging_params['stdout_warning'])
      assert_equal(true, logging_params['stdout_notice'])
      assert_equal(true, logging_params['stdout_info'])
      assert_equal(false, logging_params['stdout_debug'])
      assert_equal([], logging_params['logfiles'])
      logging_params['stdout_debug'] = true
      logging_params['logfiles'].push({ 'path' => 'server.log',
					'log_emerg' => true,
					'log_alert' => true,
					'log_crit' => true,
					'log_err' => true,
					'log_warning' => true,
					'log_notice' => true,
					'log_info' => false,
					'log_debug' => false
				      })
      logging_params['logfiles'].push({ 'path' => 'access.log',
					'log_emerg' => false,
					'log_alert' => false,
					'log_crit' => false,
					'log_err' => false,
					'log_warning' => false,
					'log_notice' => false,
					'log_info' => true,
					'log_debug' => false
				      })
      logging_params['logfiles'].push({ 'path' => 'debug.log',
					'log_emerg' => false,
					'log_alert' => false,
					'log_crit' => false,
					'log_err' => false,
					'log_warning' => false,
					'log_notice' => false,
					'log_info' => false,
					'log_debug' => true
				      })
      @control.set_logging_params(logging_params)
      assert_equal(1, @set_params_call)
      assert_equal([ 'stdout_emerg',
		     'stdout_alert',
		     'stdout_crit',
		     'stdout_err',
		     'stdout_warning',
		     'stdout_notice',
		     'stdout_info',
		     'stdout_debug',
		     'logfiles'
		   ].sort, @set_params_last_names.sort)
      assert_equal(true, @properties['stdout_emerg'])
      assert_equal(true, @properties['stdout_alert'])
      assert_equal(true, @properties['stdout_crit'])
      assert_equal(true, @properties['stdout_err'])
      assert_equal(true, @properties['stdout_warning'])
      assert_equal(true, @properties['stdout_notice'])
      assert_equal(true, @properties['stdout_info'])
      assert_equal(true, @properties['stdout_debug'])
      assert_equal(3, @properties['logfiles'].length)
      assert_equal({ 'path' => 'server.log',
		     'log_emerg' => true,
		     'log_alert' => true,
		     'log_crit' => true,
		     'log_err' => true,
		     'log_warning' => true,
		     'log_notice' => true,
		     'log_info' => false,
		     'log_debug' => false
		   }, @properties['logfiles'][0])
      assert_equal({ 'path' => 'access.log',
		     'log_emerg' => false,
		     'log_alert' => false,
		     'log_crit' => false,
		     'log_err' => false,
		     'log_warning' => false,
		     'log_notice' => false,
		     'log_info' => true,
		     'log_debug' => false
		   }, @properties['logfiles'][1])
      assert_equal({ 'path' => 'debug.log',
		     'log_emerg' => false,
		     'log_alert' => false,
		     'log_crit' => false,
		     'log_err' => false,
		     'log_warning' => false,
		     'log_notice' => false,
		     'log_info' => false,
		     'log_debug' => true
		   }, @properties['logfiles'][2])
    end

    def test_aliases
      build_control
      alias_list = @control.aliases
      assert_equal(1, @list_call)
      assert_equal('aliases', @list_last_name)
      assert(alias_list.empty?)
      alias_list.push({ 'virtual_host' => nil,
			'alias_path' => '/bar',
			'orig_path' => '/foo'
		      })
      alias_list.push({ 'virtual_host' => 'foo',
			'alias_path' => '/fuge',
			'orig_path' => '/hoge'
		      })
      @control.set_aliases(alias_list)
      assert_equal(1, @set_list_call)
      assert_equal('aliases', @set_list_last_name)
      assert_equal(2, @properties['aliases'].length)
      assert_equal({ 'virtual_host' => nil,
		     'alias_path' => '/bar',
		     'orig_path' => '/foo'
		   }, @properties['aliases'][0])
      assert_equal({ 'virtual_host' => 'foo',
		     'alias_path' => '/fuge',
		     'orig_path' => '/hoge'
		   }, @properties['aliases'][1])
    end

    def test_documents
      build_control
      document_list = @control.documents
      assert_equal(1, @list_call)
      assert_equal('documents', @list_last_name)
      assert(document_list.empty?)
      document_list.push({ 'document' => 'Page',
			   'arguments' => [
			     "Hello world.\n",
			     'text/plain'
			   ],
			   'mount_path' => '/hello',
			   'mount_mask' => nil,
			   'virtual_host' => nil
			 })
      document_list.push({ 'document' => 'LocalFile',
			   'arguments' => [
			     '/home/alice/public_html'
			   ],
			   'mount_path' => '/~alice',
			   'mount_mask' => nil,
			   'virtual_host' => 'foo'
			 })
      @control.set_documents(document_list)
      assert_equal(1, @set_list_call)
      assert_equal('documents', @set_list_last_name)
      assert_equal(2, @properties['documents'].length)
      assert_equal({ 'document' => 'Page',
		     'arguments' => [
		       "Hello world.\n",
		       'text/plain'
		     ],
		     'mount_path' => '/hello',
		     'mount_mask' => nil,
		     'virtual_host' => nil
		   }, @properties['documents'][0])
      assert_equal({ 'document' => 'LocalFile',
		     'arguments' => [
		       '/home/alice/public_html'
		     ],
		     'mount_path' => '/~alice',
		     'mount_mask' => nil,
		     'virtual_host' => 'foo'
		   }, @properties['documents'][1])
    end

    def test_filters
      build_control
      filter_list = @control.filters
      assert_equal(1, @list_call)
      assert_equal('filters', @list_last_name)
      assert(filter_list.empty?)
      filter_list.push({ 'filter' => 'ERB',
			 'arguments' => [
			   1
			 ],
			 'attach_path' => '/foo',
			 'attach_mask' => %r"\.rhtml?($|/)",
			 'virtual_host' => nil
		       })
      filter_list.push({ 'filter' => 'SSI',
			 'arguments' => [
			   { 'config' => true,
			     'include' => true,
			     'echo' => true,
			     'fsize' => true,
			     'flastmod' => true,
			     'exec' => true
			   }
			 ],
			 'attach_path' => '/bar',
			 'attach_mask' => %r"\.shtml?($|/)",
			 'virtual_host' => 'baz'
		       })
      @control.set_filters(filter_list)
      assert_equal(1, @set_list_call)
      assert_equal('filters', @set_list_last_name)
      assert_equal(2, @properties['filters'].length)
      assert_equal({ 'filter' => 'ERB',
		     'arguments' => [
		       1
		     ],
		     'attach_path' => '/foo',
		     'attach_mask' => %r"\.rhtml?($|/)",
		     'virtual_host' => nil
		   }, @properties['filters'][0])
      assert_equal({ 'filter' => 'SSI',
		     'arguments' => [
		       { 'config' => true,
			 'include' => true,
			 'echo' => true,
			 'fsize' => true,
			 'flastmod' => true,
			 'exec' => true
		       }
		     ],
		     'attach_path' => '/bar',
		     'attach_mask' => %r"\.shtml?($|/)",
		     'virtual_host' => 'baz'
		   }, @properties['filters'][1])
    end

    def test_default_server_settings
      build_control
      @control.server_setup(self)
      assert_equal(3, @params_call) # for server parameters and logging parameters and admin parameters
      assert_equal(3, @list_call) # for (1) alias list and (2) document list and (3) filter list
      assert_equal(1, @load_lib_call)
      assert_equal(1, @restart_signal_call)
      assert_instance_of(Rucy::MultiThreadRestartSignal, @restart_signal)
      assert_equal(1, @port_call)
      assert_equal(8888, @port)
      assert_equal(1, @timeout_call)
      assert_equal(300, @timeout)
      assert_equal(1, @keep_alive_call)
      assert_equal(8, @keep_alive)
      assert_equal(1, @max_requests_call)
      assert_equal(32, @max_requests)
      assert_equal(1, @queue_length_call)
      assert_equal(16, @queue_length)
      assert_equal(1, @messengers_call)
      assert_equal(8, @messengers)
      assert_equal(1, @messenger_factory_call)
      assert_equal(Rucy::MultiThreadMessenger, @messenger_factory)
      assert_equal(1, @messenger_threads_call)
      assert_equal(4, @messenger_threads)
      assert_equal(1, @messenger_queue_length_call)
      assert_equal(4, @messenger_queue_length)
      assert_equal(1, @subprocess_user_call)
      assert_equal('nobody', @subprocess_user)
      assert_equal(1, @subprocess_group_call)
      assert_equal('nobody', @subprocess_group)
      assert_equal(1, @add_logger_call)
      assert_equal(1, @close_hook_call)
    end

    def test_modified_server_settings
      @properties['server_type'] = 'multiprocess'
      @properties['port'] = 80
      @properties['timeout'] = 60
      @properties['keep_alive'] = 4
      @properties['max_requests'] = 24
      @properties['queue_length'] = 32
      @properties['messengers'] = 16
      @properties['messenger_threads'] = 8
      @properties['messenger_queue_length'] = 8
      @properties['subprocess_user'] = 'httpd'
      @properties['subprocess_group'] = 'httpd'
      build_control
      @control.server_setup(self)
      assert_equal(3, @params_call) # for server parameters and logging parameters and admin parameters
      assert_equal(3, @list_call) # for (1) alias list and (2) document list and (3) filter list
      assert_equal(1, @load_lib_call)
      assert_equal(1, @restart_signal_call)
      assert_instance_of(Rucy::MultiProcessRestartSignal, @restart_signal)
      assert_equal(1, @port_call)
      assert_equal(80, @port)
      assert_equal(1, @timeout_call)
      assert_equal(60, @timeout)
      assert_equal(1, @keep_alive_call)
      assert_equal(4, @keep_alive)
      assert_equal(1, @max_requests_call)
      assert_equal(24, @max_requests)
      assert_equal(1, @queue_length_call)
      assert_equal(32, @queue_length)
      assert_equal(1, @messengers_call)
      assert_equal(16, @messengers)
      assert_equal(1, @messenger_factory_call)
      assert_equal(Rucy::MultiProcessMessenger, @messenger_factory)
      assert_equal(1, @messenger_threads_call)
      assert_equal(8, @messenger_threads)
      assert_equal(1, @messenger_queue_length_call)
      assert_equal(8, @messenger_queue_length)
      assert_equal(1, @subprocess_user_call)
      assert_equal('httpd', @subprocess_user)
      assert_equal(1, @subprocess_group_call)
      assert_equal('httpd', @subprocess_group)
      assert_equal(1, @add_logger_call)
      assert_equal(1, @close_hook_call)
    end

    def test_admin_params
      build_control
      admin_params = @control.admin_params
      assert_equal(1, @params_call)
      assert_equal(%w[ admin_password admin_user japanese_handling localhost_only ], @params_last_names.sort)
      assert_equal('admin', admin_params['admin_user'])
      assert_equal('', admin_params['admin_password'])
      assert_equal(true, admin_params['localhost_only'])
      assert_equal(false, admin_params['japanese_handling'])
    end

    def test_set_admin_params
      build_control
      admin_params = @control.admin_params
      assert_equal(1, @params_call)
      assert_equal(%w[ admin_password admin_user japanese_handling localhost_only ], @params_last_names.sort)
      assert_equal('admin', admin_params['admin_user'])
      assert_equal('', admin_params['admin_password'])
      assert_equal(true, admin_params['localhost_only'])
      admin_params['localhost_only'] = false
      @control.set_admin_params(admin_params)
      assert_equal(1, @set_params_call)
      assert_equal(%w[ admin_password admin_user japanese_handling localhost_only ], @set_params_last_names.sort)
      assert_equal('admin', @properties['admin_user'])
      assert_equal('', @properties['admin_password'])
      assert_equal(true, @properties['localhost_only'])
      admin_params['admin_password'] = @pw_enc.crypt('nazo')
      @control.set_admin_params(admin_params)
      assert_equal(2, @set_params_call)
      assert_equal(%w[ admin_password admin_user japanese_handling localhost_only ], @set_params_last_names.sort)
      assert_equal('admin', @properties['admin_user'])
      assert(@properties['admin_password'] != 'nazo')
      assert(@properties['admin_password'] == @pw_enc.crypt('nazo', @properties['admin_password']))
      assert_equal(true, @properties['localhost_only'])
      admin_params['localhost_only'] = false
      @control.set_admin_params(admin_params)
      assert_equal(3, @set_params_call)
      assert_equal(%w[ admin_password admin_user japanese_handling localhost_only ], @set_params_last_names.sort)
      assert_equal('admin', @properties['admin_user'])
      assert(@properties['admin_password'] != 'nazo')
      assert(@properties['admin_password'] == @pw_enc.crypt('nazo', @properties['admin_password']))
      assert_equal(false, @properties['localhost_only'])
    end

    def test_ControlPanel_document
      build_control
      @control.server_setup(self)
      assert_equal(1, @mount_call)
      assert_equal(1, @attach_call)
      assert_equal(0, @virtual_mount_call)
      assert_equal(0, @virtual_attach_call)
      document = @folder.find('/control')[0]
      assert_instance_of(Rucy::FilterDocument, document)
      assert_instance_of(Rucy::RemoteAddressAllowAccessFilter, document.filter)
      assert_instance_of(Rucy::WebPageMakerDocument, document.document)
      assert_equal('/control', @folder.find('/control')[1])
    end

    def test_ControlPanel_with_BasicAuth
      @properties['admin_user'] = 'admin'
      @properties['admin_password'] = @pw_enc.crypt('nazo')
      @properties['localhost_only'] = false
      build_control
      @control.server_setup(self)
      assert_equal(1, @mount_call)
      assert_equal(1, @attach_call)
      assert_equal(0, @virtual_mount_call)
      assert_equal(0, @virtual_attach_call)
      document = @folder.find('/control')[0]
      assert_instance_of(Rucy::FilterDocument, document)
      assert_instance_of(Rucy::BasicAuth, document.filter)
      assert_instance_of(Rucy::WebPageMakerDocument, document.document)
      assert_equal('/control', @folder.find('/control')[1])
    end

    def test_ControlPanel_with_BasicAuth_and_LocalhostOnly
      @properties['admin_user'] = 'admin'
      @properties['admin_password'] = @pw_enc.crypt('nazo')
      @properties['localhost_only'] = true
      build_control
      @control.server_setup(self)
      assert_equal(1, @mount_call)
      assert_equal(2, @attach_call)
      assert_equal(0, @virtual_mount_call)
      assert_equal(0, @virtual_attach_call)
      document = @folder.find('/control')[0]
      assert_instance_of(Rucy::FilterDocument, document)
      assert_instance_of(Rucy::BasicAuth, document.filter)
      document = document.document
      assert_instance_of(Rucy::FilterDocument, document)
      assert_instance_of(Rucy::RemoteAddressAllowAccessFilter, document.filter)
      assert_instance_of(Rucy::WebPageMakerDocument, document.document)
      assert_equal('/control', @folder.find('/control')[1])
    end

    def test_logging_setup
      @properties['stdout_emerg'] = true
      @properties['stdout_alert'] = true
      @properties['stdout_crit'] = true
      @properties['stdout_err'] = true
      @properties['stdout_warning'] = true
      @properties['stdout_notice'] = true
      @properties['stdout_info'] = true
      @properties['stdout_debug'] = true
      @properties['logfiles'] = [
	{ 'path' => 'server.log',
	  'log_emerg' => true,
	  'log_alert' => true,
	  'log_crit' => true,
	  'log_err' => true,
	  'log_warning' => true,
	  'log_notice' => true,
	  'log_info' => false,
	  'log_debug' => false
	},
	{ 'path' => 'access.log',
	  'log_emerg' => false,
	  'log_alert' => false,
	  'log_crit' => false,
	  'log_err' => false,
	  'log_warning' => false,
	  'log_notice' => false,
	  'log_info' => true,
	  'log_debug' => false
	},
	{ 'path' => 'debug.log',
	  'log_emerg' => false,
	  'log_alert' => false,
	  'log_crit' => false,
	  'log_err' => false,
	  'log_warning' => false,
	  'log_notice' => false,
	  'log_info' => false,
	  'log_debug' => true
	}
      ]

      begin
	build_control
	@control.server_setup(self)
	assert_equal(4, @add_logger_call)
	assert((File.file? 'server.log'))
	assert((File.file? 'access.log'))
	assert((File.file? 'debug.log'))
	@control.logfiles_close
      ensure
	[ 'server.log',
	  'access.log',
	  'debug.log'
	].each do |filename|
	  if (File.exist? filename) then
	    File.delete(filename)
	  end
	end
      end
    end

    def test_logging_errors
      @properties['logfiles'] = [
	{ 'path' => 'detarame/server.log',
	  'log_emerg' => true,
	  'log_alert' => true,
	  'log_crit' => true,
	  'log_err' => true,
	  'log_warning' => true,
	  'log_notice' => true,
	  'log_info' => false,
	  'log_debug' => false
	}
      ]
      build_control
      @control.server_setup(self)
      assert_equal(1, @add_logger_call)
      assert_equal(1, @control.logging_errors.length)
      assert_equal('detarame/server.log', @control.logging_errors[0][:logfile])
      assert_instance_of(Errno::ENOENT, @control.logging_errors[0][:exception])
    end

    def test_alias_setup
      @properties['aliases'] = [
	{ 'virtual_host' => nil,
	  'alias_path' => '/bar',
	  'orig_path' => '/foo'
	},
	{ 'virtual_host' => 'foo',
	  'alias_path' => '/fuge',
	  'orig_path' => '/hoge'
	}
      ]
      build_control
      @control.server_setup(self)
      assert_equal(1, @set_alias_call)
      assert_equal([ [ '/bar', '/foo' ] ], @set_alias_alist)
      assert_equal(1, @set_virtual_alias_call)
      assert_equal([ [ 'foo:8888', '/fuge', '/hoge' ] ], @set_virtual_alias_alist)
    end

    def test_alias_setup_error
      @properties['aliases'] = [
	{ 'virtual_host' => nil,
	  'alias_path' => 'bar',
	  'orig_path' => 'foo'
	}
      ]
      build_control
      @control.server_setup(self)
      assert_equal(1, @set_alias_call)
      assert_equal(1, @control.alias_errors.length)
      assert_equal(nil, @control.alias_errors[0][:virtual_host])
      assert_equal('bar', @control.alias_errors[0][:alias_path])
      assert_equal('foo', @control.alias_errors[0][:orig_path])
      assert_instance_of(RuntimeError, @control.alias_errors[0][:exception])
    end

    def test_document_setup
      @properties['documents'] = [
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello world.\n",
	    'text/html'
	  ],
	  'mount_path' => '/hello',
	  'mount_mask' => nil,
	  'virtual_host' => nil
	},
	{ 'document' => 'LocalFile',
	  'arguments' => [
	    '/home/alice/public_html'
	  ],
	  'mount_path' => '/~alice',
	  'mount_mask' => nil,
	  'virtual_host' => 'foo'
	}
      ]

      page = Object.new
      class << page
	include RUNIT::Assert

	def doc_name
	  'Page'
	end

	def doc_args
	  [ [ 'content', :text, nil ],
	    [ 'content-type', :string, 'text/html' ]
	  ]
	end

	def new(content, content_type)
	  assert_equal("Hello world.\n", content)
	  assert_equal('text/html', content_type)
	  :dummy_page_document
	end
      end
      @factory.add_document(page)

      local = Object.new
      class << local
	include RUNIT::Assert

	def doc_name
	  'LocalFile'
	end

	def doc_args
	  [ [ 'local path', :string, nil ] ]
	end

	def new(local_path)
	  assert_equal('/home/alice/public_html', local_path)
	  :dummy_local_document
	end
      end
      @factory.add_document(local)

      build_control
      @control.server_setup(self)
      assert_equal(2, @mount_call)
      assert_equal(1, @virtual_mount_call)
      assert_equal(:dummy_page_document, @folder.find('/hello')[0])
      assert_equal('/hello', @folder.find('/hello')[1])
      assert_equal(:dummy_local_document, @folder.virtual_find('foo:8888', '/~alice')[0])
      assert_equal('/~alice', @folder.virtual_find('foo:8888', '/~alice')[1])
    end

    def test_document_setup_error
      error = Object.new
      def error.doc_name
	'TestError'
      end
      def error.doc_args
	[]
      end
      def error.new
	raise 'exception on creating a document.'
      end
      @factory.add_document(error)

      @properties['documents'] = [
	{ 'document' => 'TestError',
	  'arguments' => [],
	  'mount_path' => '/error',
	  'mount_mask' => nil,
	  'virtual_host' => 'foo'
	}
      ]

      build_control
      @control.server_setup(self)
      assert_equal(1, @mount_call)
      assert_equal(0, @virtual_mount_call)
      assert_nil(@folder.find('/error'))
      assert_equal(1, @control.doc_errors.length)
      assert_equal('TestError', @control.doc_errors[0][:document])
      assert_equal([], @control.doc_errors[0][:arguments])
      assert_equal('/error', @control.doc_errors[0][:mount_path])
      assert_nil(@control.doc_errors[0][:mount_mask])
      assert_equal('foo', @control.doc_errors[0][:virtual_host])
      assert_instance_of(RuntimeError, @control.doc_errors[0][:exception])
      assert_equal('exception on creating a document.', @control.doc_errors[0][:exception].message)
    end

    def test_filter_setup
      @properties['documents'] = [
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello world.\n",
	    'text/html'
	  ],
	  'mount_path' => '/',
	  'mount_mask' => nil,
	  'virtual_host' => nil
	},
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello virtual world.\n",
	    'text/html'
	  ],
	  'mount_path' => '/',
	  'mount_mask' => nil,
	  'virtual_host' => 'baz'
	}
      ]

      page = Object.new
      class << page
	include RUNIT::Assert

	def doc_name
	  'Page'
	end

	def doc_args
	  [ [ 'content', :text, nil ],
	    [ 'content-type', :string, 'text/html' ]
	  ]
	end

	def new(content, content_type)
	  assert(content == "Hello world.\n" ||
		 content == "Hello virtual world.\n")
	  assert_equal('text/html', content_type)
	  :dummy_page_document
	end
      end
      @factory.add_document(page)

      @properties['filters'] = [
	{ 'filter' => 'ERB',
	  'arguments' => [
	    1
	  ],
	  'attach_path' => '/foo',
	  'attach_mask' => %r"\.rhtml?($|/)",
	  'virtual_host' => nil
	},
	{ 'filter' => 'SSI',
	  'arguments' => [
	    { 'config' => true,
	      'include' => true,
	      'echo' => true,
	      'fsize' => true,
	      'flastmod' => true,
	      'exec' => true
	    }
	  ],
	  'attach_path' => '/bar',
	  'attach_mask' => %r"\.shtml?($|/)",
	  'virtual_host' => 'baz'
	}
      ]

      erb = Object.new
      class << erb
	include RUNIT::Assert

	def filter_name
	  'ERB'
	end

	def filter_args
	  [ [ 'safe level', :number, 1 ] ]
	end

	def new(safe_level)
	  assert_equal(1, safe_level)
	  :dummy_ERB_document
	end
      end
      @factory.add_filter(erb)

      ssi = Object.new
      class << ssi
	include RUNIT::Assert

	def filter_name
	  'SSI'
	end

	def filter_args
	  [ [ 'SSI commands', :checkset,
	      [ [ 'config',   true ],
		[ 'include',  true ],
		[ 'echo',     true ],
		[ 'fsize',    true ],
		[ 'flastmod', true ],
		[ 'exec',     true ]
	      ]
	    ]
	  ]
	end

	def new(cmds)
	  assert_equal({ 'config'   => true,
			 'include'  => true,
			 'echo'     => true,
			 'fsize'    => true,
			 'flastmod' => true,
			 'exec'     => true
		       }, cmds)
	  :dummy_SSI_document
	end
      end
      @factory.add_filter(ssi)

      build_control
      @control.server_setup(self)
      assert_equal(2, @attach_call)
      assert_equal(1, @virtual_attach_call)
      assert_equal(:dummy_ERB_document, @folder.find('/foo/index.rhtml')[0].filter)
      assert_equal('/foo', @folder.find('/foo/index.rhtml')[1])
      assert_equal(:dummy_SSI_document, @folder.virtual_find('baz:8888', '/bar/index.shtml')[0].filter)
      assert_equal('/bar', @folder.virtual_find('baz:8888', '/bar/index.shtml')[1])
    end

    def test_filter_setup_error
      @properties['documents'] = [
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello world.\n",
	    'text/html'
	  ],
	  'mount_path' => '/',
	  'mount_mask' => nil,
	  'virtual_host' => nil
	},
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello virtual world.\n",
	    'text/html'
	  ],
	  'mount_path' => '/',
	  'mount_mask' => nil,
	  'virtual_host' => 'baz'
	}
      ]

      page = Object.new
      class << page
	include RUNIT::Assert

	def doc_name
	  'Page'
	end

	def doc_args
	  [ [ 'content', :text, nil ],
	    [ 'content-type', :string, 'text/html' ]
	  ]
	end

	def new(content, content_type)
	  assert(content == "Hello world.\n" ||
		 content == "Hello virtual world.\n")
	  assert_equal('text/html', content_type)
	  :dummy_page_document
	end
      end
      @factory.add_document(page)

      error = Object.new
      def error.filter_name
	'TestError'
      end
      def error.filter_args
	[]
      end
      def error.new
	raise 'exception on creating a filter.'
      end
      @factory.add_filter(error)

      @properties['filters'] = [
	{ 'filter' => 'TestError',
	  'arguments' => [],
	  'attach_path' => '/error',
	  'attach_mask' => nil,
	  'virtual_host' => 'foo'
	}
      ]

      build_control
      @control.server_setup(self)
      assert_equal(1, @attach_call)
      assert_equal(0, @virtual_attach_call)
      assert_equal(1, @control.filter_errors.length)
      assert_equal('TestError', @control.filter_errors[0][:filter])
      assert_equal([], @control.filter_errors[0][:arguments])
      assert_equal('/error', @control.filter_errors[0][:attach_path])
      assert_nil(@control.filter_errors[0][:attach_mask])
      assert_equal('foo', @control.filter_errors[0][:virtual_host])
      assert_instance_of(RuntimeError, @control.filter_errors[0][:exception])
      assert_equal('exception on creating a filter.', @control.filter_errors[0][:exception].message)
    end
  end
end
