# $Id: test_cgi.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

=begin
Copyright (c) 2001-2004
	TOKI Yoshinori <toki@freedom.ne.jp>. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  1.  Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.

  2.  Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.

  3.  The name of the author may not be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
=end

require 'rubyunit'
require 'pseudo_req_res'
require 'pseudo_io'
require 'rucy/request'
require 'rucy/response'
require 'rucy/cgi'

module TestRucy
  module TestCGI
    CGI_DIR = 'cgi'
    TEST_CGI = 'cgi/test.cgi'

    def setup
      super
      Dir.mkdir(CGI_DIR)
    end

    def make_cgi
      File.open(TEST_CGI, 'w') { |output|
	output.binmode
	yield(output)
      }
      nil
    end
    private :make_cgi

    def teardown
      File.delete(TEST_CGI) if (File.exist? TEST_CGI)
      Dir.delete(CGI_DIR)
    end
  end

  class TestCGIDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include TestCGI

    def setup
      super
      @cgi = Rucy::CGIDocument.new(CGI_DIR, 'ruby')
    end

    def test_CGIDocument_quote_sh
      assert_equal('"Hello world."', Rucy::CGIDocument.quote_sh('Hello world.'))
      assert_equal('"\\""', Rucy::CGIDocument.quote_sh('"'))
      assert_equal('"\\$"', Rucy::CGIDocument.quote_sh('$'))
      assert_equal('"\\`"', Rucy::CGIDocument.quote_sh('`'))
      assert_equal('"\\\\"', Rucy::CGIDocument.quote_sh('\\'))
    end

    def test_cgi
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print \"Hello world.\\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end

    def test_cgi_status
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Status: 201 Created\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print \"Hello world.\\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(201, @messg_head.status)
      assert_equal('Created', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end

    def test_cgi_location
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Location: /foo/bar\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print \"Hello world.\\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('http://server:8080/foo/bar', @messg_head.header('Location'))
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end

    def test_cgi_args
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "for arg in ARGV\n"
	output.print "  print arg, \"\n\"\n"
	output.print "end\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi?foo+bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('', @messg_body)
    end

    def test_cgi_args2
      @cgi.pass_args = true
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "for arg in ARGV\n"
	output.print "  print arg, \"\n\"\n"
	output.print "end\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi?foo+bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("foo\nbar\n", @messg_body)
    end

    def test_cgi_auth
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['HTTP_AUTHORIZATION'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Authorization', 'Basic YWxpY2U6Zm9v')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("\n", @messg_body)
    end

    def test_cgi_auth2
      @cgi.pass_auth = true
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['HTTP_AUTHORIZATION'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Authorization', 'Basic YWxpY2U6Zm9v')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Basic YWxpY2U6Zm9v\n", @messg_body)
    end

    def test_cgi_path_info
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['PATH_INFO'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi/foo/bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("/foo/bar\n", @messg_body)
    end

    def test_cgi_env
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: application/octet-stream\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "Marshal.dump(ENV.to_hash, STDOUT)\n"
      }
      @cgi.env['FOO'] = 'Alice'
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('User-Agent', 'TestRucy')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('application/octet-stream', @messg_head.header('Content-Type'))
      env = Marshal.load(@messg_body)
      assert(! (ENV.include? 'FOO'))
      assert(! (ENV.include? 'HTTP_USER_AGENT'))
      assert((env.include? 'FOO'))
      assert_equal('Alice', env['FOO'])
      assert((env.include? 'HTTP_USER_AGENT'))
      assert_equal('TestRucy', env['HTTP_USER_AGENT'])
    end

    def test_cgi_post
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print STDIN.read\n"
      }
      @request.method = 'POST'
      @request.uri = 
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      messg = "Hello world.\n"
      @request.set_header('Content-Type', 'text/plain')
      @request.set_header('Content-Length', messg.length.to_s)
      post_reader = PseudoIO.new
      post_reader.write(messg)
      @request.set_reader(post_reader)
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end
  end

  class TestNonParsedHeaderCGIDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include TestCGI

    def setup
      super
      @cgi = Rucy::CGIDocument.new('cgi', 'ruby', true)
    end

    def test_cgi
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print \"Hello world.\\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end

    def test_cgi_location
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 302 Found\\r\\n\"\n"
	output.print "print \"Location: /foo/bar\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print \"Hello world.\\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(302, @messg_head.status)
      assert_equal('Found', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('/foo/bar', @messg_head.header('Location')) # not converted to absolute-URI
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end

    def test_cgi_args
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "for arg in ARGV\n"
	output.print "  print arg, \"\n\"\n"
	output.print "end\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi?foo+bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('', @messg_body)
    end

    def test_cgi_args2
      @cgi.pass_args = true
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "for arg in ARGV\n"
	output.print "  print arg, \"\n\"\n"
	output.print "end\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi?foo+bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("foo\nbar\n", @messg_body)
    end

    def test_cgi_auth
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['HTTP_AUTHORIZATION'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Authorization', 'Basic YWxpY2U6Zm9v')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("\n", @messg_body)
    end

    def test_cgi_auth2
      @cgi.pass_auth = true
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['HTTP_AUTHORIZATION'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Authorization', 'Basic YWxpY2U6Zm9v')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Basic YWxpY2U6Zm9v\n", @messg_body)
    end

    def test_cgi_path_info
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print ENV['PATH_INFO'].to_s, \"\n\"\n"
      }
      @request.method = 'GET'
      @request.uri = 
      @request.uri = '/test.cgi/foo/bar'
      @request.version = 'HTTP/1.1'
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("/foo/bar\n", @messg_body)
    end

    def test_cgi_env
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: application/octet-stream\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "Marshal.dump(ENV.to_hash, STDOUT)\n"
      }
      @cgi.env['FOO'] = 'Alice'
      @request.method = 'GET'
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('User-Agent', 'TestRucy')
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('application/octet-stream', @messg_head.header('Content-Type'))
      env = Marshal.load(@messg_body)
      assert(! (ENV.include? 'FOO'))
      assert(! (ENV.include? 'HTTP_USER_AGENT'))
      assert((env.include? 'FOO'))
      assert_equal('Alice', env['FOO'])
      assert((env.include? 'HTTP_USER_AGENT'))
      assert_equal('TestRucy', env['HTTP_USER_AGENT'])
    end

    def test_cgi_post
      make_cgi{ |output|
	output.print "#!/usr/local/bin/ruby\n"
	output.print "STDOUT.binmode\n"
	output.print "print \"HTTP/1.1 200 OK\\r\\n\"\n"
	output.print "print \"Content-Type: text/plain\\r\\n\"\n"
	output.print "print \"\\r\\n\"\n"
	output.print "print STDIN.read\n"
      }
      @request.method = 'POST'
      @request.uri = 
      @request.uri = '/test.cgi'
      @request.version = 'HTTP/1.1'
      messg = "Hello world.\n"
      @request.set_header('Content-Type', 'text/plain')
      @request.set_header('Content-Length', messg.length.to_s)
      post_reader = PseudoIO.new
      post_reader.write(messg)
      @request.set_reader(post_reader)
      @cgi.publish('', @request, @response, self)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('/test.cgi', @messg_head.doc_path)
      assert_match(@messg_head.local_path, /test\.cgi$/)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal("Hello world.\n", @messg_body)
    end
  end
end
