#!/usr/local/bin/ruby
# $Id: test_SetupDocument.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

require 'rubyunit'
require 'wpm'
require 'rucy/document'

module TestRucy
  class TestSetupDocument < RUNIT::TestCase
    def setup
      # for WPM::Driver class
      @driver_call = 0
      @params_call = 0
      @redirect_call = 0
      @redirect_last_page_name = nil
      @redirect_last_query_params = nil

      # for Rucy::DocumentFactory class
      @factory = Rucy::DocumentFactory.new
      page = Object.new
      def page.doc_name
	'Page'
      end
      def page.doc_args
	[ [ 'content', :text, nil ],
	  [ 'content-type', :string, 'text/html' ]
	]
      end
      @factory.add_document(page)
      local = Object.new
      def local.doc_name
	'LocalFile'
      end
      def local.doc_args
	[ [ 'local path', :string, nil ] ]
      end
      @factory.add_document(local)
      test_args = Object.new
      def test_args.doc_name
	'TestArguments'
      end
      def test_args.doc_args
	[ [ 'string', :string, nil ],
	  [ 'text',   :text,   nil ],
	  [ 'number', :number, nil ],
	  [ 'bool',   :bool,   false ],
	  [ 'select', :select, %w[ foo bar baz ] ],
	  [ 'checkset', :checkset, [
	      [ 'foo', false ],
	      [ 'bar', false ],
	      [ 'baz', false ]
	    ]
	  ],
	  [ 'radio', :radio, %w[ foo bar baz ] ]
	]
      end
      @factory.add_document(test_args)
      @doc_args_call = 0
      @doc_args_last_name = nil

      # for Rucy::Control class
      @modified_count_call = 0
      @set_modified_count_call = 0
      @set_modified_count_value = nil
      @document_list = [
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello world.\n",
	    'text/plain'
	  ],
	  'mount_path' => '/hello',
	  'mount_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => 'test of Page'
	},
	{ 'document' => 'LocalFile',
	  'arguments' => [
	    '/home/alice/public_html'
	  ],
	  'mount_path' => '/~alice',
	  'mount_mask' => nil,
	  'virtual_host' => 'foo',
	  'comment' => 'test of LocalFile'
	},
	{ 'document' => 'TestArguments',
	  'arguments' => [
	    'test_string',	# string
	    'test_text',	# text
	    0,			# number
	    true,		# bool
	    'bar',		# select
	    { 'foo' => true,	# checkset
	      'bar' => false,
	      'baz' => false
	    },
	    'baz'		# radio
	  ],
	  'mount_path' => '/args',
	  'mount_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => 'test of Arguments'
	}
      ]
      @documents_call = 0
      @set_documents_call = 0

      # setup WPM::PageContext class
      loader = WPM::Loader.new('../control/SetupDocument/SetupDocument.rb')
      @SetupDocument = loader.const_get('SetupDocument').new(self, :control => self, :factory => self)
      assert((@SetupDocument.is_a? WPM::PageContext))
      @SetupDocument.init_context
    end

    def driver
      @driver_call += 1
      self
    end

    def params
      @params_call += 1
      { 'doc' => 'LocalFile', 'pos' => '1' }
    end

    def redirect(page_name, query_params=nil)
      @redirect_call += 1
      @redirect_last_page_name = page_name
      @redirect_last_query_params = query_params
      nil
    end

    def doc_args(doc_name)
      @doc_args_call += 1
      @doc_args_last_name = doc_name
      @factory.doc_args(doc_name)
    end

    def modified_count
      @modified_count_call += 1
      0
    end

    def modified_count=(modified_count)
      @set_modified_count_call += 1
      @set_modified_count_value = modified_count
    end

    def documents
      @documents_call += 1
      document_list = @document_list.dup
      def document_list.setup(parent)
	@parent = parent
	self
      end
      def document_list.modified_count
	@parent.modified_count
      end
      def document_list.modified_count=(modified_count)
	@parent.modified_count = modified_count
      end
      document_list.setup(self)
    end

    def set_documents(document_list)
      @set_documents_call += 1
      @document_list = document_list
      nil
    end

    def test_reload_query
      assert_equal(1, @params_call)
      assert_equal({ 'doc' => 'LocalFile', 'pos' => '1' }, @SetupDocument.reload_query)
    end

    def test_default_path
      assert_nil(@SetupDocument.default_path)
    end

    def test_default_mask_type
      assert_nil(@SetupDocument.default_mask_type)
    end

    def test_default_mask
      assert_nil(@SetupDocument.default_mask)
    end

    def test_default_virtual_host
      assert_nil(@SetupDocument.default_virtual_host)
    end

    def test_default_comment
      assert_nil(@SetupDocument.default_comment)
    end

    def test_document_factory
      assert_equal([ [ 'content', :text, nil ],
		     [ 'content-type', :string, 'text/html' ]
		   ], @SetupDocument.doc_args('Page'))
      assert_equal(1, @doc_args_call)
      assert_equal('Page', @doc_args_last_name)
    end

    def test_document
      assert_equal(1, @params_call)
      assert_equal('1', @SetupDocument.doc_pos)
      assert_equal('LocalFile', @SetupDocument.doc_type)
    end

    def test_document_attributes
      assert_equal('document', @SetupDocument.doc_tag)
      assert_equal('document', @SetupDocument.doc_label)
      assert_equal('mount_path', @SetupDocument.path_tag)
      assert_equal('mount path', @SetupDocument.path_label)
      assert_equal('mount_mask', @SetupDocument.mask_tag)
      assert_equal('mount mask', @SetupDocument.mask_label)
    end

    def test_modified_count
      assert_equal(0, @SetupDocument.modified_count)
      assert_equal(1, @modified_count_call)
      assert_equal(1, @SetupDocument.modified_count = 1)
      assert_equal(1, @set_modified_count_call)
      assert_equal(1, @set_modified_count_value)
    end

    def test_document_list
      document_list = @SetupDocument.document_list
      assert_equal(1, @documents_call)
      assert_equal(3, document_list.length)
      # 1
      assert_equal('Page', document_list[0]['document'])
      assert_equal('/hello', document_list[0]['mount_path'])
      assert_nil(document_list[0]['mount_mask'])
      assert_nil(document_list[0]['virtual_host'])
      assert_equal('test of Page', document_list[0]['comment'])
      assert_equal(2, document_list[0]['arguments'].length)
      assert_equal("Hello world.\n", document_list[0]['arguments'][0])
      assert_equal('text/plain', document_list[0]['arguments'][1])
      # 2
      assert_equal('LocalFile', document_list[1]['document'])
      assert_equal('/~alice', document_list[1]['mount_path'])
      assert_nil(document_list[1]['mount_mask'])
      assert_equal('foo', document_list[1]['virtual_host'])
      assert_equal('test of LocalFile', document_list[1]['comment'])
      assert_equal(1, document_list[1]['arguments'].length)
      assert_equal('/home/alice/public_html', document_list[1]['arguments'][0])
      # 3
      assert_equal('TestArguments', document_list[2]['document'])
      assert_equal('/args', document_list[2]['mount_path'])
      assert_nil(document_list[2]['mount_mask'])
      assert_nil(document_list[2]['virtual_host'])
      assert_equal('test of Arguments', document_list[2]['comment'])
      assert_equal(7, document_list[2]['arguments'].length)
      assert_equal('test_string', document_list[2]['arguments'][0])
      assert_equal('test_text', document_list[2]['arguments'][1])
      assert_equal(0, document_list[2]['arguments'][2])
      assert_equal(true, document_list[2]['arguments'][3])
      assert_equal('bar', document_list[2]['arguments'][4])
      assert_equal({ 'foo' => true,
		     'bar' => false,
		     'baz' => false
		   }, document_list[2]['arguments'][5])
      assert_equal('baz', document_list[2]['arguments'][6])
    end

    def test_write_document_list
      @SetupDocument.write_document_list
      assert_equal(1, @set_documents_call)
      assert_equal(1, @driver_call)
      assert_equal(1, @redirect_call)
      assert_equal('MountMap', @redirect_last_page_name)
      assert_nil(@redirect_last_query_params)
    end

    def test_cancel
      @SetupDocument.cancel
      assert_equal(0, @set_documents_call)
      assert_equal(1, @driver_call)
      assert_equal(1, @redirect_call)
      assert_equal('MountMap', @redirect_last_page_name)
      assert_nil(@redirect_last_query_params)
    end
  end
end
