#!/usr/local/bin/ruby
# $Id: test_EditProperties.rb,v 1.4 2004/04/21 15:09:31 toki Exp $

require 'rubyunit'
require 'wpm'
require 'rucy/passwd'
require 'rucy/document'

module TestRucy
  class TestEditProperties < RUNIT::TestCase
    def setup
      # for Rucy::PasswordEncryptor class
      @pw_enc = Rucy::PasswordEncryptor.new
      @crypt_call = 0

      @factory = Rucy::DocumentFactory.new
      page = Object.new
      def page.doc_name
	'Page'
      end
      def page.doc_args
	[ [ 'content', :text, nil ],
	  [ 'content-type', :string, 'text/html' ]
	]
      end
      @factory.add_document(page)
      local = Object.new
      def local.doc_name
	'LocalFile'
      end
      def local.doc_args
	[ [ 'local path', :string, nil ] ]
      end
      @factory.add_document(local)
      test_args = Object.new
      def test_args.doc_name
	'TestArguments'
      end
      def test_args.doc_args
	[ [ 'string', :string, nil ],
	  [ 'text',   :text,   nil ],
	  [ 'number', :number, nil ],
	  [ 'bool',   :bool,   false ],
	  [ 'select', :select, %w[ foo bar baz ] ],
	  [ 'checkset', :checkset, [
	      [ 'foo', false ],
	      [ 'bar', false ],
	      [ 'baz', false ]
	    ]
	  ],
	  [ 'radio', :radio, %w[ foo bar baz ] ],
	  [ 'password', :password, nil ],
	  [ 'regexp', :regexp, nil ]
	]
      end
      @factory.add_document(test_args)

      # for SetupDocument class
      @default_path_call = 0
      @default_path = nil
      @default_mask_type_call = 0
      @default_mask_type = nil
      @default_mask_call = 0
      @default_mask = nil
      @default_virtual_host_call = 0
      @default_virtual_host = nil
      @default_comment_call = 0
      @default_comment = nil
      @doc_tag_call = 0
      @doc_label_call = 0
      @path_tag_call = 0
      @path_label_call = 0
      @mask_tag_call = 0
      @mask_label_call = 0
      @modified_count_call = 0
      @set_modified_count_call = 0
      @set_modified_count_last_value = nil
      @doc_pos_call = 0
      @doc_type_call = 0
      @document_list_call = 0
      @document_list = [
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello world.\n",
	    'text/plain'
	  ],
	  'mount_path' => '/hello',
	  'mount_mask' => 'index.html',
	  'virtual_host' => nil,
	  'comment' => 'test of Page'
	},
	{ 'document' => 'LocalFile',
	  'arguments' => [
	    '/home/alice/public_html'
	  ],
	  'mount_path' => '/~alice',
	  'mount_mask' => nil,
	  'virtual_host' => 'foo',
	  'comment' => 'test of LocalFile'
	},
	{ 'document' => 'TestArguments',
	  'arguments' => [
	    'test_string',	# string
	    'test_text',	# text
	    0,			# number
	    true,		# bool
	    'bar',		# select
	    { 'foo' => true,	# checkset
	      'bar' => false,
	      'baz' => false
	    },
	    'baz',		# radio
	    'himitsu',		# password
	    /foo/		# regexp
	  ],
	  'mount_path' => '/args',
	  'mount_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => 'test of Arguments'
	},
      ]
      @doc_args_call = 0
      @write_document_list_call = 0
      @cancel_call = 0

      # setup WPM::PageContext class
      loader = WPM::Loader.new('../control/EditProperties/EditProperties.rb')
      @EditProperties = loader.const_get('EditProperties').new(self, :control => self, :pw_enc => self)
      assert((@EditProperties.is_a? WPM::PageContext))
      @EditProperties.init_context
      @EditProperties.properties_source = self
      assert_equal(1, @doc_pos_call)
      assert_equal(1, @doc_type_call)
      #assert_equal(1, @doc_tag_call)
    end

    def crypt(password, salt=nil)
      @crypt_call += 1
      args = [ password ]
      args << salt if salt
      @pw_enc.crypt(*args)
    end

    def modified_count
      @modified_count_call += 1
      0
    end

    def modified_count=(modified_count)
      @set_modified_count_call += 1
      @set_modified_count_last_value = modified_count
    end

    def default_path
      @default_path_call += 1
      @default_path
    end

    def default_mask_type
      @default_mask_type_call += 1
      @default_mask_type
    end

    def default_mask
      @default_mask_call += 1
      @default_mask
    end

    def default_virtual_host
      @default_virtual_host_call += 1
      @default_virtual_host
    end

    def default_comment
      @default_comment_call += 1
      @default_comment
    end

    def doc_pos
      @doc_pos_call += 1
      '1'
    end

    def doc_type
      @doc_type_call += 1
      'LocalFile'
    end

    def doc_tag
      @doc_tag_call += 1
      'document'
    end

    def doc_label
      @doc_label_call += 1
      'document'
    end

    def path_tag
      @path_tag_call += 1
      'mount_path'
    end

    def path_label
      @path_label_call += 1
      'mount path'
    end

    def mask_tag
      @mask_tag_call += 1
      'mount_mask'
    end

    def mask_label
      @mask_label_call += 1
      'mount mask'
    end

    def document_list
      @document_list_call += 1
      @document_list
    end

    def doc_args(doc_name)
      @doc_args_call += 1
      @factory.doc_args(doc_name)
    end

    def write_document_list
      @write_document_list_call += 1
      nil
    end

    def cancel
      @cancel_call += 1
      nil
    end

    def test_doc_label
      assert_equal('document', @EditProperties.doc_label)
      assert_equal(1, @doc_label_call)
    end

    def test_path_label
      assert_equal('mount path', @EditProperties.path_label)
      assert_equal(1, @path_label_call)
    end

    def test_mask_label
      assert_equal('mount mask', @EditProperties.mask_label)
      assert_equal(1, @mask_label_call)
    end

    def test_modified_count
      assert_equal('0', @EditProperties.modified_count)
      assert_equal(1, @modified_count_call)
      assert_equal('1', @EditProperties.modified_count = '1')
      assert_equal(1, @set_modified_count_call)
      assert_equal(1, @set_modified_count_last_value)
    end

    def test_document
      assert_equal('1', @EditProperties.document_position)
      assert_equal('LocalFile', @EditProperties.document_type)
      @EditProperties.document_position = '0'
      @EditProperties.document_type = 'Page'
      assert_equal('0', @EditProperties.document_position)
      assert_equal('Page', @EditProperties.document_type)
      @EditProperties.document_type = 'LocalFile'
      assert_equal('LocalFile', @EditProperties.document_type)
    end

    def test_doc_path
      assert_equal('1', @EditProperties.document_position)
      assert_equal('LocalFile', @EditProperties.document_type)
      assert_equal('/~alice', @EditProperties.doc_path)
      @EditProperties.doc_path = '/~bob'
      assert_equal('/~bob', @EditProperties.doc_path)
      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      assert_equal(3, @document_list.length)
      assert_equal('/hello', @document_list[0]['mount_path'])
      assert_equal('/~bob', @document_list[1]['mount_path'])
      assert_equal('/args', @document_list[2]['mount_path'])
    end

    def test_mount_mask_nil
      @EditProperties.document_position = '0'
      @EditProperties.document_type = 'Page'
      assert_equal('string', @EditProperties.doc_mask_type_selected)
      assert_equal('index.html', @EditProperties.doc_mask)
      @EditProperties.doc_mask_type_selected = 'nil'
      assert_equal('nil', @EditProperties.doc_mask_type_selected)
      assert_equal('index.html', @EditProperties.doc_mask)
      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      assert_equal(3, @document_list.length)
      assert_nil(@document_list[0]['mount_mask'])
      assert_nil(@document_list[1]['mount_mask'])
      assert_nil(@document_list[2]['mount_mask'])
    end

    def test_mount_mask_regexp
      assert_equal('1', @EditProperties.document_position)
      assert_equal('LocalFile', @EditProperties.document_type)
      assert_equal('nil', @EditProperties.doc_mask_type_selected)
      assert_nil(@EditProperties.doc_mask)
      @EditProperties.doc_mask_type_selected = 'regexp'
      assert_equal('regexp', @EditProperties.doc_mask_type_selected)
      @EditProperties.doc_mask = "\\.cgi"
      assert_equal("\\.cgi", @EditProperties.doc_mask)
      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      assert_equal(3, @document_list.length)
      assert_equal('index.html', @document_list[0]['mount_mask'])
      assert_equal(%r"(\.cgi)(/|$)", @document_list[1]['mount_mask'])
      assert_nil(@document_list[2]['mount_mask'])
    end

    def test_mount_mask_string
      assert_equal('1', @EditProperties.document_position)
      assert_equal('LocalFile', @EditProperties.document_type)
      assert_equal('nil', @EditProperties.doc_mask_type_selected)
      assert_nil(@EditProperties.doc_mask)
      @EditProperties.doc_mask_type_selected = 'string'
      assert_equal('string', @EditProperties.doc_mask_type_selected)
      @EditProperties.doc_mask = 'index.htm'
      assert_equal('index.htm', @EditProperties.doc_mask)
      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      assert_equal(3, @document_list.length)
      assert_equal('index.html', @document_list[0]['mount_mask'])
      assert_equal('index.htm', @document_list[1]['mount_mask'])
      assert_nil(@document_list[2]['mount_mask'])
    end

    def test_mount_mask_string_empty
      assert_equal('1', @EditProperties.document_position)
      assert_equal('LocalFile', @EditProperties.document_type)
      assert_equal('nil', @EditProperties.doc_mask_type_selected)
      assert_nil(@EditProperties.doc_mask)
      @EditProperties.doc_mask_type_selected = 'string'
      assert_equal('string', @EditProperties.doc_mask_type_selected)
      @EditProperties.doc_mask = nil
      assert_nil(@EditProperties.doc_mask)
      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      assert_equal(3, @document_list.length)
      assert_equal('index.html', @document_list[0]['mount_mask'])
      assert_equal('', @document_list[1]['mount_mask'])
      assert_nil(@document_list[2]['mount_mask'])
    end

    def test_virtual_host
      assert_equal('1', @EditProperties.document_position)
      assert_equal('LocalFile', @EditProperties.document_type)
      assert_equal('foo', @EditProperties.virtual_host)
      @EditProperties.virtual_host = 'bar'
      assert_equal('bar', @EditProperties.virtual_host)
      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      assert_equal(3, @document_list.length)
      assert_nil(@document_list[0]['virtual_host'])
      assert_equal('bar', @document_list[1]['virtual_host'])
      assert_nil(@document_list[2]['virtual_host'])
    end

    def test_virtual_host_empty
      assert_equal('1', @EditProperties.document_position)
      assert_equal('LocalFile', @EditProperties.document_type)
      assert_equal('foo', @EditProperties.virtual_host)
      @EditProperties.virtual_host = ''
      assert_equal('', @EditProperties.virtual_host)
      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      assert_equal(3, @document_list.length)
      assert_nil(@document_list[0]['virtual_host'])
      assert_nil(@document_list[1]['virtual_host'])
      assert_nil(@document_list[2]['virtual_host'])
    end

    def test_comment
      assert_equal('1', @EditProperties.document_position)
      assert_equal('LocalFile', @EditProperties.document_type)
      assert_equal('test of LocalFile', @EditProperties.comment)
      @EditProperties.comment = 'Hello world.'
      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      assert_equal(3, @document_list.length)
      assert_equal('test of Page', @document_list[0]['comment'])
      assert_equal('Hello world.', @document_list[1]['comment'])
      assert_equal('test of Arguments', @document_list[2]['comment'])
    end

    def test_default_entry
      @default_path = '/'
      @default_mask_type = 'regexp'
      @default_mask = '\.html?'
      @default_virtual_host = 'bar'
      @default_comment = 'Hello world.'
      @EditProperties.document_position = '3'
      assert_equal(1, @default_path_call)
      assert_equal(1, @default_mask_type_call)
      assert_equal(1, @default_mask_call)
      assert_equal(1, @default_virtual_host_call)
      assert_equal(1, @default_comment_call)
      assert_equal('/', @EditProperties.doc_path)
      assert_equal('regexp', @EditProperties.doc_mask_type_selected)
      assert_equal('\.html?', @EditProperties.doc_mask)
      assert_equal('bar', @EditProperties.virtual_host)
      assert_equal('Hello world.', @EditProperties.comment)
    end

    def tset_arg_index
      @EditProperties.arg_index = 1
      assert_equal(1, @EditProperties.arg_index)
    end

    def test_document_arguments_string_argument
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[0]
      assert_equal('string', @EditProperties.arg_name)
      assert(@EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_string_call_index = 0
      assert_equal('test_string', @EditProperties.arg_string_value)
      @EditProperties.arg_string_value = 'alice'
      assert_equal('alice', @EditProperties.arg_string_value)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('alice', doc_args[0]) # changed!
      assert_equal('test_text', doc_args[1])
      assert_equal(0, doc_args[2])
      assert_equal(true, doc_args[3])
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('baz', doc_args[6])
      assert_equal('himitsu', doc_args[7])
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_text_argument
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[1]
      assert_equal('text', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(@EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_text_call_index = 0
      assert_equal('test_text', @EditProperties.arg_text_value)
      @EditProperties.arg_text_value = 'bob'
      assert_equal('bob', @EditProperties.arg_text_value)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('bob', doc_args[1]) # changed!
      assert_equal(0, doc_args[2])
      assert_equal(true, doc_args[3])
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('baz', doc_args[6])
      assert_equal('himitsu', doc_args[7])
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_number_argument
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[2]
      assert_equal('number', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(@EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_num_call_index = 0
      assert_equal('0', @EditProperties.arg_num_value)
      @EditProperties.arg_num_value = '1'
      assert_equal('1', @EditProperties.arg_num_value)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('test_text', doc_args[1])
      assert_equal(1, doc_args[2]) # changed!
      assert_equal(true, doc_args[3])
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('baz', doc_args[6])
      assert_equal('himitsu', doc_args[7])
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_bool_argument
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[3]
      assert_equal('bool', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(@EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_bool_call_index = 0
      assert_equal(true, @EditProperties.arg_bool_value)
      @EditProperties.arg_bool_value = false
      assert_equal(false, @EditProperties.arg_bool_value)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('test_text', doc_args[1])
      assert_equal(0, doc_args[2])
      assert_equal(false, doc_args[3]) # changed!
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('baz', doc_args[6])
      assert_equal('himitsu', doc_args[7])
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_select_argument
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[4]
      assert_equal('select', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(@EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_select_call_index = 0
      assert_equal(%w[ foo bar baz ], @EditProperties.arg_select_list)
      assert_equal('bar', @EditProperties.arg_select_value)
      @EditProperties.arg_select_value = 'baz'
      assert_equal('baz', @EditProperties.arg_select_value)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('test_text', doc_args[1])
      assert_equal(0, doc_args[2])
      assert_equal(true, doc_args[3])
      assert_equal('baz', doc_args[4]) # changed!
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('baz', doc_args[6])
      assert_equal('himitsu', doc_args[7])
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_checkset_argument
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[5]
      assert_equal('checkset', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(@EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_checkset_call_index = 0
      assert_equal(3, @EditProperties.arg_checkset_count)
      @EditProperties.arg_checkset_call_index = nil
      # foo
      @EditProperties.arg_check_call_index = 0
      assert_equal('foo', @EditProperties.arg_check_name)
      assert_equal(true, @EditProperties.arg_checked)
      @EditProperties.arg_checked = false
      assert_equal(false, @EditProperties.arg_checked)
      # bar
      @EditProperties.arg_check_call_index = 1
      assert_equal('bar', @EditProperties.arg_check_name)
      assert_equal(false, @EditProperties.arg_checked)
      @EditProperties.arg_checked = true
      assert_equal(true, @EditProperties.arg_checked)
      # baz
      @EditProperties.arg_check_call_index = 2
      assert_equal('baz', @EditProperties.arg_check_name)
      assert_equal(false, @EditProperties.arg_checked)
      @EditProperties.arg_checked = true
      assert_equal(true, @EditProperties.arg_checked)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('test_text', doc_args[1])
      assert_equal(0, doc_args[2])
      assert_equal(true, doc_args[3])
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => false, 'bar' => true, 'baz' => true }, doc_args[5]) # changed!
      assert_equal('baz', doc_args[6])
      assert_equal('himitsu', doc_args[7])
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_radio_argument
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[6]
      assert_equal('radio', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(@EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_radio_call_index = 0
      assert_equal(3, @EditProperties.arg_radio_count)
      @EditProperties.arg_radio_call_index = nil
      @EditProperties.arg_radio_item_call_index = 0
      assert_equal('RadioGroup0', @EditProperties.arg_radio_name)
      assert_equal('foo', @EditProperties.arg_radio_value)
      assert_equal('baz', @EditProperties.arg_radio_selected)
      @EditProperties.arg_radio_item_call_index = 1
      assert_equal('RadioGroup0', @EditProperties.arg_radio_name)
      assert_equal('bar', @EditProperties.arg_radio_value)
      assert_equal('baz', @EditProperties.arg_radio_selected)
      @EditProperties.arg_radio_item_call_index = 2
      assert_equal('RadioGroup0', @EditProperties.arg_radio_name)
      assert_equal('baz', @EditProperties.arg_radio_value)
      assert_equal('baz', @EditProperties.arg_radio_selected)
      @EditProperties.arg_radio_selected = 'bar'
      @EditProperties.arg_radio_item_call_index = 0
      assert_equal('bar', @EditProperties.arg_radio_selected)
      @EditProperties.arg_radio_item_call_index = 1
      assert_equal('bar', @EditProperties.arg_radio_selected)
      @EditProperties.arg_radio_item_call_index = 2
      assert_equal('bar', @EditProperties.arg_radio_selected)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('test_text', doc_args[1])
      assert_equal(0, doc_args[2])
      assert_equal(true, doc_args[3])
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('bar', doc_args[6]) # changed!
      assert_equal('himitsu', doc_args[7])
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_password_argument
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[7]
      assert_equal('password', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(@EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_password_call_index = 0
      assert_equal('himitsu', @EditProperties.arg_password_value)
      assert(! @EditProperties.arg_password_cleared)
      @EditProperties.arg_password_value = 'nazo'
      assert_equal(1, @crypt_call)
      assert(@EditProperties.arg_password_value != 'nazo')
      assert(@EditProperties.arg_password_value == @pw_enc.crypt('nazo', @EditProperties.arg_password_value))
      assert(! @EditProperties.arg_password_cleared)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('test_text', doc_args[1])
      assert_equal(0, doc_args[2])
      assert_equal(true, doc_args[3])
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('baz', doc_args[6])
      assert_equal(@pw_enc.crypt('nazo', doc_args[7]), doc_args[7]) # changed!
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_password_null
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[7]
      assert_equal('password', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(@EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_password_call_index = 0
      assert_equal('himitsu', @EditProperties.arg_password_value)
      assert(! @EditProperties.arg_password_cleared)
      @EditProperties.arg_password_value = nil
      assert_equal(0, @crypt_call)
      assert_equal('himitsu', @EditProperties.arg_password_value)
      assert(! @EditProperties.arg_password_cleared)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('test_text', doc_args[1])
      assert_equal(0, doc_args[2])
      assert_equal(true, doc_args[3])
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('baz', doc_args[6])
      assert_equal('himitsu', doc_args[7]) # not changed!
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_password_null2
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[7]
      assert_equal('password', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(@EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_password_call_index = 0
      assert_equal('himitsu', @EditProperties.arg_password_value)
      assert(! @EditProperties.arg_password_cleared)
      @EditProperties.arg_password_value = ''
      assert_equal(0, @crypt_call)
      assert_equal('himitsu', @EditProperties.arg_password_value)
      assert(! @EditProperties.arg_password_cleared)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('test_text', doc_args[1])
      assert_equal(0, doc_args[2])
      assert_equal(true, doc_args[3])
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('baz', doc_args[6])
      assert_equal('himitsu', doc_args[7]) # not changed!
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_password_clear
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[7]
      assert_equal('password', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(@EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_password_call_index = 0
      assert_equal('himitsu', @EditProperties.arg_password_value)
      assert(! @EditProperties.arg_password_cleared)
      @EditProperties.arg_password_cleared = true
      assert_equal(0, @crypt_call)
      assert_equal('himitsu', @EditProperties.arg_password_value)
      assert_equal(true, @EditProperties.arg_password_cleared)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('test_text', doc_args[1])
      assert_equal(0, doc_args[2])
      assert_equal(true, doc_args[3])
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('baz', doc_args[6])
      assert_equal(nil, doc_args[7]) # cleared!
      assert_equal(/foo/, doc_args[8])
    end

    def test_document_arguments_regexp_argument
      @EditProperties.document_position = '2'
      @EditProperties.document_type = 'TestArguments'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[8]
      assert_equal('regexp', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(@EditProperties.arg_type_regexp?)
      @EditProperties.arg_regexp_call_index = 0
      assert_equal('foo', @EditProperties.arg_regexp_value)
      @EditProperties.arg_regexp_value = 'bar'
      assert_equal('bar', @EditProperties.arg_regexp_value)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      doc_args = @document_list[2]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal('test_string', doc_args[0])
      assert_equal('test_text', doc_args[1])
      assert_equal(0, doc_args[2])
      assert_equal(true, doc_args[3])
      assert_equal('bar', doc_args[4])
      assert_equal({ 'foo' => true, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('baz', doc_args[6])
      assert_equal('himitsu', doc_args[7])
      assert_equal(/bar/, doc_args[8])
    end

    def test_new_document_with_default_arguments
      @EditProperties.document_position = '3'
      @EditProperties.document_type = 'TestArguments'
      @EditProperties.doc_path = '/default'
      doc_args = @EditProperties.doc_args
      assert_equal(9, doc_args.length)

      @EditProperties.argument = doc_args[0]
      assert_equal('string', @EditProperties.arg_name)
      assert(@EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_string_call_index = 0
      assert_nil(@EditProperties.arg_string_value)

      @EditProperties.argument = doc_args[1]
      assert_equal('text', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(@EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_text_call_index = 0
      assert_nil(@EditProperties.arg_text_value)

      @EditProperties.argument = doc_args[2]
      assert_equal('number', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(@EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_num_call_index = 0
      assert_nil(@EditProperties.arg_num_value)

      @EditProperties.argument = doc_args[3]
      assert_equal('bool', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(@EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_bool_call_index = 0
      assert_equal(false, @EditProperties.arg_bool_value)

      @EditProperties.argument = doc_args[4]
      assert_equal('select', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(@EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_select_call_index = 0
      assert_equal(%w[ foo bar baz ], @EditProperties.arg_select_list)
      assert_equal('foo', @EditProperties.arg_select_value)

      @EditProperties.argument = doc_args[5]
      assert_equal('checkset', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_password?)
      assert(@EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_checkset_call_index = 0
      assert_equal(3, @EditProperties.arg_checkset_count)
      @EditProperties.arg_checkset_call_index = nil
      # foo
      @EditProperties.arg_check_call_index = 0
      assert_equal('foo', @EditProperties.arg_check_name)
      assert_equal(false, @EditProperties.arg_checked)
      # bar
      @EditProperties.arg_check_call_index = 1
      assert_equal('bar', @EditProperties.arg_check_name)
      assert_equal(false, @EditProperties.arg_checked)
      # baz
      @EditProperties.arg_check_call_index = 2
      assert_equal('baz', @EditProperties.arg_check_name)
      assert_equal(false, @EditProperties.arg_checked)

      @EditProperties.argument = doc_args[6]
      assert_equal('radio', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(@EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_radio_call_index = 0
      assert_equal(3, @EditProperties.arg_radio_count)
      @EditProperties.arg_radio_call_index = nil
      @EditProperties.arg_radio_item_call_index = 0
      assert_equal('RadioGroup0', @EditProperties.arg_radio_name)
      assert_equal('foo', @EditProperties.arg_radio_value)
      assert_equal('foo', @EditProperties.arg_radio_selected)
      @EditProperties.arg_radio_item_call_index = 1
      assert_equal('RadioGroup0', @EditProperties.arg_radio_name)
      assert_equal('bar', @EditProperties.arg_radio_value)
      assert_equal('foo', @EditProperties.arg_radio_selected)
      @EditProperties.arg_radio_item_call_index = 2
      assert_equal('RadioGroup0', @EditProperties.arg_radio_name)
      assert_equal('baz', @EditProperties.arg_radio_value)
      assert_equal('foo', @EditProperties.arg_radio_selected)

      @EditProperties.argument = doc_args[7]
      assert_equal('password', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(@EditProperties.arg_type_password?)
      assert(! @EditProperties.arg_type_regexp?)
      @EditProperties.arg_password_call_index = 0
      assert_nil(@EditProperties.arg_password_value)

      @EditProperties.argument = doc_args[8]
      assert_equal('regexp', @EditProperties.arg_name)
      assert(! @EditProperties.arg_type_string?)
      assert(! @EditProperties.arg_type_text?)
      assert(! @EditProperties.arg_type_number?)
      assert(! @EditProperties.arg_type_bool?)
      assert(! @EditProperties.arg_type_select?)
      assert(! @EditProperties.arg_type_checkset?)
      assert(! @EditProperties.arg_type_radio?)
      assert(! @EditProperties.arg_type_password?)
      assert(@EditProperties.arg_type_regexp?)
      @EditProperties.arg_regexp_call_index = 0
      assert_nil(@EditProperties.arg_regexp_value)

      @EditProperties.write_properties
      assert_equal(1, @write_document_list_call)
      assert_equal('TestArguments', @document_list[3]['document'])
      doc_args = @document_list[3]['arguments']
      assert_equal(9, doc_args.length)
      assert_equal(nil, doc_args[0])
      assert_equal(nil, doc_args[1])
      assert_equal(nil, doc_args[2])
      assert_equal(false, doc_args[3])
      assert_equal('foo', doc_args[4])
      assert_equal({ 'foo' => false, 'bar' => false, 'baz' => false }, doc_args[5])
      assert_equal('foo', doc_args[6])
      assert_equal(nil, doc_args[7])
      assert_equal(nil, doc_args[8])
    end

    def test_cancel
      @EditProperties.cancel
      assert_equal(1, @cancel_call)
    end
  end
end
