#!/usr/local/bin/ruby
# $Id: test_DocumentTable.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

require 'rubyunit'
require 'wpm'
require 'rucy/document'

module TestRucy
  class TestDocumentTable < RUNIT::TestCase
    def setup
      # for WPM::Driver class
      @driver_call = 0
      @redirect_call = 0
      @redirect_last_page_name = nil
      @redirect_last_query_params = nil

      # for MountMap class
      @modified_count_call = 0
      @set_modified_count_call = 0
      @set_modified_count_last_value = nil
      @document_list_call = 0
      @document_list = [
	{ 'document' => 'Page',
	  'arguments' => [
	    "Hello world.\n",
	    'text/plain'
	  ],
	  'mount_path' => '/hello',
	  'mount_mask' => nil,
	  'virtual_host' => nil,
	  'comment' => 'test of Page'
	},
	{ 'document' => 'LocalFile',
	  'arguments' => [
	    '/home/alice/public_html'
	  ],
	  'mount_path' => '/~alice',
	  'mount_mask' => nil,
	  'virtual_host' => 'foo',
	  'comment' => 'test of LocalFile'
	}
      ]
      @doc_args_call = 0
      @doc_args_last_doc_name = nil
      @doc_names_call = 0
      @write_document_list_call = 0
      @setup_document_call = 0
      @setup_document_last_doc_type = nil
      @setup_document_last_pos = nil
      @doc_tag_call = 0
      @doc_label_call = 0
      @path_tag_call = 0
      @path_label_call = 0
      @mask_tag_call = 0
      @mask_label_call = 0

      @factory = Rucy::DocumentFactory.new
      page = Object.new
      def page.doc_name
	'Page'
      end
      def page.doc_args
	[ [ 'content', :text, nil ],
	  [ 'content-type', :string, 'text/html' ]
	]
      end
      @factory.add_document(page)
      local = Object.new
      def local.doc_name
	'LocalFile'
      end
      def local.doc_args
	[ [ 'local path', :string, nil ] ]
      end
      @factory.add_document(local)

      # for Rucy::DocumentFactory class
      @filter_names_call = 0
      erb = Object.new
      def erb.filter_name
	'eRuby'
      end
      def erb.filter_args
	[ [ 'safe level', :number, 1 ] ]
      end
      @factory.add_filter(erb)
      ssi = Object.new
      def ssi.filter_name
	'SSI'
      end
      def ssi.filter_args
	[ [ 'SSI commands', :checkset,
	    [ [ 'config',   true ],
	      [ 'include',  true ],
	      [ 'echo',     true ],
	      [ 'fsize',    true ],
	      [ 'flastmod', true ],
	      [ 'exec',     true ]
	    ]
	  ]
	]
      end
      @factory.add_filter(ssi)

      # for Rucy::Control class
      @filters_call = 0
      @filter_list = [
	{ 'filter' => 'eRuby',
	  'arguments' => [ 1 ],
	  'attach_path' => '/eRuby',
	  'attach_mask' => %r/\.rhtml?/,
	  'virtual_host' => nil,
	  'comment' => 'test of eRuby'
	},
	{ 'filter' => 'SSI',
	  'arguments' => [
	    { 'config'   => true,
	      'include'  => true,
	      'echo'     => true,
	      'fsize'    => true,
	      'flastmod' => true,
	      'exec'     => false
	    }
	  ],
	  'attach_path' => '/~alice',
	  'attach_mask' => %r/\.shtml?/,
	  'virtual_host' => 'foo',
	  'comment' => 'test of SSI'
	}
      ]

      # setup WPM::PageContext class
      loader = WPM::Loader.new('../control/DocumentTable/DocumentTable.rb')
      @DocumentTable = loader.const_get('DocumentTable').new(self, :control => self, :factory => self)
      assert((@DocumentTable.is_a? WPM::PageContext))
      @DocumentTable.init_context
      @DocumentTable.table_source = self
    end

    def driver
      @driver_call += 1
      self
    end

    def redirect(page_name, query_params=nil)
      @redirect_call += 1
      @redirect_last_page_name = page_name
      @redirect_last_query_params = query_params
      nil
    end

    def modified_count
      @modified_count_call += 1
      0
    end

    def modified_count=(modified_count)
      @set_modified_count_call += 1
      @set_modified_count_last_value = modified_count
    end

    def document_list
      @document_list_call += 1
      @document_list
    end

    def doc_args(doc_name)
      @doc_args_call += 1
      @doc_args_last_doc_name = doc_name
      @factory.doc_args(doc_name)
    end

    def doc_names
      @doc_names_call += 1
      @factory.doc_names
    end

    def filter_names
      @filter_names_call += 1
      @factory.filter_names
    end

    def filters
      @filters_call += 1
      @filter_list
    end

    def write_document_list
      @write_document_list_call += 1
      nil
    end

    def setup_document(doc_type, pos)
      @setup_document_call += 1
      @setup_document_last_doc_type = doc_type
      @setup_document_last_pos = pos
      nil
    end

    def doc_tag
      @doc_tag_call += 1
      'document'
    end

    def doc_label
      @doc_label_call += 1
      'document'
    end

    def path_tag
      @path_tag_call += 1
      'mount_path'
    end

    def path_label
      @path_label_call += 1
      'mount path'
    end

    def mask_tag
      @mask_tag_call += 1
      'mount_mask'
    end

    def mask_label
      @mask_label_call += 1
      'mount mask'
    end

    def test_modified_count
      assert_equal('0', @DocumentTable.modified_count)
      assert_equal(1, @modified_count_call)
      assert_equal('1', @DocumentTable.modified_count = '1')
      assert_equal(1, @set_modified_count_call)
      assert_equal(1, @set_modified_count_last_value)
    end

    def test_doc_label
      assert_equal('document', @DocumentTable.doc_label)
      assert_equal(1, @doc_label_call)
    end

    def test_path_label
      assert_equal('mount path', @DocumentTable.path_label)
      assert_equal(1, @path_label_call)
    end

    def test_mask_label
      assert_equal('mount mask', @DocumentTable.mask_label)
      assert_equal(1, @mask_label_call)
    end

    def test_has_documents?
      assert(@DocumentTable.has_documents?)
      assert_equal(1, @document_list_call)
    end

    def test_document_list
      document_list = @DocumentTable.document_list
      assert_equal(1, @document_list_call)
      assert(! document_list.empty?)
      assert_equal(2, document_list.length)
      assert_equal('Page', document_list[0]['document'])
      assert_equal(2, document_list[0]['arguments'].length)
      assert_equal("Hello world.\n", document_list[0]['arguments'][0])
      assert_equal('text/plain', document_list[0]['arguments'][1])
      assert_equal('/hello', document_list[0]['mount_path'])
      assert_nil(document_list[0]['mount_mask'])
      assert_nil(document_list[0]['virtual_host'])
      assert_equal('test of Page', document_list[0]['comment'])
      assert_equal(false, document_list[0]['show_args'])
      assert_equal('LocalFile', document_list[1]['document'])
      assert_equal(1, document_list[1]['arguments'].length)
      assert_equal('/home/alice/public_html', document_list[1]['arguments'][0])
      assert_equal('/~alice', document_list[1]['mount_path'])
      assert_nil(document_list[1]['mount_mask'])
      assert_equal('foo', document_list[1]['virtual_host'])
      assert_equal('test of LocalFile', document_list[1]['comment'])
      assert_equal(false, document_list[1]['show_args'])
    end

    def test_document
      @DocumentTable.document = {
	'document' => 'Page',
	'arguments' => [
	  "Hello world.\n",
	  'text/plain'
	],
	'mount_path' => '/hello',
	'mount_mask' => nil,
	'virtual_host' => 'foo',
	'comment' => 'test of Page',
	'show_args' => true
      }
      assert_equal('Page', @DocumentTable.doc_name)
      assert_equal(1, @doc_tag_call)
      assert_equal('/hello', @DocumentTable.doc_path)
      assert_equal(1, @path_tag_call)
      assert_nil(@DocumentTable.doc_mask)
      assert_equal(1, @mask_tag_call)
      assert_equal('foo', @DocumentTable.virtual_host)
      assert_equal('test of Page', @DocumentTable.comment)
      assert_equal(true, @DocumentTable.show_args?)
      assert_equal([ [ 'content', "Hello world.\n" ],
		     [ 'content-type', 'text/plain' ]
		   ], @DocumentTable.doc_arg_alist)
      assert_equal(1, @doc_args_call)
      assert_equal('Page', @doc_args_last_doc_name)
    end

    def test_doc_index
      @DocumentTable.doc_index = 1
      assert_equal(1, @DocumentTable.doc_index)
    end

    def test_has_comment?
      @DocumentTable.document = { 'comment' => nil }
      assert(! @DocumentTable.has_comment?)
      @DocumentTable.document = { 'comment' => '' }
      assert(! @DocumentTable.has_comment?)
      @DocumentTable.document = { 'comment' => 'Hello world.' }
      assert(@DocumentTable.has_comment?)
    end

    def test_has_args?
      @DocumentTable.document = {
	'document' => 'Page',
	'arguments' => [
	  "Hello world.\n",
	  'text/plain'
	],
	'mount_path' => '/hello',
	'mount_mask' => nil
      }
      assert(@DocumentTable.has_args?)
      @DocumentTable.document = {
	'document' => 'Empty',
	'arguments' => [],
	'mount_path' => '/empty',
	'mount_mask' => nil
      }
      assert(! @DocumentTable.has_args?)
    end

    def test_show_args_toggle
      document_list = @DocumentTable.document_list
      assert_equal(1, @document_list_call)
      assert(! document_list.empty?)
      assert_equal(2, document_list.length)
      assert_equal('Page', document_list[0]['document'])
      assert_equal(2, document_list[0]['arguments'].length)
      assert_equal("Hello world.\n", document_list[0]['arguments'][0])
      assert_equal('text/plain', document_list[0]['arguments'][1])
      assert_equal('/hello', document_list[0]['mount_path'])
      assert_nil(document_list[0]['mount_mask'])
      assert_nil(document_list[0]['virtual_host'])
      assert_equal(false, document_list[0]['show_args'])
      assert_equal('LocalFile', document_list[1]['document'])
      assert_equal(1, document_list[1]['arguments'].length)
      assert_equal('/home/alice/public_html', document_list[1]['arguments'][0])
      assert_equal('/~alice', document_list[1]['mount_path'])
      assert_nil(document_list[1]['mount_mask'])
      assert_equal('foo', document_list[1]['virtual_host'])
      assert_equal(false, document_list[1]['show_args'])

      @DocumentTable.show_args_call_index = 1
      @DocumentTable.show_args_toggle

      document_list = @DocumentTable.document_list
      assert_equal(1, @document_list_call)
      assert(! document_list.empty?)
      assert_equal(2, document_list.length)
      assert_equal('Page', document_list[0]['document'])
      assert_equal(2, document_list[0]['arguments'].length)
      assert_equal("Hello world.\n", document_list[0]['arguments'][0])
      assert_equal('text/plain', document_list[0]['arguments'][1])
      assert_equal('/hello', document_list[0]['mount_path'])
      assert_nil(document_list[0]['mount_mask'])
      assert_nil(document_list[0]['virtual_host'])
      assert_equal(false, document_list[0]['show_args'])
      assert_equal('LocalFile', document_list[1]['document'])
      assert_equal(1, document_list[1]['arguments'].length)
      assert_equal('/home/alice/public_html', document_list[1]['arguments'][0])
      assert_equal('/~alice', document_list[1]['mount_path'])
      assert_nil(document_list[1]['mount_mask'])
      assert_equal('foo', document_list[1]['virtual_host'])
      assert_equal(true, document_list[1]['show_args'])

      @DocumentTable.show_args_call_index = 1
      @DocumentTable.show_args_toggle

      document_list = @DocumentTable.document_list
      assert_equal(1, @document_list_call)
      assert(! document_list.empty?)
      assert_equal(2, document_list.length)
      assert_equal('Page', document_list[0]['document'])
      assert_equal(2, document_list[0]['arguments'].length)
      assert_equal("Hello world.\n", document_list[0]['arguments'][0])
      assert_equal('text/plain', document_list[0]['arguments'][1])
      assert_equal('/hello', document_list[0]['mount_path'])
      assert_nil(document_list[0]['mount_mask'])
      assert_nil(document_list[0]['virtual_host'])
      assert_equal(false, document_list[0]['show_args'])
      assert_equal('LocalFile', document_list[1]['document'])
      assert_equal(1, document_list[1]['arguments'].length)
      assert_equal('/home/alice/public_html', document_list[1]['arguments'][0])
      assert_equal('/~alice', document_list[1]['mount_path'])
      assert_nil(document_list[1]['mount_mask'])
      assert_equal('foo', document_list[1]['virtual_host'])
      assert_equal(false, document_list[1]['show_args'])
    end

    def test_argument
      @DocumentTable.argument = [ 'content', "Hello world.\n" ]
      assert_equal('content', @DocumentTable.arg_name)
      assert_equal("Hello world.\n", @DocumentTable.arg_value)
    end

    def test_arg_index
      @DocumentTable.arg_index = 1
      assert_equal(1, @DocumentTable.arg_index)
    end

    def test_document_type_list
      assert_equal([ 'LocalFile', 'Page' ], @DocumentTable.document_type_list)
      assert_equal(1, @doc_names_call)

      assert_nil(@DocumentTable.new_document_type_selected)
      assert_equal('Page', @DocumentTable.new_document_type_selected = 'Page')
      assert_equal('Page', @DocumentTable.new_document_type_selected)

      @DocumentTable.modify_document_type_call_index = 0
      assert_equal('Page', @DocumentTable.modify_document_type_selected)
      @DocumentTable.modify_document_type_call_index = 1
      assert_equal('LocalFile', @DocumentTable.modify_document_type_selected)

      @DocumentTable.modify_document_type_call_index = 0
      @DocumentTable.modify_document_type_selected = 'LocalFile'
      @DocumentTable.modify_document_type_call_index = 1
      @DocumentTable.modify_document_type_selected = 'Page'

      @DocumentTable.modify_document_type_call_index = 0
      assert_equal('LocalFile', @DocumentTable.modify_document_type_selected)
      @DocumentTable.modify_document_type_call_index = 1
      assert_equal('Page', @DocumentTable.modify_document_type_selected)
    end

    def test_doc_add
      @DocumentTable.new_document_type_selected = 'Page'
      @DocumentTable.doc_add
      assert_equal(1, @setup_document_call)
      assert_equal('Page', @setup_document_last_doc_type)
      assert_equal(2, @setup_document_last_pos)
    end

    def test_doc_edit
      @DocumentTable.modify_document_type_call_index = 1
      @DocumentTable.modify_document_type_selected = 'LocalFile'
      @DocumentTable.modify_document_type_call_index = nil
      @DocumentTable.doc_edit_call_index = 1
      @DocumentTable.doc_edit
      assert_equal(1, @setup_document_call)
      assert_equal('LocalFile', @setup_document_last_doc_type)
      assert_equal(1, @setup_document_last_pos)
    end

    def test_doc_up
      assert_equal(2, @document_list.length)
      # 1
      assert_equal('Page', @document_list[0]['document'])
      assert_equal(2, @document_list[0]['arguments'].length)
      assert_equal("Hello world.\n", @document_list[0]['arguments'][0])
      assert_equal('text/plain', @document_list[0]['arguments'][1])
      assert_equal('/hello', @document_list[0]['mount_path'])
      assert_nil(@document_list[0]['mount_mask'])
      assert_nil(@document_list[0]['virtual_host'])
      # 2
      assert_equal('LocalFile', @document_list[1]['document'])
      assert_equal(1, @document_list[1]['arguments'].length)
      assert_equal('/home/alice/public_html', @document_list[1]['arguments'][0])
      assert_equal('/~alice', @document_list[1]['mount_path'])
      assert_nil(@document_list[1]['mount_mask'])
      assert_equal('foo', @document_list[1]['virtual_host'])

      @DocumentTable.doc_up_call_index = 0
      @DocumentTable.doc_up
      assert_equal(0, @write_document_list_call)
      assert_equal(2, @document_list.length)
      # 1
      assert_equal('Page', @document_list[0]['document'])
      assert_equal(2, @document_list[0]['arguments'].length)
      assert_equal("Hello world.\n", @document_list[0]['arguments'][0])
      assert_equal('text/plain', @document_list[0]['arguments'][1])
      assert_equal('/hello', @document_list[0]['mount_path'])
      assert_nil(@document_list[0]['mount_mask'])
      assert_nil(@document_list[0]['virtual_host'])
      # 2
      assert_equal('LocalFile', @document_list[1]['document'])
      assert_equal(1, @document_list[1]['arguments'].length)
      assert_equal('/home/alice/public_html', @document_list[1]['arguments'][0])
      assert_equal('/~alice', @document_list[1]['mount_path'])
      assert_nil(@document_list[1]['mount_mask'])
      assert_equal('foo', @document_list[1]['virtual_host'])

      @DocumentTable.doc_up_call_index = 1
      @DocumentTable.doc_up
      assert_equal(1, @write_document_list_call)
      assert_equal(2, @document_list.length)
      # 1
      assert_equal('LocalFile', @document_list[0]['document'])
      assert_equal(1, @document_list[0]['arguments'].length)
      assert_equal('/home/alice/public_html', @document_list[0]['arguments'][0])
      assert_equal('/~alice', @document_list[0]['mount_path'])
      assert_nil(@document_list[0]['mount_mask'])
      assert_equal('foo', @document_list[0]['virtual_host'])
      # 2
      assert_equal('Page', @document_list[1]['document'])
      assert_equal(2, @document_list[1]['arguments'].length)
      assert_equal("Hello world.\n", @document_list[1]['arguments'][0])
      assert_equal('text/plain', @document_list[1]['arguments'][1])
      assert_equal('/hello', @document_list[1]['mount_path'])
      assert_nil(@document_list[1]['mount_mask'])
      assert_nil(@document_list[1]['virtual_host'])
    end

    def test_doc_up_disabled?
      @DocumentTable.doc_up_call_index = 0
      assert(@DocumentTable.doc_up_disabled?)
      @DocumentTable.doc_up_call_index = 1
      assert(! @DocumentTable.doc_up_disabled?)
    end

    def test_doc_down
      assert_equal(2, @document_list.length)
      # 1
      assert_equal('Page', @document_list[0]['document'])
      assert_equal(2, @document_list[0]['arguments'].length)
      assert_equal("Hello world.\n", @document_list[0]['arguments'][0])
      assert_equal('text/plain', @document_list[0]['arguments'][1])
      assert_equal('/hello', @document_list[0]['mount_path'])
      assert_nil(@document_list[0]['mount_mask'])
      assert_nil(@document_list[0]['virtual_host'])
      # 2
      assert_equal('LocalFile', @document_list[1]['document'])
      assert_equal(1, @document_list[1]['arguments'].length)
      assert_equal('/home/alice/public_html', @document_list[1]['arguments'][0])
      assert_equal('/~alice', @document_list[1]['mount_path'])
      assert_nil(@document_list[1]['mount_mask'])
      assert_equal('foo', @document_list[1]['virtual_host'])

      @DocumentTable.doc_down_call_index = 1
      @DocumentTable.doc_down
      assert_equal(0, @write_document_list_call)
      assert_equal(2, @document_list.length)
      # 1
      assert_equal('Page', @document_list[0]['document'])
      assert_equal(2, @document_list[0]['arguments'].length)
      assert_equal("Hello world.\n", @document_list[0]['arguments'][0])
      assert_equal('text/plain', @document_list[0]['arguments'][1])
      assert_equal('/hello', @document_list[0]['mount_path'])
      assert_nil(@document_list[0]['mount_mask'])
      assert_nil(@document_list[0]['virtual_host'])
      # 2
      assert_equal('LocalFile', @document_list[1]['document'])
      assert_equal(1, @document_list[1]['arguments'].length)
      assert_equal('/home/alice/public_html', @document_list[1]['arguments'][0])
      assert_equal('/~alice', @document_list[1]['mount_path'])
      assert_nil(@document_list[1]['mount_mask'])
      assert_equal('foo', @document_list[1]['virtual_host'])

      @DocumentTable.doc_down_call_index = 0
      @DocumentTable.doc_down
      assert_equal(1, @write_document_list_call)
      assert_equal(2, @document_list.length)
      # 1
      assert_equal('LocalFile', @document_list[0]['document'])
      assert_equal(1, @document_list[0]['arguments'].length)
      assert_equal('/home/alice/public_html', @document_list[0]['arguments'][0])
      assert_equal('/~alice', @document_list[0]['mount_path'])
      assert_nil(@document_list[0]['mount_mask'])
      assert_equal('foo', @document_list[0]['virtual_host'])
      # 2
      assert_equal('Page', @document_list[1]['document'])
      assert_equal(2, @document_list[1]['arguments'].length)
      assert_equal("Hello world.\n", @document_list[1]['arguments'][0])
      assert_equal('text/plain', @document_list[1]['arguments'][1])
      assert_equal('/hello', @document_list[1]['mount_path'])
      assert_nil(@document_list[1]['mount_mask'])
      assert_nil(@document_list[1]['virtual_host'])
    end

    def test_doc_down_disabled
      @DocumentTable.doc_down_call_index = 0
      assert(! @DocumentTable.doc_down_disabled?)
      @DocumentTable.doc_down_call_index = 1
      assert(@DocumentTable.doc_down_disabled?)
    end

    def test_doc_delete
      assert_equal(2, @document_list.length)
      # 1
      assert_equal('Page', @document_list[0]['document'])
      assert_equal(2, @document_list[0]['arguments'].length)
      assert_equal("Hello world.\n", @document_list[0]['arguments'][0])
      assert_equal('text/plain', @document_list[0]['arguments'][1])
      assert_equal('/hello', @document_list[0]['mount_path'])
      assert_nil(@document_list[0]['mount_mask'])
      assert_nil(@document_list[0]['virtual_host'])
      # 2
      assert_equal('LocalFile', @document_list[1]['document'])
      assert_equal(1, @document_list[1]['arguments'].length)
      assert_equal('/home/alice/public_html', @document_list[1]['arguments'][0])
      assert_equal('/~alice', @document_list[1]['mount_path'])
      assert_nil(@document_list[1]['mount_mask'])
      assert_equal('foo', @document_list[1]['virtual_host'])

      @DocumentTable.doc_delete_call_index = 0
      @DocumentTable.doc_delete
      assert_equal(1, @write_document_list_call)
      assert_equal(1, @document_list.length)
      # 1
      assert_equal('LocalFile', @document_list[0]['document'])
      assert_equal(1, @document_list[0]['arguments'].length)
      assert_equal('/home/alice/public_html', @document_list[0]['arguments'][0])
      assert_equal('/~alice', @document_list[0]['mount_path'])
      assert_nil(@document_list[0]['mount_mask'])
      assert_equal('foo', @document_list[0]['virtual_host'])
    end

    def test_filter_type_list
      assert_equal([ 'SSI', 'eRuby' ], @DocumentTable.filter_type_list)
      assert_equal(1, @filter_names_call)

      @DocumentTable.add_filter_type_call_index = 0
      assert_nil(@DocumentTable.add_filter_type_selected)
      @DocumentTable.add_filter_type_call_index = 1
      assert_nil(@DocumentTable.add_filter_type_selected)

      @DocumentTable.add_filter_type_call_index = 0
      @DocumentTable.add_filter_type_selected = 'eRuby'
      @DocumentTable.add_filter_type_call_index = 1
      @DocumentTable.add_filter_type_selected = 'SSI'

      @DocumentTable.add_filter_type_call_index = 0
      assert_equal('eRuby', @DocumentTable.add_filter_type_selected)
      @DocumentTable.add_filter_type_call_index = 1
      assert_equal('SSI', @DocumentTable.add_filter_type_selected)
    end

    def test_add_filter
      @DocumentTable.add_filter_type_call_index = 1
      @DocumentTable.add_filter_type_selected = 'eRuby'
      @DocumentTable.add_filter_call_index = 1
      @DocumentTable.add_filter
      assert_equal(1, @filters_call)
      assert_equal(1, @driver_call)
      assert_equal(1, @redirect_call)
      assert_equal('SetupFilter', @redirect_last_page_name)
      assert_equal({ 'filter' => 'eRuby',
		     'pos'    => '2',
		     'path'   => '/~alice',
		     'mask'   => 'nil',
		     'vhost'  => 'foo'
		   }, @redirect_last_query_params)
    end
  end
end
