#!/usr/local/bin/ruby
# $Id: test_AdminPassword.rb,v 1.1.1.1 2004/04/04 15:22:50 toki Exp $

require 'rubyunit'
require 'wpm'
require 'rucy/passwd'

module TestRucy
  class TestAdminPassword < RUNIT::TestCase
    def setup
      # for WPM::Driver class
      @driver_call = 0
      @redirect_call = 0
      @redirect_last_page_name = nil
      @redirect_last_query_params = nil

      # for Rucy::Control class
      @modified_count_call = 0
      @set_modified_count_call = 0
      @set_modified_count_value = nil
      @admin_params = {
	'admin_user' => 'admin',
	'admin_password' => '',
	'localhost_only' => true
      }
      @admin_params_call = 0
      @set_admin_params_call = 0

      # for Rucy::PasswordEncryptor
      @pw_enc = Rucy::PasswordEncryptor.new
      @crypt_call = 0
      @crypt_password_list = []

      # setup WPM::PageContext class
      loader = WPM::Loader.new('../control/AdminPassword/AdminPassword.rb')
      @AdminPassword = loader.const_get('AdminPassword').new(self, :control => self, :pw_enc => self)
      assert((@AdminPassword.is_a? WPM::PageContext))
      @AdminPassword.init_context
    end

    def driver
      @driver_call += 1
      self
    end

    def redirect(page_name, query_params=nil)
      @redirect_call += 1
      @redirect_last_page_name = page_name
      @redirect_last_query_params = query_params
      nil
    end

    def modified_count
      @modified_count_call += 1
      0
    end

    def modified_count=(modified_count)
      @set_modified_count_call += 1
      @set_modified_count_value = modified_count
    end

    def admin_params
      @admin_params_call += 1
      admin_params = @admin_params.dup
      def admin_params.setup(parent)
	@parent = parent
	self
      end
      def admin_params.modified_count
	@parent.modified_count
      end
      def admin_params.modified_count=(modified_count)
	@parent.modified_count = modified_count
      end
      admin_params.setup(self)
    end

    def set_admin_params(params)
      @set_admin_params_call += 1
      @admin_params = params
    end

    def crypt(password, salt=nil)
      @crypt_call += 1
      @crypt_password_list.push(password)
      @pw_enc.crypt(password, salt || @pw_enc.salt)
    end

    def test_modified_count
      assert_equal(1, @admin_params_call)
      assert_equal('0', @AdminPassword.modified_count)
      assert_equal(1, @modified_count_call)
      assert_equal('1', @AdminPassword.modified_count = '1')
      assert_equal(1, @set_modified_count_call)
      assert_equal(1, @set_modified_count_value)
    end

    def test_change_username
      @AdminPassword.username = 'alice'
      @AdminPassword.change_password
      assert_equal(1, @set_admin_params_call)
      assert_equal('alice', @admin_params['admin_user'])
      assert_equal(1, @driver_call)
      assert_equal(1, @redirect_call)
      assert_equal('Administrator', @redirect_last_page_name)
      assert_nil(@redirect_last_query_params)
    end

    def test_change_password
      @AdminPassword.old_password = ''
      @AdminPassword.new_password = 'nazo'
      @AdminPassword.retype_password = 'nazo'
      @AdminPassword.change_password
      assert_equal(1, @crypt_call)
      assert_equal('nazo', @crypt_password_list[0])
      assert_equal(1, @set_admin_params_call)
      assert_equal(@admin_params['admin_password'],
		   @pw_enc.crypt('nazo', @admin_params['admin_password']))
      assert_equal(1, @driver_call)
      assert_equal(1, @redirect_call)
      assert_equal('Administrator', @redirect_last_page_name)
      assert_nil(@redirect_last_query_params)

      @AdminPassword.old_password = 'nazo'
      @AdminPassword.new_password = 'himitsu'
      @AdminPassword.retype_password = 'himitsu'
      @AdminPassword.change_password
      assert_equal(3, @crypt_call)
      assert_equal('nazo', @crypt_password_list[1])
      assert_equal('himitsu', @crypt_password_list[2])
      assert_equal(2, @set_admin_params_call)
      assert_equal(@admin_params['admin_password'],
		   @pw_enc.crypt('himitsu', @admin_params['admin_password']))
      #assert_equal(2, @driver_call)
      assert_equal(2, @redirect_call)
      assert_equal('Administrator', @redirect_last_page_name)
      assert_nil(@redirect_last_query_params)
    end

    def test_mismatch_old_password_empty
      @AdminPassword.old_password = 'detarame'
      @AdminPassword.new_password = 'nazo'
      @AdminPassword.retype_password = 'nazo'
      assert_exception(RuntimeError) {
	@AdminPassword.change_password
      }
    end

    def test_mismatch_old_password
      @AdminPassword.old_password = ''
      @AdminPassword.new_password = 'nazo' # effective password
      @AdminPassword.retype_password = 'nazo'
      @AdminPassword.change_password
      assert_equal(@admin_params['admin_password'],
		   @pw_enc.crypt('nazo', @admin_params['admin_password']))

      @AdminPassword.old_password = 'detarame'
      @AdminPassword.new_password = 'nazo'
      @AdminPassword.retype_password = 'nazo'
      assert_exception(RuntimeError) {
	@AdminPassword.change_password
      }
    end

    def test_mismatch_new_password
      @AdminPassword.old_password = ''
      @AdminPassword.new_password = 'nazo'
      @AdminPassword.retype_password = 'himitsu'
      assert_exception(RuntimeError) {
	@AdminPassword.change_password
      }
    end

    def test_cancel
      @AdminPassword.cancel
      assert_equal(0, @set_admin_params_call)
      assert_equal(1, @driver_call)
      assert_equal(1, @redirect_call)
      assert_equal('Administrator', @redirect_last_page_name)
      assert_nil(@redirect_last_query_params)
    end
  end
end
