# $Id: control.rb,v 1.1.1.1 2004/04/04 15:22:49 toki Exp $

require 'wpm'
require 'wpm/rucydriver'
require 'wpm/rexml'
require 'rucy/restart'
require 'rucy/document'
require 'rucy/messenger'
require 'rucy/logger'
require 'rucy/passwd'
require 'rucy/auth'
require 'rucy/access'
require 'rucy/server'

module Rucy
  class Control
    def initialize
      @lib_loader = nil
      @pw_enc = PasswordEncryptor.new
      @doc_factory = nil
      @doc_errors = Array.new
      @filter_errors = Array.new
      @logfiles_output = Array.new
      @logging_errors = Array.new
      @alias_errors = Array.new
      @restart_signal = nil
      @properties = nil
      @page_path = nil
    end

    attr_writer :lib_loader
    attr_writer :pw_enc
    attr_writer :doc_factory
    attr_reader :doc_errors
    attr_reader :filter_errors
    attr_reader :logging_errors
    attr_reader :alias_errors
    attr_writer :properties
    attr_writer :page_path

    def server_setup(server)
      @lib_loader.load_lib
      if (@doc_factory.has_filter? 'BasicAuth') then
	@doc_factory.set_filter_option('BasicAuth', :pw_enc, @pw_enc)
      end

      server_params = self.server_params
      case (server_params['server_type'])
      when 'multithread'
	@restart_signal = MultiThreadRestartSignal.new
	server.restart_signal = @restart_signal
	server.messenger_factory = MultiThreadMessenger
      when 'multiprocess'
	@restart_signal = MultiProcessRestartSignal.new
	server.restart_signal = @restart_signal
	server.messenger_factory = MultiProcessMessenger
      else
	raise "unknown server type: #{server_params['server_type'].inspect}"
      end
      server.port = server_params['port']
      server.timeout = server_params['timeout']
      server.keep_alive = server_params['keep_alive']
      server.max_requests = server_params['max_requests']
      server.queue_length = server_params['queue_length']
      server.messengers = server_params['messengers']
      server.messenger_threads = server_params['messenger_threads']
      server.messenger_queue_length = server_params['messenger_queue_length']
      server.subprocess_user = server_params['subprocess_user']
      server.subprocess_group = server_params['subprocess_group']

      logging_params = self.logging_params
      unless ($RUCY_DAEMON) then
	stdout_logger = Logger.new(STDOUT)
	stdout_logger.log_emerg = logging_params['stdout_emerg']
	stdout_logger.log_alert = logging_params['stdout_alert']
	stdout_logger.log_crit = logging_params['stdout_crit']
	stdout_logger.log_err = logging_params['stdout_err']
	stdout_logger.log_warning = logging_params['stdout_warning']
	stdout_logger.log_notice = logging_params['stdout_notice']
	stdout_logger.log_info = logging_params['stdout_info']
	stdout_logger.log_debug = logging_params['stdout_debug']
	server.add_logger(stdout_logger)
      end
      for logfile in logging_params['logfiles']
	begin
	  output = File.open(logfile['path'], 'a')
	  @logfiles_output.push(output)
	  file_logger = Logger.new(output)
	  file_logger.log_emerg = logfile['log_emerg']
	  file_logger.log_alert = logfile['log_alert']
	  file_logger.log_crit = logfile['log_crit']
	  file_logger.log_err = logfile['log_err']
	  file_logger.log_warning = logfile['log_warning']
	  file_logger.log_notice = logfile['log_notice']
	  file_logger.log_info = logfile['log_info']
	  file_logger.log_debug = logfile['log_debug']
	  server.add_logger(file_logger)
	rescue StandardError, ScriptError
	  @logging_errors.push({ :logfile => logfile['path'],
				 :exception => $!
			       })
	end
      end

      alias_list = self.aliases
      for alias_info in alias_list
	alias_path = alias_info['alias_path']
	orig_path = alias_info['orig_path']
	host = alias_info['virtual_host']
	begin
	  if (host) then
	    host += ':' + server_params['port'].to_s
	    server.set_virtual_alias(host, alias_path, orig_path)
	  else
	    server.set_alias(alias_path, orig_path)
	  end
	rescue
	  @alias_errors.push({ :alias_path => alias_path,
			       :orig_path => orig_path,
			       :virtual_host => host,
			       :exception => $!
			     })
	end
      end

      document_list = self.documents
      for mount_info in document_list
	name = mount_info['document']
	args = mount_info['arguments']
	path = mount_info['mount_path']
	mask = mount_info['mount_mask']
	host = mount_info['virtual_host']
	begin
	  document = @doc_factory.doc_build(name, args)
	  if (host) then
	    host += ':' + server_params['port'].to_s
	    server.virtual_mount(host, document, path, mask)
	  else
	    server.mount(document, path, mask)
	  end
	rescue StandardError, ScriptError
	  @doc_errors.push({ :document => name,
			     :arguments => args,
			     :mount_path => path,
			     :mount_mask => mask,
			     :virtual_host => host,
			     :exception => $!
			   })
	end
      end

      filter_list = self.filters
      for filter_info in filter_list
	name = filter_info['filter']
	args = filter_info['arguments']
	path = filter_info['attach_path']
	mask = filter_info['attach_mask']
	host = filter_info['virtual_host']
	begin
	  filter = @doc_factory.filter_build(name, args)
	  if (host) then
	    host += ':' + server_params['port'].to_s
	    server.virtual_attach(host, filter, path, mask)
	  else
	    server.attach(filter, path, mask)
	  end
	rescue StandardError, ScriptError
	  @filter_errors.push({ :filter => name,
				:arguments => args,
				:attach_path => path,
				:attach_mask => mask,
				:virtual_host => host,
				:exception => $!
			      })
	end
      end

      admin_params = self.admin_params
      page_maker = WebPageMakerDocument.new(WPM::XMLAssistByREXML, @page_path, 'ControlPanel')
      page_maker.debug = true
      page_maker.set_page_option(:control, self)
      page_maker.set_page_option(:factory, @doc_factory)
      page_maker.set_page_option(:pw_enc, @pw_enc)
      if (admin_params['japanese_handling']) then
	require 'rucy/jconv'
	page_maker.set_input_filter{ |data| Jconv.jconv(data) }
	page_maker.set_output_filter{ |data| Uconv.u8toeuc(data) }
	page_maker.charset = 'euc-jp'
      end
      server.mount(page_maker, '/control')
      if (admin_params['localhost_only']) then
	access = Rucy::RemoteAddressAllowAccessFilter.new
	access.add_allow_address('127.0.0.1')
	server.attach(access, '/control')
      end
      if (admin_params['admin_user'] && ! admin_params['admin_user'].empty? &&
	  admin_params['admin_password'] && ! admin_params['admin_password'].empty?)
      then
	passwd = PasswordVerifier.new
	passwd.add_encrypted_user(admin_params['admin_user'], admin_params['admin_password'])
	auth = BasicAuth.new(passwd, "Administrator's password")
	server.attach(auth, '/control')
      end

      server.close_hook{ |s|
	logfiles_close
      }

      nil
    end

    def load_errors
      @lib_loader.load_errors
    end

    def server_restart
      @restart_signal.notify_restart
      nil
    end

    def server_close
      @restart_signal.notify_close
      nil
    end

    def logfiles_close
      for output in @logfiles_output
	output.close
      end
      nil
    end

    ADMIN_PARAMS = {
      'admin_user' => 'admin',
      'admin_password' => '',
      'localhost_only' => true,
      'japanese_handling' => false
    }

    def admin_params
      params = @properties.params(*ADMIN_PARAMS.keys)
      for name, default in ADMIN_PARAMS
	if (params[name] == nil) then
	  params[name] = default
	end
      end
      params
    end

    def set_admin_params(params)
      params.each_key do |name|
	unless (ADMIN_PARAMS.include? name) then
	  raise "not an administrator's parameter: #{name}."
	end
      end
      if (! params['admin_user'] || params['admin_user'].empty? ||
	  ! params['admin_password'] || params['admin_password'].empty?)
      then
	params['localhost_only'] = true
      end
      @properties.set_params(params)
      nil
    end

    SERVER_PARAMS = {
      'server_type' => 'multithread',
      'port' => 8888,
      'timeout' => 300,
      'keep_alive' => 8,
      'max_requests' => 32,
      'queue_length' => 16,
      'messengers' => 8,
      'messenger_threads' => 4,
      'messenger_queue_length' => 4,
      'subprocess_user' => 'nobody',
      'subprocess_group' => 'nobody'
    }

    def server_params
      params = @properties.params(*SERVER_PARAMS.keys)
      for name, default in SERVER_PARAMS
	if (params[name] == nil) then
	  params[name] = default
	end
      end
      params
    end

    def set_server_params(params)
      params.each_key do |name|
	unless (SERVER_PARAMS.include? name) then
	  raise "not a server parameter: #{name}."
	end
      end
      @properties.set_params(params)
      nil
    end

    STDOUT_LEVEL = {
      'stdout_emerg' => true,
      'stdout_alert' => true,
      'stdout_crit' => true,
      'stdout_err' => true,
      'stdout_warning' => true,
      'stdout_notice' => true,
      'stdout_info' => true,
      'stdout_debug' => false
    }

    def logging_params
      names = STDOUT_LEVEL.keys
      names.push('logfiles')
      params = @properties.params(*(STDOUT_LEVEL.keys + [ 'logfiles' ]))
      for level, default_flag in STDOUT_LEVEL
	if (params[level] == nil) then
	  params[level] = default_flag
	end
      end
      if (params['logfiles'] == nil) then
	params['logfiles'] = Array.new
      end
      params
    end

    def set_logging_params(params)
      params.each_key do |name|
	if (! (STDOUT_LEVEL.include? name) && name != 'logfiles') then
	  raise "not a logging parameter: #{name}"
	end
      end
      @properties.set_params(params)
      nil
    end

    def aliases
      @properties.list('aliases')
    end

    def set_aliases(alias_list)
      @properties.set_list('aliases', alias_list)
      nil
    end

    def documents
      @properties.list('documents')
    end

    def set_documents(document_list)
      @properties.set_list('documents', document_list)
      nil
    end

    def filters
      @properties.list('filters')
    end

    def set_filters(filter_list)
      @properties.set_list('filters', filter_list)
      nil
    end
  end
end
