# $Id: server.rb,v 1.2 2004/04/06 22:09:32 toki Exp $

require 'time'
require 'socket'
require 'thread'
require 'rucy/logger'
require 'rucy/messenger'
require 'rucy/document'

module Rucy
  class ServerStopException < Exception
  end

  class ServerRestartException < Exception
  end

  class Server
    include MessengerParent

    def self.daemon
      catch(:START_DAEMON) do
	unless (fork) then
	  Process.setsid
	  unless (fork) then
	    #Dir.chdir('/')
	    STDIN.reopen('/dev/null', 'r')
	    STDOUT.reopen('/dev/null', 'w')
	    STDERR.reopen('/dev/null', 'w')
	    throw :START_DAEMON
	  end
	end
	exit!
      end
    end

    PORT = 8888

    EMPTY_HOOK = proc{ |server|
      # nothing to do.
    }

    def initialize
      @open_hook = EMPTY_HOOK
      @close_hook = EMPTY_HOOK
      @accept_lock = Mutex.new
      @accept_state = :close
      @accept_thread = nil
      @socket_queue = SocketQueue.new
      reset
    end

    def open_hook(&block)
      @open_hook = block
      nil
    end

    def close_hook(&block)
      @close_hook = block
      nil
    end

    def call_hook(name, hook)
      begin
	hook.call(self)
      rescue StandardError, ScriptError
	err_messg = "[#{Time.now.httpdate}] #{name} hook error: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}"
	begin
	  STDERR.puts err_messg
	rescue StandardError, ScriptError
	  # nothing to do.
	end
	begin
	  @logger.err(err_messg)
	rescue StandardError, ScriptError
	  # nothing to do.
	end
      end

      nil
    end
    private :call_hook

    def reset
      @port = PORT
      @server = nil
      @multi_logger = MultiLogger.new
      @logger = SyncLogger.new(@multi_logger)
      @restart_signal = nil
      @folder = FolderDocument.new
      @document = @folder
      @timeout = Messenger::TIMEOUT
      @keep_alive = Messenger::KEEP_ALIVE
      @max_requests = Messenger::MAX_REQUESTS
      @queue_length = Messenger::QUEUE_LENGTH
      @messengers = Messenger::MESSENGERS
      @messenger_factory = MultiThreadMessenger
      @messenger_threads = Messenger::MESSENGER_THREADS
      @messenger_queue_length = Messenger::MESSENGER_QUEUE_LENGTH
      @subprocess_user = nil
      @subprocess_group = nil
      nil
    end
    private :reset

    def add_logger(logger)
      @multi_logger.add(logger)
      nil
    end

    def restart_signal=(restart_signal)
      @restart_signal = restart_signal
      @restart_signal.server = self
      restart_signal
    end

    attr_accessor :port
    attr_accessor :document
    attr_accessor :queue_length
    attr_accessor :timeout
    attr_accessor :keep_alive
    attr_accessor :max_requests
    attr_accessor :messengers
    attr_accessor :messenger_factory
    attr_accessor :messenger_threads
    attr_accessor :messenger_queue_length
    attr_accessor :subprocess_user
    attr_accessor :subprocess_group

    def add_filter(filter)
      @document = FilterDocument.new(@document, filter)
      nil
    end

    def set_alias(alias_path, orig_path)
      @folder.set_alias(alias_path, orig_path)
      nil
    end

    def set_virtual_alias(host, alias_path, orig_path)
      @folder.set_virtual_alias(host, alias_path, orig_path)
      nil
    end

    def mount(document, path, mask=nil)
      @folder.mount(document, path, mask)
      nil
    end

    def attach(filter, path, mask=nil)
      @folder.attach(filter, path, mask)
      nil
    end

    def umount(path, mask=nil)
      @folder.umount(path, mask)
    end

    def virtual_mount(host, document, path, mask=nil)
      @folder.virtual_mount(host, document, path, mask)
    end

    def virtual_attach(host, filter, path, mask=nil)
      @folder.virtual_attach(host, filter, path, mask)
    end

    def virtual_umount(host, path, mask=nil)
      @folder.virtual_umount(host, path, mask)
    end

    def close
      @accept_lock.synchronize{
	if (@accept_state == :open) then
	  @accept_state = :stop
	  @accept_thread.raise(ServerStopException, 'stop server')
	end
      }

      nil
    end

    def restart
      @accept_lock.synchronize{
	if (@accept_state == :open) then
	  @accept_state = :restart
	  @accept_thread.raise(ServerRestartException, 'restart server')
	end
      }

      nil
    end

    def open_document
      for document in @document
	begin
	  document.open
	rescue StandardError, ScriptError
	  logger.warning("[#{Time.now.httpdate}] failed to open a document: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	end
      end
      nil
    end
    private :open_document

    def close_document
      for document in @document
	begin
	  document.close
	rescue StandardError, ScriptError
	  logger.warning("[#{Time.now.httpdate}] failed to close a document: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	end
      end
      nil
    end
    private :close_document

    def open_server
      call_hook('open', @open_hook)
      open_document
      @socket_queue.resize(@queue_length)
      @messenger_thgrp = start_messengers(@messengers, @socket_queue, @document, @logger, @messenger_factory, @restart_signal)
      @server = TCPServer.open(@port)
      family, port, host, ip = @server.addr
      @logger.notice("[#{Time.now.httpdate}] open server: #{family}:#{host}[#{ip}]:#{port}")

      nil
    end
    private :open_server

    def close_server
      if (@server) then
	family, port, host, ip = @server.addr
	@logger.notice("[#{Time.now.httpdate}] close server: #{family}:#{host}[#{ip}]:#{port}")
	@server.close
      end
      stop_messengers(@messengers, @socket_queue, @messenger_thgrp, @restart_signal)
      close_document
      call_hook('close', @close_hook)

      nil
    end
    private :close_server

    def accept
      begin
	begin
	  open_server
	  @accept_lock.synchronize{
	    @accept_state = :open
	    @accept_thread = Thread.current
	  }
	  loop do
	    begin
	      socket = @server.accept
	      socket.setsockopt(Socket::IPPROTO_TCP, Socket::TCP_NODELAY, 1) # for fast response.
	    rescue
	      @logger.warning("[#{Time.now.httpdate}] accept error: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	      retry
	    end
	    @socket_queue.push(socket)
	  end
	rescue ServerRestartException
	  sleep(0.1)		# wait for intercepting accept
	  close_server
	  reset
	  retry
	end
      rescue ServerStopException
	sleep(0.1)		# wait for intercepting accept
      ensure
	@accept_lock.synchronize{
	  @accept_state = :close
	  @accept_thread = nil
	}
	close_server
	until (@socket_queue.empty?)
	  socket = @socket_queue.pop
	  socket.shutdown
	end
      end

      nil
    end
  end
end
