# $Id: request.rb,v 1.4 2004/04/22 08:41:36 toki Exp $

require 'rucy/error'
require 'rucy/status'
require 'rucy/version'
require 'rucy/message'

module Rucy
  class Request < Message
    def self.normalize(req_path)
      path, query = req_path.split(/\?/, 2)

      path.gsub!(/\+/, ' ')
      path.gsub!(/%([0-9A-Fa-f][0-9A-Fa-f])/) { |c| $1.hex.chr }

      while (path.gsub!(%r"/\./", '/'))
	# foo/./bar => foo/bar
      end
      while (path.gsub!(%r"[^/]+/\.\./", ''))
	# foo/../bar => bar
      end
      while (path.gsub!(%r"[^/]+/\.\.$", ''))
	# foo/bar/.. => foo/
      end
      while (path.gsub!(%r"^/\.\./", '/'))
	# /../foo => /foo
      end
      path.sub!(%r"/\.$", '/')  # foo/. => foo/
      path.sub!(%r"^/..$", '/') # /.. => /

      return path, query
    end

    def self.scan(path)
      if (path.empty? || path == '/') then
	yield('', '')
	return
      end

      if (path !~ %r"^/") then
	raise "not a path: #{path.inspect}"
      end
      path_list = path.split(%r"/", -1)
      path_info_list = Array.new
      until (path_list.empty?)
	if (path_list.length == 1) then
	  script_name = ''
	else
	  script_name = path_list.join('/')
	end
	if (path_info_list.empty?) then
	  path_info = ''
	else
	  path_info = '/' + path_info_list.join('/')
	end
	yield(script_name, path_info)
	path_info_list.unshift(path_list.pop)
      end

      nil
    end

    def initialize
      super
      @method = nil
      @uri = nil
      @version = nil
      @path = nil
      @query = nil
      @server_name = nil
      @server_address = nil
      @server_port = nil
      @client_name = nil
      @client_address = nil
      @client_port = nil
      @messg_reader = nil
      @ready_to_read = false
      @content_length = nil
      @line_buf = nil
    end

    attr_writer :method
    attr_reader :uri
    attr_accessor :version
    attr_accessor :path
    attr_accessor :query
    attr_reader :server_name
    attr_reader :server_address
    attr_reader :server_port
    attr_reader :client_name
    attr_reader :client_address
    attr_reader :client_port

    def method(*args, &block)
      if (args.empty? && block.nil?) then
	return @method
      else
	return super
      end
    end

    def uri=(new_uri)
      @uri = new_uri

      case (@uri)
      when %r"^/"
	@path, @query = Request.normalize(@uri)
      when %r"^https?://([^/\s]+)(/.*)?$"
	set_header('Host', $1)
	if ($2 && ! $2.empty?) then
	  @path, @query = Request.normalize($2)
	else
	  @path, @query = '/', nil
	end
      else
	@path, @query = nil, nil
      end

      new_uri
    end

    def line
      method = @method || '-'
      if (@uri) then
	path = @uri
      elsif (@path) then
	path = @path
	path += '?' + @query if @query
      else
	path = '-'
      end
      version = @version || '-'
      "#{method} #{path} #{version}"
    end

    def subpath(script_name)
      if (script_name == '/') then
	script_name = ''
      end
      len = script_name.length
      if (@path[0, len] != script_name) then
	raise 'mismatch script_name.'
      end
      return script_name, @path[len..-1]
    end

    def parse_line(input)
      for line in input
	line.chomp!("\n")
	line.chomp!("\r")
	next if line.empty?

	method, uri, version = line.split(/\s+/, 3)
	if (method.nil? || method.empty? || uri.nil? || uri.empty?) then
	  raise ParseError, "failed to parse a request line: #{line.inspect}"
	end

	@method = method
	self.uri = uri
	if (version) then
	  if (version =~ %r"^HTTP/\d+\.\d+$") then
	    @version = version
	  else
	    raise ParseError, "invalid HTTP version format: #{line.inspect}"
	  end
	else
	  @version = 'HTTP/0.9'
	end

	return
      end

      raise ParseError, 'closed input stream'
    end

    def parse(input)
      parse_line(input)
      if (@version != 'HTTP/0.9') then
	parse_header(input)
      end

      nil
    end

    def conn_closed?
      case (@version)
      when 'HTTP/1.0'
	if (headers('Connection').find{ |v| v =~ /close/i }) then
	  return true
	elsif (headers('Connection').find{ |v| v =~ /Keep-Alive/i }) then
	  return false
	else
	  return true
	end
      when 'HTTP/1.1'
	if (headers('Connection').find{ |v| v =~ /close/i }) then
	  return true
	else
	  return false
	end
      else
	return true
      end
    end

    def host
      header('Host') || "#{@server_name || @server_address}:#{@server_port}"
    end

    def set_server(name, addr, port)
      @server_name = name
      @server_address = addr
      @server_port = port
      nil
    end

    def set_client(name, addr, port)
      @client_name = name
      @client_address = addr
      @client_port = port
      nil
    end

    def set_reader(input)
      case (@method)
      when 'POST', 'PUT'
	if (has_header? 'Content-Length') then
	  @content_length = header('Content-Length').to_i
	  if (@content_length < 0) then
	    raise HTTPError.new(403, 'Negative Content-Length')
	  end
	elsif (conn_closed?) then
	  @content_length = nil
	else
	  raise HTTPError.new(411) # Length Required
	end
	@messg_reader = input
	@ready_to_read = true
	@line_buf = ''
      else
	# nothing to do.
      end

      nil
    end

    def has_body?
      @ready_to_read
    end

    def each_body(bufsiz=1024*16)
      unless (@ready_to_read) then
	raise 'failed to read a request message body.'
      end

      case (@method)
      when 'POST', 'PUT'
	if (@content_length) then
	  while (@content_length > bufsiz)
	    messg = @messg_reader.read(bufsiz) or break
	    @content_length -= messg.length
	    yield(messg)
	  end
	  while (@content_length > 0)
	    messg = @messg_reader.read(@content_length)
	    @content_length -= messg.length
	    yield(messg)
	  end
	else
	  while (messg = @messg_reader.read(bufsiz))
	    yield(messg)
	  end
	end
      else
	# nothing to do.
      end
      @ready_to_read = false

      nil
    end

    def fetch_body
      messg_body = ''
      each_body do |messg|
	messg_body << messg
      end

      messg_body
    end

    def each_line(eol=$/)
      if (@ready_to_read) then
	each_body do |messg|
	  @line_buf << messg
	  while (pos = @line_buf.index(eol))
	    line = @line_buf[0...(pos + eol.length)]
	    @line_buf = @line_buf[(pos + eol.length)..-1]
	    yield(line)
	  end
	end
      end

      if (@line_buf) then
	unless (@line_buf.empty?) then
	  while (pos = @line_buf.index(eol))
	    line = @line_buf[0...(pos + eol.length)]
	    @line_buf = @line_buf[(pos + eol.length)..-1]
	    yield(line)
	  end
	  unless (@line_buf.empty?) then
	    yield(@line_buf)
	  end
	end
	@line_buf = nil
      else
	raise 'failed to read a request message body.'
      end

      nil
    end

    def fetch_lines(eol=$/)
      line_list = Array.new
      each_line(eol) do |line|
	line_list.push(line)
      end

      line_list
    end

    def cgi_env(script_name, pass_auth=false)
      env = Hash.new
      env['GATEWAY_INTERFACE'] = 'CGI/1.1'
      env['REQUEST_METHOD'] = @method
      env['SCRIPT_NAME'] = script_name
      env['PATH_INFO'] = subpath(script_name)[1]
      env['QUERY_STRING'] = @query || ''
      if (has_header? 'Host') then
	host = header('Host')
	if (host =~ /:\d+$/) then
	  name, port = host.split(/:/, 2)
	  env['SERVER_NAME'] = name
	  env['SERVER_PORT'] = port
	else
	  env['SERVER_NAME'] = host
	  env['SERVER_PORT'] = '80'
	end
      else
	env['SERVER_NAME'] = @server_name || @server_address
	env['SERVER_PORT'] = @server_port.to_s
      end
      env['SERVER_PROTOCOL'] = @version
      env['SERVER_SOFTWARE'] = SERVER_TOKEN_LIST
      env['REMOTE_HOST'] = @client_name || @client_address
      env['REMOTE_ADDR'] = @client_address

      each_header do |name, value|
	case (name)
	when 'Content-Type', 'Content-Length'
	  cgi_name = name.upcase
	  cgi_name.gsub!(/-/, '_')
	  env[cgi_name] = value
	else
	  unless (pass_auth) then
	    if (name =~ /Authorization/) then
	      next
	    end
	  end
	  cgi_name = name.upcase
	  cgi_name.gsub!(/-/, '_')
	  env['HTTP_' + cgi_name] = value
	end
      end

      env
    end
  end
end
