#!/usr/local/bin/ruby
# $Id: test_writer.rb,v 1.3 2004/09/24 15:07:42 toki Exp $

require 'rubyunit'
require 'pseudo_req_res'
require 'rucy/writer'
require 'rucy/logger'
require 'rucy/request'
require 'rucy/response'

module TestRucy
  class TestHTTPThroughWriter < RUNIT::TestCase
    def setup
      super
      @request = Rucy::Request.new
      @request.method = 'GET'
      @request.uri = '/hello/world'
      @request.version = 'HTTP/1.1'
      @request.set_server('server', '192.168.0.1', 8080)
      @request.set_client('client', '192.168.0.2', 31415)
      @write_messg = ''
      @flush_call = 0
      @writer = Rucy::HTTPThroughWriter.new(self, @request)
      @response = Rucy::Response.new
      @response.set_writer(@writer)
    end

    def write(messg)
      @write_messg << messg
      nil
    end

    def flush
      assert(! @write_messg.empty?)
      @flush_call += 1
      nil
    end

    def test_response
      @response.set_header('Content-Type', 'text/html')
      @response.start_body
      @response << "<html>\n"
      @response << "Hello world.\n"
      @response << "</html>\n"
      @writer.close
      assert_equal(28, @writer.send_size)

      messg = "HTTP/1.1 200 OK\r\n"
      messg << "Content-Type: text/html\r\n"
      messg << "\r\n"
      messg << "<html>\n"
      messg << "Hello world.\n"
      messg << "</html>\n"
      assert_equal(messg, @write_messg)
      assert_equal(1, @flush_call)
    end

    def test_response_with_long_body
      long_chunk = ' ' * (Rucy::HTTPWriter::BUFFERING_THRESHOLD + 1)
      @response.set_header('Content-Type', 'application/octet-stream')
      @response.start_body
      @response << long_chunk
      @response << long_chunk
      @response << long_chunk
      @writer.close
      assert_equal(long_chunk.length * 3, @writer.send_size)

      messg = "HTTP/1.1 200 OK\r\n"
      messg << "Content-Type: application/octet-stream\r\n"
      messg << "\r\n"
      messg << long_chunk
      messg << long_chunk
      messg << long_chunk
      assert_equal(messg, @write_messg)
      assert(1, @flush_call)
    end
  end

  class TestHTTPSpoolWriter < RUNIT::TestCase
    def setup
      super
      @request = Rucy::Request.new
      @request.method = 'GET'
      @request.uri = '/hello/world'
      @request.version = 'HTTP/1.1'
      @request.set_server('server', '192.168.0.1', 8080)
      @request.set_client('client', '192.168.0.2', 31415)
      @write_messg = ''
      @write_call = 0
      @flush_call = 0
      @writer = Rucy::HTTPSpoolWriter.new(self, @request)
      @response = Rucy::Response.new
      @response.set_writer(@writer)
    end

    def write(messg)
      @write_messg << messg
      @write_call += 1
      nil
    end

    def flush
      assert(! @write_messg.empty?)
      @flush_call += 1
      nil
    end

    def test_response
      @response.set_header('Content-Type', 'text/html')
      @response.start_body
      @response << "<html>\n"
      @response << "Hello world.\n"
      @response << "</html>\n"
      @writer.close
      assert_equal(28, @writer.send_size)

      messg = "HTTP/1.1 200 OK\r\n"
      messg << "Content-Length: 28\r\n"	# auto-measured
      messg << "Content-Type: text/html\r\n"
      messg << "\r\n"
      messg << "<html>\n"
      messg << "Hello world.\n"
      messg << "</html>\n"
      assert_equal(messg, @write_messg)
      assert_equal(1, @write_call)
      assert_equal(1, @flush_call)
    end

    def test_response_with_long_body
      long_chunk = ' ' * (Rucy::HTTPWriter::BUFFERING_THRESHOLD + 1)
      @response.set_header('Content-Type', 'application/octet-stream')
      @response.start_body
      @response << long_chunk
      @response << long_chunk
      @response << long_chunk
      @writer.close
      assert_equal(long_chunk.length * 3, @writer.send_size)

      messg = "HTTP/1.1 200 OK\r\n"
      messg << "Connection: close\r\n"
      messg << "Content-Type: application/octet-stream\r\n"
      messg << "\r\n"
      messg << long_chunk
      messg << long_chunk
      messg << long_chunk
      assert_equal(messg, @write_messg)
      assert_equal(1, @flush_call)
    end
  end
end
