#!/usr/local/bin/ruby
# $Id: test_priv.rb,v 1.3 2004/12/11 15:38:04 toki Exp $

require 'rubyunit'
require 'forwarder'
require 'rucy/priv'

module TestRucy
  class TestPrivilege < RUNIT::TestCase
    def setup
      # for Etc module
      @getpwnam_call = 0
      @getpwnam_name = nil
      @getgrnam_call = 0
      @getgrnam_name = nil

      @etc = Forwarder.new(self)
      class << @etc
	def_delegator :__getobj__, :getpwnam
	def_delegator :__getobj__, :getgrnam
      end

      # for Struct::Passwd class
      @uid_call = 0

      @passwd = Forwarder.new(self)
      class << @passwd
	def_delegator :__getobj__, :uid
      end

      # for Struct::Group class
      @gid_call = 0

      @group = Forwarder.new(self)
      class << @group
	def_delegator :__getobj__, :gid
      end

      # for Process module
      @euid_call = 0
      @euid = nil
      @set_uid_call = 0
      @set_uid_id = nil
      @set_gid_call = 0
      @set_gid_id = nil
      @set_euid_call = 0
      @set_euid_id = nil
      @set_egid_call = 0
      @set_egid_id = nil

      @process = Forwarder.new(self)
      class << @process
	def_delegator :__getobj__, :euid
	def_delegator :__getobj__, :uid=
	def_delegator :__getobj__, :gid=
	def_delegator :__getobj__, :euid=
	def_delegator :__getobj__, :egid=
      end

      # for File class
      @chown_call = 0
      @chown_owner = nil
      @chown_group = nil
      @chown_path = nil
      @open_call = 0
      @open_path = 0
      @open_mode = 0
      @exist_call = 0
      @exist_path = nil

      @file = Forwarder.new(self)
      class << @file
	def_delegator :__getobj__, :chown
	def_delegator :__getobj__, :open
	def_delegator :__getobj__, :exist?
      end

      # target
      @priv = Rucy::Privilege.new('http_user', 'http_group', @etc, @process, @file)
    end

    # for Etc module

    def getpwnam(name)
      @getpwnam_call += 1
      @getpwnam_name = name
      @passwd
    end

    def getgrnam(name)
      @getgrnam_call += 1
      @getgrnam_name = name
      @group
    end

    # for Struct::Passwd class

    def uid
      @uid_call += 1
      80
    end

    # for Struct::Group class

    def gid
      @gid_call += 1
      80
    end

    # for Process module

    def euid
      @euid_call += 1
      @euid
    end

    def uid=(new_uid)
      @set_uid_call += 1
      @set_uid_id = new_uid
    end

    def gid=(new_gid)
      @set_gid_call += 1
      @set_gid_id = new_gid
    end

    def euid=(new_uid)
      @set_euid_call += 1
      @set_euid_id = new_uid
    end

    def egid=(new_gid)
      @set_egid_call += 1
      @set_egid_id = new_gid
    end

    # for File class

    def chown(owner, group, path)
      @chown_call += 1
      @chown_owner = owner
      @chown_group = group
      @chown_path = path
      nil
    end

    def open(path, mode)
      @open_call += 1
      @open_path = path
      @open_mode = mode
      yield
      nil
    end

    def exist?(path)
      @exist_call += 1
      @exist_path = path
      path == 'foo'
    end

    # test

    def test_uid_from_name
      assert_equal(80, @priv.uid('http_user'))
      assert_equal(1, @getpwnam_call)
      assert_equal('http_user', @getpwnam_name)
    end

    def test_uid_from_uid
      assert_equal(8080, @priv.uid(8080))
      assert_equal(0, @getpwnam_call)
    end

    def test_gid_from_name
      assert_equal(80, @priv.gid('http_group'))
      assert_equal(1, @getgrnam_call)
      assert_equal('http_group', @getgrnam_name)
    end

    def test_gid_from_gid
      assert_equal(8080, @priv.gid(8080))
      assert_equal(0, @getgrnam_call)
    end

    def test_privileged_user
      @euid = 0
      assert_equal(true, @priv.privileged_user?)
      assert_equal(1, @euid_call)
      @euid = 1001
      assert_equal(false, @priv.privileged_user?)
      assert_equal(2, @euid_call)
    end

    def test_non_privilege_user
      assert_equal('http_user', @priv.non_privilege_user)
    end

    def test_non_privilege_group
      assert_equal('http_group', @priv.non_privilege_group)
    end

    def test_cancel_privilege
      @priv.cancel_privilege
      assert_equal(1, @getpwnam_call)
      assert_equal('http_user', @getpwnam_name)
      assert_equal(1, @getgrnam_call)
      assert_equal('http_group', @getgrnam_name)
      assert_equal(1, @set_gid_call)
      assert_equal(80, @set_gid_id)
      assert_equal(1, @set_egid_call)
      assert_equal(80, @set_egid_id)
      assert_equal(1, @set_uid_call)
      assert_equal(80, @set_uid_id)
      assert_equal(1, @set_euid_call)
      assert_equal(80, @set_euid_id)
    end

    def test_chown
      @priv.chown('foo')
      assert_equal(1, @getpwnam_call)
      assert_equal('http_user', @getpwnam_name)
      assert_equal(1, @getgrnam_call)
      assert_equal('http_group', @getgrnam_name)
      assert_equal(1, @chown_call)
      assert_equal(80, @chown_owner)
      assert_equal(80, @chown_group)
      assert_equal('foo', @chown_path)
    end

    def test_touch_to_exist_path
      @priv.touch('foo')
      assert_equal(1, @exist_call)
      assert_equal('foo', @exist_path)
      assert_equal(1, @getpwnam_call)
      assert_equal('http_user', @getpwnam_name)
      assert_equal(1, @getgrnam_call)
      assert_equal('http_group', @getgrnam_name)
      assert_equal(1, @chown_call)
      assert_equal(80, @chown_owner)
      assert_equal(80, @chown_group)
      assert_equal('foo', @chown_path)
    end

    def test_touch_to_not_exist_path
      @priv.touch('bar')
      assert_equal(1, @exist_call)
      assert_equal('bar', @exist_path)
      assert_equal(1, @open_call)
      assert_equal('bar', @open_path)
      assert_equal(File::WRONLY | File::CREAT | File::EXCL, @open_mode)
      assert_equal(1, @getpwnam_call)
      assert_equal('http_user', @getpwnam_name)
      assert_equal(1, @getgrnam_call)
      assert_equal('http_group', @getgrnam_name)
      assert_equal(1, @chown_call)
      assert_equal(80, @chown_owner)
      assert_equal(80, @chown_group)
      assert_equal('bar', @chown_path)
    end

    def test_touch_with_no_create
      @priv.touch('bar', true)
      assert_equal(1, @exist_call)
      assert_equal('bar', @exist_path)
      assert_equal(0, @open_call)
      assert_equal(0, @getpwnam_call)
      assert_equal(0, @getgrnam_call)
      assert_equal(0, @chown_call)
    end
  end
end
