# $Id: local.rb,v 1.4 2004/11/25 05:43:34 toki Exp $

require 'time'
require 'rucy/fsep'
require 'rucy/error'
require 'rucy/document'

include Rucy

class ContentTypeResolver
  def initialize
    @suffix_map = {
      :content_type => 'application/octet-stream'
    }
  end

  def set_content_type(suffix, content_type)
    curr_map = @suffix_map
    suffix.upcase.reverse.each_byte do |c|
      curr_map[c] = Hash.new unless (curr_map.include? c)
      curr_map = curr_map[c]
    end
    curr_map[:content_type] = content_type
    nil
  end

  def set_default_content_type(content_type)
    @suffix_map[:content_type] = content_type
    nil
  end

  def content_type(path)
    curr_map = @suffix_map
    curr_type = curr_map[:content_type]
    path.upcase.reverse.each_byte do |c|
      break unless (curr_map.include? c)
      curr_map = curr_map[c]
      curr_type = curr_map[:content_type] if curr_map[:content_type]
    end
    curr_type
  end

  def clone_suffix(curr_map)
    curr_map = curr_map.dup
    for c, next_map in curr_map
      next if (c == :content_type)
      curr_map[c] = clone_suffix(next_map)
    end
    curr_map
  end
  private :clone_suffix

  def dup
    new_resolver = super
    new_resolver.instance_eval{
      @suffix_map = clone_suffix(@suffix_map)
    }
    new_resolver
  end
end

class LocalDirectory < Document
  include FileSeparator

  def initialize(root)
    @root = root
  end

  def publish(script_name, request, response, logger)
    logger.debug("enter document: #{self.class}")
    script_name2, path = request.subpath(script_name)
    if (path !~ %r"^/") then
      raise 'mismatch script_name.'
    end
    if (path !~ %r"/$") then
      raise 'not a directory path.'
    end
    if (path == '/') then
      local_dir = @root
    else
      local_dir = File.join(@root, http2local(path))
    end

    if (File.directory? local_dir) then
      case (request.method)
      when 'GET', 'HEAD'
	response.status = 200	# OK
	response.set_header('Content-Type', 'text/html')
	response.start_body
	if (request.method != 'HEAD') then
	  response << "<html>\n"
	  response << "<head><title>" << escapeHTML(request.path) << "</title></head>\n"
	  response << "<body>\n"
	  response << "<h1>" << escapeHTML(request.path) << "</h1>\n"
	  response << '<p><a href="..">parent directory...</a></p>' << "\n"
	  response << "<ul>\n"
	  name_list = Dir.entries(local_dir)
	  name_list.sort!
	  for name in name_list
	    next if (name =~ /^\./)
	    local_path = File.join(local_dir, name)
	    http_path = request.path + name
	    http_name = name
	    if (File.directory? local_path) then
	      http_path += '/'
	      http_name += '/'
	    end
	    response << '<li><a href="' << escapeHTML(http_path) << '">'
	    response << escapeHTML(http_name)
	    response << "</a></li>\n"
	  end
	  response << "</ul>\n"
	  response << "</body>\n"
	  response << "</html>\n"
	end
      else
	http_error = HTTPError.new(405) # Method Not Allowed
	http_error.set_header('Allow', 'GET, HEAD')
	raise http_error
      end
    else
      raise HTTPError.new(404) # Not Found
    end

    nil
  end
end

class LocalFileDocument < Document
  include FileSeparator

  TYPE_RESOLVER = ContentTypeResolver.new
  # from `mime.types' in Apache 1.3.27 distribution.
  [ %w[ application/andrew-inset        ez ],
    %w[ application/mac-binhex40        hqx ],
    %w[ application/mac-compactpro      cpt ],
    %w[ application/msword              doc ],
    %w[ application/octet-stream        bin dms lha lzh exe class so dll ],
    %w[ application/oda                 oda ],
    %w[ application/pdf                 pdf ],
    %w[ application/postscript          ai eps ps ],
    %w[ application/smil                smi smil ],
    %w[ application/vnd.mif             mif ],
    %w[ application/vnd.ms-excel        xls ],
    %w[ application/vnd.ms-powerpoint   ppt ],
    %w[ application/vnd.ms-project      mpp ],
    %w[ application/vnd.wap.wbxml       wbxml ],
    %w[ application/vnd.wap.wmlc        wmlc ],
    %w[ application/vnd.wap.wmlscriptc  wmlsc ],
    %w[ application/x-bcpio             bcpio ],
    %w[ application/x-cdlink            vcd ],
    %w[ application/x-chess-pgn         pgn ],
    %w[ application/x-cpio              cpio ],
    %w[ application/x-csh               csh ],
    %w[ application/x-director          dcr dir dxr ],
    %w[ application/x-dvi               dvi ],
    %w[ application/x-futuresplash      spl ],
    %w[ application/x-gtar              gtar ],
    %w[ application/x-hdf               hdf ],
    %w[ application/x-javascript        js ],
    %w[ application/x-koan              skp skd skt skm ],
    %w[ application/x-latex             latex ],
    %w[ application/x-netcdf            nc cdf ],
    %w[ application/x-sh                sh ],
    %w[ application/x-shar              shar ],
    %w[ application/x-shockwave-flash   swf ],
    %w[ application/x-stuffit           sit ],
    %w[ application/x-sv4cpio           sv4cpio ],
    %w[ application/x-sv4crc            sv4crc ],
    %w[ application/x-tar               tar ],
    %w[ application/x-tcl               tcl ],
    %w[ application/x-tex               tex ],
    %w[ application/x-texinfo           texinfo texi ],
    %w[ application/x-troff             t tr roff ],
    %w[ application/x-troff-man         man ],
    %w[ application/x-troff-me          me ],
    %w[ application/x-troff-ms          ms ],
    %w[ application/x-ustar             ustar ],
    %w[ application/x-wais-source       src ],
    %w[ application/xhtml+xml           xhtml xht ],
    %w[ application/zip                 zip ],
    %w[ audio/basic                     au snd ],
    %w[ audio/midi                      mid midi kar ],
    %w[ audio/mpeg                      mpga mp2 mp3 ],
    %w[ audio/x-aiff                    aif aiff aifc ],
    %w[ audio/x-mpegurl                 m3u ],
    %w[ audio/x-pn-realaudio            ram rm ],
    %w[ audio/x-pn-realaudio-plugin     rpm ],
    %w[ audio/x-realaudio               ra ],
    %w[ audio/x-wav                     wav ],
    %w[ chemical/x-pdb                  pdb ],
    %w[ chemical/x-xyz                  xyz ],
    %w[ image/bmp                       bmp ],
    %w[ image/gif                       gif ],
    %w[ image/ief                       ief ],
    %w[ image/jpeg                      jpeg jpg jpe ],
    %w[ image/png                       png ],
    %w[ image/tiff                      tiff tif ],
    %w[ image/vnd.djvu                  djvu djv ],
    %w[ image/vnd.wap.wbmp              wbmp ],
    %w[ image/x-cmu-raster              ras ],
    %w[ image/x-portable-anymap         pnm ],
    %w[ image/x-portable-bitmap         pbm ],
    %w[ image/x-portable-graymap        pgm ],
    %w[ image/x-portable-pixmap         ppm ],
    %w[ image/x-rgb                     rgb ],
    %w[ image/x-xbitmap                 xbm ],
    %w[ image/x-xpixmap                 xpm ],
    %w[ image/x-xwindowdump             xwd ],
    %w[ model/iges                      igs iges ],
    %w[ model/mesh                      msh mesh silo ],
    %w[ model/vrml                      wrl vrml ],
    %w[ text/css                        css ],
    %w[ text/html                       html htm shtml rhtml ],
    %w[ text/plain                      asc txt text log rb rd pl pm sh ],
    %w[ text/richtext                   rtx ],
    %w[ text/rtf                        rtf ],
    %w[ text/sgml                       sgml sgm ],
    %w[ text/tab-separated-values       tsv ],
    %w[ text/vnd.wap.wml                wml ],
    %w[ text/vnd.wap.wmlscript          wmls ],
    %w[ text/x-setext                   etx ],
    %w[ text/xml                        xml xsl ],
    %w[ video/mpeg                      mpeg mpg mpe ],
    %w[ video/quicktime                 qt mov ],
    %w[ video/vnd.mpegurl               mxu ],
    %w[ video/x-msvideo                 avi ],
    %w[ video/x-sgi-movie               movie ],
    %w[ x-conference/x-cooltalk         ice ]
  ].each do |content_type, *suffix_list|
    for suffix in suffix_list
      TYPE_RESOLVER.set_content_type(".#{suffix}", content_type)
    end
  end

  def initialize(root)
    @root = File.expand_path(root)
    @chunk_size = 1024 * 32
    @type_resolver = type_resolver=TYPE_RESOLVER.dup
    @dir_index = LocalDirectory.new(root)
  end

  attr_reader :type_resolver

  def publish(script_name, request, response, logger)
    logger.debug("enter document: #{self.class}")
    script_name2, path = request.subpath(script_name)
    local_path = File.expand_path(@root + http2local(path))
    if (local_path[0, @root.length] != @root) then
      logger.info("illegal path: #{request.path.inspect}")
      raise HTTPError.new(403, "illegal path: #{request.path.inspect}") # Forbidden
    end
    response.doc_path = request.path
    if (File.file? local_path) then
      response.local_path = local_path
      publish_file(path, local_path, request, response)
    elsif (File.directory? local_path) then
      if (path =~ %r|/$|) then
	[ 'index.html',
	  'index.htm'
	].each do |index_html|
	  index_path = File.join(local_path, index_html)
	  if (File.file? index_path) then
	    response.local_path = index_path
	    publish_file(path + index_html, index_path, request, response)
	    return
	  end
	end
	response.local_path = local_path
	@dir_index.publish(script_name, request, response, logger)
      else
	response.local_path = local_path
	response.status = 301	# Moved Permanently
	response.set_header('Location', 'http://' + request.host + request.path + '/')
	response.set_header('Content-Type', 'text/html')
	response.start_body
	if (request.method != 'HEAD') then
	  response << "Directory.\n"
	end
      end
    else
      raise HTTPError.new(404) # Not Found
    end
    nil
  end

  def publish_file(req_path, local_path, request, response)
    case (request.method)
    when 'GET', 'HEAD'
      File.open(local_path) { |file|
	stat = file.stat
	if (request.has_header? 'If-Modified-Since') then
	  publish_if_modified_since(stat, request, response) and return
	elsif (request.has_header? 'If-Unmodified-Since') then
	  publish_if_unmodified_since(stat, request, response) and return
	elsif (request.has_header? 'If-Range') then
	  if (request.has_header? 'Range') then
	    publish_if_range(file, stat, req_path, request, response) and return
	  end
	end
	if (request.has_header? 'Range') then
	  if (! (request.has_header? 'If-Range')) then
	    publish_range(file, stat, req_path, request, response) and return
	  end
	end
	response.status = 200 # OK
	response.set_header('Content-Type', @type_resolver.content_type(req_path))
	response.set_header('Content-Length', stat.size.to_s)
	response.set_header('Last-Modified', stat.mtime.httpdate)
	response.start_body
	if (request.method != 'HEAD') then
	  while (data = file.read(@chunk_size))
	    response.write(data)
	  end
	end
      }
    else
      http_error = HTTPError.new(405) # Method Not Allow
      http_error.set_header('Allow', 'GET, HEAD')
      raise http_error
    end
    nil
  end
  private :publish_file

  def publish_range(file, stat, req_path, request, response)
    if (request.header('Range') =~ /^bytes=(\d+)-(\d+)?$/) then
      first_pos = $1.to_i
      if ($2) then
	last_pos = $2.to_i
	if (last_pos > stat.size - 1) then
	  last_pos = stat.size - 1
	end
      else
	last_pos = stat.size - 1
      end
      if (first_pos <= last_pos) then
	partial_size = last_pos - first_pos + 1
	response.status = 206 # Partial Content
	response.set_header('Content-Type', @type_resolver.content_type(req_path))
	response.set_header('Content-Range', "bytes #{first_pos}-#{last_pos}/#{stat.size}")
	response.set_header('Content-Length', partial_size.to_s)
	response.set_header('Last-Modified', stat.mtime.httpdate)
	response.start_body
	if (request.method != 'HEAD') then
	  file.seek(first_pos)
	  while (partial_size > @chunk_size)
	    data = file.read(@chunk_size) or break
	    response.write(data)
	    partial_size -= data.length
	  end
	  while (partial_size > 0)
	    data = file.read(partial_size) or break
	    response.write(data)
	    partial_size -= data.length
	  end
	end
	return true
      else
	response.status = 416 # Requested Range Not Satisfiable
	response.start_body
	return true
      end
    end

    false
  end
  private :publish_range

  def publish_if_modified_since(stat, request, response)
    mtime = Time.parse(request.header('If-Modified-Since'))
    if (stat.mtime <= mtime) then
      response.status = 304 # Not Modified
      response.start_body
      return true
    end

    false
  end
  private :publish_if_modified_since

  def publish_if_unmodified_since(stat, request, response)
    mtime = Time.parse(request.header('If-Unmodified-Since'))
    if (stat.mtime > mtime) then
      response.status = 412 # Precondition Failed
      response.start_body
      return true
    end

    false
  end
  private :publish_if_unmodified_since

  def publish_if_range(file, stat, req_path, request, response)
    mtime = Time.parse(request.header('If-Range'))
    if (stat.mtime <= mtime) then
      return publish_range(file, stat, req_path, request, response)
    end

    false
  end
  private :publish_if_range
end

class LocalFileDocumentBuilder < DocumentBuilder
  NARGS = 5

  def doc_name
    'LocalFile'
  end

  def doc_args
    args = [
      [ 'local path', :string, nil ],
      [ 'default content-type (optional)', :string, nil ]
    ]
    for i in 1..NARGS
      args.push([ "suffix #{i} (optional)", :string, nil ])
      args.push([ "content-type #{i} (optional)", :string, nil ])
    end
    args
  end

  def new(local_path, default_content_type, *args)
    local_file = LocalFileDocument.new(local_path)
    if (default_content_type && ! default_content_type.strip.empty?) then
      local_file.type_resolver.set_default_content_type(default_content_type)
    end
    NARGS.times do
      suffix = args.shift
      content_type = args.shift
      if (suffix && ! suffix.strip.empty?) then
	if (content_type && ! content_type.strip.empty?) then
	  local_file.type_resolver.set_content_type(suffix, content_type)
	end
      end
    end
    local_file
  end
end
