# $Id: cgi.rb,v 1.3 2004/11/25 05:43:31 toki Exp $

require 'time'
require 'rucy/fsep'
require 'rucy/error'
require 'rucy/process'
require 'rucy/request'
require 'rucy/document'

include Rucy

class CGIDocument < Document
  include FileSeparator
  include ProcUtil

  def self.quote_sh(string)
    # for Bourne shell
    '"' + string.gsub(/["\$\`\\]/) { |special| "\\" + special } + '"'
  end

  def initialize(base_dir, run_cmd=nil, nph=false)
    @base_dir = File.expand_path(base_dir)
    @run_cmd = run_cmd
    @nph = nph
    @env = Hash.new
    @pass_args = false
    @pass_auth = false
    @local_path_check = Regexp.compile('^' + Regexp.quote(@base_dir))
  end

  attr_accessor :env
  attr_accessor :pass_args
  attr_accessor :pass_auth

  def scan_cgi(script_name, request_path)
    Request.scan(request_path) do |cgi_script_name, path_info|
      if (cgi_script_name == script_name) then
	break
      end
      subpath = cgi_script_name[(script_name.length)..-1]
      cgi_local_path = File.expand_path(@base_dir + http2local(subpath))
      if (cgi_local_path !~ @local_path_check) then
	next
      end
      if (File.file? cgi_local_path) then
	return cgi_script_name, cgi_local_path
      end
    end
    raise HTTPError.new(404)	# Not Found
  end
  private :scan_cgi

  def parse_args(query_string)
    if (query_string) then
      if (query_string.length > 1024) then
	raise HTTPError.new(414) # Request-URI Too Long
      end
      if (@pass_args) then
	if (! query_string.empty? && query_string !~ /=/) then
	  return query_string.split(/\+/, -1)
	end
      end
    end

    []
  end
  private :parse_args

  def cgi_open(path, args, env)
    if (@run_cmd) then
      cgi_cmd = CGIDocument.quote_sh(@run_cmd)
      cgi_cmd << ' '
      cgi_cmd << CGIDocument.quote_sh(path)
    else
      cgi_cmd = CGIDocument.quote_sh(path)
    end
    for cgi_arg in args
      cgi_cmd << ' '
      cgi_cmd << CGIDocument.quote_sh(cgi_arg)
    end
    begin
      cgi_io = PROC_RES_LOCK.synchronize{
	chenv(env) {
	  chdir(File.dirname(path)) {
	    IO.popen(cgi_cmd, 'r+')
	  }
	}
      }
      cgi_io.binmode
      return yield(cgi_io)
    ensure
      cgi_io.close if cgi_io
    end
  end
  private :cgi_open

  def publish(script_name, request, response, logger)
    logger.debug("enter document: #{self.class}")
    cgi_script_name, cgi_local_path = scan_cgi(script_name, request.path)
    cgi_args = parse_args(request.query)
    cgi_env = { 'PATH' => ENV['PATH'] }
    cgi_env.update(@env)
    cgi_env.update(request.cgi_env(cgi_script_name, @pass_auth))
    cgi_env['SCRIPT_FILENAME'] = cgi_local_path
    response.doc_path = cgi_script_name
    response.local_path = cgi_local_path
    cgi_open(cgi_local_path, cgi_args, cgi_env) { |cgi_io|
      # request message body
      if (request.has_body?) then
	request.each_body do |req_messg_body|
	  cgi_io.write(req_messg_body)
	end
      end
      cgi_io.flush
      cgi_io.close_write

      # response message head
      response.parse_line(cgi_io) if @nph
      response.parse_header(cgi_io)
      unless (@nph) then
	response.parse_status
	response.absolute_location(request)
      end

      # response message body
      response.start_body
      while (res_messg_body = cgi_io.read(1024*16))
	response.write(res_messg_body)
      end
    }
    nil
  end
end

class CGIDocumentBuilder < DocumentBuilder
  NARGS = 5

  def doc_name
    'CGI'
  end

  def doc_args
    args = [
      [ 'CGI directory', :string, nil ],
      [ 'CGI run command (optional)', :string, nil ],
      [ 'none parsed header', :bool, false ],
      [ 'pass CGI arguments', :bool, false ],
      [ 'pass authorization header', :bool, false ]
    ]
    for i in 1..NARGS
      args.push([ "environment variable name #{i} (optional)", :string, nil ])
      args.push([ "environment variable value #{i} (optional)", :string, nil ])
    end
    args
  end

  def new(cgi_dir, run_cmd, nph, pass_args, pass_auth, *args)
    if (run_cmd && run_cmd.empty?) then
      run_cmd = nil
    end
    cgi_doc = CGIDocument.new(cgi_dir, run_cmd, nph)
    cgi_doc.pass_args = pass_args
    cgi_doc.pass_auth = pass_auth
    NARGS.times do
      name = args.shift
      value = args.shift
      if (name && ! name.empty?) then
	cgi_doc.env[name] = value || ''
      end
    end
    cgi_doc
  end
end
