# $Id: server.rb,v 1.13 2004/12/11 15:50:44 toki Exp $

require 'time'
require 'socket'
require 'thread'
require 'forwardable'
require 'rucy/logger'
require 'rucy/messenger'
require 'rucy/document'

module Rucy
  class ServerStopException < Exception
  end

  class ServerRestartException < Exception
  end

  class Server
    extend Forwardable
    include MessengerParent
    include LoggingLevel

    def self.daemon
      catch(:START_DAEMON) do
	unless (fork) then
	  Process.setsid
	  unless (fork) then
	    #Dir.chdir('/')
	    STDIN.reopen('/dev/null', 'r')
	    STDOUT.reopen('/dev/null', 'w')
	    STDERR.reopen('/dev/null', 'w')
	    throw :START_DAEMON
	  end
	end
	exit!
      end
    end

    PORT = 8888

    EMPTY_HOOK = proc{|server|
      # nothing to do.
    }

    def initialize
      @start_time = Time.now
      @restart_count = 0
      @restart_time = nil
      @open_hook = EMPTY_HOOK
      @close_hook = EMPTY_HOOK
      @accept_lock = Mutex.new
      @accept_state = :close
      @accept_thread = nil
      @socket_queue = SocketQueue.new
      reset
    end

    attr_reader :start_time
    attr_reader :restart_count
    attr_reader :restart_time

    def open_hook(&block)
      @open_hook = block
      nil
    end

    def close_hook(&block)
      @close_hook = block
      nil
    end

    def call_hook(name, hook)
      begin
	hook.call(self)
      rescue StandardError, ScriptError
	err_messg = "#{name} hook error: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}"
	begin
	  STDERR.puts err_messg
	rescue StandardError, ScriptError
	  # nothing to do.
	end
	begin
	  @logger.err(err_messg)
	rescue StandardError, ScriptError
	  # nothing to do.
	end
      end

      nil
    end
    private :call_hook

    def reset
      @bind_address = nil
      @port = PORT
      @server = nil
      @multi_logger = MultiLogger.new
      @multi_access_log = MultiAccessLog.new
      @logger = SyncLogger.new(@multi_logger)
      @access_log = SyncAccessLog.new(@multi_access_log)
      @restart_signal = nil
      @folder = FolderDocument.new
      @document = @folder
      @timeout = Messenger::TIMEOUT
      @keep_alive = Messenger::KEEP_ALIVE
      @max_requests = Messenger::MAX_REQUESTS
      @queue_length = Messenger::QUEUE_LENGTH
      @messengers = Messenger::MESSENGERS
      @messenger_factory = MultiThreadMessenger
      @messenger_threads = Messenger::MESSENGER_THREADS
      @messenger_queue_length = Messenger::MESSENGER_QUEUE_LENGTH
      @privilege = nil
      nil
    end
    private :reset

    def_delegator(:@multi_logger, :add, :add_logger)
    def_delegator(:@multi_access_log, :add, :add_access_log)

    def logging_access(fmt=AccessLog::COMMON_LOG_FORMAT)
      @multi_access_log.add(AccessLogAdapter.new(@logger, fmt))
      nil
    end

    # logging interfaces from outside.
    def_delegator(:@logger, :messg)

    def restart_signal=(restart_signal)
      @restart_signal = restart_signal
      @restart_signal.server = self
      restart_signal
    end

    attr_accessor :bind_address
    attr_accessor :port
    attr_accessor :document
    attr_accessor :queue_length
    attr_accessor :timeout
    attr_accessor :keep_alive
    attr_accessor :max_requests
    attr_accessor :messengers
    attr_accessor :messenger_factory
    attr_accessor :messenger_threads
    attr_accessor :messenger_queue_length
    attr_accessor :privilege

    def add_filter(filter)
      @document = FilterDocument.new(@document, filter)
      nil
    end

    def_delegator(:@folder, :mount)
    def_delegator(:@folder, :attach)
    def_delegator(:@folder, :umount)
    def_delegator(:@folder, :set_alias)
    def_delegator(:@folder, :virtual_mount)
    def_delegator(:@folder, :virtual_attach)
    def_delegator(:@folder, :virtual_umount)
    def_delegator(:@folder, :set_virtual_alias)

    def close
      @accept_lock.synchronize{
	if (@accept_state == :open) then
	  @accept_state = :stop
	  @accept_thread.raise(ServerStopException, 'stop server')
	end
      }

      nil
    end

    def restart
      @accept_lock.synchronize{
	if (@accept_state == :open) then
	  @accept_state = :restart
	  @accept_thread.raise(ServerRestartException, 'restart server')
	end
      }

      nil
    end

    def open_document
      for document in @document
	begin
	  document.open
	rescue StandardError, ScriptError
	  @logger.warning("failed to open a document: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	end
      end
      nil
    end
    private :open_document

    def close_document
      for document in @document
	begin
	  document.close
	rescue StandardError, ScriptError
	  @logger.warning("failed to close a document: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	end
      end
      nil
    end
    private :close_document

    def open_server
      call_hook('open', @open_hook)
      open_document
      @socket_queue.resize(@queue_length)
      @messenger_thgrp = start_messengers(@messengers, @socket_queue, @document, @logger, @access_log, @messenger_factory, @restart_signal)
      if (@bind_address) then
	@server = TCPServer.open(@bind_address, @port)
      else
	@server = TCPServer.open(@port)
      end
      family, port, host, ip = @server.addr
      @logger.notice("open server: #{family}:#{host}[#{ip}]:#{port}")

      nil
    end
    private :open_server

    def close_server
      if (@server) then
	family, port, host, ip = @server.addr
	@logger.notice("close server: #{family}:#{host}[#{ip}]:#{port}")
	@server.close
      end
      stop_messengers(@messengers, @socket_queue, @messenger_thgrp, @restart_signal)
      close_document
      call_hook('close', @close_hook)

      nil
    end
    private :close_server

    def accept
      begin
	begin
	  open_server
	  @accept_lock.synchronize{
	    @accept_state = :open
	    @accept_thread = Thread.current
	  }
	  loop do
	    begin
	      socket = @server.accept
	      socket.setsockopt(Socket::IPPROTO_TCP, Socket::TCP_NODELAY, 1) # for fast response.
	    rescue
	      @logger.warn("accept error: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	      retry
	    end
	    @socket_queue.push(socket)
	  end
	rescue ServerRestartException
	  sleep(0.1)		# wait for intercepting accept
	  close_server
	  reset
	  @restart_count += 1
	  @restart_time = Time.now
	  retry
	end
      rescue ServerStopException
	sleep(0.1)		# wait for intercepting accept
      ensure
	@accept_lock.synchronize{
	  @accept_state = :close
	  @accept_thread = nil
	}
	close_server
	until (@socket_queue.empty?)
	  socket = @socket_queue.pop
	  socket.shutdown
	end
      end

      nil
    end
  end
end
