#!/usr/local/bin/ruby
# $Id: test_request.rb,v 1.2 2004/04/22 08:42:11 toki Exp $

require 'rubyunit'
require 'pseudo_io'
require 'test_message'
require 'rucy/request'

module TestRucy
  class TestRequest < TestMessage
    def setup
      @request = Rucy::Request.new
      @messg = @request
    end

    def test_Request_normalize
      assert_equal('/',         Rucy::Request.normalize('/')[0])
      assert_equal('/foo/bar',  Rucy::Request.normalize('/foo/./bar')[0])
      assert_equal('/bar',      Rucy::Request.normalize('/foo/../bar')[0])
      assert_equal('/foo/',     Rucy::Request.normalize('/foo/.')[0])
      assert_equal('/foo/',     Rucy::Request.normalize('/foo/bar/..')[0])
      assert_equal('/foo',      Rucy::Request.normalize('/../foo')[0])
      assert_equal('/foo',      Rucy::Request.normalize('/../foo')[0])
      assert_equal('/',         Rucy::Request.normalize('/..')[0])
      assert_equal('/foo',      Rucy::Request.normalize('/foo?bar')[0])
      assert_equal('/foo bar',  Rucy::Request.normalize('/foo+bar')[0])
      assert_equal('/foo/~bar', Rucy::Request.normalize('/foo/%7Ebar')[0])

      assert_nil(Rucy::Request.normalize('/')[1])
      assert_nil(Rucy::Request.normalize('/foo/./bar')[1])
      assert_nil(Rucy::Request.normalize('/foo/../bar')[1])
      assert_nil(Rucy::Request.normalize('/foo/.')[1])
      assert_nil(Rucy::Request.normalize('/foo/bar/..')[1])
      assert_nil(Rucy::Request.normalize('/../foo')[1])
      assert_nil(Rucy::Request.normalize('/../foo')[1])
      assert_nil(Rucy::Request.normalize('/..')[1])
      assert_nil(Rucy::Request.normalize('/foo+bar')[1])
      assert_nil(Rucy::Request.normalize('/foo/%7Ebar')[1])

      assert_equal('/foo',   Rucy::Request.normalize('/foo?')[0])
      assert_equal('',       Rucy::Request.normalize('/foo?')[1])
      assert_equal('/foo',   Rucy::Request.normalize('/foo?bar')[0])
      assert_equal('bar',    Rucy::Request.normalize('/foo?bar')[1])
      assert_equal('/foo',   Rucy::Request.normalize('/foo?+bar')[0])
      assert_equal('+bar',   Rucy::Request.normalize('/foo?+bar')[1])
      assert_equal('/foo',   Rucy::Request.normalize('/foo?%7Ebar')[0])
      assert_equal('%7Ebar', Rucy::Request.normalize('/foo?%7Ebar')[1])
    end

    def test_Request_scan
      path_pair_list = [
	[ '', '' ]
      ]
      Rucy::Request.scan('') do |script_name, path_info|
	assert_equal(path_pair_list[0][0], script_name)
	assert_equal(path_pair_list[0][1], path_info)
	path_pair_list.shift
      end
      assert(path_pair_list.empty?)

      path_pair_list = [
	[ '', '' ]
      ]
      Rucy::Request.scan('/') do |script_name, path_info|
	assert_equal(path_pair_list[0][0], script_name)
	assert_equal(path_pair_list[0][1], path_info)
	path_pair_list.shift
      end
      assert(path_pair_list.empty?)

      path_pair_list = [
	[ '/foo/bar/baz', '' ],
	[ '/foo/bar', '/baz' ],
	[ '/foo', '/bar/baz' ],
	[ '', '/foo/bar/baz' ]
      ]
      Rucy::Request.scan('/foo/bar/baz') do |script_name, path_info|
	assert_equal(path_pair_list[0][0], script_name)
	assert_equal(path_pair_list[0][1], path_info)
	path_pair_list.shift
      end
      assert(path_pair_list.empty?)

      path_pair_list = [
	[ '/foo/bar/baz/', '' ],
	[ '/foo/bar/baz', '/' ],
	[ '/foo/bar', '/baz/' ],
	[ '/foo', '/bar/baz/' ],
	[ '', '/foo/bar/baz/' ]
      ]
      Rucy::Request.scan('/foo/bar/baz/') do |script_name, path_info|
	assert_equal(path_pair_list[0][0], script_name)
	assert_equal(path_pair_list[0][1], path_info)
	path_pair_list.shift
      end
      assert(path_pair_list.empty?)
    end

    def test_uri
      assert_nil(@request.uri)

      assert_equal('/', @request.uri = '/')
      assert_equal('/', @request.uri)
      assert_equal('/', @request.path)
      assert_nil(@request.query)

      assert_equal('/foo/./bar', @request.uri = '/foo/./bar')
      assert_equal('/foo/./bar', @request.uri)
      assert_equal('/foo/bar', @request.path)
      assert_nil(@request.query)

      assert_equal('/foo/../bar', @request.uri = '/foo/../bar')
      assert_equal('/foo/../bar', @request.uri)
      assert_equal('/bar', @request.path)
      assert_nil(@request.query)

      assert_equal('/foo/.', @request.uri = '/foo/.')
      assert_equal('/foo/.', @request.uri)
      assert_equal('/foo/', @request.path)
      assert_nil(@request.query)

      assert_equal('/foo/bar/..', @request.uri = '/foo/bar/..')
      assert_equal('/foo/bar/..', @request.uri)
      assert_equal('/foo/', @request.path)
      assert_nil(@request.query)

      assert_equal('/../foo', @request.uri = '/../foo')
      assert_equal('/../foo', @request.uri)
      assert_equal('/foo', @request.path)
      assert_nil(@request.query)

      assert_equal('/../foo', @request.uri = '/../foo')
      assert_equal('/../foo', @request.uri)
      assert_equal('/foo', @request.path)
      assert_nil(@request.query)

      assert_equal('/..', @request.uri = '/..')
      assert_equal('/..', @request.uri)
      assert_equal('/', @request.path)
      assert_nil(@request.query)

      assert_equal('/foo+bar', @request.uri = '/foo+bar')
      assert_equal('/foo+bar', @request.uri)
      assert_equal('/foo bar', @request.path)
      assert_nil(@request.query)

      assert_equal('/foo/%7Ebar', @request.uri = '/foo/%7Ebar')
      assert_equal('/foo/%7Ebar', @request.uri)
      assert_equal('/foo/~bar', @request.path)
      assert_nil(@request.query)

      assert_equal('/foo?', @request.uri = '/foo?')
      assert_equal('/foo?', @request.uri)
      assert_equal('/foo', @request.path)
      assert_equal('', @request.query)

      assert_equal('/foo?bar', @request.uri = '/foo?bar')
      assert_equal('/foo?bar', @request.uri)
      assert_equal('/foo', @request.path)
      assert_equal('bar', @request.query)

      assert_equal('/foo?+bar', @request.uri = '/foo?+bar')
      assert_equal('/foo?+bar', @request.uri)
      assert_equal('/foo', @request.path)
      assert_equal('+bar', @request.query)

      assert_equal('/foo?%7Ebar', @request.uri = '/foo?%7Ebar')
      assert_equal('/foo?%7Ebar', @request.uri)
      assert_equal('/foo', @request.path)
      assert_equal('%7Ebar', @request.query)

      assert_equal('http://localhost:8080', @request.uri = 'http://localhost:8080')
      assert_equal('http://localhost:8080', @request.uri)
      assert_equal('/', @request.path)
      assert_nil(@request.query)
      assert_equal('localhost:8080', @request.header('Host'))

      assert_equal('http://localhost:8080/foo', @request.uri = 'http://localhost:8080/foo')
      assert_equal('http://localhost:8080/foo', @request.uri)
      assert_equal('/foo', @request.path)
      assert_nil(@request.query)
      assert_equal('localhost:8080', @request.header('Host'))

      assert_equal('http://localhost:8080/foo?bar', @request.uri = 'http://localhost:8080/foo?bar')
      assert_equal('http://localhost:8080/foo?bar', @request.uri)
      assert_equal('/foo', @request.path)
      assert_equal('bar', @request.query)
      assert_equal('localhost:8080', @request.header('Host'))

      assert_equal('test:hello', @request.uri = 'test:hello')
      assert_equal('test:hello', @request.uri)
      assert_nil(@request.path)
      assert_nil(@request.query)
    end

    def test_line
      assert_equal('- - -', @request.line)
      @request.method = 'GET'
      assert_equal('GET - -', @request.line)
      @request.path = '/'
      assert_equal('GET / -', @request.line)
      @request.version = 'HTTP/1.1'
      assert_equal('GET / HTTP/1.1', @request.line)
      @request.query = 'foo=1&bar=2'
      assert_equal('GET /?foo=1&bar=2 HTTP/1.1', @request.line)
    end

    def test_subpath
      @request.path = '/'
      assert_equal([ '', '/' ], @request.subpath(''))
      assert_equal([ '', '/' ], @request.subpath('/'))
      @request.path = '/foo'
      assert_equal([ '', '/foo' ], @request.subpath(''))
      assert_equal([ '', '/foo' ], @request.subpath('/'))
      assert_equal([ '/foo', '' ], @request.subpath('/foo'))
      @request.path = '/foo/'
      assert_equal([ '', '/foo/' ], @request.subpath(''))
      assert_equal([ '', '/foo/' ], @request.subpath('/'))
      assert_equal([ '/foo', '/' ], @request.subpath('/foo'))
      @request.path = '/foo/bar'
      assert_equal([ '', '/foo/bar' ], @request.subpath(''))
      assert_equal([ '', '/foo/bar' ], @request.subpath('/'))
      assert_equal([ '/foo', '/bar' ], @request.subpath('/foo'))

      assert_exception(RuntimeError) {
	@request.path = '/foo/bar'
	@request.subpath('/baz')
      }
    end

    def test_parse_line
      assert_nil(@request.method)
      assert_nil(@request.path)
      assert_nil(@request.version)

      pio = PseudoIO.new
      pio << "GET /foo/bar HTTP/1.0\r\n"
      pio << "Host: localhost:8080\r\n"

      @request.parse_line(pio)
      assert_equal('GET', @request.method)
      assert_equal('/foo/bar', @request.path)
      assert_equal('HTTP/1.0', @request.version)
      assert_equal("Host: localhost:8080\r\n", pio.read)
    end

    def test_parse_line_with_LF
      assert_nil(@request.method)
      assert_nil(@request.path)
      assert_nil(@request.version)

      pio = PseudoIO.new
      pio << "GET /foo/bar HTTP/1.0\n"
      pio << "Host: localhost:8080\n"

      @request.parse_line(pio)
      assert_equal('GET', @request.method)
      assert_equal('/foo/bar', @request.path)
      assert_equal('HTTP/1.0', @request.version)
      assert_equal("Host: localhost:8080\n", pio.read)
    end

    def test_parse_line_with_preamble
      assert_nil(@request.method)
      assert_nil(@request.path)
      assert_nil(@request.version)

      pio = PseudoIO.new
      pio << "\r\n"
      pio << "\r\n"
      pio << "\r\n"
      pio << "GET /foo/bar HTTP/1.0\r\n"
      pio << "Host: localhost:8080\r\n"

      @request.parse_line(pio)
      assert_equal('GET', @request.method)
      assert_equal('/foo/bar', @request.path)
      assert_equal('HTTP/1.0', @request.version)
      assert_equal("Host: localhost:8080\r\n", pio.read)
    end

    def test_parse_line_without_path
      pio = PseudoIO.new
      pio << "GET \r\n"

      assert_exception(Rucy::ParseError) {
	@request.parse_line(pio)
      }
    end

    def test_parse_line_HTTP_0_9
      assert_nil(@request.method)
      assert_nil(@request.path)
      assert_nil(@request.version)

      pio = PseudoIO.new
      pio << "GET /foo/bar\r\n"
      pio << "End_of_Request"

      @request.parse_line(pio)
      assert_equal('GET', @request.method)
      assert_equal('/foo/bar', @request.path)
      assert_equal('HTTP/0.9', @request.version)
      assert_equal('End_of_Request', pio.read)
    end

    def test_parse_line_with_invalid_version
      pio = PseudoIO.new
      pio << "GET /foo/bar HTTP/6.6.6\r\n"

      assert_exception(Rucy::ParseError) {
	@request.parse_line(pio)
      }
    end

    def test_parse
      assert_nil(@request.method)
      assert_nil(@request.path)
      assert_nil(@request.version)
      assert_nil(@request.header('Host'))

      pio = PseudoIO.new
      pio << "GET /foo/bar HTTP/1.0\r\n"
      pio << "Host: localhost:8080\r\n"
      pio << "\r\n"
      pio << "End_of_Request"

      @request.parse(pio)
      assert_equal('GET', @request.method)
      assert_equal('/foo/bar', @request.path)
      assert_equal('HTTP/1.0', @request.version)
      assert_equal('localhost:8080', @request.header('Host'))
      assert_equal('End_of_Request', pio.read)
    end

    def test_conn_closed
      # HTTP/0.9
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/0.9'
      assert(@request.conn_closed?)

      # HTTP/1.0
      @request.version = 'HTTP/1.0'
      assert(@request.conn_closed?)
      @request.set_header('Connection', 'Keep-Alive')
      assert(! @request.conn_closed?)
      @request.delete_header('Connection')

      # HTTP/1.1
      @request.version = 'HTTP/1.1'
      assert(! @request.conn_closed?)
      @request.set_header('Connection', 'close')
      assert(@request.conn_closed?)
    end

    def test_set_reader
      @request.method = 'POST'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Connection', 'close')
      assert(@request.conn_closed?)

      # closed connection
      assert_exception(RuntimeError) { @request.each_body{} }
      assert_exception(RuntimeError) { @request.fetch_body }
      assert_exception(RuntimeError) { @request.each_line{} }
      assert_exception(RuntimeError) { @request.fetch_lines }
      @request.set_reader(PseudoIO.new)
      @request.each_body{}	# no exception!
      @request.set_reader(PseudoIO.new)
      @request.fetch_body	# no exception!
      @request.set_reader(PseudoIO.new)
      @request.each_line{}	# no exception!
      @request.set_reader(PseudoIO.new)
      @request.fetch_lines	# no exception!

      @request.delete_header('Connection')
      assert(! @request.conn_closed?)

      # keep-alive connection
      assert_exception(Rucy::HTTPError) { @request.set_reader(PseudoIO.new) }
      @request.set_header('Content-Length', '-1')
      assert_exception(Rucy::HTTPError) { @request.set_reader(PseudoIO.new) }
      @request.set_header('Content-Length', '0')
      @request.set_reader(PseudoIO.new)	# no exception!
      
    end

    def test_each_body_with_closed_conn
      @request.method = 'POST'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Connection', 'close')
      assert(@request.conn_closed?)

      pio = PseudoIO.new
      pio << 'ABCDEFG'
      @request.set_reader(pio)

      byte_list = %w[ A B C D E F G ]
      @request.each_body(1) do |messg|
	assert_equal(byte_list.first, messg)
	byte_list.shift
      end
      assert(byte_list.empty?)
      assert_exception(RuntimeError) { @request.each_body{} }

      pio = PseudoIO.new
      pio << 'ABCDEFG'
      @request.set_reader(pio)

      byte_list1 = %w[ A B C D ]
      @request.each_body(1) do |messg|
	assert_equal(byte_list1.first, messg)
	byte_list1.shift
	break if byte_list1.empty?
      end

      byte_list2 = %w[ E F G ]
      @request.each_body(1) do |messg|
	assert_equal(byte_list2.first, messg)
	byte_list2.shift
      end
      assert(byte_list2.empty?)
      assert_exception(RuntimeError) { @request.each_body{} }
    end

    def test_each_body_with_keep_alive
      @request.method = 'POST'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      assert(! @request.conn_closed?)

      pio = PseudoIO.new
      pio << 'ABCDEFG'
      pio << 'End_of_Body'
      @request.set_header('Content-Length', '7')
      @request.set_reader(pio)

      byte_list = %w[ A B C D E F G ]
      @request.each_body(1) do |messg|
	assert_equal(byte_list.first, messg)
	byte_list.shift
      end
      assert(byte_list.empty?)
      assert_exception(RuntimeError) { @request.each_body{} }
      assert_equal('End_of_Body', pio.read)

      pio = PseudoIO.new
      pio << 'ABCDEFG'
      pio << 'End_of_Body'
      @request.set_reader(pio)

      byte_list1 = %w[ A B C D ]
      @request.each_body(1) do |messg|
	assert_equal(byte_list1.first, messg)
	byte_list1.shift
	break if byte_list1.empty?
      end

      byte_list2 = %w[ E F G ]
      @request.each_body(1) do |messg|
	assert_equal(byte_list2.first, messg)
	byte_list2.shift
      end
      assert(byte_list2.empty?)
      assert_exception(RuntimeError) { @request.each_body{} }
      assert_equal('End_of_Body', pio.read)
    end

    def test_fetch_body_with_closed_conn
      @request.method = 'POST'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Connection', 'close')
      assert(@request.conn_closed?)

      pio = PseudoIO.new
      pio << "Hello world.\n"
      @request.set_reader(pio)

      assert_equal("Hello world.\n", @request.fetch_body)
      assert_exception(RuntimeError) { @request.fetch_body }
    end

    def test_fetch_body_with_keep_alive
      @request.method = 'POST'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      assert(! @request.conn_closed?)

      pio = PseudoIO.new
      pio << "Hello world.\n"
      pio << "End_of_Body"
      @request.set_header('Content-Length', "Hello world.\n".length.to_s)
      @request.set_reader(pio)

      assert_equal("Hello world.\n", @request.fetch_body)
      assert_exception(RuntimeError) { @request.fetch_body }
      assert_equal("End_of_Body", pio.read)
    end

    def test_each_line
      @request.method = 'POST'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Connection', 'close')

      pio = PseudoIO.new
      pio << "foo\n"
      pio << "bar\n"
      pio << "baz\n"
      @request.set_reader(pio)

      line_list = [ "foo\n", "bar\n", "baz\n" ]
      @request.each_line("\n") do |line|
	assert_equal(line_list.first, line)
	line_list.shift
      end
      assert(line_list.empty?)
      assert_exception(RuntimeError) { @request.each_line("\n") {} }

      pio = PseudoIO.new
      pio << "foo\n"
      pio << "bar\n"
      pio << "baz"
      @request.set_reader(pio)

      line_list = [ "foo\n", "bar\n", "baz" ]
      @request.each_line("\n") do |line|
	assert_equal(line_list.first, line)
	line_list.shift
      end
      assert(line_list.empty?)
      assert_exception(RuntimeError) { @request.each_line("\n") {} }

      pio = PseudoIO.new
      pio << "foo\n"
      pio << "bar\n"
      pio << "baz\n"
      @request.set_reader(pio)

      line_list1 = [ "foo\n" ]
      @request.each_line("\n") do |line|
	assert_equal(line_list1.first, line)
	line_list1.shift
	break if line_list1.empty?
      end

      line_list2 = [ "bar\n", "baz\n" ]
      @request.each_line("\n") do |line|
	assert_equal(line_list2.first, line)
	line_list2.shift
      end
      assert(line_list2.empty?)
      assert_exception(RuntimeError) { @request.each_line("\n") {} }
    end

    def test_fetch_lines
      @request.method = 'POST'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Connection', 'close')

      pio = PseudoIO.new
      pio << "foo\n"
      pio << "bar\n"
      pio << "baz\n"
      @request.set_reader(pio)

      assert_equal([ "foo\n", "bar\n", "baz\n" ], @request.fetch_lines("\n"))
      assert_exception(RuntimeError) { @request.fetch_lines("\n") }
    end

    def test_set_server
      @request.set_server('server', '192.168.0.1', 8080)
      assert_equal('server', @request.server_name)
      assert_equal('192.168.0.1', @request.server_address)
      assert_equal(8080, @request.server_port)
      assert_equal('server:8080', @request.host)

      @request.set_header('Host', 'server2:8080')
      assert_equal('server2:8080', @request.host)
    end

    def test_set_client
      @request.set_client('client', '192.168.0.2', 31415)
      assert_equal('client', @request.client_name)
      assert_equal('192.168.0.2', @request.client_address)
      assert_equal(31415, @request.client_port)
    end

    def test_cgi_env
      @request.method = 'GET'
      @request.uri = '/hello/world?foo=apple&bar=banana'
      @request.version = 'HTTP/1.1'
      @request.set_header('Content-Type', 'application/x-www-form-urlencoded')
      @request.set_header('Content-Length', '64')
      @request.set_header('User-Agent', 'Mozilla')
      @request.set_server('server', '192.168.0.1', 8080)
      @request.set_client('client', '192.168.0.2', 31415)
      cgi_env = @request.cgi_env('/hello')
      assert_equal('CGI/1.1', cgi_env['GATEWAY_INTERFACE'])
      assert_equal('GET', cgi_env['REQUEST_METHOD'])
      assert_equal('/hello', cgi_env['SCRIPT_NAME'])
      assert_equal('/world', cgi_env['PATH_INFO'])
      assert_equal('foo=apple&bar=banana', cgi_env['QUERY_STRING'])
      assert_equal('application/x-www-form-urlencoded', cgi_env['CONTENT_TYPE'])
      assert_equal('64', cgi_env['CONTENT_LENGTH'])
      assert_equal('server', cgi_env['SERVER_NAME'])
      assert_equal('8080', cgi_env['SERVER_PORT'])
      assert_equal('HTTP/1.1', cgi_env['SERVER_PROTOCOL'])
      assert_equal(Rucy::SERVER_TOKEN_LIST, cgi_env['SERVER_SOFTWARE'])
      assert_equal('client', cgi_env['REMOTE_HOST'])
      assert_equal('192.168.0.2', cgi_env['REMOTE_ADDR'])
      assert_equal('Mozilla', cgi_env['HTTP_USER_AGENT'])
    end
  end
end
