#!/usr/local/bin/ruby
# $Id: test_logger.rb,v 1.19 2004/10/03 11:53:28 toki Exp $

require 'time'
require 'rubyunit'
require 'rucy/logger'

module TestRucy
  class TestLogger < RUNIT::TestCase
    def setup
      @logger = Rucy::Logger.new(self)
      @write_call = 0
      @write_list = Array.new
      @flush_call = 0
    end

    def write(messg)
      @write_call += 1
      @write_list << messg
      nil
    end

    def flush
      @flush_call += 1
      nil
    end

    def test_logging_level_constants
      assert(Rucy::LoggingLevel::LOG_EMERG)
      assert(Rucy::LoggingLevel::LOG_ALERT)
      assert(Rucy::LoggingLevel::LOG_CRIT)
      assert(Rucy::LoggingLevel::LOG_ERR)
      assert(Rucy::LoggingLevel::LOG_WARNING)
      assert(Rucy::LoggingLevel::LOG_NOTICE)
      assert(Rucy::LoggingLevel::LOG_INFO)
      assert(Rucy::LoggingLevel::LOG_DEBUG)
    end

    def test_default_logging_level
      assert_equal(Rucy::LoggingLevel::LOG_NOTICE, @logger.level)
    end

    def test_emerg
      @logger.level = Rucy::LoggingLevel::LOG_EMERG
      @logger.emerg("emerg\n")
      @logger.alert("alert\n")
      @logger.crit("crit\n")
      @logger.err("err\n")
      @logger.warning("warning\n")
      @logger.notice("notice\n")
      @logger.info("info\n")
      @logger.debug("debug\n")
      assert_equal(1, @write_call)
      assert_equal(1, @flush_call)
      assert_match(@write_list[0], /: emerg$/)
    end

    def test_alert
      @logger.level = Rucy::LoggingLevel::LOG_ALERT
      @logger.emerg("emerg\n")
      @logger.alert("alert\n")
      @logger.crit("crit\n")
      @logger.err("err\n")
      @logger.warning("warning\n")
      @logger.notice("notice\n")
      @logger.info("info\n")
      @logger.debug("debug\n")
      assert_equal(2, @write_call)
      assert_equal(2, @flush_call)
      assert_match(@write_list[0], /: emerg$/)
      assert_match(@write_list[1], /: alert$/)
    end

    def test_crit
      @logger.level = Rucy::LoggingLevel::LOG_CRIT
      @logger.emerg("emerg\n")
      @logger.alert("alert\n")
      @logger.crit("crit\n")
      @logger.err("err\n")
      @logger.warning("warning\n")
      @logger.notice("notice\n")
      @logger.info("info\n")
      @logger.debug("debug\n")
      assert_equal(3, @write_call)
      assert_equal(3, @flush_call)
      assert_match(@write_list[0], /: emerg$/)
      assert_match(@write_list[1], /: alert$/)
      assert_match(@write_list[2], /: crit$/)
    end

    def test_err
      @logger.level = Rucy::LoggingLevel::LOG_ERR
      @logger.emerg("emerg\n")
      @logger.alert("alert\n")
      @logger.crit("crit\n")
      @logger.err("err\n")
      @logger.warning("warning\n")
      @logger.notice("notice\n")
      @logger.info("info\n")
      @logger.debug("debug\n")
      assert_equal(4, @write_call)
      assert_equal(4, @flush_call)
      assert_match(@write_list[0], /: emerg$/)
      assert_match(@write_list[1], /: alert$/)
      assert_match(@write_list[2], /: crit$/)
      assert_match(@write_list[3], /: err$/)
    end

    def test_warning
      @logger.level = Rucy::LoggingLevel::LOG_WARNING
      @logger.emerg("emerg\n")
      @logger.alert("alert\n")
      @logger.crit("crit\n")
      @logger.err("err\n")
      @logger.warning("warning\n")
      @logger.notice("notice\n")
      @logger.info("info\n")
      @logger.debug("debug\n")
      assert_equal(5, @write_call)
      assert_equal(5, @flush_call)
      assert_match(@write_list[0], /: emerg$/)
      assert_match(@write_list[1], /: alert$/)
      assert_match(@write_list[2], /: crit$/)
      assert_match(@write_list[3], /: err$/)
      assert_match(@write_list[4], /: warning$/)
    end

    def test_notice
      @logger.level = Rucy::LoggingLevel::LOG_NOTICE
      @logger.emerg("emerg\n")
      @logger.alert("alert\n")
      @logger.crit("crit\n")
      @logger.err("err\n")
      @logger.warning("warning\n")
      @logger.notice("notice\n")
      @logger.info("info\n")
      @logger.debug("debug\n")
      assert_equal(6, @write_call)
      assert_equal(6, @flush_call)
      assert_match(@write_list[0], /: emerg$/)
      assert_match(@write_list[1], /: alert$/)
      assert_match(@write_list[2], /: crit$/)
      assert_match(@write_list[3], /: err$/)
      assert_match(@write_list[4], /: warning$/)
      assert_match(@write_list[5], /: notice$/)
    end

    def test_info
      @logger.level = Rucy::LoggingLevel::LOG_INFO
      @logger.emerg("emerg\n")
      @logger.alert("alert\n")
      @logger.crit("crit\n")
      @logger.err("err\n")
      @logger.warning("warning\n")
      @logger.notice("notice\n")
      @logger.info("info\n")
      @logger.debug("debug\n")
      assert_equal(7, @write_call)
      assert_equal(7, @flush_call)
      assert_match(@write_list[0], /: emerg$/)
      assert_match(@write_list[1], /: alert$/)
      assert_match(@write_list[2], /: crit$/)
      assert_match(@write_list[3], /: err$/)
      assert_match(@write_list[4], /: warning$/)
      assert_match(@write_list[5], /: notice$/)
      assert_match(@write_list[6], /: info$/)
    end

    def test_debug
      @logger.level = Rucy::LoggingLevel::LOG_DEBUG
      @logger.emerg("emerg\n")
      @logger.alert("alert\n")
      @logger.crit("crit\n")
      @logger.err("err\n")
      @logger.warning("warning\n")
      @logger.notice("notice\n")
      @logger.info("info\n")
      @logger.debug("debug\n")
      assert_equal(8, @write_call)
      assert_equal(8, @flush_call)
      assert_match(@write_list[0], /: emerg$/)
      assert_match(@write_list[1], /: alert$/)
      assert_match(@write_list[2], /: crit$/)
      assert_match(@write_list[3], /: err$/)
      assert_match(@write_list[4], /: warning$/)
      assert_match(@write_list[5], /: notice$/)
      assert_match(@write_list[6], /: info$/)
      assert_match(@write_list[7], /: debug$/)
    end

    def test_messg
      @logger.messg(Rucy::LoggingLevel::LOG_INFO, 'HALO', false)
      @logger.messg(Rucy::LoggingLevel::LOG_INFO, 'HALO', true)
      @logger.messg(Rucy::LoggingLevel::LOG_INFO, 'HALO')
      assert_equal(0, @write_call)
      assert_equal(0, @flush_call)

      @logger.messg(Rucy::LoggingLevel::LOG_NOTICE, 'HALO', false)
      assert_equal(1, @write_call)
      assert_equal(1, @flush_call)
      assert_equal('HALO', @write_list[0])
      @logger.messg(Rucy::LoggingLevel::LOG_NOTICE, 'HALO', true)
      assert_equal(2, @write_call)
      assert_equal(2, @flush_call)
      assert_match(@write_list[1], /: HALO\n/)
      @logger.messg(Rucy::LoggingLevel::LOG_NOTICE, 'HALO')
      assert_equal(3, @write_call)
      assert_equal(3, @flush_call)
      assert_match(@write_list[2], /: HALO\n/)
    end
  end

  class TestMultiLogger < RUNIT::TestCase
    class DummyOutput
      def initialize(parent, name)
	@write = parent.method("#{name}_write")
	@flush = parent.method("#{name}_flush")
      end

      def write(messg)
	@write.call(messg)
	nil
      end

      def flush
	@flush.call
	nil
      end
    end

    def setup
      @logger1_write_messg = ''
      @logger1_flush_call = 0
      @logger1 = Rucy::Logger.new(DummyOutput.new(self, 'logger1'))
      @logger2_write_messg = ''
      @logger2_flush_call = 0
      @logger2 = Rucy::Logger.new(DummyOutput.new(self, 'logger2'))
      @multi_logger = Rucy::MultiLogger.new
      @multi_logger.add(@logger1)
      @multi_logger.add(@logger2)
    end

    def logger1_write(messg)
      @logger1_write_messg << messg
    end

    def logger1_flush
      @logger1_flush_call += 1
      nil
    end

    def logger2_write(messg)
      @logger2_write_messg << messg
    end

    def logger2_flush
      @logger2_flush_call += 1
      nil
    end

    def test_multi_logger
      @multi_logger.notice('HALO')
      assert_match(@logger1_write_messg, /HALO/)
      assert_equal(1, @logger1_flush_call)
      assert_match(@logger2_write_messg, /HALO/)
      assert_equal(1, @logger2_flush_call)
    end
  end

  class TestSyncLogger < RUNIT::TestCase
    NUM_OF_THREAD = 2
    NUM_OF_CALL = 1000

    def setup
      @messg_call = 0
      @th_grp = ThreadGroup.new
      @logger = Rucy::SyncLogger.new(self)
    end

    def messg(level, messg, format=true)
      @messg_call += 1
    end

    def test_emerg
      spin_lock = true
      NUM_OF_THREAD.times do
	@th_grp.add Thread.new{
	  while (spin_lock)
	    # ready to start.
	  end
	  NUM_OF_CALL.times do
	    @logger.emerg("HALO\n")
	  end
	}
      end
      spin_lock = false
      for thread in @th_grp.list
	thread.join
      end
      assert_equal(NUM_OF_THREAD * NUM_OF_CALL, @messg_call)
    end

    def test_alert
      spin_lock = true
      NUM_OF_THREAD.times do
	@th_grp.add Thread.new{
	  while (spin_lock)
	    # ready to start.
	  end
	  NUM_OF_CALL.times do
	    @logger.alert("HALO\n")
	  end
	}
      end
      spin_lock = false
      for thread in @th_grp.list
	thread.join
      end
      assert_equal(NUM_OF_THREAD * NUM_OF_CALL, @messg_call)
    end

    def test_crit
      spin_lock = true
      NUM_OF_THREAD.times do
	@th_grp.add Thread.new{
	  while (spin_lock)
	    # ready to start.
	  end
	  NUM_OF_CALL.times do
	    @logger.crit("HALO\n")
	  end
	}
      end
      spin_lock = false
      for thread in @th_grp.list
	thread.join
      end
      assert_equal(NUM_OF_THREAD * NUM_OF_CALL, @messg_call)
    end

    def test_err
      spin_lock = true
      NUM_OF_THREAD.times do
	@th_grp.add Thread.new{
	  while (spin_lock)
	    # ready to start.
	  end
	  NUM_OF_CALL.times do
	    @logger.err("HALO\n")
	  end
	}
      end
      spin_lock = false
      for thread in @th_grp.list
	thread.join
      end
      assert_equal(NUM_OF_THREAD * NUM_OF_CALL, @messg_call)
    end

    def test_warning
      spin_lock = true
      NUM_OF_THREAD.times do
	@th_grp.add Thread.new{
	  while (spin_lock)
	    # ready to start.
	  end
	  NUM_OF_CALL.times do
	    @logger.warning("HALO\n")
	  end
	}
      end
      spin_lock = false
      for thread in @th_grp.list
	thread.join
      end
      assert_equal(NUM_OF_THREAD * NUM_OF_CALL, @messg_call)
    end

    def test_notice
      spin_lock = true
      NUM_OF_THREAD.times do
	@th_grp.add Thread.new{
	  while (spin_lock)
	    # ready to start.
	  end
	  NUM_OF_CALL.times do
	    @logger.notice("HALO\n")
	  end
	}
      end
      spin_lock = false
      for thread in @th_grp.list
	thread.join
      end
      assert_equal(NUM_OF_THREAD * NUM_OF_CALL, @messg_call)
    end

    def test_info
      spin_lock = true
      NUM_OF_THREAD.times do
	@th_grp.add Thread.new{
	  while (spin_lock)
	    # ready to start.
	  end
	  NUM_OF_CALL.times do
	    @logger.info("HALO\n")
	  end
	}
      end
      spin_lock = false
      for thread in @th_grp.list
	thread.join
      end
      assert_equal(NUM_OF_THREAD * NUM_OF_CALL, @messg_call)
    end

    def test_debug
      spin_lock = true
      NUM_OF_THREAD.times do
	@th_grp.add Thread.new{
	  while (spin_lock)
	    # ready to start.
	  end
	  NUM_OF_CALL.times do
	    @logger.debug("HALO\n")
	  end
	}
      end
      spin_lock = false
      for thread in @th_grp.list
	thread.join
      end
      assert_equal(NUM_OF_THREAD * NUM_OF_CALL, @messg_call)
    end
  end

  class TestAccessLog < RUNIT::TestCase
    def setup
      # for Rucy::Message class
      @header_call = 0
      @header_name = nil
      @header_value = nil

      # for Rucy::Request class
      @request = self
      @server_name_call = 0
      @server_address_call = 0
      @server_port_call = 0
      @host_call = 0
      @client_name_call = 0
      @client_address_call = 0
      @line_call = 0
      @method_call = 0
      @path_call = 0
      @query_call = 0
      @query_string = nil

      # for Rucy::Response class
      @response = self
      @status_call = 0
      @send_size_call = 0
      @send_size = nil
      @local_path_call = 0
      @local_path = nil

      # for IO
      @write_call = 0
      @write_data = nil
      @flush_call = 0

      # test target
      @access_log = Rucy::AccessLog.new(self)

      # misc.
      @curr_time = Time.gm(2004, 11, 19, 1, 13, 25)
    end

    def header(name)
      @header_call += 1
      @header_name = name
      @header_value
    end

    def server_name
      @server_name_call += 1
      'server'
    end

    def server_address
      @server_address_call += 1
      '192.168.0.1'
    end

    def server_port
      @server_port_call += 1
      80
    end

    def host
      @host_call += 1
      'server:80'
    end

    def client_name
      @client_name_call += 1
      'client'
    end

    def client_address
      @client_address_call += 1
      '192.168.0.2'
    end

    def line
      @line_call += 1
      'GET /foo/bar HTTP/1.1'
    end

    def method(*args, &block)
      if (args.empty? && block.nil?) then
	@method_call += 1
	return 'GET'
      else
	return super
      end
    end

    def path
      @path_call += 1
      '/foo/bar'
    end

    def query
      @query_call += 1
      @query_string
    end

    def status
      @status_call += 1
      200
    end

    def send_size
      @send_size_call += 1
      @send_size
    end

    def local_path
      @local_path_call += 1
      @local_path
    end

    def write(data)
      @write_call += 1
      @write_data = data
      nil
    end

    def flush
      @flush_call += 1
      nil
    end

    def test_AccessLog_format_plain
      assert_equal('HALO', Rucy::AccessLog.format('HALO', @request, @response, @curr_time))
    end

    def test_AccessLog_format_escape
      assert_equal('%', Rucy::AccessLog.format('%%', @request, @response, @curr_time))
    end

    def test_AccessLog_format_remote_host
      assert_equal('client', Rucy::AccessLog.format('%h', @request, @response, @curr_time))
      assert_equal(1, @client_name_call)
    end

    def test_AccessLog_format_remote_address
      assert_equal('192.168.0.2', Rucy::AccessLog.format('%a', @request, @response, @curr_time))
      assert_equal(1, @client_address_call)
    end

    def test_AccessLog_format_server_name
      assert_equal('server:80', Rucy::AccessLog.format('%v', @request, @response, @curr_time))
      assert_equal(1, @host_call)
    end

    def test_AccessLog_format_server_address
      assert_equal('192.168.0.1', Rucy::AccessLog.format('%A', @request, @response, @curr_time))
      assert_equal(1, @server_address_call)
    end

    def test_AccessLog_format_server_port
      assert_equal('80', Rucy::AccessLog.format('%p', @request, @response, @curr_time))
      assert_equal(1, @server_port_call)
    end

    def test_AccessLog_format_server_process_id
      assert_equal($$.to_s, Rucy::AccessLog.format('%P', @request, @response, @curr_time))
    end

    def test_AccessLog_format_bytes
      @send_size = 65536
      assert_equal('65536', Rucy::AccessLog.format('%B', @request, @response, @curr_time))
      assert_equal(1, @send_size_call)
      @send_size = 0
      assert_equal('0', Rucy::AccessLog.format('%B', @request, @response, @curr_time))
      assert_equal(2, @send_size_call)
    end

    def test_AccessLog_format_bytes2
      @send_size = 65536
      assert_equal('65536', Rucy::AccessLog.format('%b', @request, @response, @curr_time))
      assert_equal(1, @send_size_call)
      @send_size = 0
      assert_equal('-', Rucy::AccessLog.format('%b', @request, @response, @curr_time))
      assert_equal(2, @send_size_call)
    end

    def test_AccessLog_format_local_path
      @local_path = '/foo/bar'
      assert_equal('/foo/bar', Rucy::AccessLog.format('%f', @request, @response, @curr_time))
      assert_equal(1, @local_path_call)
      @local_path = nil
      assert_equal('-', Rucy::AccessLog.format('%f', @request, @response, @curr_time))
      assert_equal(2, @local_path_call)
    end

    def test_AccessLog_format_request_method
      assert_equal('GET', Rucy::AccessLog.format('%m', @request, @response, @curr_time))
      assert_equal(1, @method_call)
    end

    def test_AccessLog_format_query_string
      @query_string = 'foo'
      assert_equal('?foo', Rucy::AccessLog.format('%q', @request, @response, @curr_time))
      assert_equal(1, @query_call)
      @query_string = ''
      assert_equal('?', Rucy::AccessLog.format('%q', @request, @response, @curr_time))
      assert_equal(2, @query_call)
      @query_string = nil
      assert_equal('', Rucy::AccessLog.format('%q', @request, @response, @curr_time))
      assert_equal(3, @query_call)
    end

    def test_AccessLog_format_request_line
      assert_equal('GET /foo/bar HTTP/1.1', Rucy::AccessLog.format('%r', @request, @response, @curr_time))
      assert_equal(1, @line_call)
    end

    def test_AccessLog_format_request_path
      assert_equal('/foo/bar', Rucy::AccessLog.format('%U', @request, @response, @curr_time))
      assert_equal(1, @path_call)
    end

    def test_AccessLog_format_response_status
      assert_equal('200', Rucy::AccessLog.format('%s', @request, @response, @curr_time))
      assert_equal(1, @status_call)
    end

    def test_AccessLog_format_time
      assert_equal('Fri, 19 Nov 2004 01:13:25 GMT', Rucy::AccessLog.format('%t', @request, @response, @curr_time))
    end

    def test_AccessLog_format_input_header
      @header_value = 'PseudoMozilla/0.0.0'
      assert_equal('PseudoMozilla/0.0.0', Rucy::AccessLog.format('%{User-Agent}i', @request, @response, @curr_time))
      assert_equal(1, @header_call)
      assert_equal('User-Agent', @header_name)
      @header_value = nil
      assert_equal('-', Rucy::AccessLog.format('%{User-Agent2}i', @request, @response, @curr_time))
      assert_equal(2, @header_call)
      assert_equal('User-Agent2', @header_name)
    end

    def test_AccessLog_format_input_header
      @header_value = 'Test/0.0.0'
      assert_equal('Test/0.0.0', Rucy::AccessLog.format('%{Server}o', @request, @response, @curr_time))
      assert_equal(1, @header_call)
      assert_equal('Server', @header_name)
      @header_value = nil
      assert_equal('-', Rucy::AccessLog.format('%{Server2}o', @request, @response, @curr_time))
      assert_equal(2, @header_call)
      assert_equal('Server2', @header_name)
    end

    def test_AccessLog_format_unsupported
      assert_equal('-', Rucy::AccessLog.format('%D', @request, @response, @curr_time)) # elapsed time (milliseconds)
      assert_equal('-', Rucy::AccessLog.format('%T', @request, @response, @curr_time)) # elapsed time (seconds)
      assert_equal('-', Rucy::AccessLog.format('%l', @request, @response, @curr_time)) # ident
      assert_equal('-', Rucy::AccessLog.format('%u', @request, @response, @curr_time)) # remote user
    end

    def test_write_log
      @send_size = 65536
      @access_log.write_log(@request, @response, @curr_time) # CLF: %h %l %u %t "%r" %s %b
      assert_equal(1, @client_name_call) # %h
      # unsupported: %l
      # unsupported: %u
      # @curr_time: %t
      assert_equal(1, @line_call) # %r
      assert_equal(1, @status_call) # %s
      assert_equal(1, @send_size_call) # %b
      assert_equal(1, @write_call)
      assert_equal("client - - Fri, 19 Nov 2004 01:13:25 GMT \"GET /foo/bar HTTP/1.1\" 200 65536\n", @write_data)
      assert_equal(1, @flush_call)
    end
  end

  class TestMultiAccessLog < RUNIT::TestCase
    class DummyOutput
      def initialize(parent, name)
	@write = parent.method("#{name}_write")
	@flush = parent.method("#{name}_flush")
      end

      def write(messg)
	@write.call(messg)
	nil
      end

      def flush
	@flush.call
	nil
      end
    end

    def setup
      @log1_write_call = 0
      @log1_write_messg = ''
      @log1_flush_call = 0
      @log1 = Rucy::AccessLog.new(DummyOutput.new(self, 'log1'), 'foo')
      @log2_write_call = 0
      @log2_write_messg = ''
      @log2_flush_call = 0
      @log2 = Rucy::AccessLog.new(DummyOutput.new(self, 'log2'), 'bar')
      @access_log = Rucy::MultiAccessLog.new
      @access_log.add(@log1)
      @access_log.add(@log2)
    end

    def log1_write(messg)
      @log1_write_call += 1
      @log1_write_messg << messg
    end

    def log1_flush
      @log1_flush_call += 1
      nil
    end

    def log2_write(messg)
      @log2_write_call += 1
      @log2_write_messg << messg
    end

    def log2_flush
      @log2_flush_call += 1
      nil
    end

    def test_multi_access_log
      @access_log.write_log(self, self, Time.now)
      assert_equal(1, @log1_write_call)
      assert_equal("foo\n", @log1_write_messg)
      assert_equal(1, @log1_flush_call)
      assert_equal(1, @log2_write_call)
      assert_equal("bar\n", @log2_write_messg)
      assert_equal(1, @log2_flush_call)
    end
  end

  class TestSyncAccessLog < RUNIT::TestCase
    NUM_OF_THREAD = 2
    NUM_OF_CALL = 1000

    def setup
      @write_log_call = 0
      @th_grp = ThreadGroup.new
      @access_log = Rucy::SyncAccessLog.new(self)
    end

    def write_log(request, response, curr_time)
      @write_log_call += 1
      nil
    end

    def test_write_log
      spin_lock = true
      NUM_OF_THREAD.times do
	@th_grp.add Thread.new{
	  while (spin_lock)
	    # ready to start.
	  end
	  NUM_OF_CALL.times do
	    @access_log.write_log(self, self, Time.now)
	  end
	}
      end
      spin_lock = false
      for thread in @th_grp.list
	thread.join
      end
      assert_equal(NUM_OF_THREAD * NUM_OF_CALL, @write_log_call)
    end
  end

  class TestAccessLogAdapter < RUNIT::TestCase
    def setup
      # for Rucy::Logger class
      @messg_call = 0
      @messg_level = nil
      @messg_data = nil
      @messg_format = nil

      # target
      @adapter = Rucy::AccessLogAdapter.new(self, 'HALO')
    end

    def messg(level, messg, format=true)
      @messg_call += 1
      @messg_level = level
      @messg_data = messg
      @messg_format = format
      nil
    end

    def test_write_log
      @adapter.write_log(self, self, Time.now)
      assert_equal(1, @messg_call)
      assert_equal(Rucy::LoggingLevel::LOG_INFO, @messg_level)
      assert_equal("HALO\n", @messg_data)
      assert_equal(false, @messg_format)
    end
  end
end
