#!/usr/local/bin/ruby
# $Id: test_document.rb,v 1.8 2004/12/12 00:20:34 toki Exp $

require 'singleton'
require 'rubyunit'
require 'forwarder'
require 'pseudo_req_res'
require 'rucy/request'
require 'rucy/response'
require 'rucy/document'

module TestRucy
  class TestHTMLUtil < RUNIT::TestCase
    def test_escapeHTML
      assert_equal('', Rucy::HTMLUtil.escapeHTML(''))
      assert_equal('&lt;,&gt;,&amp;,&quot;', Rucy::HTMLUtil.escapeHTML('<,>,&,"'))
    end
  end

  class TestDocument < RUNIT::TestCase
    def setup
      @doc = Rucy::Document.new
    end

    def test_open
      @doc.open
    end

    def test_close
      @doc.close
    end

    def test_each
      for doc in @doc
	assert_equal(@doc, doc)
      end
    end
  end

  class TestFilter < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include PseudoDocument
    include PseudoFilter

    def setup
      super
      @filter_document = Rucy::FilterDocument.new(@document, @filter)
    end

    def test_filter_init
      @filter_document.open
      assert_equal(1, @init_call)
    end

    def test_filter_final
      @filter_document.close
      assert_equal(1, @final_call)
    end

    def test_each
      doc_list = [ @filter_document, @document ]
      for doc in @filter_document
	assert(doc_list.delete(doc))
      end
      assert(doc_list.empty?)
    end

    def test_GET
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @filter_document.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal(1, @filter_open_call)
      assert_equal(1, @filter_head_call)
      assert_equal(1, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('', @filter_script_name)
      assert_equal('/', @filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      log = "pseudo filter_open.\n"
      log += "pseudo publish.\n"
      log += "pseudo filter_head.\n"
      log += "pseudo filter_body.\n"
      log += "pseudo filter_close.\n"
      log_pat = Regexp.compile(Regexp.quote(log) + "\\z")
      assert_match(@log_debug, log_pat)
    end
  end

  class TestFilterTermination < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include PseudoDocument
    include PseudoFilter

    class FilterTermination < Rucy::Filter
      def initialize(parent)
	@parent = parent
      end

      def filter_open(context, script_name, request, response, logger)
	@parent.filter_open(context, script_name, request, response, logger)
	response.status = 404	# Not Found
	response.set_header('Content-Type', 'text/html')
	response.start_body
	response << "<html>bar</html>\n"
	terminate_filter	# skip to filter_close.
	nil
      end

      def filter_head(context, script_name, request, response, logger)
	@parent.filter_head(context, script_name, request, response, logger)
	nil
      end

      def filter_body(context, script_name, request, response, logger, messg_body)
	@parent.filter_body(context, script_name, request, response, logger, messg_body)
	nil
      end

      def filter_close(context, script_name, request, response, logger)
	@parent.filter_close(context, script_name, request, response, logger)
	nil
      end
    end

    def setup
      super
      @filter_document = Rucy::FilterDocument.new(@document, FilterTermination.new(@filter))
    end

    def test_GET
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @filter_document.publish('', @request, @response, @logger)
      assert_equal(0, @publish_call)
      assert_equal(1, @filter_open_call)
      assert_equal(0, @filter_head_call)
      assert_equal(0, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(404, @messg_head.status)
      assert_equal('Not Found', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal("<html>bar</html>\n", @messg_body)
    end
  end

  class TestDocumentFactory < RUNIT::TestCase
    def setup
      # for Rucy::DocumentBuilder class
      @doc_name_call = 0
      @doc_args_call = 0
      @set_doc_option_call = 0
      @set_doc_option_value = nil
      @doc_new_call = 0
      @doc_new_string_value = nil
      @doc_new_text_value = nil
      @doc_new_number_value = nil

      @doc_builder = Forwarder.new(self)
      class << @doc_builder
	def_delegator :__getobj__, :doc_name
	def_delegator :__getobj__, :doc_args
	def_delegator :__getobj__, :set_doc_option
	def_delegator :__getobj__, :doc_new, :new
      end

      # for Rucy::FilterBuilder class
      @filter_name_call = 0
      @filter_args_call = 0
      @set_filter_option_call = 0
      @set_filter_option_value = nil
      @filter_new_call = 0
      @filter_new_string_value = nil
      @filter_new_text_value = nil
      @filter_new_number_value = nil

      @filter_builder = Forwarder.new(self)
      class << @filter_builder
	def_delegator :__getobj__, :filter_name
	def_delegator :__getobj__, :filter_args
	def_delegator :__getobj__, :set_filter_option
	def_delegator :__getobj__, :filter_new, :new
      end

      # target
      @factory = Rucy::DocumentFactory.new
    end

    # for Rucy::DocumentBuilder class

    def doc_name
      @doc_name_call += 1
      'test_document'
    end

    def doc_args
      @doc_args_call += 1
      [ [ 'string',  :string,  nil              ],
	[ 'text',    :text,    "Hello world.\n" ],
	[ 'number',  :number,  0                ]
      ]
    end

    def set_doc_option(name, value)
      @set_doc_option_call += 1
      @set_doc_option_last_name = name
      @set_doc_option_last_value = value
      nil
    end

    def set_doc_option(option)
      @set_doc_option_call += 1
      @set_doc_option_value = option
      nil
    end

    def doc_new(string, text, number)
      @doc_new_call += 1
      @doc_new_string_value = string
      @doc_new_text_value = text
      @doc_new_number_value = number
      :test_document
    end

    # for Rucy::FilterBuilder class

    def filter_name
      @filter_name_call += 1
      'test_filter'
    end

    def filter_args
      @filter_args_call += 1
      [ [ 'string',  :string,  nil              ],
	[ 'text',    :text,    "Hello world.\n" ],
	[ 'number',  :number,  0                ]
      ]
    end

    def set_filter_option(name, value)
      @set_filter_option_call += 1
      @set_filter_option_last_name = name
      @set_filter_option_last_value = value
      nil
    end

    def set_filter_option(option)
      @set_filter_option_call += 1
      @set_filter_option_value = option
      nil
    end

    def filter_new(string, text, number)
      @filter_new_call += 1
      @filter_new_string_value = string
      @filter_new_text_value = text
      @filter_new_number_value = number
      :test_filter
    end

    # test

    def test_setup
      @factory.add_option(:foo, 'apple')
      @factory.add_option(:bar, 'banana')
      @factory.add_document(@doc_builder)
      @factory.add_filter(@filter_builder)
      @factory.setup
      assert_equal(1, @set_doc_option_call)
      assert_equal('apple', @set_doc_option_value[:foo])
      assert_equal('banana', @set_doc_option_value[:bar])
      assert_equal(1, @set_filter_option_call)
      assert_equal('apple', @set_filter_option_value[:foo])
      assert_equal('banana', @set_filter_option_value[:bar])
    end

    def test_has_document?
      assert(! (@factory.has_document? 'test_document'))
      @factory.add_document(@doc_builder)
      assert((@factory.has_document? 'test_document'))
    end

    def test_doc_names
      @factory.add_document(@doc_builder)
      assert_equal(1, @doc_name_call)
      doc_names = @factory.doc_names
      assert_equal(1, doc_names.length)
      assert_equal('test_document', doc_names[0])
    end

    def test_doc_args
      @factory.add_document(@doc_builder)
      assert_equal(1, @doc_name_call)
      doc_args = @factory.doc_args('test_document')
      assert_equal(1, @doc_args_call)
      assert_equal(3, doc_args.length)
      arg_name, arg_type, arg_default = doc_args[0]
      assert_equal('string', arg_name)
      assert_equal(:string, arg_type)
      assert_nil(arg_default)
      arg_name, arg_type, arg_default = doc_args[1]
      assert_equal('text', arg_name)
      assert_equal(:text, arg_type)
      assert_equal("Hello world.\n", arg_default)
      arg_name, arg_type, arg_default = doc_args[2]
      assert_equal('number', arg_name)
      assert_equal(:number, arg_type)
      assert_equal(0, arg_default)
    end

    def test_doc_build
      @factory.add_document(@doc_builder)
      assert_equal(1, @doc_name_call)
      dummy_doc = @factory.doc_build('test_document', [ 'string', 'text', 0 ])
      assert_equal(1, @doc_new_call)
      assert_equal('string', @doc_new_string_value)
      assert_equal('text', @doc_new_text_value)
      assert_equal(0, @doc_new_number_value)
      assert_equal(:test_document, dummy_doc)
    end

    def test_has_filter?
      assert(! (@factory.has_filter? 'test_filter'))
      @factory.add_filter(@filter_builder)
      assert((@factory.has_filter? 'test_filter'))
    end

    def test_filter_names
      @factory.add_filter(@filter_builder)
      assert_equal(1, @filter_name_call)
      filter_names = @factory.filter_names
      assert_equal(1, filter_names.length)
      assert_equal('test_filter', filter_names[0])
    end

    def test_filter_args
      @factory.add_filter(@filter_builder)
      assert_equal(1, @filter_name_call)
      filter_args = @factory.filter_args('test_filter')
      assert_equal(1, @filter_args_call)
      assert_equal(3, filter_args.length)
      arg_name, arg_type, arg_default = filter_args[0]
      assert_equal('string', arg_name)
      assert_equal(:string, arg_type)
      assert_nil(arg_default)
      arg_name, arg_type, arg_default = filter_args[1]
      assert_equal('text', arg_name)
      assert_equal(:text, arg_type)
      assert_equal("Hello world.\n", arg_default)
      arg_name, arg_type, arg_default = filter_args[2]
      assert_equal('number', arg_name)
      assert_equal(:number, arg_type)
      assert_equal(0, arg_default)
    end

    def test_filter_build
      @factory.add_filter(@filter_builder)
      assert_equal(1, @filter_name_call)
      dummy_filter = @factory.filter_build('test_filter', [ 'string', 'text', 0 ])
      assert_equal(1, @filter_new_call)
      assert_equal('string', @filter_new_string_value)
      assert_equal('text', @filter_new_text_value)
      assert_equal(0, @filter_new_number_value)
      assert_equal(:test_filter, dummy_filter)
    end
  end

  class TestDocumentLoader < RUNIT::TestCase
    def setup
      Dir.mkdir('mod_docs_for_test')

      # target
      @loader = Rucy::DocumentLoader.new('mod_docs_for_test')
    end

    def teardown
      Dir.foreach('mod_docs_for_test') do |path|
	if (! (File.directory? path)) then
	  File.delete("mod_docs_for_test/#{path}")
	end
      end
      Dir.delete('mod_docs_for_test')
    end

    def test_load_docs
      File.open('mod_docs_for_test/foo.rb', 'w') {|output|
	output.print "include Rucy\n"
	output.print "class FooDocument < Document\n"
	output.print "  def self.doc_name\n"
	output.print "    'Foo'\n"
	output.print "  end\n"
	output.print "  def self.doc_args\n"
	output.print "    []\n"
	output.print "  end\n"
	output.print "end\n"
      }
      File.open('mod_docs_for_test/bar.rb', 'w') {|output|
	output.print "include Rucy\n"
	output.print "class BarFilter < Filter\n"
	output.print "  def self.filter_name\n"
	output.print "    'Bar'\n"
	output.print "  end\n"
	output.print "  def self.filter_args\n"
	output.print "    []\n"
	output.print "  end\n"
	output.print "end\n"
      }
      @loader.load
      assert_instance_of(Class, @loader.doc_get('FooDocument'))
      assert_equal(@loader.doc_get('FooDocument'), @loader.FooDocument)
      assert_instance_of(Class, @loader.filter_get('BarFilter'))
      assert_equal(@loader.filter_get('BarFilter'), @loader.BarFilter)
      factory = @loader.factory
      factory.setup
      assert_equal([ 'Foo' ], factory.doc_names)
      assert_equal([], factory.doc_args('Foo'))
      assert_instance_of(@loader.FooDocument, factory.doc_build('Foo', []))
      assert_equal([ 'Bar' ], factory.filter_names)
      assert_equal([], factory.filter_args('Bar'))
      assert_instance_of(@loader.BarFilter, factory.filter_build('Bar', []))
    end

    def test_load_factories
      File.open('mod_docs_for_test/foo.rb', 'w') {|output|
	output.print "include Rucy\n"
	output.print "class FooDocument < Document\n"
	output.print "end\n"
	output.print "class FooDocumentBuilder < DocumentBuilder\n"
	output.print "  def doc_name\n"
	output.print "    'Foo'\n"
	output.print "  end\n"
	output.print "  def doc_args\n"
	output.print "    []\n"
	output.print "  end\n"
	output.print "  def new\n"
	output.print "    FooDocument.new\n"
	output.print "  end\n"
	output.print "end\n"
      }
      File.open('mod_docs_for_test/bar.rb', 'w') {|output|
	output.print "include Rucy\n"
	output.print "class BarFilter < Filter\n"
	output.print "end\n"
	output.print "class BarFilterBuilder < FilterBuilder\n"
	output.print "  def filter_name\n"
	output.print "    'Bar'\n"
	output.print "  end\n"
	output.print "  def filter_args\n"
	output.print "    []\n"
	output.print "  end\n"
	output.print "  def new\n"
	output.print "    BarFilter.new\n"
	output.print "  end\n"
	output.print "end\n"
      }
      @loader.load
      assert(@loader.load_errors.empty?)
      assert_instance_of(Class, @loader.doc_get('FooDocument'))
      assert_equal(@loader.doc_get('FooDocument'), @loader.FooDocument)
      assert_instance_of(Class, @loader.filter_get('BarFilter'))
      assert_equal(@loader.filter_get('BarFilter'), @loader.BarFilter)
      factory = @loader.factory
      factory.setup
      assert_equal([ 'Foo' ], factory.doc_names)
      assert_equal([], factory.doc_args('Foo'))
      assert_instance_of(@loader.FooDocument, factory.doc_build('Foo', []))
      assert_equal([ 'Bar' ], factory.filter_names)
      assert_equal([], factory.filter_args('Bar'))
      assert_instance_of(@loader.BarFilter, factory.filter_build('Bar', []))
    end

    def test_load2
      File.open('mod_docs_for_test/foo.rb', 'w') {|output|
	output.print "include Rucy\n"
	output.print "class FooDocument < Document\n"
	output.print "  def self.doc_name\n"
	output.print "    'Foo'\n"
	output.print "  end\n"
	output.print "  def self.doc_args\n"
	output.print "    []\n"
	output.print "  end\n"
	output.print "end\n"
      }
      File.open('mod_docs_for_test/bar.rb', 'w') {|output|
	output.print "include Rucy\n"
	output.print "class BarFilter < Filter\n"
	output.print "  def self.filter_name\n"
	output.print "    'Bar'\n"
	output.print "  end\n"
	output.print "  def self.filter_args\n"
	output.print "    []\n"
	output.print "  end\n"
	output.print "end\n"
      }
      @loader.load
      assert(@loader.load_errors.empty?)
      fooDocument = @loader.FooDocument
      barFilter = @loader.BarFilter
      factory = @loader.factory
      factory.setup
      foo = factory.doc_build('Foo', [])
      bar = factory.filter_build('Bar', [])
      assert_instance_of(fooDocument, foo)
      assert_instance_of(barFilter, bar)
      @loader.load
      assert(@loader.FooDocument != fooDocument)
      assert(@loader.BarFilter != barFilter)
      factory2 = @loader.factory
      assert(factory2 != factory)
      foo2 = factory2.doc_build('Foo', [])
      bar2 = factory2.filter_build('Bar', [])
      assert(! (foo2.instance_of? fooDocument))
      assert(! (bar2.instance_of? barFilter))
    end

    def test_load_error
      File.open('mod_docs_for_test/foo.rb', 'w') {|output|
	output.print "raise 'foo'\n"
      }
      File.open('mod_docs_for_test/bar.rb', 'w') {|output|
	output.print "require 'no_load'\n"
      }
      @loader.load
      assert_equal([ 'mod_docs_for_test/foo.rb',
		     'mod_docs_for_test/bar.rb'
		   ].sort,
		   @loader.load_errors.map{|k,v| k }.sort)
      assert_instance_of(RuntimeError,
			 @loader.load_errors.assoc('mod_docs_for_test/foo.rb')[1])
      assert_instance_of(LoadError,
			 @loader.load_errors.assoc('mod_docs_for_test/bar.rb')[1])
    end
  end

  class TestPage < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse

    def setup
      super
      @page = Rucy::Page.new("<html>Hello world.</html>\n")
    end

    def test_GET
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @page.publish('', @request, @response, @logger)
      assert_nil(@messg_head.doc_path)
      assert_nil(@messg_head.local_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal('26', @messg_head.header('Content-Length'))
      assert_equal("<html>Hello world.</html>\n", @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::Page/)
    end

    def test_HEAD
      @request.method = 'HEAD'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @page.publish('', @request, @response, @logger)
      assert_nil(@messg_head.doc_path)
      assert_nil(@messg_head.local_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/html', @messg_head.header('Content-Type'))
      assert_equal('26', @messg_head.header('Content-Length'))
      assert_equal('', @messg_body)
      assert(@log_emerg.empty?)
      assert(@log_alert.empty?)
      assert(@log_crit.empty?)
      assert(@log_err.empty?)
      assert(@log_warning.empty?)
      assert(@log_notice.empty?)
      assert(@log_info.empty?)
      assert_match(@log_debug, /enter document: Rucy::Page/)
    end
  end

  class TestSubsetDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include PseudoDocument

    def setup
      super
      @doc = Rucy::SubsetDocument.new(@document, '/foo')
    end

    def test_each
      doc_list = [ @doc, @document ]
      for doc in @doc
	assert(doc_list.delete(doc))
      end
      assert(doc_list.empty?)
    end

    def test_publish_foo
      @request.method = 'GET'
      @request.path = '/foo'
      @request.version = 'HTTP/1.1'
      @doc.publish('/foo', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_publish_foo_bar
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @doc.publish('/foo', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/foo/bar', @publish_request_path)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_publish_bar_foo
      @request.method = 'GET'
      @request.path = '/bar/foo'
      @request.version = 'HTTP/1.1'
      @doc.publish('/bar/foo', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/bar', @publish_script_name)
      assert_equal('/bar/foo', @publish_request_path)
      assert_equal("pseudo publish.\n", @log_debug)
    end


    def test_publish_bar_foo_baz
      @request.method = 'GET'
      @request.path = '/bar/foo/baz'
      @request.version = 'HTTP/1.1'
      @doc.publish('/bar/foo', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/bar', @publish_script_name)
      assert_equal('/bar/foo/baz', @publish_request_path)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_publish_bar_foo2
      @request.method = 'GET'
      @request.path = '/bar/foo'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@doc.publish('/bar', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status)
	assert_equal(0, @publish_call)
	assert(@log_debug.empty?)
	done = true
      end
      assert(done)
    end
  end

  class TestFolderDocument < RUNIT::TestCase
    include PseudoLogger
    include PseudoRequestResponse
    include PseudoDocument
    include PseudoFilter

    def setup
      super
      @folder = Rucy::FolderDocument.new
    end

    def test_umount
      @folder.mount(@document, '/foo')
      assert_equal(@document, @folder.find('/foo')[0])
      assert_equal('/foo', @folder.find('/foo')[1])
      assert_equal(@document, @folder.find('/foo/bar')[0])
      assert_equal('/foo', @folder.find('/foo/bar')[1])
      assert_equal(@document, @folder.umount('/foo'))
      assert_nil(@folder.find('/foo'))

      @folder.mount(@document, '/foo', /\.cgi/)
      assert_nil(@folder.find('/foo/bar'))
      assert_equal(@document, @folder.find('/foo/bar.cgi')[0])
      assert_equal('/foo', @folder.find('/foo/bar.cgi')[1])
      assert_equal(@document, @folder.umount('/foo', /\.cgi/))
      assert_nil(@folder.find('/foo/bar.cgi'))
    end

    def test_each
      folder2 = Rucy::FolderDocument.new
      @folder.mount(folder2, '/foo')
      folder2.mount(@folder, '/bar')
      @folder.mount(@document, '/')
      doc_list = [ @folder, folder2, @document ]
      for doc in @folder
	assert(doc_list.delete(doc))
      end
      assert(doc_list.empty?)
    end

    def test_GET_root
      @folder.mount(@document, '/')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_root_foo
      @folder.mount(@document, '/')
      @request.method = 'GET'
      @request.path = '/foo'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_foo
      @folder.mount(@document, '/foo')
      @request.method = 'GET'
      @request.path = '/foo'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_foo_bar
      @folder.mount(@document, '/foo')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_foo_not_found
      @folder.mount(@document, '/foo')
      @request.method = 'GET'
      @request.path = '/bar'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@folder.publish('', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	assert(0, @publish_call)
	assert_match(@log_debug, /not mounted/)
	done = true
      end
      assert(done)
    end

    def tset_GET_mask
      @folder.mount(@document, '/', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo.cgi'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/foo.cgi', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_mask_not_found
      @folder.mount(@document, '/', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo.html'
      @request.version = 'HTTP/1.1'
      done = false
      begin
	@folder.publish('', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	assert(0, @publish_call)
	assert_match(@log_debug, /not mounted/)
	done = true
      end
      assert(done)
    end

    def test_GET_filter
      @folder.mount(@document, '/foo')
      @folder.attach(@filter, '/foo')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar', @publish_request_path)
      assert_equal(1, @filter_open_call)
      assert_equal(1, @filter_head_call)
      assert_equal(1, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('/foo', @filter_script_name)
      assert_equal('/foo/bar', @filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      log = "pseudo filter_open.\n"
      log += "pseudo publish.\n"
      log += "pseudo filter_head.\n"
      log += "pseudo filter_body.\n"
      log += "pseudo filter_close.\n"
      log_pat = Regexp.compile(Regexp.quote(log) + "\\z")
      assert_match(@log_debug, log_pat)
    end

    def test_GET_filter2
      @folder.mount(@document, '/foo')
      @folder.attach(@filter, '/foo/bar')
      @request.method = 'GET'
      @request.path = '/foo/bar/baz'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar/baz', @publish_request_path)
      assert_equal(1, @filter_open_call)
      assert_equal(1, @filter_head_call)
      assert_equal(1, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('/foo/bar', @filter_script_name)
      assert_equal('/foo/bar/baz', @filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      log = "pseudo filter_open.\n"
      log += "pseudo publish.\n"
      log += "pseudo filter_head.\n"
      log += "pseudo filter_body.\n"
      log += "pseudo filter_close.\n"
      log_pat = Regexp.compile(Regexp.quote(log) + "\\z")
      assert_match(@log_debug, log_pat)
    end

    def test_GET_filter_mask
      @folder.mount(@document, '/foo')
      @folder.attach(@filter, '/foo', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo/bar.cgi'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar.cgi', @publish_request_path)
      assert_equal(1, @filter_open_call)
      assert_equal(1, @filter_head_call)
      assert_equal(1, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('/foo', @filter_script_name)
      assert_equal('/foo/bar.cgi', @filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      log = "pseudo filter_open.\n"
      log += "pseudo publish.\n"
      log += "pseudo filter_head.\n"
      log += "pseudo filter_body.\n"
      log += "pseudo filter_close.\n"
      log_pat = Regexp.compile(Regexp.quote(log) + "\\z")
      assert_match(@log_debug, log_pat)
    end

    def test_GET_filter_mask2
      @folder.mount(@document, '/foo')
      @folder.attach(@filter, '/foo', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar', @publish_request_path)
      assert_equal(0, @filter_open_call)
      assert_equal(0, @filter_head_call)
      assert_equal(0, @filter_body_call)
      assert_equal(0, @filter_close_call)
      assert_nil(@filter_script_name)
      assert_nil(@filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_virtual
      @folder.virtual_mount('foo', @document, '/')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'foo:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_virtual_not_found
      @folder.virtual_mount('foo', @document, '/')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'bar')
      begin
	@folder.publish('', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	assert(0, @publish_call)
	assert_match(@log_debug, /not mounted/)
	done = true
      end
      assert(done)
    end

    def test_GET_virtual_mask
      @folder.virtual_mount('foo', @document, '/', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'foo:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/foo.cgi', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_GET_virtual_mask_not_found
      @folder.virtual_mount('foo', @document, '/', /\.cgi/)
      @request.method = 'GET'
      @request.path = '/foo.cgi'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'bar:80')
      begin
	@folder.publish('', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	assert(0, @publish_call)
	assert_match(@log_debug, /not mounted/)
	done = true
      end
      assert(done)
    end

    def test_GET_virtual_filter
      @folder.virtual_mount('foo', @document, '/foo')
      @folder.virtual_attach('foo', @filter, '/foo')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'foo:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo/bar', @publish_request_path)
      assert_equal(1, @filter_open_call)
      assert_equal(1, @filter_head_call)
      assert_equal(1, @filter_body_call)
      assert_equal(1, @filter_close_call)
      assert_equal('/foo', @filter_script_name)
      assert_equal('/foo/bar', @filter_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      log = "pseudo filter_open.\n"
      log += "pseudo publish.\n"
      log += "pseudo filter_head.\n"
      log += "pseudo filter_body.\n"
      log += "pseudo filter_close.\n"
      log_pat = Regexp.compile(Regexp.quote(log) + "\\z")
      assert_match(@log_debug, log_pat)
    end

    def test_GET_virtual_filter_not_found
      @folder.virtual_mount('foo', @document, '/foo')
      @folder.virtual_attach('foo', @document, '/foo')
      @request.method = 'GET'
      @request.path = '/foo/bar'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'bar:80')
      begin
	@folder.publish('', @request, @response, @logger)
      rescue Rucy::HTTPError
	assert_equal(404, $!.status) # Not Found
	assert(0, @publish_call)
	assert(0, @filter_open_call)
	assert(0, @filter_head_call)
	assert(0, @filter_body_call)
	assert(0, @filter_close_call)
	assert_match(@log_debug, /not mounted/)
	done = true
      end
      assert(done)
    end

    def test_alias_root
      @folder.set_alias('/foo', '/')	# /foo -> /
      @folder.mount(@document, '/')
      @request.method = 'GET'
      @request.path = '/foo'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_alias_root2
      @folder.set_alias('/', '/foo')	# / -> /foo
      @folder.mount(@document, '/foo')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_alias_foo
      @folder.set_alias('/bar', '/foo')	# /bar -> /foo
      @folder.mount(@document, '/foo')
      @request.method = 'GET'
      @request.path = '/bar'
      @request.version = 'HTTP/1.1'
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_virtual_alias_root
      @folder.set_virtual_alias('baz', '/foo', '/') # /foo -> /
      @folder.virtual_mount('baz', @document, '/')
      @request.method = 'GET'
      @request.path = '/foo'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'baz:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('', @publish_script_name)
      assert_equal('/', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_virtual_alias_root2
      @folder.set_virtual_alias('baz', '/', '/foo') # / -> /foo
      @folder.virtual_mount('baz', @document, '/foo')
      @request.method = 'GET'
      @request.path = '/'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'baz:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end

    def test_virtual_alias_foo
      @folder.set_virtual_alias('baz', '/bar', '/foo') # /bar -> /foo
      @folder.virtual_mount('baz', @document, '/foo')
      @request.method = 'GET'
      @request.path = '/bar'
      @request.version = 'HTTP/1.1'
      @request.set_header('Host', 'baz:80')
      @folder.publish('', @request, @response, @logger)
      assert_equal(1, @publish_call)
      assert_equal('/foo', @publish_script_name)
      assert_equal('/foo', @publish_request_path)
      assert_equal('HTTP/1.1', @messg_head.version)
      assert_equal(200, @messg_head.status)
      assert_equal('OK', @messg_head.reason)
      assert_equal('text/plain', @messg_head.header('Content-Type'))
      assert_equal('foo', @messg_body)
      assert_equal("pseudo publish.\n", @log_debug)
    end
  end
end
