#!/usr/local/bin/ruby
# $Id: test_Error.rb,v 1.3 2004/10/13 15:21:30 toki Exp $

require 'rubyunit'
require 'wpm'
require 'rucy/document'

module TestRucy
  class TestError < RUNIT::TestCase
    def setup
      # for Rucy::Control class
      @load_errors_call = 0
      @doc_errors_call = 0
      @filter_errors_call = 0
      @alias_errors_call = 0
      @logging_errors_call = 0

      @factory = Rucy::DocumentFactory.new
      page = Object.new
      def page.doc_name
	'Page'
      end
      def page.doc_args
	[ [ 'content', :text, nil ],
	  [ 'content-type', :string, 'text/html' ]
	]
      end
      @factory.add_document(page)

      filter = Object.new
      def filter.filter_name
	'Filter'
      end
      def filter.filter_args
	[ [ 'first', :string, nil ],
	  [ 'second', :string, nil ]
	]
      end
      @factory.add_filter(filter)

      # setup WPM::PageContext class
      loader = WPM::Loader.new('../control/Error/Error.rb')
      @Error = loader.const_get('Error').new(self, { :control => self, :factory => @factory })
      assert((@Error.is_a? WPM::PageContext))
      @Error.init_context
    end

    def load_errors
      @load_errors_call += 1
      [ [ 'foo.rb', RuntimeError.new('HALO') ] ]
    end

    def doc_errors
      @doc_errors_call += 1
      [ { :document => 'Page',
	  :arguments => [
	    "Hello world.\n",
	    'text/plain'
	  ],
	  :mount_path => '/error',
	  :mount_mask => nil,
	  :virtual_host => 'www2',
	  :exception => RuntimeError.new('test')
	}
      ]
    end

    def filter_errors
      @filter_errors_call += 1
      [ { :filter => 'Filter',
	  :arguments => [
	    'foo',
	    'bar'
	  ],
	  :attach_path => '/error',
	  :attach_mask => nil,
	  :virtual_host => 'www2',
	  :exception => RuntimeError.new('test')
	}
      ]
    end

    def alias_errors
      @alias_errors_call += 1
      [ { :alias_path => 'bar',
	  :orig_path => 'foo',
	  :virtual_host => 'www2',
	  :exception => RuntimeError.new('test')
	}
      ]
    end

    def logging_errors
      @logging_errors_call += 1
      [ { :logfile => 'server.log',
	  :exception => RuntimeError.new('server logging error')
	},
	{ :logfile => 'access.log',
	  :exception => RuntimeError.new('access logging error')
	}
      ]
    end

    def test_has_load_errors?
      assert(@Error.has_load_errors?)
      assert_equal(1, @load_errors_call)
    end

    def test_load_error_list
      load_error_list = @Error.load_error_list
      assert_equal(1, @load_errors_call)
      assert_equal(1, load_error_list.length)
      assert_equal('foo.rb', load_error_list[0][0])
      assert_instance_of(RuntimeError, load_error_list[0][1])
      assert_equal('HALO', load_error_list[0][1].message)
    end

    def test_load_error
      @Error.load_error = [ 'foo.rb', RuntimeError.new('HALO') ]
      assert_equal('foo.rb', @Error.lib_name)
      assert_equal('RuntimeError', @Error.load_error_type)
      assert_equal('HALO', @Error.load_error_msg)
    end

    def test_load_error_index
      @Error.load_error_index = 1
      assert_equal(1, @Error.load_error_index)
    end

    def test_has_document_errors?
      assert(@Error.has_document_errors?)
      assert_equal(1, @doc_errors_call)
    end

    def test_document_error_list
      document_error_list = @Error.document_error_list
      assert_equal(1, @doc_errors_call)
      assert_equal(1, document_error_list.length)
      assert_equal('Page', document_error_list[0][:document])
      assert_equal(2, document_error_list[0][:arguments].length)
      assert_equal("Hello world.\n", document_error_list[0][:arguments][0])
      assert_equal('text/plain', document_error_list[0][:arguments][1])
      assert_equal('/error', document_error_list[0][:mount_path])
      assert_equal('www2', document_error_list[0][:virtual_host])
      assert_nil(document_error_list[0][:mount_mask])
    end

    def test_document_error
      @Error.document_error = {
	:document => 'Page',
	:arguments => [
	  "Hello world.\n",
	  'text/plain'
	],
	:mount_path => '/error',
	:mount_mask => nil,
	:virtual_host => 'www2',
	:exception => RuntimeError.new('test')
      }
      assert_equal('Page', @Error.doc_name)
      assert_equal([ [ 'content', "Hello world.\n" ],
		     [ 'content-type', 'text/plain' ]
		   ], @Error.doc_args)
      assert_equal('/error', @Error.mount_path)
      assert_nil(@Error.mount_mask)
      assert_equal('www2', @Error.mount_virtual_host)
      assert_equal('RuntimeError', @Error.doc_error_type)
      assert_equal('test', @Error.doc_error_msg)
    end

    def test_doc_error_index
      @Error.doc_error_index = 1
      assert_equal(1, @Error.doc_error_index)
    end

    def test_doc_defined?
      @Error.document_error = {
	:document => 'Page',
	:arguments => [
	  "Hello world.\n",
	  'text/plain'
	],
	:mount_path => '/error',
	:mount_mask => nil,
	:exception => RuntimeError.new('test')
      }
      assert(@Error.doc_defined?)
      @Error.document_error = {
	:document => 'Foo',
	:arguments => [],
	:mount_path => '/',
	:mount_mask => nil,
	:exception => RuntimeError.new('nothing')
      }
      assert(! @Error.doc_defined?)
    end

    def test_has_doc_args?
      @Error.document_error = {
	:document => 'Page',
	:arguments => [
	  "Hello world.\n",
	  'text/plain'
	],
	:mount_path => '/error',
	:mount_mask => nil,
	:exception => RuntimeError.new('test')
      }
      assert(@Error.has_doc_args?)
      @Error.document_error = {
	:document => 'Empty',
	:arguments => [],
	:mount_path => '/empty',
	:mount_mask => nil,
	:exception => RuntimeError.new('test')
      }
      assert(! @Error.has_doc_args?)
    end

    def test_doc_argument
      @Error.doc_argument = [ 'content', "Hello world.\n" ]
      assert_equal('content', @Error.doc_arg_name)
      assert_equal("Hello world.\n", @Error.doc_arg_value)
    end

    def test_doc_arg_index
      @Error.doc_arg_index = 1
      assert_equal(1, @Error.doc_arg_index)
    end

    def test_has_filter_errors?
      assert(@Error.has_filter_errors?)
      assert_equal(1, @filter_errors_call)
    end

    def test_filter_error_list
      filter_error_list = @Error.filter_error_list
      assert_equal(1, @filter_errors_call)
      assert_equal(1, filter_error_list.length)
      assert_equal('Filter', filter_error_list[0][:filter])
      assert_equal(2, filter_error_list[0][:arguments].length)
      assert_equal('foo', filter_error_list[0][:arguments][0])
      assert_equal('bar', filter_error_list[0][:arguments][1])
      assert_equal('/error', filter_error_list[0][:attach_path])
      assert_equal('www2', filter_error_list[0][:virtual_host])
      assert_nil(filter_error_list[0][:attach_mask])
    end

    def test_filter_error
      @Error.filter_error = {
	:filter => 'Filter',
	:arguments => [
	  'foo',
	  'bar'
	],
	:attach_path => '/error',
	:attach_mask => nil,
	:virtual_host => 'www2',
	:exception => RuntimeError.new('test')
      }
      assert_equal('Filter', @Error.filter_name)
      assert_equal([ [ 'first', 'foo' ],
		     [ 'second', 'bar' ]
		   ], @Error.filter_args)
      assert_equal('/error', @Error.attach_path)
      assert_nil(@Error.attach_mask)
      assert_equal('www2', @Error.attach_virtual_host)
      assert_equal('RuntimeError', @Error.filter_error_type)
      assert_equal('test', @Error.filter_error_msg)
    end

    def test_filter_error_index
      @Error.filter_error_index = 1
      assert_equal(1, @Error.filter_error_index)
    end

    def test_filter_defined?
      @Error.filter_error = {
	:filter => 'Filter',
	:arguments => [
	  'foo',
	  'bar'
	],
	:attach_path => '/error',
	:attach_mask => nil,
	:exception => RuntimeError.new('test')
      }
      assert(@Error.filter_defined?)
      @Error.filter_error = {
	:filter => 'Foo',
	:arguments => [],
	:attach_path => '/',
	:attach_mask => nil,
	:exception => RuntimeError.new('nothing')
      }
      assert(! @Error.filter_defined?)
    end

    def test_has_filter_args?
      @Error.filter_error = {
	:filter => 'Filter',
	:arguments => [
	  'foo',
	  'bar'
	],
	:attach_path => '/error',
	:attach_mask => nil,
	:exception => RuntimeError.new('test')
      }
      assert(@Error.has_filter_args?)
      @Error.filter_error = {
	:filter => 'Empty',
	:arguments => [],
	:attach_path => '/empty',
	:attach_mask => nil,
	:exception => RuntimeError.new('test')
      }
      assert(! @Error.has_filter_args?)
    end

    def test_filter_argument
      @Error.filter_argument = [ 'first', 'foo' ]
      assert_equal('first', @Error.filter_arg_name)
      assert_equal('foo', @Error.filter_arg_value)
    end

    def test_filter_arg_index
      @Error.filter_arg_index = 1
      assert_equal(1, @Error.filter_arg_index)
    end

    def test_has_logging_errors?
      assert(@Error.has_logging_errors?)
      assert_equal(1, @logging_errors_call)
    end

    def test_logging_error_list
      logging_error_list = @Error.logging_error_list
      assert_equal(1, @logging_errors_call)
      assert_equal(2, logging_error_list.length)
      assert_equal('server.log', logging_error_list[0][:logfile])
      assert_instance_of(RuntimeError, logging_error_list[0][:exception])
      assert_equal('access logging error', logging_error_list[1][:exception].message)
      assert_equal('access.log', logging_error_list[1][:logfile])
      assert_instance_of(RuntimeError, logging_error_list[1][:exception])
      assert_equal('access logging error', logging_error_list[1][:exception].message)
    end

    def test_has_alias_errors?
      assert(@Error.has_alias_errors?)
      assert_equal(1, @alias_errors_call)
    end

    def test_alias_error_list
      alias_error_list = @Error.alias_error_list
      assert_equal(1, @alias_errors_call)
      assert_equal(1, alias_error_list.length)
      assert_equal('bar', alias_error_list[0][:alias_path])
      assert_equal('foo', alias_error_list[0][:orig_path])
      assert_equal('www2', alias_error_list[0][:virtual_host])
      assert_instance_of(RuntimeError, alias_error_list[0][:exception])
    end

    def test_alias_error
      @Error.alias_error = {
	:alias_path => 'bar',
	:orig_path => 'foo',
	:virtual_host => 'www2',
	:exception => RuntimeError.new('test')
      }
      assert_equal('bar', @Error.alias_path)
      assert_equal('foo', @Error.alias_orig_path)
      assert_equal('www2', @Error.alias_virtual_host)
      assert_equal('RuntimeError', @Error.alias_error_type)
      assert_equal('test', @Error.alias_error_msg)
    end

    def test_logging_error
      @Error.logging_error = {
	:logfile => 'server.log',
	:exception => RuntimeError.new('server logging error')
      }
      assert_equal('server.log', @Error.log_file)
      assert_equal('RuntimeError', @Error.log_error_type)
      assert_equal('server logging error', @Error.log_error_msg)
    end

    def test_log_error_index
      @Error.log_error_index = 1
      assert_equal(1, @Error.log_error_index)
    end
  end
end
