#!/usr/local/bin/ruby
# $Id: test_DocumentTable.rb,v 1.16 2004/11/18 02:23:46 toki Exp $

require 'rubyunit'
require 'forwarder'
require 'wpm'
require 'rucy/document'

module TestRucy
  class TestDocumentTable < RUNIT::TestCase
    class DummyMountParams
      def name
	'Bar'
      end

      def args2
	[]
      end

      def path
	'/bar'
      end

      def mask
	nil
      end

      def virtual_host
	nil
      end

      def comment
	nil
      end
    end

    def setup
      # for Rucy::Page class
      @driver_call = 0

      @page = Forwarder.new(self)
      class << @page
	def_delegator :__getobj__, :driver
      end

      # for WPM::Driver class
      @redirect_call = 0
      @redirect_page_name = nil
      @redirect_query_params = nil

      @driver = Forwarder.new(self)
      class << @driver
	def_delegator :__getobj__, :redirect
      end

      # for Rucy::DocumentFactory class
      @filter_names_call = 0

      @factory = Forwarder.new(self)
      class << @factory
	def_delegator :__getobj__, :filter_names
      end

      # for Rucy::DocumentList class
      @length_call = 0
      @each_call = 0
      @swap_call = 0
      @delete_at_call = 0
      @delete_at_pos = nil

      @document_list = Forwarder.new(self)
      class << @document_list
	include Enumerable
	def_delegator :__getobj__, :length
	def_delegator :__getobj__, :each
	def_delegator :__getobj__, :swap
	def_delegator :__getobj__, :delete_at
      end

      # for Rucy::DocumentList::MountParams class
      @name_call = 0
      @args2_call = 0
      @path_call = 0
      @mask_call = 0
      @mask = /mount_mask/
      @virtual_host_call = 0
      @comment_call = 0

      @mount_params = Forwarder.new(self)
      class << @mount_params
	def_delegator :__getobj__, :_name, :name
	def_delegator :__getobj__, :args2
	def_delegator :__getobj__, :path
	def_delegator :__getobj__, :mask
	def_delegator :__getobj__, :virtual_host
	def_delegator :__getobj__, :comment
      end

      @mount_params2 = DummyMountParams.new

      # for MountMap class
      @modified_count_call = 0
      @set_modified_count_call = 0
      @set_modified_count_value = nil
      @list_call = 0
      @type_list_call = 0
      @setup_call = 0
      @setup_name = nil
      @setup_pos = nil
      @write_call = 0
      @add_filter_pos_call = 0
      @name_label_call = 0
      @path_label_call = 0
      @mask_label_call = 0

      @MountMap = Forwarder.new(self)
      class << @MountMap
	def_delegator :__getobj__, :modified_count
	def_delegator :__getobj__, :modified_count=
	def_delegator :__getobj__, :list
	def_delegator :__getobj__, :type_list
	def_delegator :__getobj__, :_setup, :setup
	def_delegator :__getobj__, :write
	def_delegator :__getobj__, :add_filter_pos
	def_delegator :__getobj__, :name_label
	def_delegator :__getobj__, :path_label
	def_delegator :__getobj__, :mask_label
      end

      # setup WPM::PageContext class
      loader = WPM::Loader.new('../control/DocumentTable/DocumentTable.rb')
      options = {
	:factory => @factory
      }
      @DocumentTable = loader.const_get('DocumentTable').new(@page, options)
      assert((@DocumentTable.is_a? WPM::PageContext))
      @DocumentTable.init_context
      @DocumentTable.src = @MountMap
    end

    # for Rucy::Page class

    def driver
      @driver_call += 1
      @driver
    end

    # for WPM::Driver class

    def redirect(page_name, query_params=nil)
      @redirect_call += 1
      @redirect_page_name = page_name
      @redirect_query_params = query_params
      nil
    end

    # for Rucy::DocumentFactory class

    def filter_names
      @filter_names_call += 1
      [ 'Apple', 'Banana' ]
    end

    # for Rucy::DocumentList class

    def length
      @length_call += 1
      2
    end

    def each
      @each_call += 1
      yield(@mount_params)
      yield(@mount_params2)
      nil
    end

    def swap(pos1, pos2)
      @swap_call += 1
      @swap_pos1 = pos1
      @swap_pos2 = pos2
      nil
    end

    def delete_at(pos)
      @delete_at_call += 1
      @delete_at_pos = pos
      nil
    end

    # for Rucy::DocumentList::MountParams class

    def _name
      @name_call += 1
      'Foo'
    end

    def args2
      @args2_call += 1
      [ { :name => 'first argument',
	  :type => :string,
	  :default => 'alice',
	  :value => 'apple'
	},
	{ :name => 'second argument',
	  :type => :regexp,
	  :default => /bob/,
	  :value => /banana/
	}
      ]
    end

    def path
      @path_call += 1
      '/mount_path'
    end

    def mask
      @mask_call += 1
      @mask
    end

    def virtual_host
      @virtual_host_call += 1
      'www.foo.org'
    end

    def comment
      @comment_call += 1
      'This is a pen.'
    end

    # for MountMap class

    def modified_count
      @modified_count_call += 1
      0
    end

    def modified_count=(new_count)
      @set_modified_count_call += 1
      @set_modified_count_value = new_count
    end

    def list
      @list_call += 1
      @document_list
    end

    def type_list
      @type_list_call += 1
      [ 'Foo', 'Bar' ]
    end

    def _setup(name, pos)
      @setup_call += 1
      @setup_name = name
      @setup_pos = pos
      nil
    end

    def write
      @write_call += 1
      nil
    end

    def add_filter_pos
      @add_filter_pos_call += 1
      3
    end

    def name_label
      @name_label_call += 1
      'document'
    end

    def path_label
      @path_label_call += 1
      'mount path'
    end

    def mask_label
      @mask_label_call += 1
      'mount mask'
    end

    # test

    def test_modified_count
      assert_equal(0, @DocumentTable.modified_count)
      assert_equal(1, @modified_count_call)
      @DocumentTable.modified_count = 7
      assert_equal(1, @set_modified_count_call)
      assert_equal(7, @set_modified_count_value)
    end

    def test_has_list
      prev_length_call = @length_call
      assert_equal(true, @DocumentTable.has_list?)
      assert_equal(prev_length_call + 1, @length_call)
    end

    def test_list
      assert_equal([ { :params => @mount_params,
		       :show_args => false,
		       :edit_type_selected => 'Foo',
		       :filter_selected => 'Apple',
		       :pos => 0
		     },
		     { :params => @mount_params2,
		       :show_args => false,
		       :edit_type_selected => 'Bar',
		       :filter_selected => 'Apple',
		       :pos => 1
		     }
		   ], @DocumentTable.list)
      assert_equal(1, @each_call)
    end

    def test_index
      @DocumentTable.index = 1
      assert_equal(1, @DocumentTable.index)
      @DocumentTable.arg_index = 2
      assert_equal(2, @DocumentTable.arg_index)
    end

    def test_entry
      @DocumentTable.entry = { :params => @mount_params }
      prev_name_call = @name_call
      assert_equal('Foo', @DocumentTable.name)
      assert_equal(prev_name_call + 1, @name_call)
      assert_equal(true, @DocumentTable.has_args?)
      assert_equal(1, @args2_call)
      assert_equal([ { :name => 'first argument',
		       :type => :string,
		       :default => 'alice',
		       :value => 'apple'
		     },
		     { :name => 'second argument',
		       :type => :regexp,
		       :default => /bob/,
		       :value => /banana/
		     }
		   ], @DocumentTable.args)
      assert_equal(2, @args2_call)
      assert_equal('/mount_path', @DocumentTable.path)
      assert_equal(1, @path_call)
      assert_equal(/mount_mask/, @DocumentTable.mask)
      assert_equal(1, @mask_call)
      assert_equal('www.foo.org', @DocumentTable.virtual_host)
      assert_equal(1, @virtual_host_call)
      assert_equal(true, @DocumentTable.has_comment?)
      assert_equal(1, @comment_call)
      assert_equal('This is a pen.', @DocumentTable.comment)
      assert_equal(2, @comment_call)
    end

    def test_show_args
      @DocumentTable.entry = { :show_args => false }
      assert_equal(false, @DocumentTable.show_args)
      assert_equal('Show arguments', @DocumentTable.show_args_label)
      @DocumentTable.entry = { :show_args => true }
      assert_equal(true, @DocumentTable.show_args)
      assert_equal('Hide arguments', @DocumentTable.show_args_label)
    end

    def test_modify_show_args
      entry = { :show_args => false }
      @DocumentTable.entry = entry
      assert_equal(false, @DocumentTable.show_args)
      @DocumentTable.show_args = true
      assert_equal(true, @DocumentTable.show_args)
      assert_equal(true, entry[:show_args])
      @DocumentTable.show_args = false
      assert_equal(false, @DocumentTable.show_args)
      assert_equal(false, entry[:show_args])
    end

    def test_toggle_show_args
      entry = { :show_args => false }
      @DocumentTable.entry = entry
      assert_equal(false, @DocumentTable.show_args)
      @DocumentTable.toggle_show_args
      assert_equal(true, @DocumentTable.show_args)
      assert_equal(true, entry[:show_args])
      @DocumentTable.toggle_show_args
      assert_equal(false, @DocumentTable.show_args)
      assert_equal(false, entry[:show_args])
    end

    def test_argument
      @DocumentTable.argument = {
	:name => 'first argument',
	:type => :string,
	:default => 'alice',
	:value => 'apple'
      }
      assert_equal('first argument', @DocumentTable.arg_name)
      assert_equal('apple', @DocumentTable.arg_value)
    end

    def test_type_list
      assert_equal([ 'Foo', 'Bar' ], @DocumentTable.type_list)
      assert_equal(2, @type_list_call)
    end

    def test_add
      assert_equal('Foo', @DocumentTable.add_type_selected)
      @DocumentTable.add_type_selected = 'Bar'
      assert_equal('Bar', @DocumentTable.add_type_selected)
      prev_length_call = @length_call
      @DocumentTable.add
      assert_equal(prev_length_call + 1, @length_call)
      assert_equal(1, @setup_call)
      assert_equal('Bar', @setup_name)
      assert_equal(2, @setup_pos)
    end

    def test_edit
      entry = { :edit_type_selected => 'Foo', :pos => 0 }
      @DocumentTable.entry = entry
      assert_equal('Foo', @DocumentTable.edit_type_selected)
      @DocumentTable.edit_type_selected = 'Bar'
      assert_equal('Bar', @DocumentTable.edit_type_selected)
      assert_equal('Bar', entry[:edit_type_selected])
      @DocumentTable.edit
      assert_equal(1, @setup_call)
      assert_equal('Bar', @setup_name)
      assert_equal(0, @setup_pos)
    end

    def test_filter_list
      prev_filter_names_call = @filter_names_call
      assert_equal([ 'Apple', 'Banana' ], @DocumentTable.filter_list)
      assert_equal(prev_filter_names_call + 1, @filter_names_call)
    end

    def test_add_filter_nil
      entry = {
	:params => @mount_params,
	:filter_selected => 'Apple',
	:pos => 0
      }
      @mask = nil
      @DocumentTable.entry = entry
      assert_equal('Apple', @DocumentTable.filter_selected)
      @DocumentTable.filter_selected = 'Banana'
      assert_equal('Banana', @DocumentTable.filter_selected)
      assert_equal('Banana', entry[:filter_selected])
      @DocumentTable.add_filter
      assert_equal(1, @add_filter_pos_call)
      assert_equal(1, @driver_call)
      assert_equal(1, @redirect_call)
      assert_equal('SetupFilter', @redirect_page_name)
      assert_equal({ 'filter' => 'Banana',
		     'pos' => '3',
		     'path' => '/mount_path',
		     'mask' => 'nil',
		     'vhost' => 'www.foo.org'
		   }, @redirect_query_params)
    end

    def test_add_filter_regexp
      entry = {
	:params => @mount_params,
	:filter_selected => 'Apple',
	:pos => 0
      }
      @mask = /mount_mask/
      @DocumentTable.entry = entry
      assert_equal('Apple', @DocumentTable.filter_selected)
      @DocumentTable.filter_selected = 'Banana'
      assert_equal('Banana', @DocumentTable.filter_selected)
      assert_equal('Banana', entry[:filter_selected])
      @DocumentTable.add_filter
      assert_equal(1, @add_filter_pos_call)
      assert_equal(1, @driver_call)
      assert_equal(1, @redirect_call)
      assert_equal('SetupFilter', @redirect_page_name)
      assert_equal({ 'filter' => 'Banana',
		     'pos' => '3',
		     'path' => '/mount_path',
		     'mask' => 'regexp:mount_mask',
		     'vhost' => 'www.foo.org'
		   }, @redirect_query_params)
    end

    def test_add_filter_string
      entry = {
	:params => @mount_params,
	:filter_selected => 'Apple',
	:pos => 0
      }
      @mask = 'mount_mask'
      @DocumentTable.entry = entry
      assert_equal('Apple', @DocumentTable.filter_selected)
      @DocumentTable.filter_selected = 'Banana'
      assert_equal('Banana', @DocumentTable.filter_selected)
      assert_equal('Banana', entry[:filter_selected])
      @DocumentTable.add_filter
      assert_equal(1, @add_filter_pos_call)
      assert_equal(1, @driver_call)
      assert_equal(1, @redirect_call)
      assert_equal('SetupFilter', @redirect_page_name)
      assert_equal({ 'filter' => 'Banana',
		     'pos' => '3',
		     'path' => '/mount_path',
		     'mask' => 'string:mount_mask',
		     'vhost' => 'www.foo.org'
		   }, @redirect_query_params)
    end

    def test_add_filter_unknown
      entry = {
	:params => @mount_params,
	:filter_selected => 'Apple',
	:pos => 0
      }
      @mask = :mount_mask
      @DocumentTable.entry = entry
      assert_exception(RuntimeError) {
	@DocumentTable.add_filter
      }
    end

    def test_labels
      assert_equal('document', @DocumentTable.name_label)
      assert_equal(1, @name_label_call)
      assert_equal('mount path', @DocumentTable.path_label)
      assert_equal(1, @path_label_call)
      assert_equal('mount mask', @DocumentTable.mask_label)
      assert_equal(1, @mask_label_call)
    end

    def test_up_disabled
      @DocumentTable.entry = { :pos => 0 }
      assert_equal(true, @DocumentTable.up_disabled?)
      @DocumentTable.entry = { :pos => 1 }
      assert_equal(false, @DocumentTable.up_disabled?)
    end

    def test_down_disabled
      @DocumentTable.entry = { :pos => 1 }
      prev_length_call = @length_call
      assert_equal(true, @DocumentTable.down_disabled?)
      assert_equal(prev_length_call + 1, @length_call)
      @DocumentTable.entry = { :pos => 0 }
      prev_length_call = @length_call
      assert_equal(false, @DocumentTable.down_disabled?)
      assert_equal(prev_length_call + 1, @length_call)
    end

    def test_up
      @DocumentTable.entry = { :pos => 0 } # out of lower limit
      @DocumentTable.up
      assert_equal(0, @swap_call)
      assert_equal(0, @write_call)
      assert_equal([ { :params => @mount_params,
		       :show_args => false,
		       :edit_type_selected => 'Foo',
		       :filter_selected => 'Apple',
		       :pos => 0
		     },
		     { :params => @mount_params2,
		       :show_args => false,
		       :edit_type_selected => 'Bar',
		       :filter_selected => 'Apple',
		       :pos => 1
		     }
		   ], @DocumentTable.list)

      @DocumentTable.entry = { :pos => 2 } # out of upper limit
      @DocumentTable.up
      assert_equal(0, @swap_call)
      assert_equal(0, @write_call)
      assert_equal([ { :params => @mount_params,
		       :show_args => false,
		       :edit_type_selected => 'Foo',
		       :filter_selected => 'Apple',
		       :pos => 0
		     },
		     { :params => @mount_params2,
		       :show_args => false,
		       :edit_type_selected => 'Bar',
		       :filter_selected => 'Apple',
		       :pos => 1
		     }
		   ], @DocumentTable.list)

      @DocumentTable.entry = { :pos => 1 }
      @DocumentTable.up
      assert_equal(1, @swap_call)
      assert_equal(1, @swap_pos1)
      assert_equal(0, @swap_pos2)
      assert_equal(1, @write_call)
      assert_equal([ { :params => @mount_params2,
		       :show_args => false,
		       :edit_type_selected => 'Bar',
		       :filter_selected => 'Apple',
		       :pos => 0
		     },
		     { :params => @mount_params,
		       :show_args => false,
		       :edit_type_selected => 'Foo',
		       :filter_selected => 'Apple',
		       :pos => 1
		     }
		   ], @DocumentTable.list)
    end

    def test_down
      @DocumentTable.entry = { :pos => -1 } # out of lower limit
      @DocumentTable.down
      assert_equal(0, @swap_call)
      assert_equal(0, @write_call)
      assert_equal([ { :params => @mount_params,
		       :show_args => false,
		       :edit_type_selected => 'Foo',
		       :filter_selected => 'Apple',
		       :pos => 0
		     },
		     { :params => @mount_params2,
		       :show_args => false,
		       :edit_type_selected => 'Bar',
		       :filter_selected => 'Apple',
		       :pos => 1
		     }
		   ], @DocumentTable.list)

      @DocumentTable.entry = { :pos => 1 } # out of upper limit
      @DocumentTable.down
      assert_equal(0, @swap_call)
      assert_equal(0, @write_call)
      assert_equal([ { :params => @mount_params,
		       :show_args => false,
		       :edit_type_selected => 'Foo',
		       :filter_selected => 'Apple',
		       :pos => 0
		     },
		     { :params => @mount_params2,
		       :show_args => false,
		       :edit_type_selected => 'Bar',
		       :filter_selected => 'Apple',
		       :pos => 1
		     }
		   ], @DocumentTable.list)

      @DocumentTable.entry = { :pos => 0 }
      @DocumentTable.down
      assert_equal(1, @swap_call)
      assert_equal(0, @swap_pos1)
      assert_equal(1, @swap_pos2)
      assert_equal(1, @write_call)
      assert_equal([ { :params => @mount_params2,
		       :show_args => false,
		       :edit_type_selected => 'Bar',
		       :filter_selected => 'Apple',
		       :pos => 0
		     },
		     { :params => @mount_params,
		       :show_args => false,
		       :edit_type_selected => 'Foo',
		       :filter_selected => 'Apple',
		       :pos => 1
		     }
		   ], @DocumentTable.list)
    end

    def test_delete
      assert_equal([ { :params => @mount_params,
		       :show_args => false,
		       :edit_type_selected => 'Foo',
		       :filter_selected => 'Apple',
		       :pos => 0
		     },
		     { :params => @mount_params2,
		       :show_args => false,
		       :edit_type_selected => 'Bar',
		       :filter_selected => 'Apple',
		       :pos => 1
		     }
		   ], @DocumentTable.list)
      @DocumentTable.entry = { :pos => 0 }
      @DocumentTable.delete
      assert_equal(1, @delete_at_call)
      assert_equal(0, @delete_at_pos)
      assert_equal(1, @write_call)
      assert_equal([ { :params => @mount_params2,
		       :show_args => false,
		       :edit_type_selected => 'Bar',
		       :filter_selected => 'Apple',
		       :pos => 0
		     }
		   ], @DocumentTable.list)
    end
  end
end
